/*
*********************************************************************************************************
*                                              EXAMPLE CODE
*
*                               (c) Copyright 2007; Micrium, Inc.; Weston, FL
*
*               All rights reserved.  Protected by international copyright laws.
*               Knowledge of the source code may NOT be used to develop a similar product.
*               Please help us continue to provide the Embedded community with the finest
*               software available.  Your honesty is greatly appreciated.
*********************************************************************************************************
*/
/**
  *
  * Ported to the Texas Instruments cc2430 as part of Applied IT Project Semester 2, 2010
  * Version: V1.00  
  * Programmer(s): some sucker
  *
*/

/*
*********************************************************************************************************
*
*                                         HARDWARE DRIVER
*
*                                  Hitachi HM55B digital compass
*
*                                             with the
*
*                                      Luminary Micro LM3S2965
*
* Filename      : hm55b.c
* Version       : V1.00
* Programmer(s) : EHS
*                 BAN
*********************************************************************************************************


/*
*********************************************************************************************************
*                                             INCLUDE FILES
*********************************************************************************************************
*/

#include "hal_mcu.h"
#include "hal_board.h"
#include "hal_hm55b.h"
#include "hal_defs.h"



/*
*********************************************************************************************************
*                                            LOCAL DEFINES
*********************************************************************************************************
*/
                                  ///1200 baud
#define  BAUD_RATE_HZ             (1200)                         /* --------- Desired Baud Rate in Hz ---------------------- */
                                                                        /* --------- GPIO Bit Definitions ------------------------- */
#define  CLK                       P0_7                             /* GPIO pin 0 is attached to the clk pin                    */
#define  EN                        P0_5                             /* GPIO pin 1 is attached to the enable pin                 */
#define  DATA                      P0_6                             /* GPIO pin 2 is attached to the DATA pin                    */

#define HIGH 1
#define LOW  0
                                                                        /* --------- HM55B Command Values ------------------------- */
#define  CMD_RESET                 (0x0000)       
#define  CMD_MEASURE               (0x0008)
#define  CMD_REPORT                (0x000C)
                                                                        /* --------- HM55B Response / Status Values --------------- */
#define  RESP_BUSY                 (0x0000)  
#define  RESP_ERR                  (0x0003)
#define  RESP_READY                (0x000C)
                                                                      
#define  NBR_RETRIES               (0x0004) 
/*
*********************************************************************************************************
*                                       LOCAL GLOBAL VARIABLES
*********************************************************************************************************
*/

static uint16 ShiftClkDlyCnts;

///local functions
void ShiftClkDly(void);
uint16 TimerValueGet(void);
uint8  ShiftIn(uint16  *var, uint8 bits);         
uint8  ShiftOut(uint16 *var, uint8 bits);

/*
*********************************************************************************************************
*                                       HM55B_Init()
*
* Description : This function intializes the I/O necessary to communicate with the HM55B
*
* Argument(s) : None.
*
* Return(s)   : None.
*
* Notes(s)    : None.
*********************************************************************************************************
*/

//76543210

/*En => Port0.5
Dat => Port0.6
Clk => port0.7
*/
void HalHm55BInit (void)
{
    uint32  pclkfrq;
   
    P0SEL|=~(0xE0);
    P0DIR|=(0xE0);
    
   // T1CTL &= ~0x10; // T1CTL.OVFIF = 0 ///need to switch of interrupts etc.
    //IEN1  &=  ~0x02; // IEN1.T1EN = 0
    T1CTL|=(0x01);
    
   // pclkfrq = 32000;
    pclkfrq = 32000000;  ///system clock in hz     
                                                                    /* Determine dlycnts for half of a clock cycle              */                                                                        
    ShiftClkDlyCnts = (uint16)(pclkfrq / (BAUD_RATE_HZ * 2));  /* Truncation may occur resulting in a faster baud rate     */    
       

}

/**
  *
******************************************************************************************************
*                                  GetMeasurement()
* This function activates the hm55b measurement process and returns the axis values.
*
* Argument(s) : result          An array that receives the final axis measurements.
*
* Return(s)   : Error code      SHIFT_ERR_NONE  - No error
*                               SHIFT_ERR_ARG   - Function argument error
  ****************************************************************************************************** 
  */
extern uint8 GetMeasurement(uint16* result){
   uint16  data;
   uint16  resp; 
   uint8  i;
   
   data = CMD_RESET;        
   EN = HIGH;                                                          /* Disable the device                                       */
   EN = LOW;                                                           /* Enable the device                                        */
   ShiftOut(&data, 4);                                                 /* Shift out 4 bits to reset the device                     */
   EN = HIGH;                                                          /* Disable the device                                       */
                                                                           /* ---------- Start of Measurement ------------------------ */    
   data = CMD_MEASURE;                                            /* Measurement command = %1000                              */
   EN = LOW;                                                           /* Enable the device, leave enabled until the Report stage  */
   ShiftOut(&data, 4);                                                 /* Shift out the measurement command                        */    
                                                                        /* ---------- Check Measurement Status (Report) ----------- */        
   data = CMD_REPORT;                                             /* Report / Status command = %1100                          */
   resp = RESP_BUSY;                                              /* Assume device is initially busy                          */
   for (i = 0; i < NBR_RETRIES; i++) {                                 /* Poll until the measurement complete or retries expired   */
       EN = HIGH;                                                      /* Create a positive pulse on EN' (disable the device)      */
       EN = LOW;                                                       /* Re-enable the device, leave enabled until data read      */
       ShiftOut(&data, 4);                                             /* Shift out the report command                             */    
       ShiftIn(&resp, 4);                                              /* Shift in the response / status code                      */
       if (resp != RESP_BUSY) {                                        /* Stop polling if the response is ready                    */
           break;
       } 
   }
    
   if ((i == NBR_RETRIES) || (resp == RESP_ERR)) {                     /* If an error occured, disable the device, set an error    */
       EN = HIGH;                                                      /* code, and return (0)                                     */
       return SHIFT_ERR_ARG;
   }
  
   ///get the measurement
   ShiftIn(&data, 11);
   result[0]=data; ///x axis
   ShiftIn(&data, 11);
   result[1]=data; ///y axis
      
   return SHIFT_ERR_NONE;
}
        
/*
*********************************************************************************************************
*                                           ShiftIn()
*
* Description : The function uses the GPIO port defined above to replicate a synchronous serial port
*               such as SPI using bit banging techniques. This routine has been designed with the
*               Parallax SHIFTIN function in mind in order to facilitate the programming of Parallax
*               sensors in 'C' for non Parallax micro-controllers.
*
* Argument(s) : var             A pointer to an unsigned 16 bit variable to store received data.
*
*               bits            The number of bits to receive.
*
* Return(s)   : Error code      SHIFT_ERR_NONE  - No error
*                               SHIFT_ERR_ARG   - Function argument error
*
* Note(s)     : 1) ONLY 16 bits maximum may be shifted in at a time.
*********************************************************************************************************
*/

uint8  ShiftIn (uint16 *var, uint8 bits)
{
    uint8  i;
    uint8  data;
        
    if ((var == (void *)0) || (bits > 16)) {
        return (SHIFT_ERR_ARG);
    }
    
   P0DIR&=~(0x40);    
   *var                 =   0;                                          /* Initialize the return data to 0                          */
   
    CLK = LOW;                                                          /* Init clock. MSBPOST, clock idles low                     */
    for (i = bits; i > 0; i--) {                                        /* Begin shifting in data highest bit first                 */
        CLK = HIGH;                                                     /* Send the high part of the clock pulse                    */
        ShiftClkDly();                                                  /* Wait for 1/2 of a clock period                           */
        data = DATA;                                       /* Read the single bit input data AFTER sending the clock   */
        if (data > 0) {                                                 /* If the bit is 1, write it into the return value          */
           *var |= (1 << (i - 1));                                  /* Save the bit in the next most significant bit position   */
        }
        CLK = LOW;                                                      /* Send the low part of the clock pulse (clock idles low)   */
        ShiftClkDly();                                                  /* Wait for 1/2 of a clock period (clock period now over)   */
    }
             
    return (SHIFT_ERR_NONE);         
}

/*
*********************************************************************************************************
*                                           ShiftOut()
*
* Description : The function uses the GPIO port defined above to replicate a synchronous serial port
*               such as SPI using bit banging techniques. This routine has been designed with the
*               Parallax SHIFTOUT function in mind in order to facilitate the programming of
*               Pallax sensors in 'C' for non Parallax micro-controllers.
*
* Argument(s) : var             A pointer to an unsigned 16 bit variable containing the data to send.
*
*               bits            The number of bits to transmit.
*
* Return(s)   : Error code      SHIFT_ERR_NONE  - No error
*                               SHIFT_ERR_ARG   - Function argument error
*
* Note(s)     : 1) ONLY 16 bits maximum may be shifted out at a time.
*********************************************************************************************************
*/
         
uint8  ShiftOut (uint16 *var, uint8 bits)
{
    uint8 i;
    
    if ((var==(void *)0)||(bits>16)){
        return (SHIFT_ERR_ARG);
    }
    P0DIR|=(0xE0);
    
    CLK = LOW;                                                          /* Init clock. MSBFIRST, clock idles low                    */
    for (i = bits;i > 0; i--){                                        /* Begin shifting out data highest bit first                */
        if((*var & (1 << (i - 1))) > 0) {                              /* If the next bit to shift is a 1                          */
            DATA = HIGH;
                                               /* Set the Tx pin high                                      */   
        } else {
            DATA = LOW;
                                                 /* Next bit is a 0, set the Tx pin low                      */
        }
        CLK = HIGH;                                                     /* Send the high part of the clk pulse (clock the data in)  */
        ShiftClkDly();                                                  /* Wait for 1/2 of a clock period                           */                
        CLK = LOW;                                                      /* Send the low part of the clock pulse (clock idles low)   */
        ShiftClkDly();                                                  /* Wait for 1/2 of a clock period (clock period now over)   */
    }
    
    return (SHIFT_ERR_NONE);
}

/*
*********************************************************************************************************
*                                       HM55B_ClkDly()
*
* Description : The function waits for one half of a HM55B clock cycle
*
* Argument(s) : none.
*
* Return(s)   : none.
*********************************************************************************************************
*/

void  ShiftClkDly (void)
{
  uint16 startVal;
  uint16 curVal;
  
  startVal=TimerValueGet();
  curVal=TimerValueGet();
    
  while ((curVal-startVal)<ShiftClkDlyCnts) {
        curVal  = TimerValueGet();
  }
}

/**
  *
  ******************************************************************************************************
  *                            TimerValueGet()
  * This function is used to access the cc2430 timer 1
  *
  * Argument(s) : none.
  *
  * Return(s)   : uint16 that contains the current timer value
  ****************************************************************************************************** 
  */

uint16 TimerValueGet(void){
   ///have to access the low byte of the 
  ///the timer 1 register first 
    uint8 low=T1CNTL;
    uint8 high=T1CNTH;
    return BUILD_UINT16(low, high);
}