/**
 *  \file   hs_mmcsd.c
 *
 *  \brief  Device abstraction layer for HS MMC/SD
 *
 *   This file contains the device abstraction layer APIs for HS MMC/SD.
 */

/* Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED
 */


/* #include "soc_AM335x.h" */
#include "hw_types.h"
#include "mmcsd.h"
#include "hw_mmcsd.h"

/*******************************************************************************
*                        API FUNCTION DEFINITIONS
*******************************************************************************/


/**
 * \brief   Place CMD and DAT logic to reset
 *
 * \param   baseAddr      Base Address of the MMC/SD controller Registers.
 *
 * \return  None.
 **/
void MMCSDPlaceInReset(unsigned int baseAddr)
{
	HWREG(baseAddr + MMCSD_MMCCTL) |= MMCSD_MMCCTL_CMDRST | MMCSD_MMCCTL_DATRST;
}

/**
 * \brief   Place CMD and DAT logic to reset
 *
 * \param   baseAddr      Base Address of the MMC/SD controller Registers.
 *
 * \return  None.
 **/
void MMCSDClearReset(unsigned int baseAddr)
{
	HWREG(baseAddr + MMCSD_MMCCTL) &= ~(MMCSD_MMCCTL_CMDRST | MMCSD_MMCCTL_DATRST);
}



/**
 * \brief   Configure the MMC/SD bus width
 *
 * \param   baseAddr      Base Address of the MMC/SD controller Registers.
 * \param   width         SD/MMC bus width
 *
 * width can take the values \n 
 *     MMCSD_BUS_WIDTH_4BIT \n
 *     MMCSD_BUS_WIDTH_1BIT \n
 *
 * \return  None.
 **/
void MMCSDBusWidthSet(unsigned int baseAddr, unsigned int width)
{
    switch (width)
    {
        case MMCSD_BUS_WIDTH_4BIT:
        	/* reset width bits  */
            HWREG(baseAddr + MMCSD_MMCCTL) &= ~MMCSD_MMCCTL_WIDTH1_WIDTH0;
            /* set bit 2 for 4-wire mode */
            HWREG(baseAddr + MMCSD_MMCCTL) |= MMCSD_MMCCTL_WIDTH1_WIDTH0_4BIT;
        break;

        case MMCSD_BUS_WIDTH_1BIT:
        	/* reset width bits  */
            HWREG(baseAddr + MMCSD_MMCCTL) &= ~MMCSD_MMCCTL_WIDTH1_WIDTH0;
            /* set bit 2 for 4-wire mode */
            HWREG(baseAddr + MMCSD_MMCCTL) |= MMCSD_MMCCTL_WIDTH1_WIDTH0_8BIT;
        break;
    }
}


/**
 * \brief   Turn clock signal on the clock pin on / off
 *
 * \param   baseAddr      Base Address of the MMC/SD controller Registers.
 * \param   pwr           clock on / off setting
 *
 * pwr can take the values \n
 *     MMCSD_CLOCK_ON \n
 *     MMCSD_CLOCK_OFF \n
 *
 * \return  0 if the operation succeeded
 *         -1 if the operation failed
 *
 **/
int MMCSDBusClock(unsigned int baseAddr, unsigned int pwr)
{
    HWREG(baseAddr + MMCSD_MMCCLK) =
            (HWREG(baseAddr + MMCSD_MMCCLK) & ~MMCSD_MMCCLK_CLKEN) | pwr;

    if (pwr == MMCSD_CLOCK_ON)
    {
        if(MMCSDIsClockStable(baseAddr, 0xFFFF) == 0)
        {
            return -1;
        }
    }

    return 0;
}


/**
 * \brief   Get the internal clock stable status
 *
 * \param   baseAddr      Base Address of the MMC/SD controller Registers.
 * \param   retry         retry times to poll for stable
 *
 * \note: if retry is zero the status is not polled. If it is non-zero status
 *        is polled for retry times
 *
 * \return  1 if the clock is stable
 *          0 if the clock is not stable
 *
 **/
unsigned int MMCSDIsClockStable(unsigned int baseAddr, unsigned int retry)
{
    volatile unsigned int status = 0;

    do {
            /* read status bit
             * CLKSTP = 1 -> clock pin is held low
             * so invert bit for capability
             * */
    		status = !((HWREG(baseAddr + MMCSD_MMCST1) & MMCSD_MMCST1_CLKSTP)
            									>> MMCSD_MMCST1_CLKSTP_SHIFT);
            if ((status == 1) || (retry == 0))
            {
                break;
            }
    } while (retry--);

    return status;
}


/**
 * \brief   Set data timeout value
 *
 * \param   baseAddr      Base Address of the MMC/SD controller Registers.
 * \param   timeout       the data time out value
 *
 * \return  None.
 *
 **/
void MMCSDDataTimeoutSet(unsigned int baseAddr, unsigned int timeout)
{
    /* write upper 10 bits */
	HWREG(baseAddr + MMCSD_MMCTOR) = (HWREG(baseAddr + MMCSD_MMCTOD) &
    		~MMCSD_MMCTOR_TOD_25_16) | (timeout & MMCSD_MMCTOR_TOD_25_16);
	/* write lower 16 bits */
    HWREG(baseAddr + MMCSD_MMCTOD)= (timeout & MMCSD_MMCTOD_TOD_15_0);
}    


/**
 * \brief   Set output bus frequency
 *
 * \param   baseAddr      Base Address of the MMC/SD controller Registers.
 * \param   freq_out      The required output frequency on the bus
 *
 * \return   0  on clock enable success
 *          -1  on clock enable fail
 *
 * \note: Use MMCSD_IN_CLOCK_DEFAULT for 150 MHz sys clock,
 * MMCSD_OUT_CLOCK_STARTUP    -> 400 kHz
 * MMCSD_OUT_CLOCK_OPERATION  -> 25 MHz
 * If the clock is set properly, the clocks are enabled to the card with
 * the return of this function
 **/
int MMCSDBusFreqSet(unsigned int baseAddr, unsigned int freq_in, unsigned int freq_out)
{
    	volatile unsigned int clkd = 0;
    	volatile unsigned int regVal = 0;

    	/* store bit 8  of MMCCLK
    	 * DIV4 bit is 0
    	 */
    	regVal = HWREG(baseAddr + MMCSD_MMCCLK) &
    					~(MMCSD_MMCCLK_CLKRT | MMCSD_MMCCLK_DIV4) ;

    	/* divider formula is
    	 * freq_out = freq_in/ (2 * (1 + clkd))
    	 */

        /* Calculate and program the divisor */
        clkd = (freq_in / (2 *freq_out)) - 1;
        // clkd = (clkd < 2) ? 2 : clkd;
        clkd = (clkd > 255) ? 255 : clkd;

		/* Do not cross the required freq */
		while((freq_in/ (2 * (1 + clkd))) > freq_out)
		{
			if (clkd == 255)
			{
				/* Return we cannot set the clock freq */
			   return -1;
			}

			clkd++;
		}

		/* set new freg */
		HWREG(baseAddr + MMCSD_MMCCLK) = regVal | (clkd << MMCSD_MMCCLK_CLKRT_SHIFT);

        /* Wait for the interface clock stabilization */
        if(MMCSDIsClockStable(baseAddr, 0xFFFF) == 0)
        {
            return -1;
        }
        
        /* Enable clock to the card */
    return 0;
}    


/**
 * \brief   Enables the controller events to generate a h/w interrupt request
 *
 * \param   baseAddr    Base Address of the MMC/SD controller Registers.
 * \param   flag        Specific event required;
 *
 * flag can take the following (or combination of) values \n
 * MMCSD_MMCIM_xxx
 *
 * \return   none
 *
 **/
void MMCSDIntrEnable(unsigned int baseAddr, unsigned int flag)
{
    HWREG(baseAddr + MMCSD_MMCIM) |= flag;
}


/**
 * \brief   Gets the status bits from the controller 
 *
 * \param   baseAddr    Base Address of the MMC/SD controller Registers.
 * \param   flag        Specific status required;
 *
 * flag can take the following (or combination of) values \n
 * MMCSD_MMCST0_xxx
 *
 * \return   status flags
 *
 **/
unsigned int MMCSDIntrStatusGet(unsigned int baseAddr, unsigned int flag)
{
    return HWREG(baseAddr + MMCSD_MMCST0) & flag;
}


/**
 * \brief   Clear the status bits from the controller 
 *
 * \param   baseAddr    Base Address of the MMC/SD controller Registers.
 * \param   flag        Specific status required;
 *
 * flag can take the following (or combination of) values \n
 * HS_MMCSD_STAT_xxx
 *
 * \return   none
 *
 **/
void MMCSDIntrStatusClear(unsigned int baseAddr, unsigned int flag)
{
    HWREG(baseAddr + MMCSD_MMCST0) = (HWREG(baseAddr + MMCSD_MMCST0) & ~flag );
}


/**
 * \brief    Checks if the command is complete
 *
 * \param    baseAddr    Base Address of the MMC/SD controller Registers
 * \param    retry       retry times to poll for completion
 *
 * \return   1 if the command is complete
 *           0 if the command is not complete
 **/
unsigned int MMCSDIsCmdComplete(unsigned int baseAddr, unsigned int retry)
{
    volatile unsigned int status = 0;

    do {
        status = (HWREG(baseAddr + MMCSD_MMCST0) & MMCSD_MMCST0_CCS) >>
        									MMCSD_MMCST0_CCS_SHIFT;
        if ((status == 1) || (retry == 0))
        {
            break;
        }
    } while (retry--);

    return status;
}



unsigned int MMCSDIsDataTXReady(unsigned int baseAddr, unsigned int retry,
				unsigned int *dataDone, unsigned int *status, unsigned int *transferDone, unsigned int *writeCRCerror)
{
    volatile unsigned int status0 = 0;
    volatile unsigned int transmitReady = 0;

    do {
        status0 = (HWREG(baseAddr + MMCSD_MMCST0) &
        		(MMCSD_MMCST0_DXRDY | MMCSD_MMCST0_TRNDNE | MMCSD_MMCST0_DATDNE | MMCSD_MMCST0_CRCWR));

        if (status0)
        {
        	*status |= status0; // accumulate events

        	transmitReady |= ((status0 & MMCSD_MMCST0_DXRDY) >> MMCSD_MMCST0_DXRDY_SHIFT);
        	*transferDone |= ((status0 & MMCSD_MMCST0_TRNDNE) >> MMCSD_MMCST0_TRNDNE_SHIFT);
        	*dataDone |= ((status0 & MMCSD_MMCST0_DATDNE) >> MMCSD_MMCST0_DATDNE_SHIFT);
        	*writeCRCerror |= ((status0 & MMCSD_MMCST0_CRCWR) >> MMCSD_MMCST0_CRCWR_SHIFT);

        	if ((*transferDone == 1)||(*dataDone == 1)||(transmitReady == 1)) break;

        }
    } while (retry--);

    return transmitReady;
}



/***********************************************************************************************************/

unsigned int MMCSDIsDataRXReady(unsigned int baseAddr, unsigned int retry,
				unsigned int *dataDone, unsigned int *status, unsigned int *readCRCerror)
{
    volatile unsigned int status0 = 0;
    volatile unsigned int receiveReady = 0;

    do {
        status0 = (HWREG(baseAddr + MMCSD_MMCST0) &
        		(MMCSD_MMCST0_DRRDY | MMCSD_MMCST0_DATDNE | MMCSD_MMCST0_CRCRD));

        if (status0)
        {
        	*status |= status0; // accumulate events

        	receiveReady |= ((status0 & MMCSD_MMCST0_DRRDY) >> MMCSD_MMCST0_DRRDY_SHIFT);
        	*dataDone |= ((status0 & MMCSD_MMCST0_DATDNE) >> MMCSD_MMCST0_DATDNE_SHIFT);
        	*readCRCerror |= ((status0 & MMCSD_MMCST0_CRCRD) >> MMCSD_MMCST0_CRCRD_SHIFT);

        	if ((*dataDone == 1)||(receiveReady == 1)) break;

        }
    } while (retry--);

    return receiveReady;
}




/**
 * \brief    Checks if the transfer is complete
 *
 * \param    baseAddr    Base Address of the MMC/SD controller Registers
 * \param    retry       retry times to poll for completion
 *
 * \return   1 if the transfer is complete
 *           0 if the transfer is not complete
 **/
unsigned int MMCSDIsXferComplete(unsigned int baseAddr, unsigned int retry)
{
    volatile unsigned int status = 0;

    do {
        status = (HWREG(baseAddr + MMCSD_MMCST0) & MMCSD_MMCST0_TRNDNE) >>
        										MMCSD_MMCST0_TRNDNE_SHIFT;
        if ((status == 1) || (retry == 0))
        {
            break;
        }
    } while (retry--);

    return status;
}


unsigned int MMCSDIsDataComplete(unsigned int baseAddr, unsigned int retry)
{
    volatile unsigned int status;

    do {
        status = (HWREG(baseAddr + MMCSD_MMCST0) & MMCSD_MMCST0_DXRDY) >>
        										MMCSD_MMCST0_DXRDY_SHIFT;

        if ((status == 1) || (retry == 0))
        {
            break;
        }
    } while (retry--);

    return status;
}

/**
 * \brief    Set the block length/size for data transfer
 *
 * \param    baseAddr    Base Address of the MMC/SD controller Registers
 * \param    blklen      Command to be passed to the controller/card
 * 
 * \note: blklen should be within the limits specified by the controller/card
 *
 * \return   none
 **/
void MMCSDBlkLenSet(unsigned int baseAddr, unsigned int blklen)
{
   HWREG(baseAddr + MMCSD_MMCBLEN) &= ~MMCSD_MMCBLEN_BLEN;
   HWREG(baseAddr + MMCSD_MMCBLEN) |= blklen;
}



void MMCSDFIFODirTX(unsigned int baseAddr)
{
	/* reset FIFO */
	HWREG(baseAddr + MMCSD_MMCFIFOCTL) |= MMCSD_MMCFIFOCTL_FIFORST;
	/* set FIFO direction to write */
	HWREG(baseAddr + MMCSD_MMCFIFOCTL) |= MMCSD_MMCFIFOCTL_FIFODIR;
}


void MMCSDFIFODirRX(unsigned int baseAddr)
{
	/* reset FIFO */
	HWREG(baseAddr + MMCSD_MMCFIFOCTL) |= MMCSD_MMCFIFOCTL_FIFORST;
	/* set FIFO direction to write */
	HWREG(baseAddr + MMCSD_MMCFIFOCTL) &= ~MMCSD_MMCFIFOCTL_FIFODIR;
}



/**
 * \brief    Pass the MMC/SD command to the controller/card
 *
 * \param   baseAddr    Base Address of the MMC/SD controller Registers
 * \param   cmd         Command to be passed to the controller/card
 * \param   cmdArg      argument for the command
 * \param   data        data pointer, if it is a data command, else must be null
 * \param   nblks       data length in number of blocks (multiple of BLEN)
 * \param   dmaEn       Should dma be enabled (1) or disabled (0)
 *
 * \note: Please use MMCSD_CMD(cmd, type, restype, rw) to form command
 *
 * \return   none
 **/
void MMCSDCommandSend(unsigned int baseAddr, unsigned int cmd,
                        unsigned int cmdarg, void *data,
                        unsigned int nblks, unsigned int dmaEn)
{
	// data command
	if (data != NULL)
    {
		// set transfer indicator and crear previous status
        cmd |= (MMCSD_MMCCMD_WDATX | MMCSD_MMCCMD_DCLR);
        //MMCSDFIFODirTX(baseAddr);
    }

    //enable DMA transfer
    if (dmaEn == 1) cmd |= MMCSD_MMCCMD_DMATRIG;

    // Set the block information; block length is specified separately
    HWREG(baseAddr + MMCSD_MMCNBLK) &= ~MMCSD_MMCNBLK_NBLK;
    HWREG(baseAddr + MMCSD_MMCNBLK) |= nblks << MMCSD_MMCNBLK_NBLK_SHIFT;

    // Set the command/command argument
    HWREG(baseAddr + MMCSD_MMCARGHL) = cmdarg;
    HWREG(baseAddr + MMCSD_MMCCMD) = cmd;
}


/**
 * \brief    Get the command response from the conntroller
 *
 * \param    baseAddr    Base Address of the MMC/SD controller Registers
 * \param    rsp         pointer to buffer which is to be filled with the response
 *
 * \note: that this function shall return the values from all response registers.
 * Hence, rsp, must be a pointer to memory which can hold max response length.
 * It is the responsibility of the caller to use only the required/relevant
 * parts of the response
 *
 * \return   return RSPDNE bit in MMCSTO register. 0 -> no receiving response is done
 * 1 -> response successfully has received or command has send without response
**/
unsigned int MMCSDIsResponseGet(unsigned int baseAddr, unsigned int *rsp)
{
	unsigned int response;
	unsigned int retry = 0x0000FFFF;

    do {
    	response = (HWREG(baseAddr + MMCSD_MMCST0)
    			 & MMCSD_MMCST0_RSPDNE) >> MMCSD_MMCST0_RSPDNE_SHIFT;

        if ((response == 1) || (retry == 0)) break;

    } while (retry--);

	if ((response == 0)|(rsp == NULL)) return response;

    unsigned int i;
    for (i = 0; i <=3; i++)
    {
        rsp[i] = HWREG(baseAddr + MMCSD_MMCRSP(i));
    }

	return response;
}


/**
 * \brief    Send the data to the card from the controller
 *
 * \param    baseAddr    Base Address of the MMC/SD controller Registers
 * \param    data        pointer to buffer which is to be filled with the data
 * \param    len         length of the data
 *
 * \note: this function reads the data in chunks of 32 bits (4-byte words).
 * Hence, the len should be multiple of 4-byte words
 *
 * \return   none
 **/
void MMCSDDataGet(unsigned int baseAddr, unsigned char *data, unsigned int len)
{
	unsigned int i;

	for (i = 0; i < len/4; i++)
	{
		((unsigned int*)data)[i] = HWREG(baseAddr + MMCSD_MMCDRR);
	}
}


/**
 * \brief    Send the data to the card from the conntroller
 *
 * \param    baseAddr    Base Address of the MMC/SD controller Registers
 * \param    data        pointer to buffer which is to be filled with the data
 * \param    len         length of the data
 *
 * \note: this function reads the data in chunks of 32 bits (4-byte words).
 * Hence, the len should be multiple of 4-byte words
 *
 * \return   none
 **/
void MMCSDDataSend(unsigned int baseAddr, unsigned char *data, unsigned int len)
{
	volatile unsigned int i;

	for (i = 0; i < len/4; i++)
	{
		HWREG(baseAddr + MMCSD_MMCDXR) = 0xFFFFFFFF; // ((unsigned int*)data)[i]
	}
}



void MMCSDFIFOFill(unsigned int baseAddr)
{
	unsigned int retry = 0xFFFF;

	do{

		HWREG(baseAddr + MMCSD_MMCDXR) = 0x5F5F5F5F;

        if (HWREG(baseAddr + MMCSD_MMCST1) & MMCSD_MMCST1_FIFOFUL)
        {
        	break;
        }

	}while(retry--);

	retry = 0;
}



/**
 * \brief    Check if the card is inserted and detected
 *
 * \param    baseAddr    Base Address of the MMC/SD controller Registers
 *
 * \return   0  if the card is not inserted and detected
 *           1  if the card is inserted and detected
 *
 * \note: that this functional may not be available for all instances of the
 * controler. This function, is only useful of the controller has a dedicated
 * card detect pin. If not, the card detection mechanism is application
 * implementation specific
 **/
unsigned int MMCSDIsCardInserted(unsigned int baseAddr)
{
   // return (HWREG(baseAddr + MMCHS_PSTATE) & MMCHS_PSTATE_CINS) >>
    //            MMCHS_PSTATE_CINS_SHIFT;
	return 1;
}


/**
 * \brief    Check if the card is write protected
 *
 * \param    baseAddr    Base Address of the MMC/SD controller Registers
 *
 * \return   0  if the card is not write protected
 *           1  if the card is write protected
 * \note: that this functional may not be available for all instances of the
 * controler. This function, is only useful of the controller has a dedicated
 * write protect detect pin. If not, the write protect detection mechanism is
 * application implementation specific
 **/
unsigned int MMCSDIsCardWriteProtected(unsigned int baseAddr)
{
    //return (HWREG(baseAddr + MMCHS_PSTATE) & MMCHS_PSTATE_WP) >>
     //           MMCHS_PSTATE_WP_SHIFT;
	return 0;
}

