/**
 * \file    gpioCardDetect.c
 *
 * \brief   This is a sample application file demonstrating the use of
 *          a GPIO pin to generate an interrupt whenever an MMC/SD card
 *          is inserted or ejected from the Evaluation Module(EVM).
 */

/* Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED */

#include "interrupt.h"
#include "uartStdio.h"
#include "gpio.h"
#include "psc.h"

#include "delay.h"
#include "mmcsd.h"

#include "soc_OMAPL138.h"
#include "evmOMAPL138.h"

/****************************************************************************/
/*              LOCAL FUNCTION PROTOTYPES                                   */
/****************************************************************************/
static void Delay(volatile unsigned int delay);
static void ConfigureIntGPIO(void);
static void CheckCardStatus(void);
static void SetupInt(void);
static void GPIOIsr(void);

/****************************************************************************/
/*              GLOBAL VARIABLES                                            */
/****************************************************************************/
volatile unsigned char flag = 0;

/****************************************************************************/
/*             LOCAL FUNCTION DEFINITIONS                                   */
/****************************************************************************/

int main(void)
{
	unsigned int response[4];
	unsigned int respDone;
	unsigned int transmitReady;
	unsigned int receiveReady;
	unsigned int transferDone;
	unsigned int writeCRCerror;
	unsigned int readCRCerror;

	unsigned int transferFinished;

	unsigned int retry;
	unsigned int rca;
	unsigned int dataDone;

	unsigned int status_0 = 0;

	unsigned char data[512];
	//unsigned int adress;


	/* The Local PSC number for GPIO is 3. GPIO belongs to PSC1 module.*/
    PSCModuleControl(SOC_PSC_1_REGS, HW_PSC_GPIO, PSC_POWERDOMAIN_ALWAYS_ON,
		     PSC_MDCTL_NEXT_ENABLE);

    /* power up MMCSD controller 0 */
    PSCModuleControl(SOC_PSC_0_REGS, HW_PSC_MMCSD0, PSC_POWERDOMAIN_ALWAYS_ON,
    		     PSC_MDCTL_NEXT_ENABLE);

    /* power up Timer 0 */
    PSCModuleControl(SOC_PSC_0_REGS, HW_PSC_CC0, PSC_POWERDOMAIN_ALWAYS_ON,
		     PSC_MDCTL_NEXT_ENABLE);

    PSCModuleControl(SOC_PSC_0_REGS, HW_PSC_TC0, PSC_POWERDOMAIN_ALWAYS_ON,
       		     PSC_MDCTL_NEXT_ENABLE);


    /* Initializes the UART instance.*/
    UARTStdioInit();   

    /* Pin Multiplexing of pin 0 of GPIO Bank 4.*/
    GPIOBank4Pin0PinMuxSetup();

    /* Sets the pin 65(GP4[0]) as input.*/
    GPIODirModeSet(SOC_GPIO_0_REGS, 65, GPIO_DIR_INPUT);


    MMCSD0PinMuxSetup();

    MMCSDPlaceInReset(SOC_MMCSD_0_REGS);

    MMCSDBusWidthSet(SOC_MMCSD_0_REGS, MMCSD_BUS_WIDTH_4BIT);

    MMCSDBusFreqSet(SOC_MMCSD_0_REGS,
    		MMCSD_IN_CLOCK_DEFAULT, MMCSD_OUT_CLOCK_STARTUP);

    MMCSDClearReset(SOC_MMCSD_0_REGS);
    MMCSDBusClock (SOC_MMCSD_0_REGS, MMCSD_CLOCK_ON);


    /*
    ** Configure rising edge and falling edge triggers on pin 65 to generate
    ** an interrupt
    */
    GPIOIntTypeSet(SOC_GPIO_0_REGS, 65, GPIO_INT_TYPE_BOTHEDGE);

    /* Enable interrupts for Bank 4.*/
    GPIOBankIntEnable(SOC_GPIO_0_REGS, 4);

    /* Configuring the AINTC to handle interrupts.*/
    SetupInt();

    /* Configure GPIO interrupts */
    ConfigureIntGPIO();

    DelayTimerSetup();

    delay(100);
/*************************************0**********************************************************/
    /* send GO_IDLE */
    MMCSDCommandSend(SOC_MMCSD_0_REGS,
    		MMCSD_CMD(MMCSD_CMD_GO_IDLE_STATE, MMCSD_NO_RESPONSE, MMCSD_CMD_DIR_DONTCARE),
    		0, 0, 0, 0);

    respDone = MMCSDIsResponseGet(SOC_MMCSD_0_REGS, 0);

    if (respDone == 0) while(1);

    //MMCSDCommandSend(baseAddr, cmd, cmdarg, *data, nblks, dmaEn)

    delay(500);

/*************************************8**********************************************************/
	/* send SEND_IF_COND */
	MMCSDCommandSend(SOC_MMCSD_0_REGS,
			MMCSD_CMD(MMCSD_CMD_SEND_IF_COND, MMCSD_R7_RESPONSE, MMCSD_CMD_DIR_DONTCARE),
			0x0000010F, 0, 0, 0);

	respDone = MMCSDIsResponseGet(SOC_MMCSD_0_REGS, 0);

	if (respDone == 0) while(1);



	do{
	/*************************************55**********************************************************/
		/* send APP_CMD-55*/
		MMCSDCommandSend(SOC_MMCSD_0_REGS,
				MMCSD_CMD(MMCSD_CMD_APP_CMD, MMCSD_R1_RESPONSE, MMCSD_CMD_DIR_DONTCARE),
				0, 0, 0, 0);

		respDone = MMCSDIsResponseGet(SOC_MMCSD_0_REGS, 0);
		if (respDone == 0)while(1);

	/*************************************41**********************************************************/
		/* send APP_CMD-41 with default VDD window */
		MMCSDCommandSend(SOC_MMCSD_0_REGS,
				MMCSD_CMD(MMCSD_CMD_SD_SEND_OP_COND, MMCSD_R3_RESPONSE, MMCSD_CMD_DIR_DONTCARE),
				(MMCSD_OCR_VDD_DEFAULT_WINDOW | MMCSD_OCR_CCS_SDHC | MMCSD_OCR_SDXC), 0, 0, 0);

		respDone = MMCSDIsResponseGet(SOC_MMCSD_0_REGS, response);
		if (respDone == 0)while(1);

		delay(1000);

	}while (!(response[3] & 0x80000000));



/*************************************2**********************************************************/
    /* send ALL_SEND_CID CMD-2 */

    MMCSDCommandSend(SOC_MMCSD_0_REGS,
        	MMCSD_CMD(MMCSD_CMD_ALL_SEND_CID, MMCSD_R2_RESPONSE, MMCSD_CMD_DIR_DONTCARE),
        	0, 0, 0, 0);

    respDone = MMCSDIsResponseGet(SOC_MMCSD_0_REGS, response);
    if (respDone == 0)while(1);

/*************************************3**********************************************************/
	/* send SEND_RELATIVE_ADDR CMD-3 */
	MMCSDCommandSend(SOC_MMCSD_0_REGS,
			MMCSD_CMD(MMCSD_CMD_SEND_RELATIVE_ADDR, MMCSD_R6_RESPONSE, MMCSD_CMD_DIR_DONTCARE),
			0, 0, 0, 0);

	respDone = MMCSDIsResponseGet(SOC_MMCSD_0_REGS, response);
	if (respDone == 0)while(1);

	rca = response[3] >> 16;

	MMCSDBusFreqSet(SOC_MMCSD_0_REGS, MMCSD_IN_CLOCK_DEFAULT, 25000000); // MMCSD_OUT_CLOCK_OPERATION
																						// MMCSD_OUT_CLOCK_STARTUP

	unsigned int i;

/*************************************7**********************************************************/
	MMCSDCommandSend(SOC_MMCSD_0_REGS,
    			MMCSD_CMD(7, MMCSD_R1_BUSY_RESPONSE, MMCSD_CMD_DIR_WRITE),
    			(rca << 16), 0, 0, 0);

    respDone = MMCSDIsResponseGet(SOC_MMCSD_0_REGS, response);
    if (respDone == 0)while(1);


    delay (200);


    for (i = 0; i <= 511; i++)
    {
    	data[i] = 0x00;
    }


#define NUM_OF_BLOCKS 1





/******************************************************************************************
 * 							read block
*******************************************************************************************/


	for (i = 0; i < 10; i++)
	{
		MMCSDFIFODirRX(SOC_MMCSD_0_REGS);

		//MMCSDDataSend(SOC_MMCSD_0_REGS, data, 512);

		// MMCSDFIFOFill (SOC_MMCSD_0_REGS);

		status_0 = 0;
		dataDone = 0;
		transferDone = 0;
		writeCRCerror = 0;

		transferFinished = 0;

		retry = 10;

/*************************************17**********************************************************/
		//MMCSDCommandSend(baseAddr, cmd, cmdarg, *data, nblks, dmaEn)
		MMCSDCommandSend(SOC_MMCSD_0_REGS,
				MMCSD_CMD(17, MMCSD_R1_RESPONSE, MMCSD_CMD_DIR_READ),
				(i * NUM_OF_BLOCKS), &data, NUM_OF_BLOCKS, 0);

		do{
			receiveReady = MMCSDIsDataRXReady(SOC_MMCSD_0_REGS, 0xFFFF, &dataDone, &status_0, &readCRCerror);

			if(receiveReady)
			{
				MMCSDDataGet (SOC_MMCSD_0_REGS, data, 256);
			}

			if(dataDone  )  // || writeCRCerror || transferDone
			{
				transferFinished = 1;
				break;
			}

		}while(retry--);
	}







/******************************************************************************************
 * 							write block
*******************************************************************************************/


    for (i = 0; i < 10; i++)
    {
    	MMCSDFIFODirTX(SOC_MMCSD_0_REGS);

    	MMCSDDataSend(SOC_MMCSD_0_REGS, data, 512);

    	// MMCSDFIFOFill (SOC_MMCSD_0_REGS);

    	status_0 = 0;
    	dataDone = 0;
    	transferDone = 0;
    	writeCRCerror = 0;

    	transferFinished = 0;

    	retry = 10;

/*************************************24**********************************************************/
        //MMCSDCommandSend(baseAddr, cmd, cmdarg, *data, nblks, dmaEn)
    	MMCSDCommandSend(SOC_MMCSD_0_REGS,
    			MMCSD_CMD(24, MMCSD_R1_RESPONSE, MMCSD_CMD_DIR_WRITE),
    			(i * NUM_OF_BLOCKS), &data, NUM_OF_BLOCKS, 0);

    	do{ 			// MMCSDIsDataTXReady(baseAddr,			retry,*dataDone,  *status,   *transferDone)
    		transmitReady = MMCSDIsDataTXReady(SOC_MMCSD_0_REGS, 0xFFFF, &dataDone, &status_0, &transferDone, &writeCRCerror);

			if(transmitReady)
			{
				MMCSDDataSend(SOC_MMCSD_0_REGS, data, 256);
			}

			if(dataDone  )  // || writeCRCerror || transferDone
			{
				transferFinished = 1;
				break;
			}

    	}while(retry--);

/*
    	do{
    		MMCSDIsDataTXReady(SOC_MMCSD_0_REGS,0xFF, &dataDone, &status_0, &transferDone);
    	}while(!(dataDone));

    	MMCSDCommandSend(SOC_MMCSD_0_REGS,
    	    			MMCSD_CMD(12, MMCSD_R1_BUSY_RESPONSE, MMCSD_CMD_DIR_WRITE),
    	    			0, 0, 0, 0);
*/

    	// (status_0 & ((1 << 12)|(1 << 0)|(1 << 5)) )
    	//respDone = MMCSDIsResponseGet(SOC_MMCSD_0_REGS, response);
    	//if (respDone == 0)while(1);
    }

    UARTPuts("StarterWare GPIO Demo Application.\r\n", -2);
    UARTPuts("Insert an MMC/SD card.\r\n", -2);

    while(1)
    {
        if(flag == 1)
        {
            CheckCardStatus();
        }
    }



} // end of main




/*
** \brief   This function invokes necessary functions to configure the ARM 
**          processor and ARM Interrupt Controller(AINTC) to receive and
**          handle interrupts.
*/

static void SetupInt(void)
{
	// Setup ARM or DSP interrupt controller

#ifdef _TMS320C6X
	// Initialize the DSP Interrupt Controller
	IntDSPINTCInit();

	// Enable DSP Interrupts Globally
	IntGlobalEnable();
#else
    /* Initialize the ARM Interrupt Controller.*/
    IntAINTCInit();

     /* Enable IRQ in CPSR.*/
    IntMasterIRQEnable();

    /* Enable the interrupts in GER of AINTC.*/
    IntGlobalEnable();

    /* Enable the interrupts in HIER of AINTC.*/
    IntIRQEnable();
#endif
}


/*
** \brief  This function configures the AINTC to receive the GPIO interrupt.
*/

static void ConfigureIntGPIO(void)
{
	// Configure GPIO interrupts for ARM or DSP

#ifdef _TMS320C6X
	// Register the ISR in the Interrupt Vector Table
	IntRegister(C674X_MASK_INT4, GPIOIsr);

	// Map the system interrupt to the DSP maskable interrupt
	IntEventMap(C674X_MASK_INT4, SYS_INT_GPIO_B4INT);

	// Enable DSP maskable interrupt
	IntEnable(C674X_MASK_INT4);
#else
    // Register the ISR in the Interrupt Vector Table.
    IntRegister(SYS_INT_GPIOB4, GPIOIsr);

    // Map the channnel number 2 of AINTC to GPIO BANK 4 system interrupt.
    IntChannelSet(SYS_INT_GPIOB4, 2);

    // Enable the System Interrupts for AINTC.
    IntSystemEnable(SYS_INT_GPIOB4);
#endif
}


/*
** \brief   Interrupt Service Routine to be executed on GPIO interrupts.
**          This disables the bank interrupts, clears the system interrupt
**          status and pin interrupt status. This also sets flag as 1.
*/
static void GPIOIsr(void)
{
    /* Disable the interrupts for pins of bank 4 in GPIO.*/
    GPIOBankIntDisable(SOC_GPIO_0_REGS, 4);

#ifdef _TMS320C6X
    // Clear the system interrupt status in the DSPINTC
    IntEventClear(SYS_INT_GPIO_B4INT);
#else
    /* Clears the system interrupt status of GPIO in AINTC.*/
    IntSystemStatusClear(SYS_INT_GPIOB4);
#endif

    /* Clears the Interrupt Status of GP4[0] in GPIO.*/
    GPIOPinIntClear(SOC_GPIO_0_REGS, 65);

    flag = 1;
}

/*
** \brief  This function checks the insertion status of the MMC/SD card
**         in the device and prints related statements on the serial
**         commuincation console of the external device.
**         
*/

static void CheckCardStatus(void)
{
    Delay(0x1FFF);
    
#ifdef _TMS320C6X
    // Clear the system interrupt status in the DSPINTC
    IntEventClear(SYS_INT_GPIO_B4INT);
#else
    /* Clears the system interrupt status of GPIO in AINTC.*/
    IntSystemStatusClear(SYS_INT_GPIOB4);
#endif

    /* Clears the Interrupt Status of GP4[0] in GPIO.*/
    GPIOPinIntClear(SOC_GPIO_0_REGS, 65);
    
    /* 
    ** 'GPIOPinRead' here returns the value on the GP4[0].
    ** If value returned is 1, it implies the card is removed.
    ** If value returned is 0, it implies the card is inserted.
    */

    if (GPIOPinRead(SOC_GPIO_0_REGS, 65))
    {
        UARTPuts("MMC/SD card is removed.\n\r", -2);
    }
    else
    {
        UARTPuts("MMC/SD card inserted.\n\r", -2);
    }

    flag = 0;

    /* Enable interrupts for pins of bank 4.*/
    GPIOBankIntEnable(SOC_GPIO_0_REGS, 4);
}

/*
** \brief   This function can be called to generate a delay.
*/

static void Delay(volatile unsigned int delay)
{
    while(delay--);
}


/*****************************END OF FILE************************************/
