/**
 *  \file   mmcsd.h
 *
 *  \brief  MMC/SD APIs and macros.
 *
 *   This file contains the driver API prototypes and macro definitions.
 */

/* Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/
 * ALL RIGHTS RESERVED
 */
#ifndef __HS_MMCSD_H__
#define __HS_MMCSD_H__

#include "hw_mmcsd.h"
#include "hw_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/*****************************************************************************/

/*****************************************************************************/

/*****************************************************************************/
/*
** Macros for bus clock confoguration
*/
#define MMCSD_IN_CLOCK_DEFAULT 			(150000000)
#define MMCSD_OUT_CLOCK_STARTUP 		(400000)
#define MMCSD_OUT_CLOCK_OPERATION		(25000000)
#define MMCSD_CLOCK_ON            (MMCSD_MMCCLK_CLKEN)
#define MMCSD_CLOCK_OFF           (0)

/*
** Macros that can be used for SD controller DMA request configuration
*/
#define HS_MMCSD_DMAREQ_EDGETRIG    (MMCHS_CON_SDMA_LNE_EARLYDEASSERT << MMCHS_CON_DMA_LNE_SHIFT)
#define HS_MMCSD_DMAREQ_LVLTRIG     (MMCHS_CON_SDMA_LNE_LATEDEASSERT << MMCHS_CON_DMA_LNE_SHIFT)

/*
** Macros that can be used for selecting the bus/data width
*/
#define MMCSD_BUS_WIDTH_4BIT    (0x4)
#define MMCSD_BUS_WIDTH_1BIT    (0x1)

/*
** Macros that can be used for forming the MMC/SD command
*/
/* Here\n
** cmd     : SD/MMC command number enumeration
** type    : specifies suspend upon CMD52 or resume upon CMD52 or abort upon CMD52/12.
** restype : no response, or response with or without busy
** rw      : direction of data transfer
*/
#define MMCSD_CMD(cmd, restype, rw)    ((cmd << MMCSD_MMCCMD_CMD_SHIFT) | rw | restype)

/*
** Some commands and their codes
*/
/* SD ver 2.0 initialisation commands */
#define MMCSD_CMD_GO_IDLE_STATE 			(0)
#define MMCSD_CMD_SEND_IF_COND 				(8)
#define MMCSD_CMD_APP_CMD 					(55)
#define MMCSD_CMD_SD_SEND_OP_COND 			(41)
#define MMCSD_CMD_ALL_SEND_CID 				(2)
#define MMCSD_CMD_SEND_RELATIVE_ADDR 		(3)
/* write data commands */
#define MMCSD_CMD_SET_BLOCK_COUNT 			(23)
#define MMCSD_CMD_WRITE_MULTIPLE_BLOCK 		(25)


/* Macros for response format */
#define MMCSD_NO_RESPONSE           (MMCSD_MMCCMD_RSPFMT_NORSP << MMCSD_MMCCMD_RSPFMT_SHIFT)
#define MMCSD_R1_RESPONSE       	(MMCSD_MMCCMD_RSPFMT_R1 << MMCSD_MMCCMD_RSPFMT_SHIFT)
#define MMCSD_R2_RESPONSE       	(MMCSD_MMCCMD_RSPFMT_R2 << MMCSD_MMCCMD_RSPFMT_SHIFT)
#define MMCSD_R3_RESPONSE       	(MMCSD_MMCCMD_RSPFMT_R3 << MMCSD_MMCCMD_RSPFMT_SHIFT)
#define MMCSD_R4_RESPONSE       	(MMCSD_MMCCMD_RSPFMT_R4 << MMCSD_MMCCMD_RSPFMT_SHIFT)
#define MMCSD_R5_RESPONSE       	(MMCSD_MMCCMD_RSPFMT_R5 << MMCSD_MMCCMD_RSPFMT_SHIFT)
#define MMCSD_R6_RESPONSE       	(MMCSD_MMCCMD_RSPFMT_R6 << MMCSD_MMCCMD_RSPFMT_SHIFT)
#define MMCSD_R7_RESPONSE       	(MMCSD_MMCCMD_RSPFMT_R7 << MMCSD_MMCCMD_RSPFMT_SHIFT)
#define MMCSD_R1_BUSY_RESPONSE      ((MMCSD_MMCCMD_RSPFMT_R1 << MMCSD_MMCCMD_RSPFMT_SHIFT)|\
									 MMCSD_MMCCMD_BSYEXP)

#define MMCSD_CMD_DIR_READ           (0x00000000u)
#define MMCSD_CMD_DIR_WRITE          (MMCSD_MMCCMD_DTRW)
#define MMCSD_CMD_DIR_DONTCARE       (MMCSD_MMCCMD_DTRW)

/* operation conditions register (OCR) */

#define MMCSD_OCR_VDD_DEFAULT_WINDOW       (0x00FF8000) // 2,7V - 3,6V

#define MMCSD_OCR_CCS       		(1)		// Card Capacity Status bit
#define MMCSD_OCR_CCS_SDSC_SHIFT    (30) 	// Card Capacity Status bit position
#define MMCSD_OCR_CCS_SDSC       	(0) 	// 0 indicates that the card is SDSC
// 1 indicates that the card is SDHC or SDXC
#define MMCSD_OCR_CCS_SDHC       	(MMCSD_OCR_CCS << MMCSD_OCR_CCS_SDSC_SHIFT)

#define MMCSD_OCR_XPC			(1)		// SDXC Power Control
#define MMCSD_OCR_XPC_SHIFT		(28)	// SDXC Power Control bit position
// 1 indicates that the controller suports 150 mA supply to card
#define MMCSD_OCR_SDXC     		(MMCSD_OCR_XPC << MMCSD_OCR_XPC_SHIFT)

/*
** Macros that can be used for checking the present state of the host controller
*/
#define HS_MMCSD_CARD_WRITEPROT        (MMCHS_PSTATE_WP)
#define HS_MMCSD_CARD_INSERTED         (MMCHS_PSTATE_CINS)        
#define HS_MMCSD_CARD_STABLE           (MMCHS_PSTATE_CSS)        
#define HS_MMCSD_BUFFER_READABLE       (MMCHS_PSTATE_BRE)        
#define HS_MMCSD_BUFFER_WRITEABLE      (MMCHS_PSTATE_BWE)        
#define HS_MMCSD_READ_INPROGRESS       (MMCHS_PSTATE_RTA)        
#define HS_MMCSD_WRITE_INPROGRESS      (MMCHS_PSTATE_WTA)        

/*
** Macros that can be used for setting the clock, timeout values
*/
#define HS_MMCSD_DATA_TIMEOUT(n)        ((((n) - 13) & 0xF) << MMCHS_SYSCTL_DTO_SHIFT)
#define HS_MMCSD_CLK_DIVIDER(n)         ((n & 0x3FF) << MMCHS_SYSCTL_CLKD_SHIFT)
#define HS_MMCSD_CARDCLOCK_ENABLE       (MMCHS_SYSCTL_CEN_ENABLE << MMCHS_SYSCTL_CEN_SHIFT)
#define HS_MMCSD_CARDCLOCK_DISABLE      (MMCHS_SYSCTL_CEN_DISABLE << MMCHS_SYSCTL_CEN_SHIFT)

/*
** Macros that can be used for interrupt enable/disable and status get operations
*/
#define HS_MMCSD_INTR_BADACCESS         (MMCHS_IE_BADA_ENABLE)
#define HS_MMCSD_INTR_CARDERROR         (MMCHS_IE_CERR_ENABLE)
#define HS_MMCSD_INTR_ADMAERROR         (MMCHS_IE_ADMAE_ENABLE)
#define HS_MMCSD_INTR_ACMD12ERR         (MMCHS_IE_ACE_ENABLE)
#define HS_MMCSD_INTR_DATABITERR        (MMCHS_IE_DEB_ENABLE)
#define HS_MMCSD_INTR_DATACRCERR        (MMCHS_IE_DCRC_ENABLE)
#define HS_MMCSD_INTR_DATATIMEOUT       (MMCHS_IE_DTO_ENABLE)
#define HS_MMCSD_INTR_CMDINDXERR        (MMCHS_IE_CIE_ENABLE)
#define HS_MMCSD_INTR_CMDBITERR         (MMCHS_IE_CEB_ENABLE)
#define HS_MMCSD_INTR_CMDCRCERR         (MMCHS_IE_CCRC_ENABLE)
#define HS_MMCSD_INTR_CMDTIMEOUT        (MMCHS_IE_CTO_ENABLE)
#define HS_MMCSD_INTR_CARDINS           (MMCHS_IE_CINS_ENABLE)                    
#define HS_MMCSD_INTR_BUFRDRDY          (MMCHS_IE_BRR_ENABLE)                    
#define HS_MMCSD_INTR_BUFWRRDY          (MMCHS_IE_BWR_ENABLE)                    
#define HS_MMCSD_INTR_TRNFCOMP          (MMCHS_IE_TC_ENABLE)
#define HS_MMCSD_INTR_CMDCOMP           (MMCHS_IE_CC_ENABLE)

#define HS_MMCSD_STAT_BADACCESS         (MMCHS_STAT_BADA)
#define HS_MMCSD_STAT_CARDERROR         (MMCHS_STAT_CERR)
#define HS_MMCSD_STAT_ADMAERROR         (MMCHS_STAT_ADMAE)
#define HS_MMCSD_STAT_ACMD12ERR         (MMCHS_STAT_ACE)
#define HS_MMCSD_STAT_DATABITERR        (MMCHS_STAT_DEB)
#define HS_MMCSD_STAT_DATACRCERR        (MMCHS_STAT_DCRC)
#define HS_MMCSD_STAT_ERR				(MMCHS_STAT_ERRI)
#define HS_MMCSD_STAT_DATATIMEOUT       (MMCHS_STAT_DTO)
#define HS_MMCSD_STAT_CMDINDXERR        (MMCHS_STAT_CIE)
#define HS_MMCSD_STAT_CMDBITERR         (MMCHS_STAT_CEB)
#define HS_MMCSD_STAT_CMDCRCERR         (MMCHS_STAT_CCRC)
#define HS_MMCSD_STAT_CMDTIMEOUT        (MMCHS_STAT_CTO)
#define HS_MMCSD_STAT_CARDINS           (MMCHS_STAT_CINS)                    
#define HS_MMCSD_STAT_BUFRDRDY          (MMCHS_STAT_BRR)                    
#define HS_MMCSD_STAT_BUFWRRDY          (MMCHS_STAT_BWR)                    
#define HS_MMCSD_STAT_TRNFCOMP          (MMCHS_STAT_TC)
#define HS_MMCSD_STAT_CMDCOMP           (MMCHS_STAT_CC)

#define HS_MMCSD_SIGEN_BADACCESS        (MMCHS_ISE_BADA_SIGEN)
#define HS_MMCSD_SIGEN_CARDERROR        (MMCHS_ISE_CERR_SIGEN)
#define HS_MMCSD_SIGEN_ADMAERROR        (MMCHS_ISE_ADMAE_SIGEN)
#define HS_MMCSD_SIGEN_ACMD12ERR        (MMCHS_ISE_ACE_SIGEN)
#define HS_MMCSD_SIGEN_DATABITERR       (MMCHS_ISE_DEB_SIGEN)
#define HS_MMCSD_SIGEN_DATACRCERR       (MMCHS_ISE_DCRC_SIGEN)
#define HS_MMCSD_SIGEN_DATATIMEOUT      (MMCHS_ISE_DTO_SIGEN)
#define HS_MMCSD_SIGEN_CMDINDXERR       (MMCHS_ISE_CIE_SIGEN)
#define HS_MMCSD_SIGEN_CMDBITERR        (MMCHS_ISE_CEB_SIGEN)
#define HS_MMCSD_SIGEN_CMDCRCERR        (MMCHS_ISE_CCRC_SIGEN)
#define HS_MMCSD_SIGEN_CMDTIMEOUT       (MMCHS_ISE_CTO_SIGEN)
#define HS_MMCSD_SIGEN_CARDINS          (MMCHS_ISE_CINS_SIGEN)                    
#define HS_MMCSD_SIGEN_BUFRDRDY         (MMCHS_ISE_BRR_SIGEN)                    
#define HS_MMCSD_SIGEN_BUFWRRDY         (MMCHS_ISE_BWR_SIGEN)                    
#define HS_MMCSD_SIGEN_TRNFCOMP         (MMCHS_ISE_TC_SIGEN)
#define HS_MMCSD_SIGEN_CMDCOMP          (MMCHS_ISE_CC_SIGEN)



/*
** Function prototypes
*/

extern void MMCSDDataGet(unsigned int baseAddr, unsigned char *data, unsigned int len);
extern void MMCSDDataSend(unsigned int baseAddr, unsigned char *data, unsigned int len);
extern void MMCSDFIFOFill(unsigned int baseAddr);

extern void MMCSDBusWidthSet(unsigned int baseAddr, unsigned int width);
extern void MMCSDBlkLenSet(unsigned int baseAddr, unsigned int blklen);
extern void MMCSDDataTimeoutSet(unsigned int baseAddr, unsigned int timeout);
extern int MMCSDBusFreqSet(unsigned int baseAddr, unsigned int freq_in, unsigned int freq_out);
extern int MMCSDBusClock(unsigned int baseAddr, unsigned int pwr);

extern unsigned int MMCSDIsClockStable(unsigned int baseAddr, unsigned int retry);
extern unsigned int MMCSDIsCardInserted(unsigned int baseAddr);
extern unsigned int MMCSDIsCardWriteProtected(unsigned int baseAddr);

extern void MMCSDIntrEnable(unsigned int baseAddr, unsigned int flag);
extern unsigned int MMCSDIntrStatusGet(unsigned int baseAddr, unsigned int flag);
extern void MMCSDIntrStatusClear(unsigned int baseAddr, unsigned int flag);

extern unsigned int MMCSDIsResponseGet(unsigned int baseAddr, unsigned int *rsp);
extern unsigned int MMCSDIsXferComplete(unsigned int baseAddr, unsigned int retry);
extern unsigned int MMCSDIsDataComplete(unsigned int baseAddr, unsigned int retry);
extern unsigned int MMCSDIsCmdComplete(unsigned int baseAddr, unsigned int retry);
extern unsigned int MMCSDIsDataTXReady(unsigned int baseAddr, unsigned int retry,
		unsigned int *dataDone, unsigned int *status, unsigned int *transferDone, unsigned int *writeCRCerror);

extern unsigned int MMCSDIsDataRXReady(unsigned int baseAddr, unsigned int retry,
				unsigned int *dataDone, unsigned int *status, unsigned int *readCRCerror);

extern void MMCSDPlaceInReset(unsigned int baseAddr);
extern void MMCSDClearReset(unsigned int baseAddr);

extern void MMCSDFIFODirTX(unsigned int baseAddr);
extern void MMCSDFIFODirRX(unsigned int baseAddr);
extern void MMCSDCommandSend(unsigned int baseAddr, unsigned int cmd,
                               unsigned int cmdarg, void *data,
                               unsigned int nblks, unsigned int dmaEn);

//extern void MMCSDBusVoltSet(unsigned int baseAddr, unsigned int volt);
//extern int HSMMCSDBusPower(unsigned int baseAddr, unsigned int pwr);
// extern unsigned int HSMMCSDIsHSupported(unsigned int baseAddr);
//extern int HSMMCSDInitStreamSend(unsigned int baseAddr);
//extern int MMCSDLinesReset(unsigned int baseAddr, unsigned int flag);
//extern void MMCSDSupportedVoltSet(unsigned int baseAddr, unsigned int volt);
//extern void MMCSDSystemConfig(unsigned int baseAddr, unsigned int config);

#ifdef __cplusplus
}
#endif

#endif
