/*
 * nandwriter.c
*/

/*
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
*/
/* 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/* --------------------------------------------------------------------------
    FILE        : nandwriter.c 				                             	 	        
    PURPOSE     : NAND writer main program
    PROJECT     : DA8xx CCS NAND Flashing Utility (for use on DM355 EVM)
    AUTHOR      : Daniel Allred
    DESC	      : CCS-based utility to flash the DM644x in preparation for 
                  NAND booting
 ----------------------------------------------------------------------------- */

// This module's header file 
#include "nandwriter.h"


/************************************************************
* Explicit External Declarations                            *
************************************************************/


/************************************************************
* Local Macro Declarations                                  *
************************************************************/

//#define NANDWIDTH_16
#define NANDWIDTH_8

/************************************************************
* Local Typedef Declarations                                *
************************************************************/


/************************************************************
* Local Function Declarations                               *
************************************************************/

static Uint32 nandwriter(void);
static Uint32 LOCAL_writeData(NAND_InfoHandle hNandInfo, Uint8 *srcBuf, Uint32 totalPageCnt);


/************************************************************
* Global Variable Definitions
************************************************************/
Uint8 *data_array;

Uint8 *p_data_array_readback;

extern VUint32 __FAR__ DDRStart;
//extern VUint32 __FAR__ NANDStart;

// Global variables for page buffers 
static Uint8* gNandTx;
static Uint8* gNandRx;

Uint32 Error_Erase;
Uint32 Error_Write_Page;
Uint32 Error_Verify;


NAND_InfoHandle  hNandInfo;

/************************************************************
* Global Function Definitions                               *
************************************************************/

void main( void )
{
  int status;
  //
  // Init memory alloc pointer
  //
  UTIL_setCurrMemPtr(0);
  //
  // System init
  //
  if (DEVICE_init() !=E_PASS)
  {
    exit();
  }
  //
  // Execute the NAND flashing
  //
  status = nandwriter();

  if (status != E_PASS)
  {
    DEBUG_printString("\n\nNAND flashing failed!\r\n");
  }
  else
  {
    DEBUG_printString( "\n\nNAND boot preparation was successful!\r\n" );
  }
}


/************************************************************
* Local Function Definitions                                *
************************************************************/

static Uint32 nandwriter()
{
  FILE	*fPtr;
  Uint8	*aisPtr;
  Int32	aisFileSize = 0,aisAllocSize = 0;
  Uint32 numPagesAIS;
  Int32 numBytes;
  Uint32 filepos;
  Uint32 readsize;
  int i;
  //
  // Initialize NAND Flash
  //
  hNandInfo = NAND_open((Uint32)NANDStart, DEVICE_BUSWIDTH_8BIT );
  
  if (hNandInfo == NULL)
  {
    return E_FAIL;
  }

  if (NAND_globalErase(hNandInfo) != E_PASS)
  {
      return E_FAIL;
    }
  //
  // Read the const table of data OMAPL138-DSP-LED-NAND-TEST.bin, VibeCardFirmware0_7.ais
  //
  //fPtr = fopen("C:/Users/Lumen/DSP/VibeCardWorkspace/NANDWriter_ARM/Firmware/OMAPL138-DSP-LED-NAND-TEST.bin", "rb");
  
 // fPtr = fopen("C:/Users/Lumen/ARM/BootL138Test/ArmBoot/Debug/OMAPL138-DSP-LED-NAND-SPI-UART.bin", "rb");
  fPtr = fopen("C:/Users/Lumen/ARM/BootL138Test/ArmBoot/Release/ArmBoot.bin", "rb");

  if(fPtr == NULL)
  {
      DEBUG_printString("\tERROR: File Open Failed.\r\n");
 	  return E_FAIL;
  }
  if (0)
  {
  	fclose (fPtr);
 	return E_FAIL;
  }
  // Read file size
  fseek(fPtr,0,SEEK_END);
  aisFileSize = ftell(fPtr);
	
  if(aisFileSize == 0)
  {
  	fclose (fPtr);
    return E_FAIL;
  }

	  numPagesAIS = 0;
	  while ( (numPagesAIS * hNandInfo->dataBytesPerPage)  < aisFileSize )
	  {
		  numPagesAIS++;
	  }

     //We want to allocate an even number of pages.
    aisAllocSize = numPagesAIS * hNandInfo->dataBytesPerPage;

    // Setup pointer in RAM
    aisPtr = (Uint8 *) UTIL_allocMem(aisAllocSize);
    
    
     // Clear memory
	  for (i=0; i<aisAllocSize; i++)
	    aisPtr[i]=0xFF;
	//
    // Go to start of file
    //
    filepos = 0;
    fseek(fPtr,filepos,SEEK_SET);
	numBytes = aisFileSize;	//
    // Read file data
    //
    /*if (aisFileSize != fread(aisPtr, 1, aisFileSize, fPtr)) // was 1
    {
  		fclose (fPtr);
    	return E_FAIL;
    }*/
    readsize = 0x100;
    //
    // Loop
    //
    while (numBytes >= readsize)
    {
    	if (readsize != fread(&aisPtr[filepos], 1, readsize, fPtr)) // was 1
   	 	{
  			fclose (fPtr);
    		return E_FAIL;
    	}
    	filepos += readsize;
  		fseek(fPtr,filepos,SEEK_SET);	
		numBytes -= readsize;
    }
    
    if (numBytes != fread(&aisPtr[filepos], 1, numBytes, fPtr)) // was 1
    {
  		fclose (fPtr);
    	return E_FAIL;
    }

 	p_data_array_readback = (Uint8 *) UTIL_allocMem(aisAllocSize);
	for (i=0; i<aisAllocSize; i++)
	{
		p_data_array_readback[i] = aisPtr[i];
	}
	//
    // Close file
    //
	 fclose (fPtr);
 
//   hNandInfo = NAND_open((Uint32)NANDStart, DEVICE_BUSWIDTH_8BIT );
 
	//
    // Write the file data to the NAND flash
    //
	if (LOCAL_writeData(hNandInfo, (Uint8 *)p_data_array_readback, numPagesAIS) != E_PASS)
	{
		printf("\tERROR: Write failed.\r\n");
		return E_FAIL;
	}
	//
	//
	//
	for (i=0; i<aisAllocSize; i++)
	{
		p_data_array_readback[i] = 0;
	}
	//
	// Read back from flash from block 1
	//
	for (i=0; i<numPagesAIS; i++)
	{
		if (NAND_readPage(hNandInfo, 1, i, &p_data_array_readback[i*hNandInfo->dataBytesPerPage]) != E_PASS)
		{
			printf("\tERROR: Read Back failed.\r\n");
			break;			
		}
	}
	//
	// Check Reab Back vs Original
	//
	for (i=0; i<aisAllocSize; i++)
	{
		if (p_data_array_readback[i] != aisPtr[i])
		{
			printf("\tERROR: Check Read Back failed: %d.\r\n",i);			
			break;		
		}
	}

	return E_PASS;
}

// Generic function to write a UBL or Application header and the associated data
static Uint32 LOCAL_writeData(NAND_InfoHandle hNandInfo, Uint8 *srcBuf, Uint32 totalPageCnt)
{
  Uint32    blockNum,pageNum,pageCnt;
  Uint32    numBlks;
  Uint32    i;
  Uint8     *dataPtr;

  gNandTx = (Uint8 *) UTIL_allocMem(NAND_MAX_PAGE_SIZE);
  gNandRx = (Uint8 *) UTIL_allocMem(NAND_MAX_PAGE_SIZE);

  for (i=0; i<NAND_MAX_PAGE_SIZE; i++)  
  {
    gNandTx[i]=0xff;
	  gNandRx[i]=0xff;
  }	

	//
	// Dbug Var
	//
	Error_Erase = 0;
	Error_Write_Page = 0;
	Error_Verify = 0;
		
  // Get total number of blocks needed
  numBlks = 0;
  while ( (numBlks*hNandInfo->pagesPerBlock)  < totalPageCnt )
  {
    numBlks++;
  }


  //
  // Start in block 1 (leave block 0 alone)
  //
  blockNum = 1;

  // Unprotect all blocks of the device
//	if (NAND_unProtectBlocks(hNandInfo, blockNum, (hNandInfo->numBlocks-1)) != E_PASS)
	if (NAND_unProtectBlocks(hNandInfo, 0, (hNandInfo->numBlocks-1)) != E_PASS)
	{
		blockNum++;
//		DEBUG_printString("Unprotect failed.\r\n");
    return E_FAIL;
	}

    // Erase block 0
    NAND_eraseBlocks(hNandInfo,0,1);
   /* NAND_eraseBlocks(hNandInfo,1,1);
    NAND_eraseBlocks(hNandInfo,2,1);
    NAND_eraseBlocks(hNandInfo,3,1);
    NAND_eraseBlocks(hNandInfo,4,1);
    NAND_eraseBlocks(hNandInfo,5,1);

	while(1);*/
	
  while (blockNum < hNandInfo->numBlocks)
  {
    // Find first good block
    while (NAND_badBlockCheck(hNandInfo,blockNum) != E_PASS)
    {
      blockNum++;
    }

    // Erase the current block
    NAND_eraseBlocks(hNandInfo,blockNum,1);

	  // Start writing in page 0 of current block
    pageNum = 0;
    pageCnt = 0;

    // Setup data pointer
    dataPtr = srcBuf;

    // Start page writing loop
    do
    {
//      DEBUG_printString((Uint8 *)"Writing image data to block ");
//      DEBUG_printHexInt(blockNum);
 //     DEBUG_printString((Uint8 *)", page ");
 ////     DEBUG_printHexInt(pageNum);
  //    DEBUG_printString((Uint8 *)"\r\n");

		  // Write the AIS image data to the NAND device
     //if( NAND_writePage_ubl_header(hNandInfo, blockNum,  pageNum, dataPtr) != E_PASS)
      if (NAND_writePage(hNandInfo, blockNum,  pageNum, dataPtr) != E_PASS)
      {
      	Error_Write_Page++;
        //DEBUG_printString("Write failed. Marking block as bad...\n");
        NAND_reset(hNandInfo);
        NAND_badBlockMark(hNandInfo,blockNum);
        dataPtr -=  pageNum * hNandInfo->dataBytesPerPage;
        blockNum++;
		    continue;
      }
    
      UTIL_waitLoop(200);
		
      // Verify the page just written
      if (NAND_verifyPage(hNandInfo, blockNum, pageNum, dataPtr, gNandRx) != E_PASS)
      {
      	Error_Verify++;
        //DEBUG_printString("Verify failed. Marking block as bad...\n");
        NAND_reset(hNandInfo);
        NAND_badBlockMark(hNandInfo,blockNum);
        dataPtr -=  pageNum * hNandInfo->dataBytesPerPage;
        blockNum++;
		    continue;
      }
		
      pageNum++;
      pageCnt++;
      dataPtr +=  hNandInfo->dataBytesPerPage;
  
      if (pageNum == hNandInfo->pagesPerBlock)
      {
        // A block transition needs to take place; go to next good block
        do
        {
          blockNum++;
        }
        while (NAND_badBlockCheck(hNandInfo,blockNum) != E_PASS);

        // Erase the current block
        NAND_eraseBlocks(hNandInfo,blockNum,1);

        pageNum = 0;
		  }
	  } while (pageCnt < totalPageCnt);

    NAND_protectBlocks(hNandInfo);
    break;
  }
  if (Error_Write_Page)
  	return E_FAIL;
  
  if (Error_Verify)
   	return E_FAIL;
  
  
  return E_PASS;
}


/***********************************************************
* End file                                                 
************************************************************/


