/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2010
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** ============================================================================
 *
 *   @file  edmaIntDispatcher.h
 *
 *   @path  
 *
 *   @desc  Header file for EDMA interrupt example
 *
 * =============================================================================
 */
 
#ifndef _EDMAINTDISPATCHER_H_
#define _EDMAINTDISPATCHER_H_

#ifdef __cplusplus
extern "C" {
#endif

// if a shadow region is used, define the index value of that region, n
#define SHADOW_REGION	CSL_chipReadReg( CSL_CHIP_DNUM )

// select which region this EDMA dispatcher will be using, CSL_EDMA3_REGION_GLOBAL or a shadow CSL_EDMA3_REGION_n
//#define EDMA_REGION		SHADOW_REGION
#define EDMA_REGION		CSL_EDMA3_REGION_GLOBAL
/*
	The dispatcher can handle race conditions either of two ways. The race condition comes up when one or more EDMA
	interrupt bits are set in IPR at the start of this dispatcher, and then more come in after IPR is read and during
	the processing of the first set of interrupt bits. Without special handling, interrupts could get missed or
	never be activated again.


	1. Re-read IPR after clearing the processed bits. If any bits are set, repeat the processing until IPR == 0, 
	which will guarantee that any new bits will assert the interrupt flag. 
	
	Advantage: if it is likely that more IPR bits will be set after processing other IPR bits, then this loop 
	will save the overhead of another interrupt context save/restore.

	Disadvantage: 2 or 4 ConfigBus reads will occur at the end when no more IPR bits are left, and these are
	very slow reads.


	2. Assert IEVAL = 1 after processing all enabled IPR bits. This will guarantee that any recently-set IPR bits
	will properly assert the interrupt flag. 
	
	Advantage: if it is common for only one EDMA interrupt flag to be asserted at a time, this requires a single 
	ConfigBus write to IEVAL (which is very fast) and avoids four (4) ConfigBus reads of IPR/IPRH/IER/IERH 
	(which are very slow).

	Disadvantage: a race condition is unavoidable that could lead to a double pulse on the EDMAINT line to the
	CPU interrupt input. This can cause an indication of a dropped interrupt and assert the INTERR system 
	interrupt. It is the author's recommendation to mask out the EDMA interrupt from causing the IDROP/INTERR 
	event. This will have to be done outside of this routine, probably in main() or wherever the EDMA 
	system initialization is done.
*/
#define EDMADISPATCH_REREAD_IPR 1
#define EDMADISPATCH_IEVAL      2
#define EDMADISPATCH_METHOD     EDMADISPATCH_IEVAL


/* Macro for tcc handler */
#define InvokeHandle(num)       TccHandlerTable[num]()
typedef void (*EdmaTccHandler)(void);

/* Funtion which registers individual event handlers in a table */
void edmaEventHook(Uint16 ,EdmaTccHandler);

/* Forward declaration */
void edmaIntDispatcher();

#ifdef __cplusplus
}
#endif

#endif  /* _EDMAINTDISPATCHER_H_ */
