/****************************************************************************\
 *           Copyright (C) 2009 Texas Instruments Incorporated.             *
 *                           All Rights Reserved                            *
 *                                                                          *
 * GENERAL DISCLAIMER                                                       *
 * -------------------------------------------------------------------      *
 * All software and related documentation is provided "AS IS" and without   *
 * warranty or support of any kind and Texas Instruments expressly disclaims*
 * all other warranties, express or implied, including, but not limited to, *
 * the implied warranties of merchantability and fitness for a particular   *
 * purpose.  Under no circumstances shall Texas Instruments be liable for   *
 * any incidental, special or consequential damages that result from the    *
 * use or inability to use the software or related documentation, even if   *
 * Texas Instruments has been advised of the liability.                     *
 ****************************************************************************
 *                                                                          *
 * Written by :                                                             *
 *            Albert Bae                                                    *
 *            Texas Instruments                                             *
 *            01 Feb, 2011                                                 *
 *                                                                          *
 ***************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h> 

/* QMSS LLD include */
#include <ti/drv/qmss/qmss_drv.h>
#include <ti/drv/qmss/qmss_firmware.h>

/* Navigator LLD include */
#include <ti/drv/cppi/cppi_drv.h>
#include <ti/drv/cppi/cppi_desc.h>

/* CSL RL includes */
#include <ti/csl/csl_chip.h>
/* INTC CSL include */
#include <ti/csl/src/intc/csl_intc.h>
/* HyperLink CSL include */
#include "Hyperlink_config.h"
#include "psc_util.h"
   
/* Define queues for common FDQs */
#define MONO_TX_FDQ            2000
#define MONO_RX_FDQ            2001

#define NUM_MONO_DESC               32
#define SIZE_MONO_DESC             4112
#define SIZE_DATA_BUFFER           4096
#define MONOLITHIC_DESC_DATA_OFFSET  16 
#define NUM_PACKETS                  10

//#pragma DATA_SECTION(mono_region,".intData_sect")//use MSMC memory for test mode
#pragma DATA_ALIGN (mono_region, 16)
Uint8   mono_region[32 * 4112]; 

#pragma DATA_ALIGN (txDataBuff, 16)
Uint8   txDataBuff[SIZE_DATA_BUFFER];
Uint8   *rxBuffPtr;

#define iLoopModeOn //comment this line for non loopback mode

volatile CSL_Uint64 StartTime = 0, EndTime = 0;

Qmss_AccCmdCfg          cfg;
/* List address for accumulator - twice the number of entries for Ping and Pong page */
#pragma DATA_ALIGN (hiPrioList, 16)
Uint32  hiPrioList[(NUM_PACKETS + 1) * 2];

/* Intc variable declarartion */
CSL_IntcObj    intcObj[2];
CSL_IntcHandle   hIntc[2];
CSL_IntcEventHandlerRecord  EventHandler[2];
CSL_IntcGlobalEnableState state;

/* MNavigator handle for this test */
Cppi_Handle             cppiHnd;
Cppi_ChHnd              rxChHnd, txChHnd;
Qmss_QueueHnd           txQueHnd, rxQueHnd, freeQueHnd, txFreeQueHnd;
Cppi_FlowHnd            rxFlowHnd;
/* PKTDMA configuration */
Cppi_CpDmaInitCfg       cpdmaCfg;
/* Tx channel configuration */
Cppi_TxChInitCfg        txChCfg;
/* Rx channel configuration */
Cppi_RxChInitCfg        rxChCfg;
/* Rx flow configuration */
Cppi_RxFlowCfg          rxFlowCfg;
/* QMSS configuration */
Qmss_InitCfg            qmssInitConfig;
/* Memory region configuration information */
Qmss_MemRegInfo         monoMemInfo;
/* Core number */
Uint32                  coreNum;
Cppi_Desc               *monoDescPtr;

/* qmss and cppi global config parameters */
extern Qmss_GlobalConfigParams  qmssGblCfgParams;
extern Cppi_GlobalConfigParams  cppiGblCfgParams;

/* Global structures and variables  */
extern void CSL_tscEnable();
volatile unsigned int int10_result = 0;

interrupt void int10_isr(){
	int channel = 0; //accumulator channel num 
	Qmss_ackInterrupt (channel, 1);
    Qmss_setEoiVector (Qmss_IntdInterruptType_HIGH, channel);

    EndTime = DSP_ReadTime64();//check test end time
    
    /* Clear the Interrupt */
    CSL_intcHwControl(hIntc[0],CSL_INTC_CMD_EVTCLEAR,NULL);
    int10_result++;
}


void Intc_config(void)
{
   CSL_IntcParam    vectId;
   CSL_IntcContext  context;
   //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
   //! GEM0 Intc Configuration              !//
   //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!//
   /* Setup the global Interrupt */
   context.numEvtEntries = 8;    
   context.eventhandlerRecord = EventHandler; 
   CSL_intcInit(&context);
   /* Enable NMIs  */
   CSL_intcGlobalNmiEnable();
   /* Enable Global Interrupts  */
   CSL_intcGlobalEnable(&state);
  
   /* VectorID for the General Event  */
   vectId = CSL_INTC_VECTID_10;
   /* Opening a handle for the QMSS Hi priority accumulator Interrupt Event */                                      
   hIntc[0]   = CSL_intcOpen(&intcObj[0],
                           FIRST_HI_GEM_EVENT, // Hi priority GEM Event for Accumulator
                           &vectId,
                           NULL);
   //Hook the ISRs
   CSL_intcHookIsr(vectId,  &int10_isr);
   
   // Clear the Interrupt    
   CSL_intcHwControl(hIntc[0], CSL_INTC_CMD_EVTCLEAR,  NULL);
  
   //Enable the Event & the interrupt 
   CSL_intcHwControl(hIntc[0], CSL_INTC_CMD_EVTENABLE,  NULL);
	
}

void MNav_config(void){
    Qmss_Result             result;
    Uint32                  numAllocated;
    Uint8                   isAllocated;
    Cppi_DescCfg            descCfg;
    Qmss_Queue              queInfo;
    int i;
    /* Get the core number. */
    coreNum = CSL_chipReadReg(CSL_CHIP_DNUM); 
    /* Initialize the heap in shared memory for CPPI data structures */
    initCppiMem ();// init MNavigator memory
    initQmssMem ();// init QMSS memory
    
    memset ((void *) &qmssInitConfig, 0, sizeof (Qmss_InitCfg));
    /* Use Internal LinkRAM  */
    qmssInitConfig.linkingRAM0Base = 0;
    qmssInitConfig.linkingRAM0Size = 0;
    qmssInitConfig.linkingRAM1Base = 0;
    qmssInitConfig.maxDescNum      = NUM_MONO_DESC*2;
    /* Use PDSP firmware for little endian */
    qmssInitConfig.pdspFirmware[0].pdspId = Qmss_PdspId_PDSP1;
    qmssInitConfig.pdspFirmware[0].firmware = &acc48_le;
    qmssInitConfig.pdspFirmware[0].size = sizeof (acc48_le);
    
    /* Initialize Queue Manager SubSystem */
    result = Qmss_init (&qmssInitConfig, &qmssGblCfgParams);
    if (result != QMSS_SOK)
    {
        printf ("Error Core %d : Initializing Queue Manager SubSystem error code : %d\n", coreNum, result);
        return;
    }
    /* Start Queue Manager SubSystem */
    result = Qmss_start ();
    if (result != QMSS_SOK)
    printf ("Core %d : Error starting Queue Manager error code : %d\n", coreNum, result);

    /* Initialize MNavigator LLD */
    result = Cppi_init (&cppiGblCfgParams);
    if (result != CPPI_SOK)
    printf ("Error Core %d : Initializing CPPI LLD error code : %d\n", coreNum, result);

    /* Set up QMSS PKTDMA configuration */
    memset ((void *) &cpdmaCfg, 0, sizeof (Cppi_CpDmaInitCfg));
    cpdmaCfg.dmaNum = Cppi_CpDma_QMSS_CPDMA;
    cpdmaCfg.qm1BaseAddress = 0x34030000;//local QM area 
    cpdmaCfg.qm2BaseAddress = 0x40000000;// vUSR remote QM area 
    
    /* Open QMSS PKTDMA */
    cppiHnd = (Cppi_Handle) Cppi_open (&cpdmaCfg);
    if (cppiHnd == NULL)
    {
        printf ("Error Core %d : Initializing QMSS CPPI PKTPDMA %d\n", coreNum, cpdmaCfg.dmaNum);
        return;
    }

    /* Setup memory region 0 for local QM */
    memset ((void *) &mono_region, 0, SIZE_MONO_DESC * NUM_MONO_DESC);
    monoMemInfo.descBase = (Uint32 *) &mono_region[0];
    monoMemInfo.descSize = SIZE_MONO_DESC;
    monoMemInfo.descNum = NUM_MONO_DESC;
    monoMemInfo.manageDescFlag = Qmss_ManageDesc_MANAGE_DESCRIPTOR;
    monoMemInfo.memRegion = Qmss_MemRegion_MEMORY_REGION0;
    monoMemInfo.startIndex = 0;

    result = Qmss_insertMemoryRegion (&monoMemInfo);
    if (result < QMSS_SOK)
    printf ("Error Core %d : Inserting memory region %d error code : %d\n", coreNum, monoMemInfo.memRegion, result);
    
    /* Setup memory region 1 for remote QM */
    //memset ((void *) 0x40200000, 0, SIZE_MONO_DESC * NUM_MONO_DESC);
    monoMemInfo.descBase = (Uint32 *) 0x40200000;//remote DSP memory region base in vUSR
    monoMemInfo.descSize = SIZE_MONO_DESC;
    monoMemInfo.descNum = NUM_MONO_DESC;
    monoMemInfo.manageDescFlag = Qmss_ManageDesc_MANAGE_DESCRIPTOR;
    monoMemInfo.memRegion = Qmss_MemRegion_MEMORY_REGION1;
    monoMemInfo.startIndex = 32;

    result = Qmss_insertMemoryRegion (&monoMemInfo);
    if (result < QMSS_SOK)
    printf ("Error Core %d : Inserting memory region %d error code : %d\n", coreNum, monoMemInfo.memRegion, result);
    
    /* Setup the descriptors for transmit free queue */
    memset(&descCfg, 0, sizeof(descCfg));
    descCfg.memRegion = Qmss_MemRegion_MEMORY_REGION0;
    descCfg.descNum = NUM_MONO_DESC/2;// 16 descriptors for this test
    descCfg.destQueueNum = MONO_TX_FDQ;
    descCfg.queueType = Qmss_QueueType_GENERAL_PURPOSE_QUEUE;
    descCfg.initDesc = Cppi_InitDesc_INIT_DESCRIPTOR;
    descCfg.descType = Cppi_DescType_MONOLITHIC;
    descCfg.epibPresent = Cppi_EPIB_NO_EPIB_PRESENT;
    descCfg.cfg.mono.dataOffset = MONOLITHIC_DESC_DATA_OFFSET;
    /* Descriptor should be recycled back to Queue Number 2000 */
    descCfg.returnQueue.qMgr = 0;
    descCfg.returnQueue.qNum = MONO_TX_FDQ;
    
    /* Initialize the descriptors and push to free Queue */
    if ((txFreeQueHnd = Cppi_initDescriptor (&descCfg, &numAllocated)) < 0)
	{
        printf ("Error Core %d : Initializing Tx descriptor error code: %d \n", coreNum, txFreeQueHnd);
		return;
	}

    /* Setup the descriptors for Region 1 Rx free queue */
    memset(&descCfg, 0, sizeof(descCfg));
    descCfg.memRegion = Qmss_MemRegion_MEMORY_REGION1;
    descCfg.descNum = NUM_MONO_DESC/2; // 16 descriptors fot this test
    descCfg.destQueueNum = MONO_RX_FDQ;
    descCfg.queueType = Qmss_QueueType_GENERAL_PURPOSE_QUEUE;
    descCfg.initDesc = Cppi_InitDesc_INIT_DESCRIPTOR;
    descCfg.descType = Cppi_DescType_MONOLITHIC;
    descCfg.epibPresent = Cppi_EPIB_NO_EPIB_PRESENT;
    descCfg.cfg.mono.dataOffset = MONOLITHIC_DESC_DATA_OFFSET;
    descCfg.returnQueue.qMgr = QMSS_PARAM_NOT_SPECIFIED;
    descCfg.returnQueue.qNum = QMSS_PARAM_NOT_SPECIFIED;
    
    /* Initialize the descriptors and push to free Queue */
    if ((freeQueHnd = Cppi_initDescriptor (&descCfg, &numAllocated)) < 0)
	{
        printf ("Error Core %d : Initializing Rx descriptor error code: %d \n", coreNum, freeQueHnd);
		return;
	}
	
    /* Set up QMSS PKTDMA Rx Channel parameters */
    rxChCfg.channelNum = 0;// PKTDMA channel 0 
    rxChCfg.rxEnable = Cppi_ChState_CHANNEL_DISABLE;
    
    /* Open Rx Channel */
    rxChHnd = (Cppi_ChHnd) Cppi_rxChannelOpen (cppiHnd, &rxChCfg, &isAllocated);
    if (rxChHnd == NULL)
    {
        printf ("Error Core %d : Opening Rx channel : %d\n", coreNum, rxChCfg.channelNum);
        return;
    }
    
    /* Set up Tx Channel parameters */
    txChCfg.channelNum = 0;// PKTDMA channel 0
    txChCfg.priority = 0;
    txChCfg.filterEPIB = 0;
    txChCfg.filterPS = 0;
    txChCfg.aifMonoMode = 0;
    txChCfg.txEnable = Cppi_ChState_CHANNEL_DISABLE;
    
    /* Open Tx Channel */
    txChHnd = (Cppi_ChHnd) Cppi_txChannelOpen (cppiHnd, &txChCfg, &isAllocated);
    if (txChHnd == NULL)
    {
        printf ("Error Core %d : Opening Tx channel : %d\n", coreNum, txChCfg.channelNum);
        return;
    }
    
    /* Opens transmit queue. This is infrastructure Tx queue */
    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_INFRASTRUCTURE_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
	{
        printf ("Error Core %d : Opening Transmit Queue Number\n", coreNum);
		return;
	}
    else
        printf ("Core %d : Transmit Queue Number : %d opened\n", coreNum, txQueHnd);

    /* Opens receive queue (General purpose queue)*/
    if ((rxQueHnd = Qmss_queueOpen (Qmss_QueueType_HIGH_PRIORITY_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
	{
        printf ("Error Core %d : Opening Receive Queue Number\n", coreNum);
        return;
	}
    else
        printf ("Core %d : Receive Queue Number : %d opened\n", coreNum, rxQueHnd);

    printf ("Core %d : Rx Free Queue Number : %d opened\n", coreNum, freeQueHnd);
    printf ("Core %d : Transmit Free Queue Number : %d opened\n", coreNum, txFreeQueHnd);

    Qmss_configureAccTimer (Qmss_PdspId_PDSP1, 3500);// configure accumulator timer delay to 20 us

    // program the high priority accumulator 
    memset ((void *) &hiPrioList, 0, sizeof (hiPrioList));
    cfg.channel = 0;
    cfg.command = Qmss_AccCmd_ENABLE_CHANNEL;
    cfg.queueEnMask = 0;
    cfg.listAddress = (Uint32) hiPrioList; // Should be global if reading on another core
    // Get queue manager and queue number from handle 
    queInfo = Qmss_getQueueNumber (rxQueHnd);
    cfg.queMgrIndex = queInfo.qNum;
    cfg.maxPageEntries = NUM_PACKETS + 1 ;
    cfg.timerLoadCount = 0;
    cfg.interruptPacingMode = Qmss_AccPacingMode_NONE;
    cfg.listEntrySize = Qmss_AccEntrySize_REG_D;
    cfg.listCountMode = Qmss_AccCountMode_ENTRY_COUNT;
    cfg.multiQueueMode = Qmss_AccQueueMode_SINGLE_QUEUE;
    
    if ((result = Qmss_programAccumulator (Qmss_PdspId_PDSP1, &cfg)) != QMSS_ACC_SOK)
	{
       printf ("Error Core %d : Programming high priority accumulator for channel : %d queue : %d error code : %d\n",
                        coreNum, cfg.channel, cfg.queMgrIndex, result);
		return;
	}
	
    /* Setup Rx flow parameters */
    memset ((void *) &rxFlowCfg, 0, sizeof (Cppi_RxFlowCfg));
    rxFlowCfg.flowIdNum = 0;// flow ID number should be matched with AIF2 DB channel number
    /* Get queue manager and queue number from handle */
    queInfo = Qmss_getQueueNumber (rxQueHnd);
    rxFlowCfg.rx_dest_qnum = queInfo.qNum;
    rxFlowCfg.rx_dest_qmgr = 2;//QM2 for Rx
    rxFlowCfg.rx_sop_offset = MONOLITHIC_DESC_DATA_OFFSET;
    rxFlowCfg.rx_desc_type = Cppi_DescType_MONOLITHIC; 
    /* Get queue manager and queue number from handle */
    queInfo = Qmss_getQueueNumber (freeQueHnd);
    rxFlowCfg.rx_fdq0_sz0_qnum = queInfo.qNum; 
    rxFlowCfg.rx_fdq0_sz0_qmgr = 2;//QM2 for Rx

    /* Configure Rx flow */
    rxFlowHnd = (Cppi_FlowHnd) Cppi_configureRxFlow (cppiHnd, &rxFlowCfg, &isAllocated);
    if (rxFlowHnd == NULL)
    {
        printf ("Error Core %d : Opening Rx flow : %d\n", coreNum, rxFlowCfg.flowIdNum);
        return;
    }
  
    
    /* Fill in some data in Tx buffer */
    for (i = 0; i < SIZE_DATA_BUFFER; i++) 
        txDataBuff[i] = i;
    
    /***** Send out 10 packets. pop 10 Tx free descriptors and push into the AIF2 Tx Queue  ***************/
    for (i = 0; i < NUM_PACKETS; i++)
    {
        /* Get a free descriptor */
        if ((monoDescPtr = (Cppi_Desc *) Qmss_queuePop (txFreeQueHnd)) == NULL)
        {
           printf ("Error Core %d : Getting descriptor from Queue Number %d \n", coreNum, txFreeQueHnd);
           return;
        }   
        /* Add data into the descriptor payload */
        Cppi_setData (Cppi_DescType_MONOLITHIC, monoDescPtr, (Uint8 *) &txDataBuff, SIZE_DATA_BUFFER);
     
        /* Set mono packet length */
        Cppi_setPacketLen (Cppi_DescType_MONOLITHIC, monoDescPtr, SIZE_DATA_BUFFER);//packet length is 512 byte

        /* Push descriptor to QMSS Tx queue */   
        Qmss_queuePushDescSize (txQueHnd, (Uint32 *) monoDescPtr, 16);  
    }    
    
    /* Enable transmit and receive channel */
    Cppi_channelEnable (txChHnd);
    Cppi_channelEnable (rxChHnd);
    
    StartTime = DSP_ReadTime64(); //check test start time
    //next step : verify data through remote DSP packet accumulator
        
}


void Hyperlink_config(void)
{
	
    /************ Initialize Hyperlink ************************************************/ 
    
	/* Start Hyperlink module Serdes */
    Configure_vUSR_SerDes();
    
	/*	Enable all four lanes */
    *((int *)CSL_MCM_LPMC) |= 0x4;//0x4 quad lane, 0x2 single lane
    //*((int *)CSL_MCM_LPMC) &= 0xFFFFFFA;// single lane
    *((int *)CSL_MCM_LPMC) &= 0xFFFFFFC; // quad lane
    
#ifdef iLoopModeOn
    vUSR_InternalLoopBackEnable(); //internal loopback mode on
#else
    vUSR_NonLoopBack_LinkOnWait(); //wait until the vUSR in other DSP is turned on
#endif

    /**** local vUSR Tx, Rx address mapping for remote DSP 
    * 0x40000000 => 0x34020000 and 0x40200000 ==> 0x0C010000  **/   
    configure_address_mappings(0xD, 4,(Uint32)0x34020000,(Uint32)0x0C010000);
}


void Hyperlink_Close(void)
{
   vUSR_ModuleReset();//reset vUSR module
}

void Test_result(void){
	Uint32 i,result,destLen,base,index,count, test_pass = 1,Tput = 0, Clock =0;
	Cppi_Desc               *rxPkt;
	Qmss_Queue              queInfo;
    
    /* Read Entry count in accumulator */
    base = 0; //get data from only ping buffer
    count = hiPrioList[base];
    if(count == 0)test_pass = 0;
    hiPrioList[base] = 0;
    for (index = base + 1; index <= base + count; index++)
    {
        rxPkt = (Cppi_Desc *) QMSS_DESC_PTR (hiPrioList[index]);
        hiPrioList[index] = 0;

        /* Get data buffer */
        Cppi_getData (Cppi_DescType_MONOLITHIC, rxPkt, &rxBuffPtr, &destLen);

        /* Compare */
        for (i = 0; i < destLen; i++)
        {
            if (txDataBuff[i] != rxBuffPtr[i])
            test_pass = 0;
        }
    
        /* Recycle the Rx host descriptors */
        queInfo.qMgr = 0;
        queInfo.qNum = MONO_RX_FDQ;
        /* Push descriptor back to Rx free queue */
        Qmss_queuePushDesc (Qmss_getQueueHandle(queInfo), (Uint32 *) rxPkt);
    }
    
    if (test_pass == 1)
      printf(" Test a) Packet Data Send/Recv: PASS\n");
    else
      printf(" Test a) Packet Data Send/Recv: FAIL\n");

    result = Qmss_getQueueEntryCount (txQueHnd);
    if (result != 0) printf(" Test b1) Mono Packet Tx Descriptor Counts:%d FAIL\n",result);
    else printf(" Test b1) Mono Packet Tx Descriptor Counts:%d PASS\n",result);
    
    result = Qmss_getQueueEntryCount (txFreeQueHnd);
    if (result != 16) printf(" Test b2) Mono Packet Tx Free Descriptor Counts:%d FAIL\n",result);
    else printf(" Test b2) Mono Packet Tx Free Descriptor Counts:%d PASS\n",result);
    
    result = Qmss_getQueueEntryCount (freeQueHnd);
    if (result != 16) printf(" Test b3) Mono Packet Rx Free Descriptor Counts:%d FAIL\n",result);
    else printf(" Test b3) Mono Packet Rx Free Descriptor Counts:%d PASS\n",result);

    result = Qmss_getQueueEntryCount (rxQueHnd);
    if (result != 0) printf(" Test b4) Mono Packet Rx Descriptor Counts:%d FAIL\n",result);
    else printf(" Test b4) Mono Packet Rx Descriptor Counts:%d PASS\n",result);
    
    Clock = (Uint32)(EndTime - StartTime);//check consumed time
    Tput = (4096*8*10)/Clock;
    printf("Data throughput is %d Gbps \n",Tput);
    
    printf("Ending Hyperlink CPPI complex throughput test\n");
}

void Intc_Close(void)
{ 
   CSL_intcHwControl(hIntc[0], CSL_INTC_CMD_EVTDISABLE, NULL);//Disable the Event & the interrupt 
   CSL_intcGlobalDisable(&state);// Disable Global Interrupts  
   CSL_intcClose(hIntc[0]); // Close handles 
}

void MNav_Close(void){
	
	/* Close Tx, Rx channel */
    Cppi_channelClose (txChHnd);
    Cppi_channelClose (rxChHnd);
    /* Close Rx flow */
    Cppi_closeRxFlow (rxFlowHnd);
    //disable accumulator
    Qmss_disableAccumulator (Qmss_PdspId_PDSP1, cfg.channel);
	
    /* Close the queues */
    Qmss_queueClose (rxQueHnd);
    Qmss_queueClose (txQueHnd);
    Qmss_queueClose (freeQueHnd);
    Qmss_queueClose (txFreeQueHnd);
      
    /* Close AIF2 PKTDMA instance */
    Cppi_close (cppiHnd);
    /* Deinitialize Navigator LLD */
    Cppi_exit ();
}

void main(void)
{

    printf("\nBeginning Hyperlink CPPI complex throughput test:\n");
    
    enable_module(mcm_pdctl, mcm_mdctl);//Enable Hyperlink module power
    CSL_tscEnable(); /* Enable the TSC */    
    Intc_config();
    
    memset(mono_region, 0xFF, sizeof(mono_region));
    memset(txDataBuff, 0x00, sizeof(txDataBuff));
    
    Hyperlink_config();//hyperlink configuration for Cppi mode
    
    MNav_config();//Navigator and PKTDMA configuration
    
     
    /* Wait for completion. */
    while(1)
    {
        asm (" NOP 9 ");
	    asm (" NOP 9 ");
	    
        if(int10_result > 0)//Wait until accumulator is filled with packets
        {
            break;
        }
    }
    
    Test_result(); //compare Tx, Rx data

    MNav_Close();
    
    Hyperlink_Close();
        
    Intc_Close();
}

