/******************************************************************************
*              	 File Description
*	        	 ================
*
*   File Name		:	hyperlink_functions.c
*
*   Description		:	Common functions for MCM(Hyperlink).
*
*******************************************************************************/
#include <stdio.h>
#include <ti/csl/cslr.h>
#include <ti/csl/cslr_tpcc.h>
#include <ti/csl/csl_edma3.h>
#include <ti/csl/csl_edma3Aux.h>
#include <ti/csl/csl_cache.h>
#include "Hyperlink_config.h"

#define btcfg_cfg_base  0x02620000 //Chip level registers base

CSL_Edma3Handle hModule, hModule1, hModule2;
CSL_Edma3ParamHandle paramHandle0;
CSL_Edma3ParamHandle paramHandle1;

CSL_Edma3ChannelAttr  chParam;
CSL_Edma3ChannelObj  ChObj0;
CSL_Edma3ChannelHandle  hChannel0;
CSL_Edma3ChannelObj  ChObj1;
CSL_Edma3ChannelHandle  hChannel1;
CSL_Edma3HwDmaChannelSetup chSetup;
CSL_Edma3ParamSetup paramSetup; 
CSL_Edma3Obj  moduleObj;
CSL_Edma3CmdIntr regionIntr,regionIpr;
CSL_Status EdmaStat, handleStatus;

static int mcm_reg_wr(volatile int *reg, int val)
{
	*reg = val;
	return PASS;
}

static int mcm_reg_rd(volatile int *reg)
{
	int val;
	 val = *reg;
	 return val;
}

void write_local_reg (unsigned int offset, unsigned int bit_position)
{
    unsigned int wr_val;
    unsigned int i = 0x1;
    
	wr_val = ((mcm_reg_rd((volatile int *)(CSL_MCM_CONFIG_REGS + offset))) | (i << bit_position));

    mcm_reg_wr(((volatile int *)(CSL_MCM_CONFIG_REGS + offset)),wr_val);
}


void Configure_vUSR_SerDes( void )
{
  int val;
  
  //unlock device control configuration area to setup Hyperlink SERDES (TI internal registers)
  mcm_reg_wr((volatile int *)(btcfg_cfg_base+0x38),0x83e70b13); //KICK0 key setup
  mcm_reg_wr((volatile int *)(btcfg_cfg_base+0x3c),0x95a4f1e0); //KICK1 key setup

  /*** Hyper bridge SERDES configuration (User may use SRIO SERDES register map for this)*****/ 
  /* Configure SERDES PLL 
  CFGPLL[0] ENPLL = 0 (VUSR overrides this value)
  CFGPLL[8:1] MPY = 40(10x)for 312.50MHz input clock, 80(20x)for 156.25MHz, 50(12.5x) for 250MHz to get max speed 3.125GHz
  CFGPLL[9] VRANGE = 1 for full rate and half rate. 0 for quad rate and eighth rate  
  CFGPLL[10] SLEEPPLL = 0
  CFGPLL[12:11] LOOP BANDWIDTH = 0   CFGPLL[14:13] sCLKBYP = 00 no Bypass */
  mcm_reg_wr((volatile int *)(btcfg_cfg_base+VUSR_SERDES_CFGPLL), 0x250);

  /* Configure CFGRX
  CFGRX [0] ENRX = 1 Don't care (Overwritten automatically by VUSR)  
  CFGRX [3:1] BUSWIDTH = 2
  CFGRX [5:4] RATE = 0 full rate (4 bit per clock) 3.125GHz * 4 = 12.5Gbps max speed per lane  
  CFGRX [6] INVPAIR = 0  CFGRX 
  [9:7] TERM =  1 (1 for AC couple and 6 for DC couple)
  CFGRX [11:10] ALIGN = 01 Don't care (Overwritten by VUSR in run time)  
  CFGRX [14:12] LOS = 100 (vUSR automatically sets to 0 for loopback. User must set to 4 for non-loopack)
  CFGRX [17:15] CDR = 101      
  CFGRX [20:18] EQ = 001 
  CFGRX [21] EQHLD = 0     
  CFGRX [22] ENOC = 1   
  CFGRX [24:23] LOOPBACK = 00 Automatically configured by vUSR HW
  CFGRX [27:25] TESTPATTERN = 000  */
  mcm_reg_wr((volatile int *)(btcfg_cfg_base+VUSR_SERDES_CFGRX0), 0x0046C485);
  mcm_reg_wr((volatile int *)(btcfg_cfg_base+VUSR_SERDES_CFGRX1), 0x0046C485);
  mcm_reg_wr((volatile int *)(btcfg_cfg_base+VUSR_SERDES_CFGRX2), 0x0046C485);
  mcm_reg_wr((volatile int *)(btcfg_cfg_base+VUSR_SERDES_CFGRX3), 0x0046C485);
  
  /* Configure CFGTX 
  CFGTX [0] ENTX = 1 Don't care (Overwritten automatically by VUSR)   
  CFGTX [3:1] BUSWIDTH = 2
  CFGTX [5:4] RATE = 0 full rate (4 bit per clock) 3.125GHz * 4 = 12.5Gbps max speed per lane  
  CFGTX [6] INVPAIR = 0  
  CFGTX [10:7] SWING = 1110b  7 for short trace length and 15 for long trace length
  CFGTX [13:11] TWPRE = 001b Precursor Tap weight. (-2.5%)
  CFGTX [18:14] TWPST1 = 18(10010b) Adjacent Post Cursor Tap Weight 
  If trace length is 4 or less, use 23 (-10%). If trace length is between 4 and 10, use 27 (-27.5%).
  CFGTX [19] FIRUPT = 1   Transmitter pre and post cursor FIR filter update    
  CFGTX [20] MSYNC = 1 (vUSR will automatically set this value)     
  CFGTX [22:21] LOOPBACK = 00 (vUSR will automatically set this field)
  CFGTX [25:23] TESTPATTERN = 000  */
  mcm_reg_wr((volatile int *)(btcfg_cfg_base+VUSR_SERDES_CFGTX0), 0x001C8F05 );
  mcm_reg_wr((volatile int *)(btcfg_cfg_base+VUSR_SERDES_CFGTX1), 0x001C8F05 );
  mcm_reg_wr((volatile int *)(btcfg_cfg_base+VUSR_SERDES_CFGTX2), 0x001C8F05 );
  mcm_reg_wr((volatile int *)(btcfg_cfg_base+VUSR_SERDES_CFGTX3), 0x001C8F05 );
  
  //check for PLL lock
  val =  mcm_reg_rd((volatile int *)(btcfg_cfg_base+STS_VUSR));
  while(!(val & 1))
  {
  	val =  mcm_reg_rd((volatile int *)(btcfg_cfg_base+STS_VUSR));
  }
  	
}


void vUSR_InternalLoopBackEnable()
{
  int val;
  write_local_reg(4, 2);
  write_local_reg(4, 1);

  val = mcm_reg_rd((volatile int *)CSL_MCM_CR);
  val = val & 0xFFFFFFFB;
  mcm_reg_wr(((volatile int *)CSL_MCM_CR), val);

}

void vUSR_NonLoopBack_LinkOnWait()
{
  int val, retryCount = 0;
  uint64_t wait_start, time_waited;
  
  /* Wait for other peripheral to come up */
  wait_start = DSP_ReadTime64();
  do {
    val = mcm_reg_rd((volatile int *)CSL_MCM_SR);
    val = val & 0x00003001;
  
    time_waited = (DSP_ReadTime64() - wait_start);
    if ((! time_waited) || (time_waited > 1000000000)) {

    printf ("Waiting for other side to come up (%8d)\n", retryCount);
    wait_start = DSP_ReadTime64();
    retryCount++;
    }
  } while (val != 0x1);
	
}

void vUSR_ModuleReset()
{
  int val,i;
  write_local_reg(4, 2);
  write_local_reg(4, 0);//reset

  for(i=0;i<100;i++)asm(" NOP 5 ");
  val = mcm_reg_rd((volatile int *)CSL_MCM_CR);
  val = val & 0xFFFFFFFA;
  mcm_reg_wr(((volatile int *)CSL_MCM_CR), val);
  
   volatile Uint32 *pi;
   // Empty queues (to avoid not working the 2nd time)... */
   for (pi = (volatile Uint32 *)0x02a20000; pi < (volatile Uint32 *)(0x02a20000 + 0x20000); pi += 4)
   {
      pi[2] = 0;
      pi[3] = 0;
   }
}


void CheckLinkStatus(void)
{
    while ((mcm_reg_rd((volatile int *)CSL_MCM_SR) & (0x1)) != 0x1);//check link enable
	
    while ((mcm_reg_rd((volatile int *)CSL_MCM_SR) & (0xF000)) != 0x0);//check serial halt and pll unlock
}	


void configure_address_mappings(Uint32 Txigmask, Uint32 rxsegsel, Uint32 RxBufAddr1, Uint32 RxBufAddr2)
{

	Uint32 rxAddrOffset1 = 0, rxAddrOffset2 = 0;
	Uint32 rxlen_val;
	
	rxAddrOffset1 = RxBufAddr1;
	rxAddrOffset2 = RxBufAddr2;
		
	mcm_reg_wr((volatile int *)CSL_MCM_TXADDOVLCTL, Txigmask);//configure Tx address overlay (PrivID = 0, Sec = 0)
	CheckLinkStatus();
	
    //Rx Address Selector Control (PrivID = 0, Sec = 0)
    if(rxsegsel == 0x1)rxlen_val = 16;
    else if(rxsegsel == 0x2)rxlen_val = 17;
    else if(rxsegsel == 0x3)rxlen_val = 18;
    else if(rxsegsel == 0x4)rxlen_val = 19;
    
	mcm_reg_wr((volatile int *)CSL_MCM_RXADDSELCTL, rxsegsel);
	
    //Remote Rx Adress PrivID Index
	mcm_reg_wr((volatile int *)CSL_MCM_RXADDPRVIDX, 0x0);

    //Remote Rx Adress PrivID Value Register
	mcm_reg_wr((volatile int *)CSL_MCM_RXADDPRVVAL, 0x0);

    //Remote Rx Adress Segment Index Register for remote QM
	mcm_reg_wr((volatile int *)CSL_MCM_RXADDSEGIDX, 0x0);
	  	
    //Rx Adress Segment Value Register for remote QM
	mcm_reg_wr((volatile int *)CSL_MCM_RXADDSEGVAL, ((rxAddrOffset1 & 0xFFFF0000)+rxlen_val));
	 
	//Remote Rx Adress Segment Index Register for remote MSMC
	mcm_reg_wr((volatile int *)CSL_MCM_RXADDSEGIDX, 0x2);
	  	
    //Rx Adress Segment Value Register for remote MSMC
	mcm_reg_wr((volatile int *)CSL_MCM_RXADDSEGVAL, ((rxAddrOffset2 & 0xFFFF0000)+rxlen_val));
	
}

void configure_remote_address_mappings(Uint32 Txigmask, Uint32 rxsegsel, Uint32 RxBufAddr1, Uint32 RxBufAddr2)
{

	Uint32 rxAddrOffset1 = 0, rxAddrOffset2 = 0;
	Uint32 rxlen_val;
	
	rxAddrOffset1 = RxBufAddr1;
	rxAddrOffset2 = RxBufAddr2;
		
	mcm_reg_wr((volatile int *)CSL_RMCM_TXADDOVLCTL, Txigmask);//configure Tx address overlay (PrivID = 0, Sec = 0)
	
    //Rx Address Selector Control (PrivID = 0, Sec = 0)
    if(rxsegsel == 0x1)rxlen_val = 16;
    else if(rxsegsel == 0x2)rxlen_val = 17;
    else if(rxsegsel == 0x3)rxlen_val = 18;
    else if(rxsegsel == 0x4)rxlen_val = 19;
    
	mcm_reg_wr((volatile int *)CSL_RMCM_RXADDSELCTL, rxsegsel);
	
    //Remote Rx Adress PrivID Index
	mcm_reg_wr((volatile int *)CSL_RMCM_RXADDPRVIDX, 0x0);

    //Remote Rx Adress PrivID Value Register
	mcm_reg_wr((volatile int *)CSL_RMCM_RXADDPRVVAL, 0x0);

    //Remote Rx Adress Segment Index Register for remote QM
	mcm_reg_wr((volatile int *)CSL_RMCM_RXADDSEGIDX, 0x0);
	  	
    //Rx Adress Segment Value Register for remote QM
	mcm_reg_wr((volatile int *)CSL_RMCM_RXADDSEGVAL, ((rxAddrOffset1 & 0xFFFF0000)+rxlen_val));
	 
	//Remote Rx Adress Segment Index Register for remote MSMC
	mcm_reg_wr((volatile int *)CSL_RMCM_RXADDSEGIDX, 0x2);
	  	
    //Rx Adress Segment Value Register for remote MSMC
	mcm_reg_wr((volatile int *)CSL_RMCM_RXADDSEGVAL, ((rxAddrOffset2 & 0xFFFF0000)+rxlen_val));
	
}


void configure_address_mappings_EDMA(Uint32 Txigmask, Uint32 rxadrsize1, Uint32 RxAddrOffset)
{

	Uint32 rxadroffset1 = 0x0;
	Uint32 rxregsel,rxlen_val;
	if(((Uint32)RxAddrOffset & 0xFFF00000) == 0x00800000) 
		rxadroffset1 = 0x10000000 + (Uint32)RxAddrOffset;//core 0
	else
		rxadroffset1 = RxAddrOffset;
		
	mcm_reg_wr((int *)CSL_MCM_TXADDOVLCTL, Txigmask);
	CheckLinkStatus();
    //Rx Address Selector Control
    if (rxadrsize1 < 0x1FFFF)
      {
      rxregsel = 0x1;
      rxlen_val = 16;
      }
    else if (rxadrsize1 < 0x3FFFF)
      {
      rxregsel = 0x2;
      rxlen_val = 17;
      }
    else if (rxadrsize1 < 0x7FFFF)
      {
      rxregsel = 0x3;
      rxlen_val = 18;
      }
    else if (rxadrsize1 < 0xFFFFF)
      {
      rxregsel = 0x4;
      rxlen_val = 19;
      }
	mcm_reg_wr((int *)CSL_RMCM_RXADDSELCTL, rxregsel);
	
  //Remote Rx Adress PrivID Index
	mcm_reg_wr((int *)CSL_RMCM_RXADDPRVIDX, 0x0);

  //Remote Rx Adress PrivID Value Register
	mcm_reg_wr((int *)CSL_RMCM_RXADDPRVVAL, 0x0);

  //Remote Rx Adress Segment Index Register
	mcm_reg_wr((int *)CSL_RMCM_RXADDSEGIDX, 0x0);
	  	
  //Rx Adress Segment Value Register
	mcm_reg_wr((int *)CSL_RMCM_RXADDSEGVAL, ((rxadroffset1 & 0xFFFFF000)+rxlen_val));
		
}

void trigger_edma_rd_channel()
{
	   	CSL_edma3HwChannelControl(hChannel0,CSL_EDMA3_CMD_CHANNEL_SET,NULL);
}
void trigger_edma_wr_channel()
{
	CSL_edma3HwChannelControl(hChannel1,CSL_EDMA3_CMD_CHANNEL_SET,NULL);
}


void vusr_edma_rd_setup_all_payload(Uint32 srcBuf, Uint32 dstBuf, Uint32 payload, Uint32 channel_num)
{
    // EDMA Module Initialization
	CSL_edma3Init(NULL);
 
 	// EDMA Module Open	
    hModule = CSL_edma3Open(&moduleObj,CSL_TPCC_0,NULL,&EdmaStat);

	chParam.regionNum  = CSL_EDMA3_REGION_GLOBAL;
	//chSetup.que        = CSL_EDMA3_QUE_0;
	chParam.chaNum     = channel_num;

    /* NySh Tx Channel */
	hChannel0 = CSL_edma3ChannelOpen(&ChObj0, CSL_TPCC_0, &chParam, &EdmaStat);
	chSetup.paramNum   = chParam.chaNum;
    CSL_edma3HwChannelSetupParam(hChannel0,chSetup.paramNum);
	paramHandle0            = CSL_edma3GetParamHandle(hChannel0,channel_num,&EdmaStat);

    paramSetup.aCntbCnt     = CSL_EDMA3_CNT_MAKE(payload*4,1);       		
	paramSetup.srcDstBidx   = CSL_EDMA3_BIDX_MAKE(4,4 );  
	paramSetup.srcDstCidx   = CSL_EDMA3_CIDX_MAKE(0,0);     		
	paramSetup.cCnt         = 1;
	paramSetup.option       = CSL_EDMA3_OPT_MAKE(FALSE,FALSE,FALSE,TRUE,channel_num,CSL_EDMA3_TCC_NORMAL, \
	      CSL_EDMA3_FIFOWIDTH_NONE,FALSE,CSL_EDMA3_SYNC_AB,CSL_EDMA3_ADDRMODE_INCR,CSL_EDMA3_ADDRMODE_INCR);
		      
	paramSetup.srcAddr      = (Uint32)srcBuf;
	if( ((Uint32)dstBuf & 0xFFF00000) == 0x00800000) 
		paramSetup.dstAddr      = (Uint32)(0x10000000 + (Uint32)dstBuf);
	else
		paramSetup.dstAddr      = (Uint32)dstBuf;	
	paramSetup.linkBcntrld  = CSL_EDMA3_LINKBCNTRLD_MAKE(CSL_EDMA3_LINK_NULL,0);     

	CSL_edma3ParamSetup(paramHandle0,&paramSetup);
	CSL_edma3HwChannelControl(hChannel0,CSL_EDMA3_CMD_CHANNEL_ENABLE,NULL);
	trigger_edma_rd_channel();
}

void vusr_edma_wr_setup_all_payload(Uint32 srcBuf, Uint32 dstBuf, Uint32 payload, Uint32 channel_num)
{
    // EDMA Module Initialization
	CSL_edma3Init(NULL);
 
 	// EDMA Module Open	
    hModule = CSL_edma3Open(&moduleObj,CSL_TPCC_0,NULL,&EdmaStat);

	chParam.regionNum  = CSL_EDMA3_REGION_GLOBAL;
    //chSetup.que        = CSL_EDMA3_QUE_0;
	chParam.chaNum     = channel_num;

    /* NySh Tx Channel */
	hChannel1 = CSL_edma3ChannelOpen(&ChObj1, CSL_TPCC_0, &chParam, &EdmaStat);
	chSetup.paramNum   = chParam.chaNum; 
    CSL_edma3HwChannelSetupParam(hChannel1,chSetup.paramNum);
	paramHandle1            = CSL_edma3GetParamHandle(hChannel1,channel_num,&EdmaStat);

    paramSetup.aCntbCnt     = CSL_EDMA3_CNT_MAKE(payload*4,1);       		
	paramSetup.srcDstBidx   = CSL_EDMA3_BIDX_MAKE(4,4 );  
	paramSetup.srcDstCidx   = CSL_EDMA3_CIDX_MAKE(0,0);     		
	paramSetup.cCnt         = 1;
	paramSetup.option       = CSL_EDMA3_OPT_MAKE(FALSE,FALSE,FALSE,TRUE,channel_num,CSL_EDMA3_TCC_NORMAL, \
	      CSL_EDMA3_FIFOWIDTH_NONE,FALSE,CSL_EDMA3_SYNC_AB,CSL_EDMA3_ADDRMODE_INCR,CSL_EDMA3_ADDRMODE_INCR);
	
	if( ((Uint32)srcBuf & 0xFFF00000) == 0x00800000)
		paramSetup.srcAddr      = (Uint32)(0x10000000 + (Uint32)srcBuf);
	else
		paramSetup.srcAddr      = (Uint32)srcBuf;

		paramSetup.dstAddr      = (Uint32)dstBuf;
	paramSetup.linkBcntrld  = CSL_EDMA3_LINKBCNTRLD_MAKE(CSL_EDMA3_LINK_NULL,0);     

	CSL_edma3ParamSetup(paramHandle1,&paramSetup);
	CSL_edma3HwChannelControl(hChannel1,CSL_EDMA3_CMD_CHANNEL_ENABLE,NULL);
	trigger_edma_wr_channel();
}

void verify_edma_complete(int ch_num)
{
    Uint32 val;
    regionIpr.region  = CSL_EDMA3_REGION_GLOBAL;
	regionIpr.intr    = 0;
	regionIpr.intrh   = 0;
	
	val = 1 << (ch_num);
	
	do{
		CSL_edma3GetHwStatus(hModule,CSL_EDMA3_QUERY_INTRPEND,&regionIpr);
	}while ((regionIpr.intr & val) != val);// channel 8 ~
}
		 




