/* ======================================================================== */
/*  TEXAS INSTRUMENTS, INC.                                                 */
/*                                                                          */
/*  \file aet_counters.c                                                    */
/*                                                                          */
/*  \brief AET common support functions                                     */
/*                                                                          */
/*  \date  2nd Apr 2004                                                     */
/*                                                                          */
/*  \author Oliver Sohm                                                     */
/*                                                                          */
/*  DESCRIPTION                                                             */
/*                                                                          */
/*  LAST MODIFIED                                                           */
/*       4th Nov 2004                                                       */
/*                                                                          */
/*  MODIFICATIONS 1                                                         */
/*      - Oliver Sohm                                                       */
/*      - Created                                                           */
/*      - Date:  4th Nov 2004                                               */
/*                                                                          */
/* ------------------------------------------------------------------------ */
/*             Copyright (c) 2004 Texas Instruments, Incorporated.          */
/*                            All Rights Reserved.                          */
/* ======================================================================== */

#include <aet_counters_iar.h>
#include <aet_counters_mmr.h>
#include <aet_counters.h>

/* ------------------------------------------------------------------------ */
/*  Claim AET                                                               */
/* ------------------------------------------------------------------------ */
void AET_claim()       
{
  AET_MMR_PID = CSL_FMKT(AET_MMR_PID_WRT_STATE, CLAIM); 
}

/* ------------------------------------------------------------------------ */
/*  Enable AET                                                              */
/* ------------------------------------------------------------------------ */
void AET_enable()       
{
  AET_MMR_PID = CSL_FMKT(AET_MMR_PID_WRT_STATE, ENABLE); 
}

/* ------------------------------------------------------------------------ */
/*  Read indirect AET register                                              */
/*  ireg:  index of register to be read                                     */
/*  Return value: contents of register                                      */
/* ------------------------------------------------------------------------ */
unsigned int AET_iregRead(unsigned int ireg)
{
  AET_MMR_IAR_ADD = ireg;

  return (AET_MMR_IAR_DAT);
}

/* ------------------------------------------------------------------------ */
/*  Write indirect AET register                                             */
/*  ireg:  index of register to be written                                  */
/*  val:   value to be written to register                                  */
/* ------------------------------------------------------------------------ */
void AET_iregWrite(unsigned int ireg, unsigned int val)
{
  AET_MMR_IAR_ADD = ireg;

  AET_MMR_IAR_DAT = val;
}

/* ------------------------------------------------------------------------ */
/*  Configure counters for event counting                                   */
/*                                                                          */
/*  input_set   :selects the input set that contains the desired signal(s), */
/*               0 - Memory Events, 1 - Stalls, 2 - Misc Events             */
/*  input_mask_0:32-bit field where each bit corresponds to one of the 32   */
/*               inputs of the AEGs. If multiple bits are set on OR-type    */
/*               AEGs, then the output will be the OR of the chosen input   */
/*               signals. For counter 0.                                    */
/*  input_mask_1:For counter 1.                                             */ 
/* ------------------------------------------------------------------------ */
void AET_countEvents(int input_set, unsigned int input_mask_0, unsigned int input_mask_1)
{
    /* ----------------------------------------- */
    /* Configure Counters                        */
    /* ----------------------------------------- */

    /* set counter reload value to max (counter is decrementing) */
    AET_iregWrite(AET_CNT_RLD0, 0xFFFFFFFF); /* counter 0 */
    AET_iregWrite(AET_CNT_RLD1, 0xFFFFFFFF); /* counter 1 */

    /* configure counter modes */
    AET_iregWrite(AET_FUNC_CNTL, 
                  /* counter 0 */
                  CSL_FMKT(AET_FUNC_CNTL_CTM0, EVENT)   /* event mode */
                | CSL_FMKT(AET_FUNC_CNTL_CTC0, RUN)     /* don't stop on HALT */
                | CSL_FMKT(AET_FUNC_CNTL_CTE0, ENABLE)  /* enable counter */
                  /* counter 1 (programmed through bitfields #2) */
                | CSL_FMKT(AET_FUNC_CNTL_CTM2, EVENT)   /* event mode */
                | CSL_FMKT(AET_FUNC_CNTL_CTC2, RUN)     /* don't stop on HALT */
                | CSL_FMKT(AET_FUNC_CNTL_CTE2, ENABLE)  /* enable counter */
    );

    /* ----------------------------------------- */
    /* Configure AEGs                            */
    /* ----------------------------------------- */
    /* configure input signal set (memory, stall or misc), affects both counters */
    AET_iregWrite(AET_CMPI_SEL, CSL_FMK(AET_CMPI_SEL_OAE, input_set));

    /* configure AEG 1 mode */
    AET_iregWrite(AET_AE_CNTL1,                         
                  CSL_FMKT(AET_AE_CNTL_EP0, OUT0)   /* configure output sense and polarity */
                | CSL_FMKT(AET_AE_CNTL_EP1, OUT0)
                | CSL_FMKT(AET_AE_CNTL_EP2, OUT1)
                | CSL_FMKT(AET_AE_CNTL_EP3, OUT1)
                | CSL_FMKT(AET_AE_CNTL_EP4, OUT0)
                | CSL_FMKT(AET_AE_CNTL_EP5, OUT0)
                | CSL_FMKT(AET_AE_CNTL_EP6, OUT1)
                | CSL_FMKT(AET_AE_CNTL_EP7, OUT1) 
                | CSL_FMKT(AET_AE_CNTL_ALIGN, DISABLE) /* no S&H of events */
                | CSL_FMKT(AET_AE_CNTL_EDGE, LEVEL)    /* detect levels */
                | CSL_FMKT(AET_AE_CNTL_SYNC, DISABLE)  /* no sync */
    );

    /* configure AEG 1 input */
    AET_iregWrite(AET_AE_ENA1, input_mask_0); 

    /* configure AEG 2 mode */
    AET_iregWrite(AET_AE_CNTL2,                         
                  CSL_FMKT(AET_AE_CNTL_EP0, OUT0)   /* configure output sense and polarity */
                | CSL_FMKT(AET_AE_CNTL_EP1, OUT0)
                | CSL_FMKT(AET_AE_CNTL_EP2, OUT1)
                | CSL_FMKT(AET_AE_CNTL_EP3, OUT1)
                | CSL_FMKT(AET_AE_CNTL_EP4, OUT0)
                | CSL_FMKT(AET_AE_CNTL_EP5, OUT0)
                | CSL_FMKT(AET_AE_CNTL_EP6, OUT1)
                | CSL_FMKT(AET_AE_CNTL_EP7, OUT1) 
                | CSL_FMKT(AET_AE_CNTL_ALIGN, DISABLE) /* no S&H of events */
                | CSL_FMKT(AET_AE_CNTL_EDGE, LEVEL)    /* detect levels */
                | CSL_FMKT(AET_AE_CNTL_SYNC, DISABLE)  /* no sync */
    );

    /* configure AEG 2 input */
    AET_iregWrite(AET_AE_ENA2, input_mask_1); 
}

/* ------------------------------------------------------------------------ */
/*  Read counter 0                                                          */
/* ------------------------------------------------------------------------ */
int AET_counter0Read()
{
  return (AET_iregRead(AET_CNT_VAL0));
}

/* ------------------------------------------------------------------------ */
/*  Read counter 1                                                          */
/* ------------------------------------------------------------------------ */
int AET_counter1Read()
{
  return (AET_iregRead(AET_CNT_VAL1));
}

#if 0
/* ------------------------------------------------------------------------ */
/*  Set PC range for counters                                               */
/*                                                                          */
/*  Both counters shall start counting when PC>=start_pc and stop counting  */
/*  when PC>end_pc.                                                         */
/* ------------------------------------------------------------------------ */
void AET_countersSetRange(unsigned int start_pc, unsigned int end_pc)
{
 /* "exclusive" count - requires PC comparator */
}

/* ------------------------------------------------------------------------ */
/*  Start counting                                                          */
/*                                                                          */
/*  counter   : 0 - start counter 0                                         */
/*              1 - start counter 1                                         */
/*              ALL - start both counters simultaneously                    */
/* ------------------------------------------------------------------------ */
void AET_countersStart(int counter)
{
  /* "inclusive" count - this would require the state machine */
}

/* ------------------------------------------------------------------------ */
/*  Stop counting                                                           */
/*                                                                          */
/*  counter   : 0 - stop counter 0                                          */
/*              1 - stop counter 1                                          */
/*              ALL - stop both counters simultaneously                     */
/* ------------------------------------------------------------------------ */
void AET_countersStop(int counter)
{
  /* "inclusive" count - this would require the state machine */
}

/* ------------------------------------------------------------------------ */
/*  Reset counters                                                          */
/*                                                                          */
/*  counter   : 0 - reset counter 0                                         */
/*              1 - reste counter 1                                         */
/*              ALL - reset both counters simultaneously                    */
/* ------------------------------------------------------------------------ */
void AET_countersReset(int counter)
{

}
#endif


/*****************************************************************************\
* Setup AET:- Closes all the channels/modules etc.
******************************************************************************/

 




/* ======================================================================== */
/*             Copyright (c) 2004 Texas Instruments, Incorporated.          */
/*                            All Rights Reserved.                          */
/* ======================================================================== */


