/**************************************************************************************************
  Filename:       masterSlaveSwitch.c
  Revised:        $Date: 2010-08-06 08:56:11 -0700 (Fri, 06 Aug 2010) $
  Revision:       $Revision: 23333 $

  Description:    This file contains the Master Slave Switch sample application
                  for use with the CC2540 Bluetooth Low Energy Protocol Stack.

  Copyright 2010 - 2012 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/

/*********************************************************************
 * INCLUDES
 */

#include "bcomdef.h"
#include "OSAL.h"
#include "OSAL_PwrMgr.h"

#include "OnBoard.h"
#include "hal_adc.h"
#include "hal_led.h"
#include "hal_key.h"
#include "hal_lcd.h"

#include "gatt.h"

#include "hci.h"

#include "gapgattserver.h"
#include "gattservapp.h"
#include "devinfoservice.h"

#include "central.h"
#include "peripheral.h"
#include "gapbondmgr.h"

#include "masterSlaveSwitch.h"

// Proprietary services

#include "exampleService.h"


/*********************************************************************
 * MACROS
 */

/*********************************************************************
 * CONSTANTS
 */
// What is the advertising interval when device is discoverable (units of 625us, 160=100ms)
#define DEFAULT_ADVERTISING_INTERVAL          160

// Whether to enable automatic parameter update request when a connection is formed
#define DEFAULT_ENABLE_UPDATE_REQUEST         FALSE

// Limited discoverable mode advertises for 30.72s, and then stops
// General discoverable mode advertises indefinitely

#define DEFAULT_DISCOVERABLE_MODE             GAP_ADTYPE_FLAGS_GENERAL

// Minimum connection interval (units of 1.25ms, 80=100ms) if automatic parameter update request is enabled
#define DEFAULT_DESIRED_MIN_CONN_INTERVAL     80

// Maximum connection interval (units of 1.25ms, 800=1000ms) if automatic parameter update request is enabled
#define DEFAULT_DESIRED_MAX_CONN_INTERVAL     800

// Slave latency to use if automatic parameter update request is enabled
#define DEFAULT_DESIRED_SLAVE_LATENCY         0

// Supervision timeout value (units of 10ms, 1000=10s) if automatic parameter update request is enabled
#define DEFAULT_DESIRED_CONN_TIMEOUT          1000

#define INVALID_CONNHANDLE                    0xFFFF

// Length of bd addr as a string
#define B_ADDR_STR_LEN                        15

#if defined ( PLUS_BROADCASTER )
  #define ADV_IN_CONN_WAIT                    500 // delay 500 ms
#endif

/***************************** CENTRAL defines ***********************/

// Default service discovery timer delay in ms
#define DEFAULT_SVC_DISCOVERY_DELAY           1000

// TRUE to filter discovery results on desired service UUID
#define DEFAULT_DEV_DISC_BY_SVC_UUID          FALSE

#define WANTED_SERVICE_UUID                   0xABBA

// Maximum number of scan responses
#define DEFAULT_MAX_SCAN_RES                  8

// Scan duration in ms
#define DEFAULT_SCAN_DURATION                 4000

// Discovey mode (limited, general, all)
#define DEFAULT_DISCOVERY_MODE                DEVDISC_MODE_ALL

// TRUE to use active scan
#define DEFAULT_DISCOVERY_ACTIVE_SCAN         TRUE

// TRUE to use white list during discovery
#define DEFAULT_DISCOVERY_WHITE_LIST          FALSE

// TRUE to use high scan duty cycle when creating link
#define DEFAULT_LINK_HIGH_DUTY_CYCLE          FALSE

// TRUE to use white list when creating link
#define DEFAULT_LINK_WHITE_LIST               FALSE

// Default RSSI polling period in ms
#define DEFAULT_RSSI_PERIOD                   1000



/*********************************************************************
 * TYPEDEFS
 */
typedef enum {
  ROLE_PERIPHERAL = 1,
  ROLE_CENTRAL    = 2
} deviceRole_t;


typedef enum {
  GAPROLE_CENTRAL_UNITIALIZED,
  GAPROLE_CENTRAL_INIT_DONE,
  GAPROLE_CENTRAL_SCANNING,
  GAPROLE_CENTRAL_CONNECTING,
  GAPROLE_CENTRAL_CONNECTED,
  GAPROLE_CENTRAL_DISCONNECTING,
  GAPROLE_CENTRAL_DISCONNECTED
} gapCentralRole_States_t;

// Discovery states
typedef enum
{
  BLE_DISC_STATE_IDLE,                // Idle
  BLE_DISC_STATE_SVC,                 // Service discovery
  BLE_DISC_STATE_CHAR                 // Characteristic discovery
} gapCentralDiscStates_t;

/*********************************************************************
 * GLOBAL VARIABLES
 */

/*********************************************************************
 * EXTERNAL VARIABLES
 */

/*********************************************************************
 * EXTERNAL FUNCTIONS
 */

/*********************************************************************
 * LOCAL VARIABLES
 */
static uint8 masterSlaveSwitch_TaskID;   // Task ID for internal task/event processing

static gaprole_States_t             gapPeripheralState = GAPROLE_INIT;
static gapCentralRole_States_t      gapCentralState = GAPROLE_CENTRAL_UNITIALIZED;

// GAP - SCAN RSP data (max size = 31 bytes)
static uint8 scanRspData[] =
{
  // complete name
  14,   // length of this data
  GAP_ADTYPE_LOCAL_NAME_COMPLETE,
    'M',  'a',  's',  't',  'e',  'r',  'S',  'l',  'a',  'v',  'e',  'S',  'w',
  // connection interval range
  0x05,   // length of this data
  GAP_ADTYPE_SLAVE_CONN_INTERVAL_RANGE,
  LO_UINT16( DEFAULT_DESIRED_MIN_CONN_INTERVAL ),   // 100ms
  HI_UINT16( DEFAULT_DESIRED_MIN_CONN_INTERVAL ),
  LO_UINT16( DEFAULT_DESIRED_MAX_CONN_INTERVAL ),   // 1s
  HI_UINT16( DEFAULT_DESIRED_MAX_CONN_INTERVAL ),

  // Tx power level
  0x02,   // length of this data
  GAP_ADTYPE_POWER_LEVEL,
  0       // 0dBm
};

// GAP - Advertisement data (max size = 31 bytes, though this is
// best kept short to conserve power while advertisting)
static uint8 advertData[] =
{
  // Flags; this sets the device to use limited discoverable
  // mode (advertises for 30 seconds at a time) instead of general
  // discoverable mode (advertises indefinitely)
  0x02,   // length of this data
  GAP_ADTYPE_FLAGS,
  DEFAULT_DISCOVERABLE_MODE | GAP_ADTYPE_FLAGS_BREDR_NOT_SUPPORTED,

  // service UUID, to notify central devices what services are included
  // in this peripheral
  0x03,   // length of this data
  GAP_ADTYPE_16BIT_MORE,      // some of the UUID's, but not all
  LO_UINT16( DEVINFO_SERV_UUID ), // Device Info. Completely un-interesting
  HI_UINT16( DEVINFO_SERV_UUID ),
};

// GAP GATT Attributes
static uint8 attDeviceName[GAP_DEVICE_NAME_LEN] = "Master Slave Switch";

// Initial State
static deviceRole_t deviceRole = ROLE_PERIPHERAL;

// Connection handle of active connection
static uint16 connHandle = INVALID_CONNHANDLE;


/**************** Variables concerning the Central mode ***************/
/**********************************************************************/
// Number of scan results and scan result index
static uint8 simpleBLEScanRes;
static uint8 simpleBLEScanIdx;

// Scan result list
static gapDevRec_t simpleBLEDevList[DEFAULT_MAX_SCAN_RES];

// Scanning state
static uint8 simpleBLEScanning = FALSE;

// RSSI polling state
static uint8 simpleBLERssi = FALSE;

// Connection handle of current connection 
static uint16 simpleBLEConnHandle = GAP_CONNHANDLE_INIT;

// Discovery state
static uint8 simpleBLEDiscState = BLE_DISC_STATE_IDLE;

// Discovered service start and end handle
static uint16 simpleBLESvcStartHdl = 0;
static uint16 simpleBLESvcEndHdl = 0;

// Discovered characteristic handle
static uint16 simpleBLECharHdl = 0;

// Value to write
static uint8 simpleBLECharVal = 0;

// Value read/write toggle
static bool simpleBLEDoWrite = FALSE;

// GATT read/write procedure state
static bool simpleBLEProcedureInProgress = FALSE;



/*********************************************************************
 * LOCAL FUNCTIONS
 */
static void masterSlaveSwitch_ProcessOSALMsg( osal_event_hdr_t *pMsg );
static void peripheralStateNotificationCB( gaprole_States_t newState );
static void masterSlaveSwitch_HandleKeys( uint8 shift, uint8 keys );

static void simpleBLECentralRssiCB( uint16 connHandle, int8  rssi );
static void simpleBLECentralEventCB( gapCentralRoleEvent_t *pEvent );

static void exampleServiceChangeCB( uint8 paramID );
char *bdAddr2Str( uint8 *pAddr );

/*********************************************************************
 * PROFILE CALLBACKS
 */

// GAP Role Callbacks PERIPHERAL
static gapRolesCBs_t masterSlaveSwitch_PeripheralCBs =
{
  peripheralStateNotificationCB,  // Profile State Change Callbacks
  NULL                            // When a valid RSSI is read from controller (not used by application)
};

// GAP Bond Manager Callbacks
static gapBondCBs_t masterSlaveSwitch_BondMgrCBs =
{
  NULL,                     // Passcode callback (not used by application)
  NULL                      // Pairing / Bonding state Callback (not used by application)
};

// GAP Role Callbacks CENTRAL
static const gapCentralRoleCB_t simpleBLERoleCB =
{
  simpleBLECentralRssiCB,       // RSSI callback
  simpleBLECentralEventCB       // Event callback
};


// Example Service Callbacks
static exampleServiceCBs_t masterSlaveSwitch_exampleServiceCBs =
{
  exampleServiceChangeCB    // Charactersitic value change callback
};


/*********************************************************************
 * PUBLIC FUNCTIONS
 */

/*********************************************************************
 * @fn      MasterSlaveSwitch_Init
 *
 * @brief   Initialization function for the Master Slave Switch App Task.
 *          This is called during initialization and should contain
 *          any application specific initialization (ie. hardware
 *          initialization/setup, table initialization, power up
 *          notificaiton ... ).
 *
 * @param   task_id - the ID assigned by OSAL.  This ID should be
 *                    used to send messages and set timers.
 *
 * @return  none
 */
void MasterSlaveSwitch_Init( uint8 task_id )
{
  masterSlaveSwitch_TaskID = task_id;

  // Setup the GAP Peripheral Role Profile
  {

    // For other hardware platforms, device starts advertising upon initialization
    uint8 initial_advertising_enable = FALSE; // We set this to FALSE to bypass peripheral.c's auto-start

    // By setting this to zero, the device will go into the waiting state after
    // being discoverable for 30.72 second, and will not being advertising again
    // until the enabler is set back to TRUE
    uint16 gapRole_AdvertOffTime = 0;

    uint8 enable_update_request = DEFAULT_ENABLE_UPDATE_REQUEST;
    uint16 desired_min_interval = DEFAULT_DESIRED_MIN_CONN_INTERVAL;
    uint16 desired_max_interval = DEFAULT_DESIRED_MAX_CONN_INTERVAL;
    uint16 desired_slave_latency = DEFAULT_DESIRED_SLAVE_LATENCY;
    uint16 desired_conn_timeout = DEFAULT_DESIRED_CONN_TIMEOUT;

    // Set the GAP Role Parameters
    GAPRole_SetParameter( GAPROLE_ADVERT_ENABLED, sizeof( uint8 ), &initial_advertising_enable );
    GAPRole_SetParameter( GAPROLE_ADVERT_OFF_TIME, sizeof( uint16 ), &gapRole_AdvertOffTime );

    GAPRole_SetParameter( GAPROLE_SCAN_RSP_DATA, sizeof ( scanRspData ), scanRspData );
    GAPRole_SetParameter( GAPROLE_ADVERT_DATA, sizeof( advertData ), advertData );

    GAPRole_SetParameter( GAPROLE_PARAM_UPDATE_ENABLE, sizeof( uint8 ), &enable_update_request );
    GAPRole_SetParameter( GAPROLE_MIN_CONN_INTERVAL, sizeof( uint16 ), &desired_min_interval );
    GAPRole_SetParameter( GAPROLE_MAX_CONN_INTERVAL, sizeof( uint16 ), &desired_max_interval );
    GAPRole_SetParameter( GAPROLE_SLAVE_LATENCY, sizeof( uint16 ), &desired_slave_latency );
    GAPRole_SetParameter( GAPROLE_TIMEOUT_MULTIPLIER, sizeof( uint16 ), &desired_conn_timeout );
  }

  // Set the GAP Characteristics
  GGS_SetParameter( GGS_DEVICE_NAME_ATT, GAP_DEVICE_NAME_LEN, attDeviceName );

  // Set advertising interval
  {
    uint16 advInt = DEFAULT_ADVERTISING_INTERVAL;

    GAP_SetParamValue( TGAP_LIM_DISC_ADV_INT_MIN, advInt );
    GAP_SetParamValue( TGAP_LIM_DISC_ADV_INT_MAX, advInt );
    GAP_SetParamValue( TGAP_GEN_DISC_ADV_INT_MIN, advInt );
    GAP_SetParamValue( TGAP_GEN_DISC_ADV_INT_MAX, advInt );
  }

  // Setup the GAP Bond Manager
  {
    uint32 passkey = 0; // passkey "000000"
    uint8 pairMode = GAPBOND_PAIRING_MODE_WAIT_FOR_REQ;
    uint8 mitm = TRUE;
    uint8 ioCap = GAPBOND_IO_CAP_DISPLAY_ONLY;
    uint8 bonding = TRUE;
    GAPBondMgr_SetParameter( GAPBOND_DEFAULT_PASSCODE, sizeof ( uint32 ), &passkey );
    GAPBondMgr_SetParameter( GAPBOND_PAIRING_MODE, sizeof ( uint8 ), &pairMode );
    GAPBondMgr_SetParameter( GAPBOND_MITM_PROTECTION, sizeof ( uint8 ), &mitm );
    GAPBondMgr_SetParameter( GAPBOND_IO_CAPABILITIES, sizeof ( uint8 ), &ioCap );
    GAPBondMgr_SetParameter( GAPBOND_BONDING_ENABLED, sizeof ( uint8 ), &bonding );
  }

  // Initialize GATT attributes
  GGS_AddService( GATT_ALL_SERVICES );            // GAP
  GATTServApp_AddService( GATT_ALL_SERVICES );    // GATT attributes
  DevInfo_AddService();                           // Device Information Service

  // Proprietary service
  
  ExampleService_AddService();
  ExampleService_RegisterAppCBs( &masterSlaveSwitch_exampleServiceCBs );
  
  // Receive key presses
  RegisterForKeys( masterSlaveSwitch_TaskID );
  // Enable clock divide on halt
  // This reduces active current while radio is active and CC254x MCU
  // is halted
  HCI_EXT_ClkDivOnHaltCmd( HCI_EXT_ENABLE_CLK_DIVIDE_ON_HALT );

#if defined ( DC_DC_P0_7 )
  // Enable stack to toggle bypass control on TPS62730 (DC/DC converter)
  HCI_EXT_MapPmIoPortCmd( HCI_EXT_PM_IO_PORT_P0, HCI_EXT_PM_IO_PORT_PIN7 );
#endif // defined ( DC_DC_P0_7 )

  // Setup a delayed profile startup
  osal_set_event( masterSlaveSwitch_TaskID, MSS_START_DEVICE_EVT );
  
  // Setup Central Profile
  {
    uint8 scanRes = DEFAULT_MAX_SCAN_RES;
    GAPCentralRole_SetParameter ( GAPCENTRALROLE_MAX_SCAN_RES, sizeof( uint8 ), &scanRes );
  }
  
  // Initialize GATT Client
  VOID GATT_InitClient();
}

/*********************************************************************
 * @fn      MasterSlaveSwitch_ProcessEvent
 *
 * @brief   Master Slave Switch Application Task event processor.  This function
 *          is called to process all events for the task.  Events
 *          include timers, messages and any other user defined events.
 *
 * @param   task_id  - The OSAL assigned task ID.
 * @param   events - events to process.  This is a bit map and can
 *                   contain more than one event.
 *
 * @return  events not processed
 */
uint16 MasterSlaveSwitch_ProcessEvent( uint8 task_id, uint16 events )
{

  VOID task_id; // OSAL required parameter that isn't used in this function

  if ( events & SYS_EVENT_MSG )
  {
    uint8 *pMsg;

    if ( (pMsg = osal_msg_receive( masterSlaveSwitch_TaskID )) != NULL )
    {
      masterSlaveSwitch_ProcessOSALMsg( (osal_event_hdr_t *)pMsg );

      // Release the OSAL message
      VOID osal_msg_deallocate( pMsg );
    }

    // return unprocessed events
    return (events ^ SYS_EVENT_MSG);
  }

  if ( events & MSS_START_DEVICE_EVT )
  {
    // Start the Device
    VOID GAPRole_StartDevice( &masterSlaveSwitch_PeripheralCBs );

    // Start Bond Manager
    VOID GAPBondMgr_Register( &masterSlaveSwitch_BondMgrCBs );

    return ( events ^ MSS_START_DEVICE_EVT );
  }
  
  if ( events & MSS_CHANGE_ROLE_EVT )
  {
   if (deviceRole == ROLE_PERIPHERAL)
     VOID GAPRole_StartDevice( &masterSlaveSwitch_PeripheralCBs );
   else
     VOID GAPCentralRole_StartDevice( (gapCentralRoleCB_t *) &simpleBLERoleCB );
     
    return ( events ^ MSS_CHANGE_ROLE_EVT );
  }

  // Discard unknown events
  return 0;
}

/*********************************************************************
 * @fn      masterSlaveSwitch_ProcessOSALMsg
 *
 * @brief   Process an incoming task message.
 *
 * @param   pMsg - message to process
 *
 * @return  none
 */
static void masterSlaveSwitch_ProcessOSALMsg( osal_event_hdr_t *pMsg )
{
  switch ( pMsg->event )
  {
    case KEY_CHANGE:
      masterSlaveSwitch_HandleKeys( ((keyChange_t *)pMsg)->state, ((keyChange_t *)pMsg)->keys );
      break;
    default:
      // do nothing
      break;
  }
}

/*********************************************************************
 * @fn      peripheralStateNotificationCB
 *
 * @brief   Notification from the profile of a state change.
 *
 * @param   newState - new state
 *
 * @return  none
 */
static void peripheralStateNotificationCB( gaprole_States_t newState )
{
  switch ( newState )
  {
    case GAPROLE_STARTED:
      {
        uint8 ownAddress[B_ADDR_LEN];
        uint8 systemId[DEVINFO_SYSTEM_ID_LEN];

        GAPRole_GetParameter(GAPROLE_BD_ADDR, ownAddress);

        // use 6 bytes of device address for 8 bytes of system ID value
        systemId[0] = ownAddress[0];
        systemId[1] = ownAddress[1];
        systemId[2] = ownAddress[2];

        // set middle bytes to zero
        systemId[4] = 0x00;
        systemId[3] = 0x00;

        // shift three bytes up
        systemId[7] = ownAddress[5];
        systemId[6] = ownAddress[4];
        systemId[5] = ownAddress[3];

        DevInfo_SetParameter(DEVINFO_SYSTEM_ID, DEVINFO_SYSTEM_ID_LEN, systemId);
        
        #if (defined HAL_LCD) && (HAL_LCD == TRUE)
          // Display device address
          HalLcdWriteString( "BLE Peripheral", HAL_LCD_LINE_1 );
          HalLcdWriteString( bdAddr2Str( ownAddress ),  HAL_LCD_LINE_2 );
          HalLcdWriteString( "Initialized",  HAL_LCD_LINE_3 );
        #endif // (defined HAL_LCD) && (HAL_LCD == TRUE)
          
        // Set up advertising to start
         uint8 advertising_enable = TRUE;
         GAPRole_SetParameter( GAPROLE_ADVERT_ENABLED, sizeof( uint8 ), &advertising_enable );
      }
      break;

    case GAPROLE_ADVERTISING:
      {
        #if (defined HAL_LCD) && (HAL_LCD == TRUE)
          HalLcdWriteString( "Advertising",  HAL_LCD_LINE_3 );
        #endif // (defined HAL_LCD) && (HAL_LCD == TRUE)

      }
      break;

    case GAPROLE_CONNECTED:
      {
        #if (defined HAL_LCD) && (HAL_LCD == TRUE)
          HalLcdWriteString( "Connected",  HAL_LCD_LINE_3 );
        #endif // (defined HAL_LCD) && (HAL_LCD == TRUE)
      }
      break;

    case GAPROLE_WAITING:                 // Disconnected or stopped adv
    case GAPROLE_WAITING_AFTER_TIMEOUT:   // Disconnected due to superv. timeout
      {
        #if (defined HAL_LCD) && (HAL_LCD == TRUE)
          HalLcdWriteString( "Disconnected",  HAL_LCD_LINE_3 );
        #endif // (defined HAL_LCD) && (HAL_LCD == TRUE)
        
        if ( deviceRole == ROLE_CENTRAL )
        {
          // We are going to change role. So do that.
          osal_set_event(masterSlaveSwitch_TaskID, MSS_CHANGE_ROLE_EVT);
        }

      }
      break;

    case GAPROLE_ERROR:
      {
      }
      break;

    default:
      {
      }
      break;

  }

  gapPeripheralState = newState;
  VOID gapPeripheralState;
}



/*********************************************************************
 * @fn      exampleServiceChangeCB
 *
 * @brief   Callback from exampleService indicating a value change
 *
 * @param   paramID - parameter ID of the value that was changed.
 *
 * @return  none
 */
static void exampleServiceChangeCB( uint8 paramID )
{
  uint8 newValue[20];

  switch( paramID )
  {

    case EXAMPLESERVICE_CHAR1:
      ExampleService_GetParameter( EXAMPLESERVICE_CHAR1, &newValue );

      HalLcdWriteString( "Data received", HAL_LCD_LINE_1 );
      HalLcdWriteStringValue( "Char1: ", (uint16)newValue, 10, HAL_LCD_LINE_2 );

      break;

    default:
      // should not reach here!
      break;
  }
}

/*********************************************************************
 * @fn      masterSlaveSwitch_HandleKeys
 *
 * @brief   Handles all key events for this device.
 *
 * @param   shift - true if in shift/alt.
 * @param   keys - bit field for key events. Valid entries:
 *                 HAL_KEY_SW_2
 *                 HAL_KEY_SW_1
 *
 * @return  none
 */
volatile uint8 SK_Keys = 0;
static void masterSlaveSwitch_HandleKeys( uint8 shift, uint8 keys )
{
  (void)shift;  // Intentionally unreferenced parameter

  if ( keys & HAL_KEY_UP )
  {
    if (deviceRole == ROLE_CENTRAL)
    {
      // Start or stop discovery
      if ( gapCentralState != GAPROLE_CENTRAL_CONNECTED )
      {
        if ( !simpleBLEScanning )
        {
          simpleBLEScanning = TRUE;
          simpleBLEScanRes = 0;
          
          HalLcdWriteString( "Discovering...", HAL_LCD_LINE_1 );
          HalLcdWriteString( "", HAL_LCD_LINE_2 );
          
          GAPCentralRole_StartDiscovery( DEFAULT_DISCOVERY_MODE,
                                         DEFAULT_DISCOVERY_ACTIVE_SCAN,
                                         DEFAULT_DISCOVERY_WHITE_LIST );      
        }
        else
        {
          GAPCentralRole_CancelDiscovery();
        }
      }
      else if ( gapCentralState == GAPROLE_CENTRAL_CONNECTED &&
                simpleBLECharHdl != 0 &&
                simpleBLEProcedureInProgress == FALSE )
      {
        uint8 status;
        
        // Do a read or write as long as no other read or write is in progress
        if ( simpleBLEDoWrite )
        {
          // Do a write
          attWriteReq_t req;
          
          req.handle = simpleBLECharHdl;
          req.len = 1;
          req.value[0] = simpleBLECharVal;
          req.sig = 0;
          req.cmd = 0;
          status = GATT_WriteCharValue( connHandle, &req, masterSlaveSwitch_TaskID );         
        }
        else
        {
          // Do a read
          attReadReq_t req;
          
          req.handle = simpleBLECharHdl;
          status = GATT_ReadCharValue( connHandle, &req, masterSlaveSwitch_TaskID );
        }
        
        if ( status == SUCCESS )
        {
          simpleBLEProcedureInProgress = TRUE;
          simpleBLEDoWrite = !simpleBLEDoWrite;
        }
      }    
    }
  }

  if ( keys & HAL_KEY_LEFT )
  {
    if (deviceRole == ROLE_CENTRAL)
    {
      // Display discovery results
      if ( !simpleBLEScanning && simpleBLEScanRes > 0 )
      {
          // Increment index of current result (with wraparound)
          simpleBLEScanIdx++;
          if ( simpleBLEScanIdx >= simpleBLEScanRes )
          {
            simpleBLEScanIdx = 0;
          }
          
          HalLcdWriteStringValue( "Device", simpleBLEScanIdx + 1,
                                  10, HAL_LCD_LINE_1 );
          HalLcdWriteString( bdAddr2Str( simpleBLEDevList[simpleBLEScanIdx].addr ),
                            HAL_LCD_LINE_2 );
      }
    }
  }

  if ( keys & HAL_KEY_RIGHT )
  {
    if (deviceRole == ROLE_CENTRAL)
    {
    
    // Connection update
      if ( gapCentralState == GAPROLE_CENTRAL_CONNECTED )
      {
        GAPCentralRole_UpdateLink( connHandle,
                                   DEFAULT_DESIRED_MIN_CONN_INTERVAL,
                                   DEFAULT_DESIRED_MAX_CONN_INTERVAL,
                                   DEFAULT_DESIRED_SLAVE_LATENCY,
                                   DEFAULT_DESIRED_CONN_TIMEOUT );
      }
    
    }
  }
  
  if ( keys & HAL_KEY_CENTER )
  {
    if (deviceRole == ROLE_CENTRAL)
    {
      
      uint8 addrType;
      uint8 *peerAddr;
      
      // Connect or disconnect
      if ( gapCentralState == GAPROLE_CENTRAL_INIT_DONE || gapCentralState == GAPROLE_CENTRAL_DISCONNECTED )
      {
        // if there is a scan result
        if ( simpleBLEScanRes > 0 )
        {
          // connect to current device in scan result
          peerAddr = simpleBLEDevList[simpleBLEScanIdx].addr;
          addrType = simpleBLEDevList[simpleBLEScanIdx].addrType;
        
          gapCentralState = GAPROLE_CENTRAL_CONNECTING;
          
          GAPCentralRole_EstablishLink( DEFAULT_LINK_HIGH_DUTY_CYCLE,
                                        DEFAULT_LINK_WHITE_LIST,
                                        addrType, peerAddr );
    
          HalLcdWriteString( "Connecting", HAL_LCD_LINE_1 );
          HalLcdWriteString( bdAddr2Str( peerAddr ), HAL_LCD_LINE_2 ); 
        }
      }
      else if ( gapCentralState ==  GAPROLE_CENTRAL_CONNECTING ||
                gapCentralState ==  GAPROLE_CENTRAL_CONNECTED )
      {
        // disconnect
        gapCentralState = GAPROLE_CENTRAL_DISCONNECTING;

        GAPCentralRole_TerminateLink( connHandle );
        
        HalLcdWriteString( "Disconnecting", HAL_LCD_LINE_1 ); 
      }
    }
  }
  
  if ( keys & HAL_KEY_DOWN )
  {
    asm("nop");
    if (deviceRole == ROLE_PERIPHERAL)
    {
      deviceRole = ROLE_CENTRAL;
      uint8 adv_enabled_status = FALSE;
      // Disable advertising if active
      if (gapPeripheralState == GAPROLE_ADVERTISING )
      {
        GAPRole_SetParameter( GAPROLE_ADVERT_ENABLED, sizeof( uint8 ), &adv_enabled_status ); // To stop advertising
        // Role will be switched on Peripheral callback
      }
      else if (gapPeripheralState == GAPROLE_CONNECTED)
      {
        GAPRole_SetParameter( GAPROLE_ADVERT_ENABLED, sizeof( uint8 ), &adv_enabled_status ); // To avoid auto-reenabling of advertising
        GAPRole_TerminateConnection();
        // Role will be switched on Peripheral callback
      }
      else
      {
        osal_set_event(masterSlaveSwitch_TaskID, MSS_CHANGE_ROLE_EVT);
      }
    }
    else
    {
      deviceRole = ROLE_PERIPHERAL;
      if (gapCentralState == GAPROLE_CENTRAL_INIT_DONE || gapCentralState == GAPROLE_CENTRAL_DISCONNECTED )
      {
        osal_set_event(masterSlaveSwitch_TaskID, MSS_CHANGE_ROLE_EVT);
      }
      else
      {
        // disconnect all
        GAPCentralRole_TerminateLink( GAP_CONNHANDLE_ALL );
        // Expect the Central disconnect callback to switch the role

      }
    }
    
  }
}

/*********************************************************************
       CENTRAL FUNCTIONS
*********************************************************************/


/*********************************************************************
 * @fn      simpleBLECentralRssiCB
 *
 * @brief   RSSI callback.
 *
 * @param   connHandle - connection handle
 * @param   rssi - RSSI
 *
 * @return  none
 */
static void simpleBLECentralRssiCB( uint16 connHandle, int8 rssi )
{

}

/*********************************************************************
 * @fn      simpleBLECentralEventCB
 *
 * @brief   Central event callback function.
 *
 * @param   pEvent - pointer to event structure
 *
 * @return  none
 */
static void simpleBLECentralEventCB( gapCentralRoleEvent_t *pEvent )
{
  switch ( pEvent->gap.opcode )
  {
    case GAP_DEVICE_INIT_DONE_EVENT:  
      {
        HalLcdWriteString( "BLE Central", HAL_LCD_LINE_1 );
        HalLcdWriteString( bdAddr2Str( pEvent->initDone.devAddr ),  HAL_LCD_LINE_2 );
        HalLcdWriteString( "Initialized", HAL_LCD_LINE_3 );
        gapCentralState = GAPROLE_CENTRAL_INIT_DONE;
      }
      break;

    case GAP_DEVICE_INFO_EVENT:
      {
        // if filtering device discovery results based on service UUID
        if ( DEFAULT_DEV_DISC_BY_SVC_UUID == TRUE )
        {
          if ( simpleBLEFindSvcUuid( WANTED_SERVICE_UUID,
                                     pEvent->deviceInfo.pEvtData,
                                     pEvent->deviceInfo.dataLen ) )
          {
            simpleBLEAddDeviceInfo( pEvent->deviceInfo.addr, pEvent->deviceInfo.addrType );
          }
        }
      }
      break;
      
    case GAP_DEVICE_DISCOVERY_EVENT:
      {
        // discovery complete
        simpleBLEScanning = FALSE;

        // if not filtering device discovery results based on service UUID
        if ( DEFAULT_DEV_DISC_BY_SVC_UUID == FALSE )
        {
          // Copy results
          simpleBLEScanRes = pEvent->discCmpl.numDevs;
          osal_memcpy( simpleBLEDevList, pEvent->discCmpl.pDevList,
                       (sizeof( gapDevRec_t ) * pEvent->discCmpl.numDevs) );
        }
        
        HalLcdWriteStringValue( "Devices Found", simpleBLEScanRes,
                                10, HAL_LCD_LINE_1 );
        if ( simpleBLEScanRes > 0 )
        {
          HalLcdWriteString( "<- To Select", HAL_LCD_LINE_2 );
        }

        // initialize scan index to last device
        simpleBLEScanIdx = simpleBLEScanRes;

      }
      break;

    case GAP_LINK_ESTABLISHED_EVENT:
      {
       if ( pEvent->gap.hdr.status == SUCCESS )
        {          
          simpleBLEProcedureInProgress = TRUE;   
          HalLcdWriteString( bdAddr2Str( pEvent->linkCmpl.devAddr ), HAL_LCD_LINE_2 );
          HalLcdWriteString( "Connected", HAL_LCD_LINE_3 );
          gapCentralState = GAPROLE_CENTRAL_CONNECTED;
          connHandle = pEvent->linkCmpl.connectionHandle;
       }
       else
       {
          gapCentralState = GAPROLE_CENTRAL_DISCONNECTED;
          HalLcdWriteString( "Connect Failed", HAL_LCD_LINE_1 );
          HalLcdWriteStringValue("Reason:",  pEvent->gap.hdr.status, 10, HAL_LCD_LINE_2 );
          HalLcdWriteString( "Disconnected", HAL_LCD_LINE_3 );
          simpleBLEDiscState = BLE_DISC_STATE_IDLE;
          connHandle = INVALID_CONNHANDLE;
       }

      }
      break;

    case GAP_LINK_TERMINATED_EVENT:
      {
        gapCentralState = GAPROLE_CENTRAL_DISCONNECTED;
        connHandle = INVALID_CONNHANDLE;
        simpleBLERssi = FALSE;
        simpleBLEDiscState = BLE_DISC_STATE_IDLE;
        simpleBLECharHdl = 0;
        simpleBLEProcedureInProgress = FALSE;
          
        HalLcdWriteString( "Disconnected", HAL_LCD_LINE_1 );
        HalLcdWriteStringValue( "Reason:", pEvent->linkTerminate.reason,
                                10, HAL_LCD_LINE_2 );
        HalLcdWriteString( "Disconnected", HAL_LCD_LINE_3 );
        
        
        // Start role switching if applicable
        if (deviceRole == ROLE_PERIPHERAL)
        {
          osal_set_event(masterSlaveSwitch_TaskID, MSS_CHANGE_ROLE_EVT);
        }
        else
        {
          // Do normal Central link terminated tasks. E.g restart scanning.
        }
      }
      break;

    case GAP_LINK_PARAM_UPDATE_EVENT:
      {
      }
      break;
      
    default:
      break;
  }
}

#if (defined HAL_LCD) && (HAL_LCD == TRUE)
/*********************************************************************
 * @fn      bdAddr2Str
 *
 * @brief   Convert Bluetooth address to string. Only needed when
 *          LCD display is used.
 *
 * @return  none
 */
char *bdAddr2Str( uint8 *pAddr )
{
  uint8       i;
  char        hex[] = "0123456789ABCDEF";
  static char str[B_ADDR_STR_LEN];
  char        *pStr = str;

  *pStr++ = '0';
  *pStr++ = 'x';

  // Start from end of addr
  pAddr += B_ADDR_LEN;

  for ( i = B_ADDR_LEN; i > 0; i-- )
  {
    *pStr++ = hex[*--pAddr >> 4];
    *pStr++ = hex[*pAddr & 0x0F];
  }

  *pStr = 0;

  return str;
}
#endif // (defined HAL_LCD) && (HAL_LCD == TRUE)