/*
 * uartSample_main.c
 *
 * This file contains the test / demo code to demonstrate the UART driver
 * functionality on SYS/BIOS 6
 *
 * Copyright (C) 2009 Texas Instruments Incorporated - http://www.ti.com/
 *
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 * \file    uartSample_main.c
 *
 * \brief   This file contains the test / demo code to demonstrate the
 *          UART driver implemented for SYS/BIOS 6
 *
 *          (C) Copyright 2009, Texas Instruments, Inc
 *
 * \author  PSG, modified by MZ to use UART1 instead of UART2
 *
 * \version 0.1   created for the OMAPL138 and C6748 platforms
 */

/* ========================================================================== */
/*                              INCLUDE FILES                                 */
/* ========================================================================== */ 

#include <xdc/std.h>
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/io/GIO.h>
#include <xdc/runtime/log.h>
#include <ti/sysbios/knl/Task.h>
#include <xdc/runtime/Error.h>
#include <xdc/runtime/System.h>
#include <ti/sdo/edma3/drv/edma3_drv.h>

#include <psc/include/Psc.h>
#include <uart/include/Uart.h>
#include <uart/include/UartLocal.h>
#include <platforms/evm6748/Uart_evmInit.h>

/* ========================================================================== */
/*                            EXTERNAL VARIABLES                              */
/* ========================================================================== */ 

/* ========================================================================== */
/*                            EXTERNAL REFERENCES                             */
/* ========================================================================== */ 

Void startUartSample(Void);
Void askBLEforInfo( Void);
EDMA3_DRV_Handle edma3init(unsigned int edma3Id, EDMA3_DRV_Result *);

/* ========================================================================== */
/*                             MACRO DEFINITIONS                              */
/* ========================================================================== */ 

#define UART_NUM_IOP         (1u)

/* ========================================================================== */
/*                           LOCAL FUNCTION PROTOTYPES                        */
/* ========================================================================== */  

static Void uartEdmaInit(Void);
static Void tskHeartBeat(Void);
 
/* ========================================================================== */
/*                             GLOBAL VARIABLES                               */
/* ========================================================================== */  

/* Handle to the EDMA driver instance                                         */
EDMA3_DRV_Handle hEdma;
 
/* UART handle for input stream                                               */
GIO_Handle uartRxHandle;

/* UART handle for output stream                                              */
GIO_Handle uartTxHandle;

Uart_Params   uartParams;

/* Global function prototypes */
void user_uart_init(void);

/* ========================================================================== */
/*                           FUNCTION DEFINITIONS                             */
/* ========================================================================== */ 

/**
 *  \brief  Void main(Void)
 *
 *   Main function of the sample application. This function enables
 *   the uart instance in the power sleep controller and also
 *   enables the pinmux for the Uart 0 instance.
 *
 *  \param  None
 *  \return None
 */
Void main(Void)
{
    System_printf("\r\nUart Sample Main\n");

    /* Call the EVM specific initialization function                          */
    configureUart();
    
    BIOS_start();

    return;
}

/**
 * \brief      Entry point for the main task. This creates the required channels
 *             and calls the required functions for demonstrating the UART
 *             driver
 *
 * \param      arg1  [IN]   unused
 * \param      arg2  [IN]   unused 
 *
 * \return     None
 */
Void uartSampleTask(UArg arg1, UArg arg2)
{
    Uart_ChanParams    	chanParams;
	Error_Block 		eb;
 	GIO_Params 			ioParams;

    Error_init(&eb);

    /*
     * Initialize channel attributes.
     */
    GIO_Params_init( &ioParams);
    /* initialise the edma library and get the EDMA handle                    */
    uartEdmaInit();

    /* update the edma Handle                                                 */
    chanParams.hEdma = hEdma;
    ioParams.chanParams = ( Ptr) &chanParams;

   	/* create the required streams for the UART demo                           */
    // uartTxHandle = GIO_create( "/uart2", GIO_OUTPUT, &ioParams, &eb);
    // uartRxHandle = GIO_create( "/uart2", GIO_INPUT, &ioParams, &eb);
    uartTxHandle = GIO_create( "/uart1", GIO_OUTPUT, &ioParams, &eb);
    uartRxHandle = GIO_create( "/uart1", GIO_INPUT, &ioParams, &eb);

    if ((NULL == uartRxHandle) || (NULL == uartTxHandle))
    {
        System_printf("\nStream creation failed\n");
    }
    else
    {
        /* Run the uart sample application                                    */
        startUartSample();

        /* Start the Heart Beat Print                                         */
        tskHeartBeat();
    }
}

/**
 * \brief      Function to initialize the edma driver and get the handle to the
 *             edma driver;
 *
 * \param      arg1  [IN]   unused
 * \param      arg2  [IN]   unused 
 *
 * \return     None
 */
static Void uartEdmaInit(Void)
{
    EDMA3_DRV_Result    edmaResult      = 0;

    hEdma = edma3init(0, &edmaResult);
    
    if (edmaResult != EDMA3_DRV_SOK)
    {
        /* Report EDMA Error */
        System_printf("\nEDMA driver initialization FAIL\n");
    }
    else
    {
        System_printf("\nEDMA driver initialization PASS.\n");
    }
}

/*
 * UART0 init function called when creating the driver.
 */
void user_uart_init()
{
    Uart_init();

    uartParams = Uart_PARAMS;
	uartParams.opMode = Uart_OpMode_POLLED;
	uartParams.baudRate = Uart_BaudRate_115_2K;
	uartParams.hwiNumber = 9;
	uartParams.rxThreshold = Uart_RxTrigLvl_1;
	uartParams.softTxFifoThreshold = 1;

	if(Uart_OpMode_POLLED == uartParams.opMode)
	{
		System_printf("\r\nUart is configured in polled mode");
	}
	else if (Uart_OpMode_INTERRUPT == uartParams.opMode)
	{
		System_printf("\r\nUart is configured in interrupt mode");
	}
	else if (Uart_OpMode_DMAINTERRUPT == uartParams.opMode)
	{
		System_printf("\r\nUart is configured in dma mode");
	}
	else
	{
		System_printf("\r\nError: Unknown mode of operation!!!!!!!!!!");
	}
}

/**
 * \brief      Function to print an log to the console every one second(Approx)
 *
 * \param      None
 *
 * \return     None
 *
 */
static Void tskHeartBeat()
{
    static Uint32 counter = 0;

    while (counter < 0xFFFFFFFF)
    {
        Task_sleep(1000u);
        
        System_printf("\r\n\r\n!!! PSP HrtBt %l", counter);
        counter++;
    }
}

/* ========================================================================== */
/*                              END OF FILE                                   */
/* ========================================================================== */
