//*****************************************************************************
//! @file       cc120x_transparent_serial_mode_rx.c
//! @brief      
//!
//
//  Copyright (C) 2013 Texas Instruments Incorporated - http://www.ti.com/
//
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//    Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
//
//    Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
//    Neither the name of Texas Instruments Incorporated nor the names of
//    its contributors may be used to endorse or promote products derived
//    from this software without specific prior written permission.
//
//  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
//  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
//  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
//  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
//  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
//  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
//  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
//  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//****************************************************************************/

/*****************************************************************************
* INCLUDES
*/
#include "msp430.h"
#include "hal_spi_rf_trxeb.h"
#include "bsp.h"
#include "cc120x_spi.h"
#include "cc120x_serial_mode_reg_config.h"

/******************************************************************************
* DEFINES
*/

/******************************************************************************
* GLOBAL VARIABLES
*/

/******************************************************************************
* STATIC FUNCTIONS
*/
static void initRX(void);
static void initMCU(void);
static void registerConfig(void);
static void transparentMode(void);
/******************************************************************************
* @fn          main
*
* @brief       Runs the main routine
*                
* @param       none
*
* @return      none
*/
void main(void){
  
  //Initialize MCU and peripherals
  initMCU();
  
  //Write radio registers
  registerConfig();
  
  //Setup ports, timers, calculate delay and set the radio in RX mode.
  initRX();
  
  while(1) {}
}

/******************************************************************************
* @fn          transparentMode
*
*@brief        Writes all the register settings necessary for transparent mode
*
*
* @param       none
*
*
* @return      none
*/
static void transparentMode(void) {
  uint8 writeByte;
  // Carrier sense on GPIO3
  writeByte = 0x11;
  cc120xSpiWriteReg(CC120X_IOCFG3, &writeByte, 1);
  // Serial data on GPIO2
  writeByte = 0x09;
  cc120xSpiWriteReg(CC120X_IOCFG2, &writeByte, 1);
  // FIFO_EN = 0;
  writeByte = 0x06;
  cc120xSpiWriteReg(CC120X_MDMCFG1, &writeByte, 1);
  // Transparent mode enable 
  writeByte = 0x65;
  cc120xSpiWriteReg(CC120X_MDMCFG0, &writeByte, 1);
  // Carrier sense threshold 
  writeByte = 0x16;
  cc120xSpiWriteReg(CC120X_AGC_CS_THR, &writeByte, 1);
  // Transparent mode
  writeByte = 0x07;
  cc120xSpiWriteReg(CC120X_PKT_CFG2, &writeByte, 1);
}

/******************************************************************************
* @fn          initRX
*
*@brief        Setup pin directions, systemclock, transparent mode specific registers,
*              timer A, calculates delays and puts the radio in RX 
*
*
* @param       none
*
*
* @return      none
*/
static void initRX(void) {
  
  //Set direction on P1: Carrier sense on GPIO_3 and serial data on GPIO_2
  P1DIR &= ~0x0C;
  P1SEL &= ~0x0C;
  P1REN &= ~0x0C;  
  
  //Used for debugging
  P1DIR |= 0x13;
  P1DS |= 0x13;
  
  //Set transparent serial mode specific registers
  transparentMode();
 
  // Set up timer A
  TA0CTL = TASSEL_2 + MC_1;//SMCLK clock source + Up mode
  
  //Set radio in RX mode
  trxSpiCmdStrobe(CC120X_SRX);  
}

/*******************************************************************************
* @fn          registerConfig
*
* @brief       Write register settings as given by SmartRF Studio found in
*              cc120x_easy_link_reg_config.h
*
* @param       none
*
* @return      none
*/
static void registerConfig(void) {
  
  uint8 writeByte;
  
  // Reset radio
  trxSpiCmdStrobe(CC120X_SRES);
  
  // Write registers to radio
  for(uint16 i = 0; i < (sizeof  preferredSettings/sizeof(registerSetting_t)); i++) {
    writeByte =  preferredSettings[i].data;
    cc120xSpiWriteReg( preferredSettings[i].addr, &writeByte, 1);
  }
}

/******************************************************************************
 * @fn          initMCU
 *
 * @brief       Initialize MCU and board peripherals
 *                
 * @param       input, output parameters
 *
 * @return      describe return value, if any
 */
static void initMCU(void){
  
  // Init clocks and I/O 
  bspInit(BSP_SYS_CLK_25MHZ);
  
  // Initialize SPI interface to LCD (shared with SPI flash)
  bspIoSpiInit(BSP_FLASH_LCD_SPI, BSP_FLASH_LCD_SPI_SPD);  

  // Instantiate tranceiver RF spi interface to SCLK ~ 4 MHz */
  //input clockDivider - SMCLK/clockDivider gives SCLK frequency
  trxRfSpiInterfaceInit(0x04);

  // Enable global interrupt
  _BIS_SR(GIE);
}