/******************************************************************************
* RF2500 Range Test for Ez430-RF2500 EVM
*
* To use this demo please plug in the USB port to a laptop and open a telnet
* session using 9600/8/1 and follow the instructions on the on the
*
* Thomas Almholt, Texas Instruments Inc, February 2010.
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
#include "stdio.h"

#include "msp430.h"
#include "CCx_hal.h"
#include "hal_usb.h"
#include "TRX_ui.h"
#include "stdlib.h"

/* define the maximum number of hopping channels */
#define MAX_HOP_CHANNELS  50                 // Sets the number of channels
#define SYSTEM_ID          1                 // 1 and 255
#define CHANNEL_MULTIPLIER 5                 // Multiply the channel sequence
#define ENABLE_HOPPING      TRUE
#define ENABLE_TWO_WAY_LINK FALSE

/*
* Enable Debug Uart will send RF performance data to the UART this is very
* useful but some cases it takes too long to write the data to the UART port
* and the MCU can no longer keep its timeline that is needed for the RF     
*/
#define ENABLE_DEBUG_UART  

/*
* burst timing - Master burst rate control setting
* This value is calculated as follows 20312/(6.5M/8) = 25ms
* Burst length + PLL calibration : 3.75ms + 1ms = 4.75ms
*/
//#define MASTER_BURST_VALUE 20312   // use for 6.5M operation
#define MASTER_BURST_VALUE 8192    // 250ms using 32K ACLK
/*
* burst timeout - if it has not happen, just turn off the RX. 
* This value is calculated as follows 4063/(6.5M/8) = 5ms
*/
//#define MAX_WAIT_NORMAL  4563      // use for 6.5M operation
#define MAX_WAIT_NORMAL  819         // 25ms using 32k ACLK
/*
* Seach timeout - this setting is long enough to have at least 1 burst and 
* dwell time inside the window.
* This value is calculated as follows 40625/(6.5M/8) = 50ms
*/
//#define MAX_WAIT_SEARCH  36562     // use for 6.5M operation
#define MAX_WAIT_SEARCH  7500       // 500ms using 32k ACLK

/*
* burst timing - This calibrates the timer to the burst duration and period
* This value is calculated as follows 3860/(6.5M/8) = 4.75ms
* Burst length + PLL calibration : 3.75ms + 1ms = 4.75ms.
*/
// #define TIMER_ALIGNMENT_VALUE 4100 // use for 6.5M operation
#define TIMER_ALIGNMENT_VALUE 327     // 10ms using 32K ACLK

/*
* The options lets the RX skip a number of burst to save power
* however if is does not get the burst it expected it will try the next burst 
* without skipping until it finds it again. 
*/
#define BURST_SKIP_NUMBER         10   // We will skip 30 burst to save power
#define BURST_LOST_SOFT_LIMIT      4   // we will alert the user at this limit
#define BURST_LOST_HARD_LIMIT    100   // assume the RF link is broken and restart


/* these should not normally be needed, but this demo is accessing the hardware*/
/* directly to enable some of the features */
#include "CCxx00_def.h"
#include "CCxx00_spi.h"
//#include "CC112x_def.h"
//#include "CC112x_spi.h"
//#include "CC2520_def.h"
//#include "CC2520_spi.h"


/* *****************************************************************************
* Internal functions definitions
******************************************************************************/
void uartSendString(char *UART_Text);
char radio_mode;
char radio_mode_idle_count = 0;
extern char user_button_pushed;        // indicator that button has been pressed
extern char wakeup_on_wdt;
extern const unsigned char rand_data[];
unsigned char chan_hop_seq[MAX_HOP_CHANNELS];
char u_str[80];
unsigned char burst_skip_mode;
/******************************************************************************
* To use this demo please plug in the USB port to a laptop and open a telnet
* session using 9600/8/1 and follow the instructions on the on the
*
* MENU_IDLE, MENU_UNMOD_TX, MENU_TX_nMCU, MENU_TX_wMCU, MENU_RX_STATIC
* MENU_RX_SWEEP, MENU_TX_BER, MENU_RX_BER, MENU_RESTART
*
*  uartSendString("1) Idle Radio\n\r");
*  uartSendString("2) Unmodulated TX      (line)\n\r");
*  uartSendString("3) Static RX  (line)\n\r");
*  uartSendString("4) Modulated TX no MCU (line)\n\r");
*  uartSendString("5) TX burst BER test   (line, bursts) eg.(5 1000)\n\r");
*  uartSendString("6) RX burst BER test   (line, bursts) eg.(6 1000)\n\r");
*  uartSendString("7) Sweep RX, like SA   (line, sweeps) eg.(7 10)\n\r");
*  uartSendString("8) Single channel RX   (line, sweeps) eg.(8 10)\n\r");
*  uartSendString("9) Setup Radio page 1  (line, channel)\n\r");

* Thomas Almholt, Texas Instruments Inc, February 2010.
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
char parse_ui_cmd(trx_cfg_struct *trx_cfg, char *ui_cmd) {
  long line, s1, s2, s3;                // clear the command line from previous
  
  sscanf(ui_cmd, "%ld", &line);
  switch(line){
  case 1:                                                  /* "1) Idle Radio" */
    trx_cfg->trx_mode  = MENU_RF_IDLE;
    trx_cfg->cc_state  = CC_IDLE;
    break;
  case 2:                                              /* "2) Unmodulated TX" */
    trx_cfg->trx_mode  = MENU_TX_UNMOD;
    trx_cfg->cc_state  = CC_IDLE;
    break;
  case 3:                                         /* "3) Modulated TX no MCU" */
    trx_cfg->trx_mode  = MENU_RX_STATIC; 
    trx_cfg->cc_state  = CC_IDLE;
    break;
  case 4:                                         /* "4) Modulated TX w. MCU" */
    trx_cfg->trx_mode  = MENU_TX_nMCU; 
    trx_cfg->cc_state  = CC_IDLE;
    break;
  case 5:                                           /* "5) TX burst BER test" */
    trx_cfg->trx_mode  = MENU_TX_BER; 
    trx_cfg->cc_state  = CC_IDLE;
    sscanf(ui_cmd, "%ld %ld", &line, &s1);
    trx_cfg->no_burst = s1;
    break;
  case 6:                                           /* "6) RX burst BER test" */
    trx_cfg->trx_mode  = MENU_RX_BER; 
    trx_cfg->cc_state  = CC_IDLE;
    sscanf(ui_cmd, "%ld %ld", &line, &s1);
    trx_cfg->no_burst = s1;
    break;
  case 7:                                           /* "7) Sweep RX, like SA" */
    trx_cfg->trx_mode  = MENU_RX_SWEEP; 
    trx_cfg->cc_state  = CC_IDLE;
    sscanf(ui_cmd, "%ld %ld", &line, &s1);
    trx_cfg->no_burst = s1;
    break;
  case 8:                                           /* "8) Single channel RX" */
    trx_cfg->trx_mode  = MENU_RX_SINGLE; 
    trx_cfg->cc_state  = CC_IDLE;
    sscanf(ui_cmd, "%ld %ld", &line, &s1);
    trx_cfg->no_burst = s1;
    break;
  case 9:                                          /* "9) Setup Radio page 1" */
    trx_cfg->trx_mode  = MENU_RESTART; 
    trx_cfg->cc_state  = CC_IDLE;
    sscanf(ui_cmd, "%ld %ld %ld %ld", &line, &s1, &s2, &s3);
    trx_cfg->start_freq = s1;
    trx_cfg->stop_freq = s2;
    trx_cfg->tx_pwr = s3;
    rf_set_freq(trx_cfg->start_freq, trx_cfg->cc_device_id);
    trx_cfg->tx_pwr = set_tx_pwr(trx_cfg->tx_pwr, trx_cfg->cc_device_id, RF_CC430); 
    break;
  case 10:                                        /* "10) Setup Radio page 2" */
    trx_cfg->trx_mode  = MENU_RESTART; 
    trx_cfg->cc_state  = CC_IDLE;
    sscanf(ui_cmd, "%ld %ld %ld %ld", &line, &s1, &s2, &s3);
    trx_cfg->rx_bw = s1;
    trx_cfg->ch_spc = s2;
    trx_cfg->freq_dev = s3;
    trx_cfg->rx_bw = set_rx_bw(trx_cfg->rx_bw, RF_CC430);          
    trx_cfg->ch_spc = set_ch_spc(trx_cfg->ch_spc, RF_CC430);       
    trx_cfg->freq_dev = set_freq_dev(trx_cfg->freq_dev, RF_CC430);     
    break;
  default: 
    trx_cfg->trx_mode  = MENU_RESTART; 
    trx_cfg->cc_state  = CC_IDLE;
    break;
  }
  
  return trx_cfg->trx_mode;
}

/******************************************************************************
* default setup 
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void rf_default_setup(trx_cfg_struct *trx_cfg, char RF_IDx) {
  trx_cfg->trx_mode   = MENU_RESTART;
  trx_cfg->cc_state   = CC_IDLE;  
  trx_cfg->no_burst   = 50000;
  trx_cfg->tx_pwr     = 0;
  trx_cfg->bit_rate   = 50;
  trx_cfg->rx_bw      = 200;
  trx_cfg->ch_spc     = 200;
  trx_cfg->freq_dev   = 25;
  trx_cfg->rf_channel = 0;
  
  
  trx_cfg->cc_device_id = get_device_id(RF_IDx);
  switch (trx_cfg->cc_device_id) {
  case DEV_CC2500:
    // input default setting in the configuration array.
    trx_cfg->start_freq = 2405000;
    trx_cfg->stop_freq  = 2500000;
    trx_cfg->ch_spc     = 300;
    set_ch_spc(trx_cfg->ch_spc, RF_IDx);
    rf_set_freq(trx_cfg->start_freq, RF_IDx); 
    break;
  case DEV_CC430x:
    // input default setting in the configuration array.
    trx_cfg->start_freq = 902750;
    trx_cfg->stop_freq = 927500;
    rf_set_freq(trx_cfg->start_freq, RF_IDx);     
    break;    
  case DEV_CC1100:
    // input default setting in the configuration array.
    trx_cfg->start_freq = 902750;
    trx_cfg->stop_freq = 927500;
    rf_set_freq(trx_cfg->start_freq, RF_IDx);     
    break;
  case DEV_CC1101:
    // input default setting in the configuration array.
    trx_cfg->start_freq = 902750;
    trx_cfg->stop_freq = 927500;
    rf_set_freq(trx_cfg->start_freq, RF_IDx);     
    break;
  case DEV_CC1125:
    // input default setting in the configuration array.
    trx_cfg->start_freq = 902750;
    trx_cfg->stop_freq = 927500;
    rf_set_freq(trx_cfg->start_freq, RF_IDx);     
    break;
  case DEV_CC1120:
    // input default setting in the configuration array.
    trx_cfg->start_freq = 902750;
    trx_cfg->stop_freq = 927500;
    rf_set_freq(trx_cfg->start_freq, RF_IDx);     
    break;
    
  default:
    break;
  }  
  return;
}

/******************************************************************************
* unmodulated TX 
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void unmodulated_tx(trx_cfg_struct *trx_cfg, char RF_IDx) {
  
  // Make sure the radio is IDLE, then reprogram the registers for umodulated tx
  radio_idle(RF_IDx);
  set_tx_unmodulated_test_mode(RF_IDx);
  radio_transmit(RF_IDx);
  
  uartSendString("\n\r");  
  uartSendString("Started unmodulated TX, Press any key to stop\n\r");
  return;
}


/******************************************************************************
* modulated TX 
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void modulated_tx_no_mcu(trx_cfg_struct *trx_cfg, char RF_IDx) {
  
  // Make sure the radio is IDLE, then reprogram the registers for modulated tx
  radio_idle(RF_IDx);
  set_tx_modulated_test_mode(RF_IDx);  
  radio_transmit(RF_IDx);
  
  uartSendString("\n\r");  
  uartSendString("Started modulated TX using PN9 inside CC device\n\r");  
  uartSendString("Press any key to stop\n\r");
  return;
}


/******************************************************************************
* rx_static 
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void rx_static(trx_cfg_struct *trx_cfg, char RF_IDx) {
  
  radio_receive_on(RF_IDx);
  uartSendString("\n\r");  
  uartSendString("Started static RX, Press any key to stop\n\r");
  return;
}
/******************************************************************************
* main_memu displays the text based menu using a uart link.
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void main_menu(trx_cfg_struct *trx_cfg) {
  
  uartSendString(" \n\r");
  //uartSendString("RF demonstration based on the Ez430-RF2500T EVM\n\r");
  uartSendString("RF demonstration based on the Ez430-CC430F5137 MVK\n\r");  
  uartSendString("Hardware version 1.0, Software version 1.x\n\r");
  uartSendString("Select from items below:\n\r");
  uartSendString("*************************************************************\n\r");
  uartSendString(" 1) Reset and Idle Radio\n\r");
  uartSendString(" 2) Unmodulated TX      (line)\n\r");
  uartSendString(" 3) Static RX           (line)\n\r");
  uartSendString(" 4) Modulated TX no MCU (line)\n\r");
  uartSendString(" 5) TX burst BER test   (line, bursts) eg.(5 1000)\n\r");
  uartSendString(" 6) RX burst BER test   (line, bursts) eg.(6 1000)\n\r");
  uartSendString(" 7) Sweep RX, like SA   (line, sweeps) eg.(7 10)\n\r");
  uartSendString(" 8) Single channel RX   (line, sweeps) eg.(8 10)\n\r");
  uartSendString(" 9) Setup Radio page 1  (line, start freq, stop freq, tx pwr)\n\r");
  uartSendString("10) Setup Radio page 2  (line, rx bw, chan spac, freq dev)\n\r");
  uartSendString("*************************************************************\n\r");
  
  if(trx_cfg->cc_device_id == DEV_UNKNOWN) {
    sprintf(u_str, "Current configuration: (Device = Unknown)\n\r");
  } 
  if(trx_cfg->cc_device_id == DEV_CC2500) {
    sprintf(u_str, "Current configuration: (Device = CC2500)\n\r");
  } 
  if(trx_cfg->cc_device_id == DEV_CC1100) {
    sprintf(u_str, "Current configuration: (Device = CC1100)\n\r");
  } 
  if(trx_cfg->cc_device_id == DEV_CC1101) {
    sprintf(u_str, "Current configuration: (Device = CC1101)\n\r");
  } 
  if(trx_cfg->cc_device_id == DEV_CC430x) {
    sprintf(u_str, "Current configuration: (Device = CC430x)\n\r");
  } 
  if(trx_cfg->cc_device_id == DEV_CC1125) {
    sprintf(u_str, "Current configuration: (Device = CC1125)\n\r");
  } 
  if(trx_cfg->cc_device_id == DEV_CC1120) {
    sprintf(u_str, "Current configuration: (Device = CC1120)\n\r");
  } 
  
  uartSendString(u_str);
  uartSendString("   Current configuration:\n\r");
  sprintf(u_str, "   TX : 9 %7ld %ld %i\n\r", trx_cfg->start_freq, 
          trx_cfg->stop_freq, trx_cfg->tx_pwr);
  uartSendString(u_str);
  sprintf(u_str, "   RX : 10 %i %i %i\n\r", trx_cfg->rx_bw, trx_cfg->ch_spc, 
          trx_cfg->freq_dev);
  uartSendString(u_str);
  uartSendString("*************************************************************\n\r");
  uartSendString("CMD >\n\r");
  return;
}

/******************************************************************************
* Start TX burst timer using Timer A1
* 250 millisecond timer using ACLK.
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void generate_hopping_table(unsigned char *hop_table) {
  int channel_found,ii,jj;
  unsigned int chan_int;
  unsigned char length_table = MAX_HOP_CHANNELS;
  int rand_seed = SYSTEM_ID;

  /* if the rand_seed is positive generate a PN hopping table */
  if(ENABLE_HOPPING == TRUE) {
    srand(rand_seed);                            // start by placing the seed
    for (ii=0;ii<length_table;ii++) {            // ii is the table index
      channel_found = 1;                         // we have to look for overlaps
      while(channel_found > 0) {
        chan_int =  MAX_HOP_CHANNELS * (rand()>>7);           // find a new random number
        hop_table[ii] =  chan_int >> 8;       
        channel_found = 0;
        for (jj=0;jj<ii;jj++) {                  // check to see if already exists
          if(hop_table[ii] == hop_table[jj]) channel_found++;
        }
      }
    }
  } else {
    /* generate a simple table of all the same values based on the negative input */
    for (ii=0;ii<length_table;ii++) { 
      hop_table[ii] = -rand_seed;
    }
  }
  /* Multiply the channel hop sequence by a fixed number to enable larger     */
  /* channels spacings that allowed by just using the built in channel width  */
  /* be very careful that the final results remains smaller than 255          */
  for (ii=0;ii<length_table;ii++) {                     // ii is the table index
    hop_table[ii] = hop_table[ii] * CHANNEL_MULTIPLIER;
  }
  return;
}

#ifdef MSP430_F5xx 
/******************************************************************************
* Start TX burst timer using Timer A1
* 250 millisecond timer using ACLK.
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void start_mac_timer(void) {
  // Start Timer 0 using the ACLK as a source (this enables the use of
  // various low power modes). Timer 0 will be used to keep RF burst time
  TA1CCR0 = MASTER_BURST_VALUE-1;           // Seting for 250 millisecond timing
  TA1CCR1 = 1;                              // RX START (Start TX imidiately)
  TA1CCR2 = 2;                              // TX START (enable 2ms drift (1%)
  TA1CTL = TASSEL_1 + MC_1 + TACLR;         // ACLK, upmode, clear TAR
  TA1CCTL1 = CCIE;                          // interrupt enabled
  TA1CCTL2 = CCIE;                          // interrupt enabled  
  
  return;
}

/******************************************************************************
* Start TX burst timer using Timer A1
* 250 millisecond timer using ACLK.
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void start_mac_timer_mclk(void) {
  // Start Timer 0 using the ACLK as a source (this enables the use of
  // various low power modes). Timer 0 will be used to keep RF burst time
  TA1CCR0  = MASTER_BURST_VALUE-1;          // Seting for MASTER BURST SCHEDULE
  TA1CCR1  = 1;                             // RX START (Start TX imidiately)
  TA1CCR2  = 635;                           // TX START (a little for drift)
  TA1CTL   = TASSEL_2 + MC_1 + TACLR + ID_3;// SMCLK, upmode, clear TAR, div/8
  TA1CCTL1 = CCIE;                          // interrupt enabled
  TA1CCTL2 = CCIE;                          // interrupt enabled  
  
  return;  
}

/******************************************************************************
* Stop the timer
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void stop_mac_timer(void) {
  TA1CTL = 0;  
  
  return;
}

#else

/******************************************************************************
* Start TX burst timer using Timer A1
* 250 millisecond timer using ACLK.
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void start_mac_timerA_mclk(void) {
  // Start Timer 0 using the ACLK as a source (this enables the use of
  // various low power modes). Timer 0 will be used to keep RF burst time
  TACCR0  = MASTER_BURST_VALUE-1;          // Seting for MASTER SCHEDULE
  TACCR1  = 1;                             // RX START (Start TX imidiately)
  TACCR2  = 20;                            // TX START (a little for drift)
  TACTL   = TASSEL_2 + MC_1 + TACLR + ID_3;// SMCLK, upmode, clear TAR, div/8
  TACCTL1 = CCIE;                          // interrupt enabled
  TACCTL2 = CCIE;                          // interrupt enabled  
  
  return;
}

/******************************************************************************
* Start TX burst timer using Timer A1
* 250 millisecond timer using ACLK.
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void start_mac_timer(void) {
  // Start Timer 0 using the ACLK as a source (this enables the use of
  // various low power modes). Timer 0 will be used to keep RF burst time
  TBCCR0  = MASTER_BURST_VALUE-1;          // Seting for MASTER SCHEDULE
  TBCCR1  = 1;                             // RX START (Start TX imidiately)
  TBCCR2  = 20;                            // TX START (a little for drift)
  TBCTL   = TBSSEL_2 + MC_1 + TBCLR + ID_3;// SMCLK, upmode, clear TAR, div/8
  TBCCTL1 = CCIE;                          // interrupt enabled
  TBCCTL2 = CCIE;                          // interrupt enabled  
  
  return;
}



/******************************************************************************
* Stop the timer
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void stop_mac_timer(void) {
  //  TACTL = 0;  
  TBCTL = 0;
  
  return;
}

#endif


/******************************************************************************
* Transmit : Perform RF burst transmit function with frequency hopping
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void tx_ber(unsigned char *txBuffer, trx_cfg_struct *trx_cfg, char RF_IDx) {
  unsigned int bb, cc, dd;
  unsigned int cc_sweeps;
  unsigned short rx_length, data_error;
  signed int rssi_forward, rssi_reverse;
  unsigned int status, total_pkt_count;
  unsigned long error_forward, error_reverse;
  unsigned long total_pay_tx, total_pay_rx;
  signed long   avg_rssi_forward, avg_rssi_reverse;
  unsigned char *current_hop_seq;
  unsigned char current_hop_length;
  
  uartSendString("\n\r");  
  uartSendString("Perform RF burst transmit function with frequency hopping\n\r");
  uartSendString("Using frequency hopping (50 channels, spaced at 5x channel spacing)\n\r");

  // We have to make sure we are using the correct hopping sequence, due to 
  // stack limitation that generation of the sequence is already done
  current_hop_seq = chan_hop_seq;
  current_hop_length = MAX_HOP_CHANNELS;
  for(bb=0;bb<MAX_HOP_CHANNELS;bb++) {
    sprintf(u_str, "%3d %3d\n\r", bb, chan_hop_seq[bb]);
    uartSendString(u_str);
  }  

  HAL_LED1_OFF();                                             // Initialize LED1
  HAL_LED2_OFF();                                             // Initialize LED2
  
  error_forward = 0;
  error_reverse = 0;
  avg_rssi_forward = 0;
  avg_rssi_reverse = 0;
  total_pkt_count = 0;
  // we know that we will be performing 50 channels per sweeps, lets figure out
  // how many sweeps we have to do based on number of burst requested by user.
  cc_sweeps = trx_cfg->no_burst / 50;
  // start the timer
  start_mac_timer();
  // start transmitting burst, one on each of the 50 channels in the PN sequence
  
  
  for(cc=0;cc<cc_sweeps;cc++){
    for(bb=0;bb<current_hop_length;bb++){
      
      // enter low power mode and wait for TimerA1
      radio_mode = RADIO_TX_MODE;
      _BIS_SR(LPM3_bits + GIE);                                    // Enter LPM0
      radio_mode = RADIO_TX_ACTIVE;      
      
      // Wake radio from sleep
      radio_idle(RF_IDx);
      __delay_cycles(8000);  // 1ms delay while the XTAL starts up
      
      // put some data into the payload
      txBuffer[0] =  SYSTEM_ID;                            // System ID
      dd = cc * current_hop_length + bb;
      txBuffer[1] = (dd>>8) & 0xFF;                        // Burst number (MSB)
      txBuffer[2] =  dd     & 0xFF;                        // Burst number (LSB)
      for(dd=3;dd<trx_cfg->b_length;dd++){
        txBuffer[dd] = rand_data[dd];
      }
      // Transmit packet. The MCU does not exit this until the TX is complete.
      HAL_LED2_ON();
      radio_set_channel(current_hop_seq[bb], RF_IDx);
      radio_send(txBuffer, trx_cfg->b_length, RF_IDx);
      radio_wait_for_idle(0,  RF_IDx);         // 0 = no timeout, just wait
      HAL_LED2_OFF();
      
      if(ENABLE_TWO_WAY_LINK == TRUE) {
        HAL_LED2_ON();
        radio_receive_on(RF_IDx);              // Change state to RX, recieve packet
        // wait until end_of_packet is found or timeout (if packet is not found)
        status = radio_wait_for_idle(MAX_WAIT_NORMAL,  RF_IDx);  
        HAL_LED2_OFF();
        
        if(status < MAX_WAIT_NORMAL) {
          
          rx_length = trx_cfg->b_length + 2;
          radio_read(txBuffer, &rx_length, RF_IDx);
          
          /* calculate the RSSI from the information from the RADIO registers */
          rssi_reverse = calculate_rssi(txBuffer[rx_length+RSSI_RX]);
          rssi_forward = calculate_rssi(txBuffer[3]);
          
          data_error = 0;
          for(dd=4;dd<trx_cfg->b_length;dd++){
            if( txBuffer[dd] != rand_data[dd] ) {
              data_error++;
            }
          }
          dd = (txBuffer[1] << 8) + txBuffer[2];
          error_forward = error_forward + txBuffer[4];
          error_reverse = error_reverse + data_error;
          avg_rssi_forward = avg_rssi_forward + rssi_forward;
          avg_rssi_reverse = avg_rssi_reverse + rssi_reverse;
          total_pkt_count++;
        } else {
          HAL_LED1_ON();
          __delay_cycles(10000);         // this is just so we can see the LED blink
          radio_idle(RF_IDx);           // Force IDLE state and Flush the RX FIFO's
          rssi_forward = 0;
          rssi_reverse = 0;
          data_error   = trx_cfg->b_length;
          HAL_LED1_OFF();
        }                  
      }
      // Force the radio into sleep mode for ultra low power operation
      radio_sleep(RF_IDx);

#ifdef ENABLE_DEBUG_UART
      sprintf(u_str, "%5d %5d %3i %3i %5d %3i %3i %5u\n\r", cc, chan_hop_seq[bb], 
              rssi_forward, txBuffer[4], dd, rssi_reverse, data_error, status);
      uartSendString(u_str);
#endif
    }  
  }
  stop_mac_timer();
  radio_mode = RADIO_IDLE;
  
  total_pay_tx = trx_cfg->no_burst;
  total_pay_tx = total_pay_tx * trx_cfg->b_length;  
  total_pay_rx = total_pkt_count;
  total_pay_rx = total_pay_rx * trx_cfg->b_length;  
  
  sprintf(u_str, "Total packets Sent     : %5d\n\r", trx_cfg->no_burst);
  uartSendString(u_str);
  sprintf(u_str, "Total packets Received : %5d\n\r", total_pkt_count);
  uartSendString(u_str);
  sprintf(u_str, "Total Payload Bytes TX : %7ld\n\r", total_pay_tx);
  uartSendString(u_str);
  sprintf(u_str, "Total Payload Bytes RX : %7ld\n\r", total_pay_rx);
  uartSendString(u_str);
  sprintf(u_str, "Forward Error count    : %7ld\n\r", error_forward);
  uartSendString(u_str);
  sprintf(u_str, "Reverse Error count    : %7ld\n\r", error_reverse);
  uartSendString(u_str);
  sprintf(u_str, "Forward Error rate     : %5d\n\r", 
          (int)((error_forward*1000)/ (trx_cfg->no_burst * trx_cfg->b_length)));
  uartSendString(u_str);
  sprintf(u_str, "Reverse Error rate     : %5d\n\r", 
          (int)((error_reverse*1000)/ (trx_cfg->no_burst * trx_cfg->b_length)));
  uartSendString(u_str);
  sprintf(u_str, "Forward Average RSSI   : %5d\n\r", 
          (int)(avg_rssi_forward/total_pkt_count));
  uartSendString(u_str);
  sprintf(u_str, "Reverse Average RSSI   : %5d\n\r", 
          (int)(avg_rssi_reverse/total_pkt_count));
  uartSendString(u_str);
  
  return;
}


/******************************************************************************
* rx_ber : Implement base station function with frequency hopping
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
* #define RADIO_RX_MODE           0x82
* #define RADIO_TX_MODE           0x83
* #define RADIO_RX_ACTIVE         0x84
* #define RADIO_TX_ACTIVE         0x85
* BURST_SKIP_NUMBER         30   // We will skip 30 burst to save power
* BURST_LOST_SOFT_LIMIT      4   // we will alert the user at this limit
* BURST_LOST_HARD_LIMIT    100   // assume the RF link is broken and restart
******************************************************************************/
void rx_ber(unsigned char *txBuffer, trx_cfg_struct *trx_cfg, char RF_IDx) {
  int channel_index, mis_packet_counter, bb, dd, burst_skip_counter; 
  unsigned short rx_length, max_wait_current;
  unsigned char done = FALSE;
  unsigned int status;
  unsigned char *current_hop_seq;
  unsigned char current_hop_length;
#ifdef ENABLE_DEBUG_UART
  unsigned int curr_pkt_id;
#endif   
  uartSendString("\n\r");  
  uartSendString("Perform RF burst recieve function with frequency hopping\n\r");
  uartSendString("Using frequency hopping (50 channels, spaced at 5x channel spacing)\n\r");
  
  // We have to make sure we are using the correct hopping sequence, due to 
  // stack limitation that generation of the sequence is already done
  current_hop_seq = chan_hop_seq;
  current_hop_length = MAX_HOP_CHANNELS;
  for(bb=0;bb<MAX_HOP_CHANNELS;bb++) {
    sprintf(u_str, "%3d %3d\n\r", bb, chan_hop_seq[bb]);
    uartSendString(u_str);
  }  
  
  HAL_LED1_OFF();                        // Initialize LED1
  HAL_LED2_OFF();                        // Initialize LED2
  
  user_button_pushed = 0;
  
  // Initialize various control variable used in the FHSS algorithm
  channel_index = 0;      // determines the current channels index (lookup value)
  mis_packet_counter = 0;     // counts how many packets we have missed in a row
  // Sets teh mode in which we operate: 
  // MAX_WAIT_SEARCH; indicates that we are not locked to a basestation and we 
  //                  need to perform adaptive seeking of basestation signal
  // MAX_WAIT_NORMAL: We are locked to a basestation and should proceed in 
  //                  normal operation
  max_wait_current = MAX_WAIT_SEARCH;   // sets the mode it which we operate
  
  // start the medium access control timer, this timer is used to figure out when
  // to wake up to recieve a new burst.
  start_mac_timer();
  done = FALSE;
  burst_skip_counter = BURST_SKIP_NUMBER;
  burst_skip_mode = FALSE; // start out by not skipping any channels
  while (done == FALSE) {    
    // Wait for the next burst time slot
    HAL_LED1_OFF();                     
    radio_mode = RADIO_RX_MODE;
    _BIS_SR(LPM3_bits + GIE);           // Enter LPM0
    radio_mode = RADIO_RX_ACTIVE;
    HAL_LED1_ON();                      
    HAL_LED2_OFF();                     // turn off the pass indictor
    // Update the RX channel and enable RX
    if (channel_index < (current_hop_length-1) ) {
      channel_index++;
    } else {
      channel_index = 0;
    }
    
    // Enable the skipping of burst to save power, this statement figures out
    // if we need this particular burst or not. If not the entire RX is skipped
    // and the MCU returns to sleep (measured to about 10us todal on time)
    if(burst_skip_counter == 0) {
      burst_skip_counter = BURST_SKIP_NUMBER;
    } else {
      burst_skip_counter--;
    }
    if((burst_skip_mode == FALSE) || (burst_skip_counter == BURST_SKIP_NUMBER) ) {
      
      // Wake radio from sleep
      radio_idle(RF_IDx);
      __delay_cycles(8000);  // 1ms delay while the XTAL starts up
      
      radio_set_channel(current_hop_seq[channel_index], RF_IDx); // set the channel
      
      radio_receive_on(RF_IDx);              // Change state to RX, recieve packet
      
      // wait until end_of_packet is found, but with timeout (if packet is not found)
      status = radio_wait_for_idle(max_wait_current,  RF_IDx);
      
      //the "mac_wait_trx" functions is being used in both modes, the two modes are:
      // max_wait = 0 : infinite wait for RX burst to arrive
      // max_wait > 0 : wait on RX for a specified amount of time
      switch (max_wait_current) {
        // This case we have been waiting for a burst with RX continuesly running
      case MAX_WAIT_SEARCH:
        // This means we have found a packet and should syncronize to it
        if(status < MAX_WAIT_SEARCH) { 
          // Aligns timer for next burst to arrive 204 bit at 38400bit/s 
#ifdef MSP430_F5xx
          TA1R = TIMER_ALIGNMENT_VALUE;      
#else 
          TBR = TIMER_ALIGNMENT_VALUE;      
#endif
          max_wait_current = MAX_WAIT_NORMAL;         // 8192 = 8ms (256 bit/38400 = 7ms)      
          rx_length = trx_cfg->b_length + 2;
          radio_read(txBuffer, &rx_length, RF_IDx);         // read content of FIFO
          
          if(txBuffer[0] == SYSTEM_ID) {
            txBuffer[3] = txBuffer[rx_length + RSSI_RX];      // insert measurement in
            txBuffer[4] = 0;                                  // return data
            
            for(dd=5;dd<trx_cfg->b_length;dd++){              // count errors in data
              if( txBuffer[dd] != rand_data[dd] ) {
                txBuffer[4] = txBuffer[4] + 1;                // error count is byte 3
                txBuffer[dd] = rand_data[dd];                 // clean data for return
              }
            }
            if(ENABLE_TWO_WAY_LINK == TRUE) {
              radio_send(txBuffer, trx_cfg->b_length, RF_IDx);  // standard packet send
              // wait until end_of_packet is sent
              radio_wait_for_idle(0,  RF_IDx);         // here not strictly nessecary
            }
            
#ifdef ENABLE_DEBUG_UART
            curr_pkt_id = (txBuffer[1]<<8) + txBuffer[2];
            sprintf(u_str, "FB:%d,%i,%d,%d,%d,%i\n\r", txBuffer[0], curr_pkt_id, 
                    channel_index ,current_hop_seq[channel_index], 
                    calculate_rssi(txBuffer[3]), status);
            uartSendString(u_str);    
#endif
            mis_packet_counter = 0;                
          }
        }
        // force a backwards channel hopping sequence. This helps with multipath and 
        // shortens the connection worst case wait for connect.
        if(status == MAX_WAIT_SEARCH) { 
          radio_idle(RF_IDx);                          // force idle and flush fifos
          if(channel_index >= 1) {
            channel_index = channel_index - 2;
          } else {
            channel_index = current_hop_length - 2;
          }
        }
        break;
        // default setting is that we have been running a standard RX burst, check
        // to see if the burs has been found or not.
      case MAX_WAIT_NORMAL:
        if(status <  max_wait_current ) {
          mis_packet_counter = 0;
          if(BURST_SKIP_NUMBER > 0) {
             burst_skip_mode = TRUE;
          }
          // This step is for trying to calibrate the internal 32K reference to
          // reference of the transmitter. If it is noted that the error is more
          // 4 values from the target, we change the system timing by 1/2 of that
          // value. This makes a very stabil but slowly locking system.
#ifdef MSP430_F5xx
          //if(TA1R > TA1R_ALIGNMENT_VALUE + 4 ) {
          //  TA1CCR0 = TA1CCR0 + 1;
          //}
          //if(TA1R < TA1R_ALIGNMENT_VALUE - 4 ) {
          //  TA1CCR0 = TA1CCR0 - 1;
          //}        
          TA1R = TIMER_ALIGNMENT_VALUE;     // Aligns timer for next burst to arrive 
#else 
          //if(TBR > TIMER_ALIGNMENT_VALUE + 4 ) {
          //  TBCCR0 = TBCCR0 + 1;
          //}
          //if(TAR < TIMER_ALIGNMENT_VALUE - 4 ) {
          //  TBCCR0 = TBCCR0 - 1;
          //}        
          TBR = TIMER_ALIGNMENT_VALUE;     // Aligns timer for next burst to arrive 
#endif        
          rx_length = trx_cfg->b_length + 2;
          radio_read(txBuffer, &rx_length, RF_IDx);         // read content of FIFO
          
          txBuffer[3] = txBuffer[rx_length + RSSI_RX];      // insert measurement in
          txBuffer[4] = 0;                                  // return data
          
          for(dd=5;dd<trx_cfg->b_length;dd++){              // count errors in data
            if( txBuffer[dd] != rand_data[dd] ) {
              txBuffer[4] = txBuffer[4] + 1;                // error count is byte 3
              txBuffer[dd] = rand_data[dd];                 // clean data for return
            }
          }          
          if(ENABLE_TWO_WAY_LINK == TRUE) {
            radio_send(txBuffer, trx_cfg->b_length, RF_IDx);  // standard packet send
            // wait until end_of_packet is sent
            status = radio_wait_for_idle(0,  RF_IDx);         // here not strictly nessecary
          }
          
#ifdef ENABLE_DEBUG_UART        
          curr_pkt_id = (txBuffer[1]<<8) + txBuffer[2];
            sprintf(u_str, "NB:%d,%i,%d,%d,%d,%i\n\r", txBuffer[0], curr_pkt_id, 
                    channel_index ,current_hop_seq[channel_index], 
                    calculate_rssi(txBuffer[3]), status);
          uartSendString(u_str);    
#else
          if(txBuffer[4]==0) {
            uartSendString("1\n\r");
          } else {
            uartSendString("0\n\r");
          }
#endif          
        } else { 
          // Check to see if we have lost the connection and we need to stop and hold
          radio_idle(RF_IDx);                          // force idle and flush fifos
          // if max_wait == 0 that means we are waiting for first sync burst to appear
          
          if(mis_packet_counter > BURST_LOST_HARD_LIMIT) {
            max_wait_current = MAX_WAIT_SEARCH;  
          } else {
            mis_packet_counter++;
            burst_skip_mode = FALSE;
          }
          // If we have somehow gotten a hold of TX ID that does not belong to us
          // go back and perform a search for ours TX ID again.
          //if(txBuffer[0] != SYSTEM_ID) {
          //  max_wait_current = MAX_WAIT_SEARCH;  
          //}
          // Only light the RED led when we have missed 4 packets in a row.
          if(mis_packet_counter > BURST_LOST_SOFT_LIMIT) {
            HAL_LED1_ON();
          }
#ifdef ENABLE_DEBUG_UART
            sprintf(u_str, "MB:%i,%d,%d,%i,%i\n\r", mis_packet_counter, 
                    channel_index ,current_hop_seq[channel_index], 
                    calculate_rssi(txBuffer[3]), status);
          uartSendString(u_str);    
#endif
        }    
        break;
      }
      // Force the radio into sleep mode for ultra low power operation
      radio_sleep(RF_IDx);
    }
    // Check to see if the user button has been pressed, exit the rx_ber()
    if(user_button_pushed == 1) {
      user_button_pushed = 0;
      //done = TRUE;
    }
  }
  stop_mac_timer();
  
  return;   
}

#ifdef MSP430_F5xx
/*******************************************************************************
* @brief  Timer_A3 Interrupt Vector (TAIV) handler
* 
* @param  none
* 
* @return none
*******************************************************************************/
#pragma vector=TIMER1_A1_VECTOR
__interrupt void TIMER1_A1_ISR(void) {
  
  switch(__even_in_range(TA1IV,16)) {
  case TA1IV_NONE:                        // No interrupt
    break;                       
  case TA1IV_TA1CCR1:                     // Used to wake up radio from sleep
    if(radio_mode == RADIO_RX_MODE) {
      _BIC_SR_IRQ(LPM3_bits);             // Clear LPM0 bits from 0(SR)      
    }
    //HAL_LED1_ON();                   // LED 1 on
    break;    
  case  TA1IV_TA1CCR2:                    // 1 Second pulse (TX or RX)
    if(radio_mode == RADIO_TX_MODE) {
      _BIC_SR_IRQ(LPM3_bits);             // Clear LPM0 bits from 0(SR)      
    }
    //HAL_LED1_OFF();                  // LED 1 off
    break;     
  case  TA1IV_3: break;          // reserved
  case  TA1IV_4: break;          // reserved
  case  TA1IV_5: break;          // reserved
  case  TA1IV_6: break;          // reserved
  case  TA1IV_TA1IFG:            // TA1IFG
    break;
  default: 
    break; 
  }
  return;
}

#else 
/*******************************************************************************
* @brief  Timer_B0 Interrupt Vector (TAIV) handler
* 
* @param  none
* 
* @return none
*******************************************************************************/
#pragma vector=TIMERB1_VECTOR
__interrupt void TIMERB1_VECTOR_ISR(void) {
  
  switch(__even_in_range(TBIV,16)) {
  case TBIV_NONE:                        // No interrupt
    break;                       
  case TBIV_TBCCR1:                     // Used to wake up radio from sleep
    if(radio_mode == RADIO_RX_MODE) {
      _BIC_SR_IRQ(LPM3_bits);             // Clear LPM0 bits from 0(SR)      
    }
    //HAL_LED1_ON();                   // LED 1 on
    break;    
  case  TBIV_TBCCR2:                    // 1 Second pulse (TX or RX)
    if(radio_mode == RADIO_TX_MODE) {
      _BIC_SR_IRQ(LPM0_bits);             // Clear LPM0 bits from 0(SR)      
    }
    //HAL_LED1_OFF();                  // LED 1 off
    break;     
  case  TBIV_3: break;          // reserved
  case  TBIV_4: break;          // reserved
  case  TBIV_5: break;          // reserved
  case  TBIV_6: break;          // reserved  
  case  TBIV_TBIFG:            // TA1IFG
    break;
  default: 
    break; 
  }
  
  return;
}

/*******************************************************************************
* @brief  Timer_B0 Interrupt Vector (TAIV) handler
* 
* @param  none
* 
* @return none
*******************************************************************************/
#pragma vector=TIMERB0_VECTOR
__interrupt void TIMERB0_VECTOR_ISR(void) {
  
  switch(__even_in_range(TBIV,16)) {
  case TBIV_NONE:                        // No interrupt
    break;                       
  case  TBIV_TBCCR1:                     // Used to wake up radio from sleep
  case  TBIV_TBCCR2:                    // 1 Second pulse (TX or RX)
  case  TBIV_3: break;          // reserved
  case  TBIV_4: break;          // reserved
  case  TBIV_5: break;          // reserved
  case  TBIV_6: break;          // reserved  
  case  TBIV_TBIFG:            // TA1IFG
    break;
  default: 
    break; 
  }
  
  return;
}
#endif

/******************************************************************************
* Transmit : Perform RF burst transmit function
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void rx_sweep(unsigned char *txBuffer, trx_cfg_struct *trx_cfg, char RF_IDx) {
  int cc,dd, rssi;
  unsigned char t_chan;
  long rx_freq, freq_span, freq_steps, new_rx_freq, t_wrap;
  
  uartSendString("\n\r");  
  uartSendString("Started RX sweep analyzer:\n\r");
  
  for(dd=0;dd<trx_cfg->no_burst;dd++){
    
    radio_set_channel(0, RF_IDx);
    radio_receive_on(RF_IDx);
    uartSendString("\n\r");
    uartSendString("FREQ   RSSI\n\r");
    
    
    freq_span = trx_cfg->stop_freq - trx_cfg->start_freq;
    freq_steps = freq_span / trx_cfg->ch_spc;
    t_wrap = 0;
    for(cc=0;cc<freq_steps;cc++){
      
      rssi = get_rssi(RF_IDx);
      radio_idle(RF_IDx);                        // force idle and flush fifos
      
      /* calculate the new channel number, if wrapping is needed program a new*/
      /* base frequency and then start the channel numbers from start again */
      t_chan = cc&0xFF;
      if(t_chan == 0) {
        new_rx_freq = t_wrap * 256L * trx_cfg->ch_spc;
        new_rx_freq = trx_cfg->start_freq + new_rx_freq;
        rf_set_freq( new_rx_freq, RF_IDx);
        t_wrap++;
      }
      radio_set_channel(t_chan, RF_IDx);       // channel rf channel
      radio_receive_on(RF_IDx);                // enable RX
      /* calculate the actual RF frequency that was used for RSSI             */
      rx_freq = trx_cfg->ch_spc;
      rx_freq = rx_freq * cc;
      rx_freq = trx_cfg->start_freq + rx_freq;
      /* print the RSSI results on the UART */
      sprintf(u_str, "%ld %i\n\r", rx_freq, rssi); 
      uartSendString(u_str);                
    }
  }

  return;
}
/******************************************************************************
* Transmit : Perform RF burst transmit function
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void rx_scan(unsigned char *txBuffer, trx_cfg_struct *trx_cfg, char RF_IDx) {
  int cc, rssi;
  long rx_freq;
  int no_bursts;
  
  uartSendString("\n\r");  
  uartSendString("Started RX static channel RF level analyzer:\n\r");
  
  radio_set_channel(trx_cfg->rf_channel, RF_IDx);       // channel rf channel
  radio_receive_on(RF_IDx);                // enable RX
  
  uartSendString("\n\r");
  /* calculate the actual RF frequency that was used for RSSI             */
  rx_freq = trx_cfg->ch_spc;
  rx_freq = rx_freq * trx_cfg->rf_channel;
  rx_freq = trx_cfg->start_freq + rx_freq;
  sprintf(u_str, "FREQ = %ld\n\r", rx_freq);
  uartSendString(u_str);
  uartSendString("COUNT  RSSI\n\r");
  
  no_bursts = trx_cfg->no_burst;
  for(cc=0;cc<no_bursts;cc++){
    rssi = get_rssi(RF_IDx);
    radio_idle(RF_IDx);                        // force idle and flush fifos
    radio_receive_on(RF_IDx);                // enable RX
    
    /* calculate the RSSI and print the results on the UART */
    
    sprintf(u_str, "%i %i\n\r", cc, rssi);
    uartSendString(u_str);
  }
  
  return;
}

/******************************************************************************
* Transmit : Perform RF burst transmit function
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void uartSendString(char *UART_Text) {
  unsigned char ii;
  
  for(ii=0;ii<UART_BUFF_SIZE;ii++) {
    if(UART_Text[ii] == 13) {
      halUsbSendString(UART_Text, ii+1);
      ii = UART_BUFF_SIZE;
    }
  }
  
  return;
}


/******************************************************************************
* Calculate RSSI from unsigned char value from Radio
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
int calculate_rssi(unsigned char cc_rssi) {
  int rssi;
  
  if (cc_rssi >= 128) {
    rssi = ((cc_rssi-256)>>1) - 72;
  } else {
    rssi = (cc_rssi>>1) - 72;
  }
  
  return rssi; 
}

/******************************************************************************
* print_rf_settings
*
* Thomas Almholt, Texas Instruments Inc.
* February 2010
* Built with IAR 5.10.1 (5.10.1.50144)
******************************************************************************/
void print_rf_settings(char RF_IDx) {
  unsigned char status;
  
  // Read register settings
  status = CC_SPIReadReg(IOCFG2, RF_IDx); // GDO2 output pin config.
  sprintf(u_str, "IOCFG2,%d,%d\n\r", status, 0x06);
  uartSendString(u_str);
  status = CC_SPIReadReg(IOCFG0,  RF_IDx); // GDO0 output pin config.
  sprintf(u_str, "IOCFG0,%d,%d\n\r", status, 0x06);
  uartSendString(u_str);
  status = CC_SPIReadReg(PKTLEN,  RF_IDx); // Packet length.
  sprintf(u_str, "PKTLEN,%d,%d\n\r", status, 0x0A);
  uartSendString(u_str);
  status = CC_SPIReadReg(PKTCTRL1, RF_IDx); // Packet automation control.
  sprintf(u_str, "PKTCTRL1,%d,%d\n\r", status, 0x04);
  uartSendString(u_str);
  status = CC_SPIReadReg(PKTCTRL0, RF_IDx); // Packet automation control.
  sprintf(u_str, "PKTCTRL0,%d,%d\n\r", status, 0x44);
  uartSendString(u_str);
  status = CC_SPIReadReg(ADDR, RF_IDx); // Device address.
  sprintf(u_str, "ADDR,%d,%d\n\r", status, 0x00);
  uartSendString(u_str);
  status = CC_SPIReadReg(CHANNR, RF_IDx); // Channel number.
  sprintf(u_str, "CHANNR,%d,%d\n\r", status, 0x00);
  uartSendString(u_str);
  status = CC_SPIReadReg(FSCTRL1, RF_IDx); // Freq synthesizer control.
  sprintf(u_str, "FSCTRL1,%d,%d\n\r", status, 0x06);
  uartSendString(u_str);
  status = CC_SPIReadReg(FSCTRL0, RF_IDx); // Freq synthesizer control.
  sprintf(u_str, "FSCTRL0,%d,%d\n\r", status, 0x00);
  uartSendString(u_str);
  status = CC_SPIReadReg(FREQ2, RF_IDx); // Freq control word, high byte
  sprintf(u_str, "FREQ2,%d,%d\n\r", status, 0x22);
  uartSendString(u_str);
  status = CC_SPIReadReg(FREQ1,  RF_IDx); // Freq control word, mid byte.
  sprintf(u_str, "FREQ1,%d,%d\n\r", status, 0xB1);
  uartSendString(u_str);
  status = CC_SPIReadReg(FREQ0,  RF_IDx); // Freq control word, low byte.
  sprintf(u_str, "FREQ0,%d,%d\n\r", status, 0x3B);
  uartSendString(u_str);
  status = CC_SPIReadReg(MDMCFG4, RF_IDx); // Modem configuration.
  sprintf(u_str, "MDMCFG4,%d,%d\n\r", status, 0xCA);
  uartSendString(u_str);
  status = CC_SPIReadReg(MDMCFG3, RF_IDx); // Modem configuration.
  sprintf(u_str, "MDMCFG3,%d,%d\n\r", status, 0x83);
  uartSendString(u_str);
  status = CC_SPIReadReg(MDMCFG2, RF_IDx); // Modem configuration.
  sprintf(u_str, "MDMCFG2,%d,%d\n\r", status, 0x13);
  uartSendString(u_str);
  status = CC_SPIReadReg(MDMCFG1, RF_IDx); // Modem configuration.
  sprintf(u_str, "MDMCFG1,%d,%d\n\r", status, 0x21);
  uartSendString(u_str);
  status = CC_SPIReadReg(MDMCFG0, RF_IDx); // Modem configuration.
  sprintf(u_str, "MDMCFG0,%d,%d\n\r", status, 0xF8);
  uartSendString(u_str);
  status = CC_SPIReadReg(DEVIATN, RF_IDx); // Modem dev (when FSK mod en)
  sprintf(u_str, "DEVIATN,%d,%d\n\r", status, 0x35);
  uartSendString(u_str);
  status = CC_SPIReadReg(MCSM2 , RF_IDx); // MainRadio Cntrl State Machine    
  sprintf(u_str, "MCSM2,%d,%d\n\r", status, 0x07);
  uartSendString(u_str);
  status = CC_SPIReadReg(MCSM1 ,  RF_IDx); // MainRadio Cntrl State Machine , normal
  sprintf(u_str, "MCSM1,%d,%d\n\r", status, 0x30);
  uartSendString(u_str);
  status = CC_SPIReadReg(MCSM0 , RF_IDx); // MainRadio Cntrl State Machine
  sprintf(u_str, "MCSM0,%d,%d\n\r", status, 0x10);
  uartSendString(u_str);
  status = CC_SPIReadReg(FOCCFG, RF_IDx); // Freq Offset Compens. Config
  sprintf(u_str, "FOCCFG,%d,%d\n\r", status, 0x16);
  uartSendString(u_str);
  status = CC_SPIReadReg(BSCFG,  RF_IDx); // Bit synchronization config.
  sprintf(u_str, "BSCFG,%d,%d\n\r", status, 0x6C);
  uartSendString(u_str);
  status = CC_SPIReadReg(AGCCTRL2, RF_IDx); // AGC control.
  sprintf(u_str, "AGCCTRL2,%d,%d\n\r", status, 0x43);
  uartSendString(u_str);
  status = CC_SPIReadReg(AGCCTRL1, RF_IDx); // AGC control.
  sprintf(u_str, "AGCCTRL1,%d,%d\n\r", status, 0x40);
  uartSendString(u_str);
  status = CC_SPIReadReg(AGCCTRL0, RF_IDx); // AGC control.
  sprintf(u_str, "AGCCTRL0,%d,%d\n\r", status, 0x91);
  uartSendString(u_str);
  status = CC_SPIReadReg(FREND1, RF_IDx); // Front end RX configuration.
  sprintf(u_str, "FREND1,%d,%d\n\r", status, 0x56);
  uartSendString(u_str);
  status = CC_SPIReadReg(FREND0,  RF_IDx); // Front end RX configuration.
  sprintf(u_str, "FREND0,%d,%d\n\r", status, 0x10);
  uartSendString(u_str);
  status = CC_SPIReadReg(FSCAL3, RF_IDx); // Frequency synthesizer cal.
  sprintf(u_str, "FSCAL3,%d,%d\n\r", status, 0xE9);
  uartSendString(u_str);
  status = CC_SPIReadReg(FSCAL2, RF_IDx); // Frequency synthesizer cal.
  sprintf(u_str, "FSCAL2,%d,%d\n\r", status, 0x2A);
  uartSendString(u_str);
  status = CC_SPIReadReg(FSCAL1, RF_IDx); // Frequency synthesizer cal.
  sprintf(u_str, "FSCAL1,%d,%d\n\r", status, 0x00);
  uartSendString(u_str);
  status = CC_SPIReadReg(FSCAL0, RF_IDx); // Frequency synthesizer cal.
  sprintf(u_str, "FSCAL0,%d,%d\n\r", status, 0x1F);
  uartSendString(u_str);
  status = CC_SPIReadReg(FSTEST, RF_IDx); // Frequency synthesizer cal.
  sprintf(u_str, "FSTEST,%d,%d\n\r", status, 0x59);
  uartSendString(u_str);
  status = CC_SPIReadReg(TEST2,  RF_IDx); // Various test settings.
  sprintf(u_str, "TEST2,%d,%d\n\r", status, 0x81);
  uartSendString(u_str);
  status = CC_SPIReadReg(TEST1,  RF_IDx); // Various test settings.
  sprintf(u_str, "TEST1,%d,%d\n\r", status, 0x35);
  uartSendString(u_str);
  status = CC_SPIReadReg(TEST0, RF_IDx); // Various test settings.
  sprintf(u_str, "TEST0,%d,%d\n\r", status, 0x09);
  uartSendString(u_str);
  
  status = CC_SPIReadReg(PATABLE, RF_IDx);     //Write PATABLE
  sprintf(u_str, "PATABLE,%d,%d\n\r", status, 0x50);
  uartSendString(u_str);
}


/* RF settings SoC  CC1120 RF IC*/
/*
void print_rf_settings(char RF_IDx) {
char u_str[40];
sprintf(u_str, "IOCFG3           = %3d\n\r", CC_SPIReadReg(IOCFG3,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IOCFG2           = %3d\n\r", CC_SPIReadReg(IOCFG2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IOCFG1           = %3d\n\r", CC_SPIReadReg(IOCFG1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IOCFG0           = %3d\n\r", CC_SPIReadReg(IOCFG0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "SYNC3            = %3d\n\r", CC_SPIReadReg(SYNC3,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "SYNC2            = %3d\n\r", CC_SPIReadReg(SYNC2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "SYNC1            = %3d\n\r", CC_SPIReadReg(SYNC1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "SYNC0            = %3d\n\r", CC_SPIReadReg(SYNC0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "SYNC_CFG1        = %3d\n\r", CC_SPIReadReg(SYNC_CFG1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "SYNC_CFG0        = %3d\n\r", CC_SPIReadReg(SYNC_CFG0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "DEVIATION_M      = %3d\n\r", CC_SPIReadReg(DEVIATION_M,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "MODCFG_DEV_E     = %3d\n\r", CC_SPIReadReg(MODCFG_DEV_E,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "DCFILT_CFG       = %3d\n\r", CC_SPIReadReg(DCFILT_CFG,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PREAMBLE_CFG1    = %3d\n\r", CC_SPIReadReg(PREAMBLE_CFG1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PREAMBLE_CFG0    = %3d\n\r", CC_SPIReadReg(PREAMBLE_CFG0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FREQ_IF_CFG      = %3d\n\r", CC_SPIReadReg(FREQ_IF_CFG,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IQIC             = %3d\n\r", CC_SPIReadReg(IQIC,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "CHAN_BW          = %3d\n\r", CC_SPIReadReg(CHAN_BW,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "MDMCFG1          = %3d\n\r", CC_SPIReadReg(MDMCFG1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "MDMCFG0          = %3d\n\r", CC_SPIReadReg(MDMCFG0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "DRATE2           = %3d\n\r", CC_SPIReadReg(DRATE2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "DRATE1           = %3d\n\r", CC_SPIReadReg(DRATE1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "DRATE0           = %3d\n\r", CC_SPIReadReg(DRATE0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "AGC_REF          = %3d\n\r", CC_SPIReadReg(AGC_REF,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "AGC_CS_THR       = %3d\n\r", CC_SPIReadReg(AGC_CS_THR,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "AGC_GAIN_ADJUST  = %3d\n\r", CC_SPIReadReg(AGC_GAIN_ADJUST,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "AGC_CFG3         = %3d\n\r", CC_SPIReadReg(AGC_CFG3,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "AGC_CFG2         = %3d\n\r", CC_SPIReadReg(AGC_CFG2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "AGC_CFG1         = %3d\n\r", CC_SPIReadReg(AGC_CFG1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "AGC_CFG0         = %3d\n\r", CC_SPIReadReg(AGC_CFG0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FIFO_CFG         = %3d\n\r", CC_SPIReadReg(FIFO_CFG,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "DEV_ADDR         = %3d\n\r", CC_SPIReadReg(DEV_ADDR,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "SETTLING_CFG     = %3d\n\r", CC_SPIReadReg(SETTLING_CFG,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_CFG           = %3d\n\r", CC_SPIReadReg(FS_CFG,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "WOR_CFG1         = %3d\n\r", CC_SPIReadReg(WOR_CFG1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "WOR_CFG0         = %3d\n\r", CC_SPIReadReg(WOR_CFG0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "WOR_EVENT0_MSB   = %3d\n\r", CC_SPIReadReg(WOR_EVENT0_MSB,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "WOR_EVENT0_LSB   = %3d\n\r", CC_SPIReadReg(WOR_EVENT0_LSB,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PKT_CFG2         = %3d\n\r", CC_SPIReadReg(PKT_CFG2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PKT_CFG1         = %3d\n\r", CC_SPIReadReg(PKT_CFG1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PKT_CFG0         = %3d\n\r", CC_SPIReadReg(PKT_CFG0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "RFEND_CFG1       = %3d\n\r", CC_SPIReadReg(RFEND_CFG1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "RFEND_CFG0       = %3d\n\r", CC_SPIReadReg(RFEND_CFG0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PA_CFG2          = %3d\n\r", CC_SPIReadReg(PA_CFG2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PA_CFG1          = %3d\n\r", CC_SPIReadReg(PA_CFG1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PA_CFG0          = %3d\n\r", CC_SPIReadReg(PA_CFG0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PKT_LEN          = %3d\n\r", CC_SPIReadReg(PKT_LEN,  RF_IDx));    
uartSendString(u_str);

sprintf(u_str, "IF_MIX_CFG       = %3d\n\r", CC_SPIReadExtReg(IF_MIX_CFG,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FREQOFF_CFG      = %3d\n\r", CC_SPIReadExtReg(FREQOFF_CFG,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "TOC_CFG          = %3d\n\r", CC_SPIReadExtReg(TOC_CFG,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "MARC_SPARE       = %3d\n\r", CC_SPIReadExtReg(MARC_SPARE,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "ECG_CFG          = %3d\n\r", CC_SPIReadExtReg(ECG_CFG,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "SOFT_TX_DATA_CFG = %3d\n\r", CC_SPIReadExtReg(SOFT_TX_DATA_CFG,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "EXT_CTRL         = %3d\n\r", CC_SPIReadExtReg(EXT_CTRL,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "RCCAL_FINE       = %3d\n\r", CC_SPIReadExtReg(RCCAL_FINE,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "RCCAL_COARSE     = %3d\n\r", CC_SPIReadExtReg(RCCAL_COARSE,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "RCCAL_OFFSET     = %3d\n\r", CC_SPIReadExtReg(RCCAL_OFFSET,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FREQOFF1         = %3d\n\r", CC_SPIReadExtReg(FREQOFF1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FREQOFF0         = %3d\n\r", CC_SPIReadExtReg(FREQOFF0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FREQ2            = %3d\n\r", CC_SPIReadExtReg(FREQ2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FREQ1            = %3d\n\r", CC_SPIReadExtReg(FREQ1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FREQ0            = %3d\n\r", CC_SPIReadExtReg(FREQ0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IF_ADC2          = %3d\n\r", CC_SPIReadExtReg(IF_ADC2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IF_ADC1          = %3d\n\r", CC_SPIReadExtReg(IF_ADC1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IF_ADC0          = %3d\n\r", CC_SPIReadExtReg(IF_ADC0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_DIG1          = %3d\n\r", CC_SPIReadExtReg(FS_DIG1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_DIG0          = %3d\n\r", CC_SPIReadExtReg(FS_DIG0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_CAL3          = %3d\n\r", CC_SPIReadExtReg(FS_CAL3,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_CAL2          = %3d\n\r", CC_SPIReadExtReg(FS_CAL2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_CAL1          = %3d\n\r", CC_SPIReadExtReg(FS_CAL1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_CAL0          = %3d\n\r", CC_SPIReadExtReg(FS_CAL0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_CHP           = %3d\n\r", CC_SPIReadExtReg(FS_CHP,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_DIVTWO        = %3d\n\r", CC_SPIReadExtReg(FS_DIVTWO,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_DSM1          = %3d\n\r", CC_SPIReadExtReg(FS_DSM1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_DSM0          = %3d\n\r", CC_SPIReadExtReg(FS_DSM0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_DVC1          = %3d\n\r", CC_SPIReadExtReg(FS_DVC1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_DVC0          = %3d\n\r", CC_SPIReadExtReg(FS_DVC0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_LBI           = %3d\n\r", CC_SPIReadExtReg(FS_LBI,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_PFD           = %3d\n\r", CC_SPIReadExtReg(FS_PFD,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_PRE           = %3d\n\r", CC_SPIReadExtReg(FS_PRE,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_REG_DIV_CML   = %3d\n\r", CC_SPIReadExtReg(FS_REG_DIV_CML,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_SPARE         = %3d\n\r", CC_SPIReadExtReg(FS_SPARE,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_VCO4          = %3d\n\r", CC_SPIReadExtReg(FS_VCO4,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_VCO3          = %3d\n\r", CC_SPIReadExtReg(FS_VCO3,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_VCO2          = %3d\n\r", CC_SPIReadExtReg(FS_VCO2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_VCO1          = %3d\n\r", CC_SPIReadExtReg(FS_VCO1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FS_VCO0          = %3d\n\r", CC_SPIReadExtReg(FS_VCO0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "GBIAS6           = %3d\n\r", CC_SPIReadExtReg(GBIAS6,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "GBIAS5           = %3d\n\r", CC_SPIReadExtReg(GBIAS5,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "GBIAS4           = %3d\n\r", CC_SPIReadExtReg(GBIAS4,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "GBIAS3           = %3d\n\r", CC_SPIReadExtReg(GBIAS3,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "GBIAS2           = %3d\n\r", CC_SPIReadExtReg(GBIAS2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "GBIAS1           = %3d\n\r", CC_SPIReadExtReg(GBIAS1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "GBIAS0           = %3d\n\r", CC_SPIReadExtReg(GBIAS0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IFAMP            = %3d\n\r", CC_SPIReadExtReg(IFAMP,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "LNA              = %3d\n\r", CC_SPIReadExtReg(LNA,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "RXMIX            = %3d\n\r", CC_SPIReadExtReg(RXMIX,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "XOSC5            = %3d\n\r", CC_SPIReadExtReg(XOSC5,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "XOSC4            = %3d\n\r", CC_SPIReadExtReg(XOSC4,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "XOSC3            = %3d\n\r", CC_SPIReadExtReg(XOSC3,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "XOSC2            = %3d\n\r", CC_SPIReadExtReg(XOSC2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "XOSC1            = %3d\n\r", CC_SPIReadExtReg(XOSC1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "XOSC0            = %3d\n\r", CC_SPIReadExtReg(XOSC0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "ANALOG_SPARE     = %3d\n\r", CC_SPIReadExtReg(ANALOG_SPARE,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IRQ0M            = %3d\n\r", CC_SPIReadExtReg(IRQ0M,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IRQ0F            = %3d\n\r", CC_SPIReadExtReg(IRQ0F,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PA_CFG3          = %3d\n\r", CC_SPIReadExtReg(PA_CFG3,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "WOR_TIME1        = %3d\n\r", CC_SPIReadExtReg(WOR_TIME1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "WOR_TIME0        = %3d\n\r", CC_SPIReadExtReg(WOR_TIME0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "WOR_CAPTURE1     = %3d\n\r", CC_SPIReadExtReg(WOR_CAPTURE1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "WOR_CAPTURE0     = %3d\n\r", CC_SPIReadExtReg(WOR_CAPTURE0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "BIST             = %3d\n\r", CC_SPIReadExtReg(BIST,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "DCFILTOFFSET_I1  = %3d\n\r", CC_SPIReadExtReg(DCFILTOFFSET_I1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "DCFILTOFFSET_I0  = %3d\n\r", CC_SPIReadExtReg(DCFILTOFFSET_I0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "DCFILTOFFSET_Q1  = %3d\n\r", CC_SPIReadExtReg(DCFILTOFFSET_Q1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "DCFILTOFFSET_Q0  = %3d\n\r", CC_SPIReadExtReg(DCFILTOFFSET_Q0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IQIE_I1          = %3d\n\r", CC_SPIReadExtReg(IQIE_I1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IQIE_I0          = %3d\n\r", CC_SPIReadExtReg(IQIE_I0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IQIE_Q1          = %3d\n\r", CC_SPIReadExtReg(IQIE_Q1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "IQIE_Q0          = %3d\n\r", CC_SPIReadExtReg(IQIE_Q0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "RSSI1            = %3d\n\r", CC_SPIReadExtReg(RSSI1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "RSSI0            = %3d\n\r", CC_SPIReadExtReg(RSSI0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "MARCSTATE        = %3d\n\r", CC_SPIReadExtReg(MARCSTATE,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "LQI_VAL          = %3d\n\r", CC_SPIReadExtReg(LQI_VAL,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PQT_SYNC_ERR     = %3d\n\r", CC_SPIReadExtReg(PQT_SYNC_ERR,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "DEM_STATUS       = %3d\n\r", CC_SPIReadExtReg(DEM_STATUS,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FREQOFF_EST1     = %3d\n\r", CC_SPIReadExtReg(FREQOFF_EST1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FREQOFF_EST0     = %3d\n\r", CC_SPIReadExtReg(FREQOFF_EST0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "AGC_GAIN3        = %3d\n\r", CC_SPIReadExtReg(AGC_GAIN3,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "AGC_GAIN2        = %3d\n\r", CC_SPIReadExtReg(AGC_GAIN2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "AGC_GAIN1        = %3d\n\r", CC_SPIReadExtReg(AGC_GAIN1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "AGC_GAIN0        = %3d\n\r", CC_SPIReadExtReg(AGC_GAIN0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "SOFT_RX_DATA_OUT = %3d\n\r", CC_SPIReadExtReg(SOFT_RX_DATA_OUT,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "SOFT_TX_DATA_IN  = %3d\n\r", CC_SPIReadExtReg(SOFT_TX_DATA_IN,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "ASK_SOFT_RX_DATA = %3d\n\r", CC_SPIReadExtReg(ASK_SOFT_RX_DATA,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "RNDGEN           = %3d\n\r", CC_SPIReadExtReg(RNDGEN,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "MAGN2            = %3d\n\r", CC_SPIReadExtReg(MAGN2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "MAGN1            = %3d\n\r", CC_SPIReadExtReg(MAGN1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "MAGN0            = %3d\n\r", CC_SPIReadExtReg(MAGN0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "ANG1             = %3d\n\r", CC_SPIReadExtReg(ANG1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "ANG0             = %3d\n\r", CC_SPIReadExtReg(ANG0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "CHFILT_I2        = %3d\n\r", CC_SPIReadExtReg(CHFILT_I2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "CHFILT_I1        = %3d\n\r", CC_SPIReadExtReg(CHFILT_I1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "CHFILT_I0        = %3d\n\r", CC_SPIReadExtReg(CHFILT_I0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "CHFILT_Q2        = %3d\n\r", CC_SPIReadExtReg(CHFILT_Q2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "CHFILT_Q1        = %3d\n\r", CC_SPIReadExtReg(CHFILT_Q1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "CHFILT_Q0        = %3d\n\r", CC_SPIReadExtReg(CHFILT_Q0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "GPIO_STATUS      = %3d\n\r", CC_SPIReadExtReg(GPIO_STATUS,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FSCAL_CTRL       = %3d\n\r", CC_SPIReadExtReg(FSCAL_CTRL,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PHASE_ADJUST     = %3d\n\r", CC_SPIReadExtReg(PHASE_ADJUST,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PARTNUMBER       = %3d\n\r", CC_SPIReadExtReg(PARTNUMBER,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PARTVERSION      = %3d\n\r", CC_SPIReadExtReg(PARTVERSION,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "SERIAL_STATUS    = %3d\n\r", CC_SPIReadExtReg(SERIAL_STATUS,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "RX_STATUS        = %3d\n\r", CC_SPIReadExtReg(RX_STATUS,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "TX_STATUS        = %3d\n\r", CC_SPIReadExtReg(TX_STATUS,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "MARC_STATUS1     = %3d\n\r", CC_SPIReadExtReg(MARC_STATUS1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "MARC_STATUS0     = %3d\n\r", CC_SPIReadExtReg(MARC_STATUS0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PA_IFAMP_TEST    = %3d\n\r", CC_SPIReadExtReg(PA_IFAMP_TEST,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FSRF_TEST        = %3d\n\r", CC_SPIReadExtReg(FSRF_TEST,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PRE_TEST         = %3d\n\r", CC_SPIReadExtReg(PRE_TEST,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "PRE_OVR          = %3d\n\r", CC_SPIReadExtReg(PRE_OVR,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "ADC_TEST         = %3d\n\r", CC_SPIReadExtReg(ADC_TEST,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "DVC_TEST         = %3d\n\r", CC_SPIReadExtReg(DVC_TEST,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "ATEST            = %3d\n\r", CC_SPIReadExtReg(ATEST,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "ATEST_LVDS       = %3d\n\r", CC_SPIReadExtReg(ATEST_LVDS,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "ATEST_MODE       = %3d\n\r", CC_SPIReadExtReg(ATEST_MODE,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "XOSC_TEST1       = %3d\n\r", CC_SPIReadExtReg(XOSC_TEST1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "XOSC_TEST0       = %3d\n\r", CC_SPIReadExtReg(XOSC_TEST0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "NUM_TXBYTES      = %3d\n\r", CC_SPIReadExtReg(NUM_TXBYTES,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "NUM_RXBYTES      = %3d\n\r", CC_SPIReadExtReg(NUM_RXBYTES,  RF_IDx));    
uartSendString(u_str);
};
*/


/* CC2520 Printer function *
void print_rf_settings(char RF_IDx) {
char u_str[40];
sprintf(u_str, "FREQCTRL     = %3d\n\r", CC_SPIReadReg(FREQCTRL,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "TXPOWER      = %3d\n\r", CC_SPIReadReg(TXPOWER,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "MDMCTRL0     = %3d\n\r", CC_SPIReadReg(MDMCTRL0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "MDMCTRL1     = %3d\n\r", CC_SPIReadReg(MDMCTRL1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "RXCTRL       = %3d\n\r", CC_SPIReadReg(RXCTRL,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FSCTRL       = %3d\n\r", CC_SPIReadReg(FSCTRL,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "AGCCTRL1     = %3d\n\r", CC_SPIReadReg(AGCCTRL1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "ADCTEST0     = %3d\n\r", CC_SPIReadReg(ADCTEST0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "ADCTEST2     = %3d\n\r", CC_SPIReadReg(ADCTEST2,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "EXTCLOCK     = %3d\n\r", CC_SPIReadReg(EXTCLOCK,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FRMCTRL0     = %3d\n\r", CC_SPIReadReg(FRMCTRL0,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "FRMCTRL1     = %3d\n\r", CC_SPIReadReg(FRMCTRL1,  RF_IDx));    
uartSendString(u_str);
sprintf(u_str, "MDMTEST0     = %3d\n\r", CC_SPIReadReg(MDMTEST0,  RF_IDx));    
uartSendString(u_str);
};

*/

/* ****************************************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS, 
* EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF 
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY 
* OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND 
* NON-INFRINGEMENT OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD
* TO THE PROGRAM OR YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR 
* INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR 
* NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY 
* OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. EXCLUDED 
* DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION, 
* COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS,
* OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S AGGREGATE 
* LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE PROGRAM 
* EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted by Texas 
* Instruments is distributed as "freeware".  You may, only under TI's copyright
* in the Program, use and modify the Program without any charge or restriction.  
* You may distribute to third parties, provided that you transfer a copy of this
* license to the third party and the third party agrees to these terms by its 
* first use of the Program. You must reproduce the copyright notice and any 
* other legend of ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains copyrighted material, 
* trade secrets and other TI proprietary information and is protected by 
* copyright laws, international copyright treaties, and trade secret laws, as
* well as other intellectual property laws.  To protect TI's rights in the 
* Program, you agree not to decompile, reverse engineer, disassemble or 
* otherwise translate any object code versions of the Program to a human-
* readable form.  You agree that in no event will you alter, remove or destroy
* any copyright notice included in the Program.  TI reserves all rights not 
* specifically granted under this license. Except as specifically provided 
* herein, nothing in this agreement shall be construed as conferring by 
* implication, estoppel, or otherwise, upon you, any license or other right
* under any TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
*******************************************************************************/
