/*******************************************************************************
*  Description:  Wireless demo application
*  
*  This wireless demo applications is meant to show the advantage of using 
*  Frequency hopping Spread Spectrum (FHSS) to implement your wireless sensor
*  network. This code base has been developed to run on a number of TI devices
*  including most MSP430F2x and MSP430F5x series microcontrolers using stand
*  alone wireless transcievers (CC2500 and CC1101/CC1100). Futhermore is also 
*  support the CC430F5 and CC430F6x series SoC's
*
*  Thomas Almholt
*  Texas Instruments, Inc, October 2010
*  Built with IAR Embedded Workbench Version: 4.21
********************************************************************************
*  Version:  1.00
*  Comments: Initial Release Version
*  Version:  1.10
*  Comments: Reduced the packet size by eliniating 2 Aux and Altimeter channels
*            and enabled the low power modes of the CC1101 radio device.
*  Version:  1.20
*  Comments: Added frequency hopping spread spectrum to tx_ber and rx_ber
*  
*******************************************************************************/

#include "msp430.h"
#include "CCx_hal.h"
#include "hal_usb.h"
#include "hal_adc.h"

#include "TRX_ui.h"

#define TX_BUFF_SIZE 10
#define REF_MVOLT 2500
//#define WDT_SKIP_VALUE 203  //use when running the 6.5MHz demo

#define WDT_SKIP_VALUE 1  //use when running normal demos
/******************************************************************************
* Select which of the three MSP configuration to use
******************************************************************************/
//#define MSP430F2x_Ez430
#define CC430F5x_Ez430
//#define MSP430F5x_Exp
//#define MSP430FR57xx_Exp
/******************************************************************************
* Prototypes
******************************************************************************/
void MSP_Setup(void);
void MSP_SMCLK_enable(void);
/******************************************************************************
* Global variable to accessed from ISR's
******************************************************************************/
unsigned char txBuffer[TX_BUFF_SIZE];
long wdt_uptime;
char user_button_pushed;
unsigned char wakeup_on_wdt;
unsigned char wdt_skip_counter;
trx_cfg_struct trx_cfg;
extern unsigned char chan_hop_seq[];

/******** Random data for both the reciever and transmitter to know ***********/
const unsigned char rand_data[60] = 
{49,231,121,199,255,153,138,232,220,203,51,253,117,172,161,191,79,58,225,215,
149,251,15,163,153,236,141,172,3,186,224,37,224,210,75,69,117,58,153,207,
61,203,19,125,76,90,143,226,11,208,16,72,109,217,100,12,128,228,185,142};


/*******************************************************************************
* @brief  
* Main GUI application level control loop is implemented in the main loop. It is
* a state machine that cycles thru various states during the interactive use by
* the operator. The presents a text based GUI, the user is then promted to type 
* in a command with a given set of parameters. The state machine then calls a 
* simpler parser that tries to figure out what the user wants to do.
* 
* @param  none
* 
* @return none
*******************************************************************************/
void main (void) {
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT
  char idle_counter = 0;
  
  MSP_Setup();                              // Setup of all IO's, ADC's and WDT
  
  halUsbInit();                             // Initialize the UART port
  
  radio_init( RF_CC430 );
  
#ifdef MSP430F2x_Ez430   // if we using the Ez430 kit we have to use the 26MHz xtal
  MSP_SMCLK_enable();    // as reference for the entire system.
#endif
  
  // Perform initial setup of the CC radio in RF_CC430
  trx_cfg.b_length = TX_BUFF_SIZE;
  rf_default_setup(&trx_cfg, RF_CC430);
  set_rf_packet_length(trx_cfg.b_length, RF_CC430);
  
  // Generate the frequency hopping sequence table that we will be using
  generate_hopping_table(chan_hop_seq);
  
  // Configure ports -- switch inputs, LEDs, GDO0 to RX packet info from CCxxxx
  CC_LED1_PxDIR |= CC_LED1;                          // LED Direction to Outputs
  CC_LED2_PxDIR |= CC_LED2;                          // LED Direction to Outputs
  
  HAL_LED1_ON();
  HAL_LED2_ON();
  
  wakeup_on_wdt = 1;                 // Generally we use the WDT time wake us up
  wdt_uptime = 0;                    // reset the wdt_uptime counter to 0
  user_button_pushed = 0;            // indicator that button has been pressed
  
  while(1) {     
    // Infinite loop with a 1 second timer
    // Put MCU in low power mode and wait for end-of-line from UART ISR
    HAL_LED2_OFF();                  // indicate low power RF+MCU mode
    _BIS_SR(LPM3_bits + GIE);        // Enter LPM0
    HAL_LED2_ON();                   // indicate normal mode
    idle_counter++;
    
    // Check to see if the user has pushed the putton and take appropriate action
    if(user_button_pushed == 1) {
      user_button_pushed = 0;
      trx_cfg.trx_mode = MENU_RX_BER;
    }   
    
    // For each idle loop (1 second) a counter is incremented, if check to see
    // if we have had 50 idle loops, if so we enter a default action. This enables
    // the board to automatically start doing something (used for becoming a 
    // reciever in a TX/RX BER system
    if((idle_counter>20) && trx_cfg.cc_state == CC_IDLE) {
      trx_cfg.trx_mode = MENU_TX_BER;
      idle_counter = 0;
    }
    
    // Check to see if the UART driver has detected an end-of-line from the user
    // if we have an end-of-line event decode the command and take action
    if (uart_eol == 1 ) { 
      idle_counter = 0;
      uart_eol = 0;
      if(bufferSize > 1) {
        trx_cfg.trx_mode = MENU_SETUP;
      } else {
        trx_cfg.trx_mode = MENU_IDLE;
      }
    } 
    // Main loop statemachine
    switch(trx_cfg.trx_mode){
    case MENU_RESTART: 
      print_rf_settings(RF_CC430);
      main_menu(&trx_cfg);
      trx_cfg.trx_mode = MENU_IDLE;
      break;
    case MENU_SETUP:
      parse_ui_cmd(&trx_cfg, halUsbReceiveBuffer);
      halUsbReceiveBuffer[0] = 0;
      bufferSize = 0;
      break;
    case MENU_IDLE:
      HAL_LED1_TOGGLE();
      break;
    case MENU_RF_IDLE:
      rf_default_setup(&trx_cfg, RF_CC430);      
      trx_cfg.trx_mode = MENU_RESTART;
      trx_cfg.cc_state = CC_IDLE;      
      break;
    case MENU_TX_UNMOD:
      wakeup_on_wdt = 0;
      unmodulated_tx(&trx_cfg, RF_CC430);
      trx_cfg.trx_mode = MENU_IDLE;
      trx_cfg.cc_state = CC_TX_ACTIVE;
      wakeup_on_wdt = 1;
      break;
    case MENU_TX_nMCU:
      wakeup_on_wdt = 0;
      modulated_tx_no_mcu(&trx_cfg, RF_CC430);
      trx_cfg.trx_mode = MENU_IDLE;
      trx_cfg.cc_state = CC_TX_ACTIVE;
      wakeup_on_wdt = 1;
      break;
    case MENU_TX_wMCU:
      wakeup_on_wdt = 0;
      trx_cfg.trx_mode = MENU_IDLE;
      trx_cfg.cc_state = CC_TX_ACTIVE;
      wakeup_on_wdt = 1;
      break;
    case MENU_RX_STATIC:
      wakeup_on_wdt = 0;
      rx_static(&trx_cfg, RF_CC430);
      trx_cfg.trx_mode = MENU_IDLE;
      trx_cfg.cc_state = CC_RX_ACTIVE;
      wakeup_on_wdt = 1;
      break;
    case MENU_RX_SWEEP:
      wakeup_on_wdt = 0;
      rx_sweep(txBuffer, &trx_cfg, RF_CC430);
      trx_cfg.trx_mode = MENU_RESTART;
      trx_cfg.cc_state = CC_IDLE;
      wakeup_on_wdt = 1;
      break;
    case MENU_RX_SINGLE:
      wakeup_on_wdt = 0;
      rx_scan(txBuffer, &trx_cfg, RF_CC430);
      trx_cfg.trx_mode = MENU_RESTART;
      trx_cfg.cc_state = CC_IDLE;
      wakeup_on_wdt = 1;
      break;
    case MENU_TX_BER:
      wakeup_on_wdt = 0;
      tx_ber(txBuffer, &trx_cfg, RF_CC430);
      wakeup_on_wdt = 1;
      trx_cfg.trx_mode = MENU_RESTART;
      trx_cfg.cc_state = CC_IDLE;      
      break;
    case MENU_RX_BER:
      wakeup_on_wdt = 0;
      rx_ber(txBuffer, &trx_cfg, RF_CC430);
      wakeup_on_wdt = 1;
      trx_cfg.trx_mode = MENU_RESTART;
      trx_cfg.cc_state = CC_IDLE;      
      break;
    default:
      trx_cfg.trx_mode = MENU_RESTART;
      break;
    }
  }
}  

#ifdef MSP430F2x_Ez430
/*******************************************************************************
* @brief  Setup all the peripherals of the MSP. This is for TI sensors
The DCO is set at 8MHz, the 32KHz is started and the WDT is at 1 sec.
*         (MSP430F2274 version)
* @param  none
* 
* @return none
*******************************************************************************/
void MSP_SMCLK_enable(void) {
  
  P2SEL |= BIT6; // Port 2.6 is XIN, we need to make this an regular GPIO input
  P2DIR &= ~BIT6; // make it an input
  
  BCSCTL3 |= (LFXT1S1 + LFXT1S0);  // select the external digital input for XT2
  BCSCTL1 &= ~XT2OFF;          // Turns the XT2 on
  
  // Have to wait for the XTAL subsystem to accept the clock as "ok"
  do  {
    BCSCTL3 &= ~(XT2OF);
    // Clear XT2,XT1,DCO fault flags
    IFG1 &= ~OFIFG;                      // Clear fault flags
  } while (IFG1&OFIFG);                  // Test oscillator fault flag
  
  BCSCTL2 |= SELS;          // Switch SMCLK to external clock source
  //  P2DIR |= BIT1;        // Places SMCLK on P2.1 for debugging purposes
  //  P2SEL |= BIT1;
}



/*******************************************************************************
* @brief  Setup all the peripherals of the MSP. This is for TI sensors
The DCO is set at 8MHz, the 32KHz is started and the WDT is at 1 sec.
*         (MSP430F2274 version)
* @param  none
* 
* @return none
*******************************************************************************/
void MSP_Setup(void) {
  
  // set system clock to 8MHz
  DCOCTL  = CALDCO_8MHZ;    // CALDCO_8MHZ is also an option
  BCSCTL1 = CALBC1_8MHZ;    // CALBC1_8MHZ is also an option
  
  // LED0_GPIO, LED1_GPIO, User_Sw0 on P1
  P1DIR = BIT7 + BIT6 + BIT5 + BIT4 + BIT3 + BIT1 + BIT0;  
  P1REN |= BIT2;                            // Set P1.0 to output direction
  P1OUT = BIT2;                            // Set P1.0 to output direction
  P1IE |= BIT2;                             // P1.2 interrupt enabled
  P1IES |= BIT2;                            // P1.2 Hi/lo edge
  P1IFG &= ~BIT2;                           // P1.2 IFG cleared
  
  // CC GDO0 input, CC GDO2 input on P2
  P2DIR = BIT5 + BIT4 + BIT3 + BIT2 + BIT1 + BIT0;   // change to outputs 
  P2OUT &= ~(BIT5 + BIT4 + BIT3 + BIT2 + BIT1 + BIT0); // change BITs low
  P2SEL &= ~(BIT7 + BIT6); // make the XTAL is disabled -there is no xtal on pcb
  
  // MSP_SPI_nCS (BIT0) MSP_SPI_SIMO (BIT1) MSP_SPI_SOMI (BIT2) 
  // MSP_SPI_CLK (BIT3) MSP_SPI_SIMO (BIT4) MSP_SPI_SOMI (BIT5)
  P3DIR |= BIT7 + BIT6 + BIT4 + BIT3 + BIT1 + BIT0;   
  P3OUT = 0x00;             // initialize the port to 0
  
  // Setup Watch dog timer for a 1 second tick using a 32KHz external XTAL   
  // BCSCTL1 |= DIVA_0;                        // ACLK/1 = 32.768KHz
  // WDTCTL = WDT_ADLY_1000;                   // WDT 1s interval timer
  // IE1 |= WDTIE;                             // Enable WDT interrupt
  // Setup Watch dog timer for a 32ms second tick using SMCLK
  WDTCTL = WDT_MDLY_32;                   // WDT 1s interval timer
  IE1 |= WDTIE;                             // Enable WDT interrupt
  
  
}
#endif

#ifdef CC430F5x_Ez430
/*******************************************************************************
* @brief  Setup all the peripherals of the MSP. This is for TI sensors
The DCO is set at 8MHz, the 32KHz is started and the WDT is at 1 sec.
*         (CC430 version)
* @param  none
* 
* @return none
********************************************************************************/
void MSP_Setup(void) {
  
  SetVCore(2); // 2 is needed for radio operation, 3 if we need to go to 24MHz.
  
  // Set the High-Power Mode Request Enable bit so LPM3 can be entered
  // with active radio enabled 
  PMMCTL0_H = 0xA5;
  PMMCTL0_L |= PMMHPMRE_L; 
  PMMCTL0_H = 0x00; 
  
  // Enable the interupts on port 2 to catch the user button
  P1DIR |= BIT7;                           // Set P1.0 to output direction
  P1REN |= BIT7;
  P1OUT |= BIT7;
  P1IE  |= BIT7;                            // P1.2 interrupt enabled
  P1IES |= BIT7;                            // P1.2 Hi/lo edge
  P1IFG &= ~BIT7;                           // P1.2 IFG cleared
  
  // Enable the interupts on port 2 to catch the user button
  // P2DIR &= ~BIT7;                           // Set P2.7 to input direction
  // P2IE  |= BIT7;                            // P2.7 interrupt enabled
  // P2IES |= BIT7;                            // P2.7 Hi/lo edge
  // P2IFG &= ~BIT7;                           // P2.7 IFG cleared
  
  // Setup the XTAL ports to use the external 32K oscillilator  
  P5SEL |= 0x03;                            // Select XT1
  UCSCTL6 &= ~(XT1DRIVE_0);                 // Max drive on XTAL
  UCSCTL6 |= XCAP_2;                        // Internal load cap
  
  // Loop until XT1,XT2 & DCO stabilizes
  do  {
    UCSCTL7 &= ~(XT2OFFG + XT1LFOFFG + XT1HFOFFG + DCOFFG);
    // Clear XT2,XT1,DCO fault flags
    SFRIFG1 &= ~OFIFG;                      // Clear fault flags
  } while (SFRIFG1&OFIFG);                  // Test oscillator fault flag
  UCSCTL6 &= ~(XT1DRIVE_3);                 // Xtal is now stable, reduce drive
  
  // Set up clock system on MCU to fit your system
  // Target specific implementation
  UCSCTL0 = 0x00;                           // Set lowest possible DCOx, MODx
  UCSCTL1 = DCORSEL_4;                      // Select suitable range  
  UCSCTL2 = 244 ;                           // DCO = 244 * 32768Hz ~= 8MHz
  UCSCTL4 = SELA__XT1CLK | SELS__DCOCLK | SELM__DCOCLK ; // use 32k xtal
  //UCSCTL4 = SELA__REFOCLK | SELS__DCOCLK | SELM__DCOCLK ; // use 32k refoclk
  
  // Setup Watch dog timer for 1 second tick using 32Khz XTAL on CC430
  WDTCTL = WDT_ADLY_1000;                   // WDT 15.6ms, ACLK, interval timer
  SFRIE1 |= WDTIE;                          // Enable WDT interrupt
}

/*******************************************************************************
* @brief  Setup all the peripherals of the MSP. This is for TI sensors
The DCO is set at 8MHz, the 32KHz is started and the WDT is at 1 sec.
*         (CC430 version)
* @param  none
* 
* @return none
********************************************************************************/
void MSP_Setup_26MHz(void) {
  
  SetVCore(2); // 2 is needed for radio operation, 3 if we need to go to 24MHz.
  
  // Set the High-Power Mode Request Enable bit so LPM3 can be entered
  // with active radio enabled 
  PMMCTL0_H = 0xA5;
  PMMCTL0_L |= PMMHPMRE_L; 
  PMMCTL0_H = 0x00; 
  
  // Enable the interupts on port 2 to catch the user button
  P1DIR |= BIT7;                           // Set P1.0 to output direction
  P1REN |= BIT7;
  P1OUT |= BIT7;
  P1IE  |= BIT7;                            // P1.2 interrupt enabled
  P1IES |= BIT7;                            // P1.2 Hi/lo edge
  P1IFG &= ~BIT7;                           // P1.2 IFG cleared
  
  // Enable the interupts on port 2 to catch the user button
  P2DIR &= ~BIT7;                           // Set P2.7 to input direction
  P2IE  |= BIT7;                            // P2.7 interrupt enabled
  P2IES |= BIT7;                            // P2.7 Hi/lo edge
  P2IFG &= ~BIT7;                           // P2.7 IFG cleared
  
  // Setup the XTAL ports to use the external 32K oscillilator  
  // P5SEL |= 0x03;                            // Select XT1
  // UCSCTL6 &= ~(XT1DRIVE_0);                 // Max drive on XTAL
  // UCSCTL6 &= ~(XT1OFF + XT2OFF);            // Set XT1 & XT2 On
  // UCSCTL6 |= XCAP_3;                        // Internal load cap
  
  // Set up clock system on MCU to fit your system
  // Target specific implementation
  UCSCTL6 &= ~XT2OFF;                       // Enable XT2
  UCSCTL3 |= SELREF__REFOCLK;               // FLLref = REFO
  // prior to stabilizing the XTALS, we need to make sure the ACLK is sourced 
  // from the internal REFCLOCK and not the XTAL. This will cause an error and
  // we will never exit the loop test below.
  UCSCTL4 |= SELA__REFOCLK;
  
  // Loop until XT1,XT2 & DCO stabilizes
  do  {
    UCSCTL7 &= ~(XT2OFFG + XT1LFOFFG + XT1HFOFFG + DCOFFG);
    // Clear XT2,XT1,DCO fault flags
    SFRIFG1 &= ~OFIFG;                      // Clear fault flags
  } while (SFRIFG1&OFIFG);                  // Test oscillator fault flag
  
  UCSCTL5 = DIVS1 + DIVS0 + DIVM1 + DIVM0;
  UCSCTL4 = SELA__REFOCLK | SELS__XT2CLK | SELM__XT2CLK;
  
  __delay_cycles(10000);
  
  WDTCTL = WDT_ADLY_1000;                   // WDT 15.6ms, ACLK, interval timer
  SFRIE1 |= WDTIE;                          // Enable WDT interrupt
}


#endif

#ifdef MSP430F5x_Exp
/*******************************************************************************
* @brief  Setup all the peripherals of the MSP. This is for TI sensors
The DCO is set at 8MHz, the 32KHz is started and the WDT is at 1 sec.
*         (MSP430F5438 version)
* @param  none
* 
* @return none
*******************************************************************************/
void MSP_Setup(void) {
  
  // Enable the interupts on port 2 to catch the user button
  P2DIR &= ~BIT3;                         // Set P2.7 to output direction
  P2OUT =   BIT3;                         // Set P2.7 to output direction
  P2IE  |=  BIT3;                         // P2.7 interrupt enabled
  P2IES |=  BIT3;                         // P2.7 Hi/lo edge
  P2REN |=  BIT3;                         // P2.7 Pull up resistor
  P2IFG &= ~BIT3;                         // P1.2 IFG cleared
  
  // Setup the XTAL ports to use the external 32K oscillilator  
  P7SEL |= 0x03;                            // Select XT1
  UCSCTL6 |= XCAP_3;                        // Internal load cap
  
  // Loop until XT1,XT2 & DCO stabilizes
  do  {
    UCSCTL7 &= ~(XT2OFFG + XT1LFOFFG + XT1HFOFFG + DCOFFG);
    // Clear XT2,XT1,DCO fault flags
    SFRIFG1 &= ~OFIFG;                      // Clear fault flags
  } while (SFRIFG1&OFIFG);                  // Test oscillator fault flag
  UCSCTL6 &= ~(XT1DRIVE_3);                 // Xtal is now stable, reduce drive
  
  // Set up clock system on MCU to fit your system
  // Target specific implementation
  UCSCTL0 = 0x00;                           // Set lowest possible DCOx, MODx
  UCSCTL1 = DCORSEL_4;                      // Select suitable range  
  UCSCTL2 = 244 ;                           // DCO = 244 * 32768Hz ~= 8MHz
  UCSCTL4 = SELA__XT1CLK | SELS__DCOCLK | SELM__DCOCLK ;
  
  // Setup Watch dog timer for 1 second tick using 32Khz XTAL on CC430
  WDTCTL = WDT_ADLY_1000;                    // WDT 15.6ms, ACLK, interval timer
  SFRIE1 |= WDTIE;                           // Enable WDT interrupt
}
#endif


#ifdef MSP430FR57xx_Exp
/*******************************************************************************
* @brief  Setup all the peripherals of the MSP. This is for TI sensors
The DCO is set at 8MHz, the 32KHz is started and the WDT is at 1 sec.
*         (MSP430F5438 version)
* @param  none
* 
* @return none
*******************************************************************************/
void MSP_Setup(void) {
  
  // Enable the interupts on port 2 to catch the user button
  P4DIR &= ~BIT3;                         // Set P2.7 to output direction
  P4OUT =   BIT3;                         // Set P2.7 to output direction
  P4IE  |=  BIT3;                         // P2.7 interrupt enabled
  P4IES |=  BIT3;                         // P2.7 Hi/lo edge
  P4REN |=  BIT3;                         // P2.7 Pull up resistor
  P4IFG &= ~BIT3;                         // P1.2 IFG cleared
  
  // XT1 Setup  
  PJSEL0 |= BIT4 + BIT5; 
  
  CSCTL0_H = 0xA5;
  CSCTL1 |= DCOFSEL0 + DCOFSEL1;            // Set max. DCO setting
  CSCTL2 = SELA_0 + SELS_3 + SELM_3;        // set ACLK = XT1; MCLK = DCO
  CSCTL3 = DIVA_0 + DIVS_1 + DIVM_1;        // set all dividers 
  CSCTL4 |= XT1DRIVE_0; 
  CSCTL4 &= ~XT1OFF;
  
  do
  {
    CSCTL5 &= ~XT1OFFG;                     // Clear XT1 fault flag
    SFRIFG1 &= ~OFIFG; 
  }while (SFRIFG1&OFIFG);                   // Test oscillator fault flag
  
  // Setup Watch dog timer for 1 second tick using 32Khz XTAL
  WDTCTL = WDT_ADLY_1000;                    // WDT 15.6ms, ACLK, interval timer
  SFRIE1 |= WDTIE;                           // Enable WDT interrupt
}
#endif


/*******************************************************************************
* @brief  ISR for Watch Dog Timer. 
* 
*******************************************************************************/
#pragma vector=WDT_VECTOR
__interrupt void watchdog_timer (void) {
  
  wdt_uptime++;
  if(wakeup_on_wdt == 1) {
    wdt_skip_counter++;
    if(wdt_skip_counter==WDT_SKIP_VALUE) {
      _BIC_SR_IRQ(LPM3_bits);                 // Clear LPM0 bits from 0(SR)      
      wdt_skip_counter = 0;
    }
  }
}

// Port 1 interrupt service routine
#pragma vector=PORT1_VECTOR
__interrupt void Button_ISR(void) {
  P1IFG &= ~BIT7;                         // P1.2 IFG cleared
  user_button_pushed = 1;
  _BIC_SR_IRQ(LPM3_bits);                 // Clear LPM0 bits from 0(SR)      
}
/*
// Port 2 interrupt service routine
#pragma vector=PORT2_VECTOR
__interrupt void Button2_ISR(void)  {
P2IFG &= ~BIT7;                         // P1.7 IFG cleared
user_button_pushed = 1;
_BIC_SR_IRQ(LPM3_bits);                 // Clear LPM0 bits from 0(SR)        
}
*/


/* ****************************************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS, 
* EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF 
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY 
* OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND 
* NON-INFRINGEMENT OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD
* TO THE PROGRAM OR YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR 
* INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR 
* NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY 
* OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. EXCLUDED 
* DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION, 
* COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS,
* OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S AGGREGATE 
* LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE PROGRAM 
* EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted by Texas 
* Instruments is distributed as "freeware".  You may, only under TI's copyright
* in the Program, use and modify the Program without any charge or restriction.  
* You may distribute to third parties, provided that you transfer a copy of this
* license to the third party and the third party agrees to these terms by its 
* first use of the Program. You must reproduce the copyright notice and any 
* other legend of ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains copyrighted material, 
* trade secrets and other TI proprietary information and is protected by 
* copyright laws, international copyright treaties, and trade secret laws, as
* well as other intellectual property laws.  To protect TI's rights in the 
* Program, you agree not to decompile, reverse engineer, disassemble or 
* otherwise translate any object code versions of the Program to a human-
* readable form.  You agree that in no event will you alter, remove or destroy
* any copyright notice included in the Program.  TI reserves all rights not 
* specifically granted under this license. Except as specifically provided 
* herein, nothing in this agreement shall be construed as conferring by 
* implication, estoppel, or otherwise, upon you, any license or other right
* under any TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
*******************************************************************************/
