/*******************************************************************************
 * @file  hal_usb.c
 * 
 * Copyright 2008 Texas Instruments, Inc.
*******************************************************************************/


/* ****************************************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS, 
* EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF 
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY 
* OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND 
* NON-INFRINGEMENT OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD
* TO THE PROGRAM OR YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR 
* INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR 
* NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY 
* OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. EXCLUDED 
* DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION, 
* COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS,
* OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S AGGREGATE 
* LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE PROGRAM 
* EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted by Texas 
* Instruments is distributed as "freeware".  You may, only under TI's copyright
* in the Program, use and modify the Program without any charge or restriction.  
* You may distribute to third parties, provided that you transfer a copy of this
* license to the third party and the third party agrees to these terms by its 
* first use of the Program. You must reproduce the copyright notice and any 
* other legend of ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains copyrighted material, 
* trade secrets and other TI proprietary information and is protected by 
* copyright laws, international copyright treaties, and trade secret laws, as
* well as other intellectual property laws.  To protect TI's rights in the 
* Program, you agree not to decompile, reverse engineer, disassemble or 
* otherwise translate any object code versions of the Program to a human-
* readable form.  You agree that in no event will you alter, remove or destroy
* any copyright notice included in the Program.  TI reserves all rights not 
* specifically granted under this license. Except as specifically provided 
* herein, nothing in this agreement shall be construed as conferring by 
* implication, estoppel, or otherwise, upon you, any license or other right
* under any TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
*******************************************************************************/
#include "msp430.h"
#include "hal_usb.h"

char halUsbReceiveBuffer[255];
unsigned char bufferSize;
unsigned char buffer_pos;
extern char uart_char_rx[];
volatile char uart_eol = 0;

/**********************************************************************//**
 * @brief  Initializes the serial communications peripheral and GPIO ports 
 *         to communicate with the TUSB3410.
 * 
 * @param  none
 * 
 * @return none
 **************************************************************************/
void halUsbInit(void) {
  volatile unsigned char i;
  
  for (i = 0;i < 255; i++)
    halUsbReceiveBuffer[i]='\0';
	  
  bufferSize = 80;
  buffer_pos = 0;
  // PMAPPWD = 0x02D52;                        // Get write-access to port mapping regs  
  // P2MAP6 = PM_UCA0RXD;                      // Map UCA0RXD output to P2.6 
  // P2MAP7 = PM_UCA0TXD;                      // Map UCA0TXD output to P2.7 
  // PMAPPWD = 0;                              // Lock port mapping registers 

  USB_PORT_SEL |= USB_PIN_RXD + USB_PIN_TXD;
  USB_PORT_DIR |= USB_PIN_TXD;
  USB_PORT_DIR &= ~USB_PIN_RXD;

  UCA0CTL1  = UCSWRST;		            // Hold module in reset state.
  UCA0CTL0  = 0x00;                         // UART mode, 1-stop bit, 8-bit data, MSB first, no parity.
  UCA0CTL1 |= UCSSEL_2;                     // SMCLK
  UCA0BR0   = 0x38;                         // 3.25MHz 57600 0x8A=57600
  UCA0BR1   = 0x00;                         // 3.25MHz 57600

  //UCA0BR0   = 0x8A;                         // 8MHz 115200, 0x8A=57600
  //UCA0BR1   = 0x00;                         // 8MHz 115200
  //UCA0BR0   = 0x41;                         // 8MHz 0x41=9600
  //UCA0BR1   = 0x03;                         // 8MHz 0x03=9600

  UCA0MCTL  = UCBRS_4;                      // Modulation UCBRSx = 5 (changed to 4 MAM, per table 16-4 5xx UG)
  UCA0CTL1 &= ~UCSWRST;                     // Enable module.
  UCA0IE   |= UCRXIE;                       // Enable USCI_A1 RX interrupt.

  __bis_SR_register(GIE);                  // Enable Interrupts
}

/**********************************************************************//**
 * @brief  Disables the serial communications peripheral and clears the GPIO
 *         settings used to communicate with the TUSB3410.
 * 
 * @param  none
 * 
 * @return none
 **************************************************************************/
void halUsbShutDown(void) {
  UCA0IE &= ~UCRXIE;
  UCA0CTL1 = UCSWRST;                          //Reset State                         
  USB_PORT_SEL &= ~( USB_PIN_RXD + USB_PIN_TXD );
  USB_PORT_DIR |= USB_PIN_TXD;
  USB_PORT_DIR |= USB_PIN_RXD;
  USB_PORT_OUT &= ~(USB_PIN_TXD + USB_PIN_RXD);
}

/**********************************************************************//**
 * @brief  Sends a character over UART to the TUSB3410.
 * 
 * @param  character The character to be sent. 
 * 
 * @return none
 **************************************************************************/
void halUsbSendChar(char character) {
  while (UCA0STAT & UCBUSY);
  UCA0TXBUF = character;
}

/**********************************************************************//**
 * @brief  Sends a string of characters to the TUSB3410
 * 
 * @param  string[] The array of characters to be transmit to the TUSB3410.
 * 
 * @param  length   The length of the string.
 * 
 * @return none
 **************************************************************************/
void halUsbSendString(char string[], unsigned char length) {
  volatile unsigned char i;
  for (i=0; i < length; i++)
    halUsbSendChar(string[i]);  
}


//************************************************************************/
#pragma vector=USCI_A0_VECTOR
__interrupt void USCI0RX_ISR(void) {
  if(UCA0RXBUF == 8) {
    if(bufferSize>0) {
      bufferSize--;
    }
    while (!(UCA0IFG&UCTXIFG));   // print a back space
    UCA0TXBUF = UCA0RXBUF;
    while (!(UCA0IFG&UCTXIFG));   // print a white space
    UCA0TXBUF = 32;
    while (!(UCA0IFG&UCTXIFG));   // print a white space
    UCA0TXBUF = UCA0RXBUF;
  } else {
    halUsbReceiveBuffer[bufferSize++] = UCA0RXBUF; // store inputs
    if(UCA0RXBUF == 13) {     // if its a "return" then activate main-loop
      uart_eol = 1;
     __bic_SR_register_on_exit(LPM3_bits);   
    } else {                  // if normal character, loop it back for ui
       UCA0TXBUF = UCA0RXBUF;
    }
  }
}
