/******************************************************************************
    Filename: hal_spi_config.c

    Copyright 2007 Texas Instruments, Inc.
******************************************************************************/

/* ****************************************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS, 
* EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF 
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY 
* OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND 
* NON-INFRINGEMENT OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD
* TO THE PROGRAM OR YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR 
* INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR 
* NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY 
* OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. EXCLUDED 
* DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION, 
* COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS,
* OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S AGGREGATE 
* LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE PROGRAM 
* EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted by Texas 
* Instruments is distributed as "freeware".  You may, only under TI's copyright
* in the Program, use and modify the Program without any charge or restriction.  
* You may distribute to third parties, provided that you transfer a copy of this
* license to the third party and the third party agrees to these terms by its 
* first use of the Program. You must reproduce the copyright notice and any 
* other legend of ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains copyrighted material, 
* trade secrets and other TI proprietary information and is protected by 
* copyright laws, international copyright treaties, and trade secret laws, as
* well as other intellectual property laws.  To protect TI's rights in the 
* Program, you agree not to decompile, reverse engineer, disassemble or 
* otherwise translate any object code versions of the Program to a human-
* readable form.  You agree that in no event will you alter, remove or destroy
* any copyright notice included in the Program.  TI reserves all rights not 
* specifically granted under this license. Except as specifically provided 
* herein, nothing in this agreement shall be construed as conferring by 
* implication, estoppel, or otherwise, upon you, any license or other right
* under any TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
*******************************************************************************/
#include "hal_spi_config.h"
#include "hal_msp430.h"


//----------------------------------------------------------------------------------
//   Generic SPI plugin module. Supports all known MSP430 SPI interfaces.
//----------------------------------------------------------------------------------

//----------------------------------------------------------------------------------
//  USART0
//----------------------------------------------------------------------------------
#if defined HAL_SPI_INTERFACE_USART0

void halSpiInit(uint8 clockrate) {
    // Keep peripheral in reset state
    U0CTL = SWRST;

    // 8-bit SPI Master 3-pin mode, with SMCLK as clock source
    U0CTL  |= CHAR + SYNC + MM;
    U0TCTL |= CKPH + SSEL1 + SSEL0 + STC;

    // Ignore clockrate argument for now, just use clock source/2
    U0BR0  = 0x02;
    U0BR1  = 0x00;
    U0MCTL = 0x00;

    // Enable SPI mode
    ME1 |= USPIE0;

    // Set up pins used by peripheral unit
    MCU_IO_PERIPHERAL(HAL_SPI_SOMI_PORT, HAL_SPI_SOMI_PIN);
    MCU_IO_PERIPHERAL(HAL_SPI_SIMO_PORT, HAL_SPI_SIMO_PIN);
    MCU_IO_PERIPHERAL(HAL_SPI_CLK_PORT,  HAL_SPI_CLK_PIN);
    MCU_IO_OUTPUT(HAL_SPI_CS_PORT, HAL_SPI_CS_PIN, 1);

    // Release for operation
    U0CTL &= ~SWRST;
}

//----------------------------------------------------------------------------------
//  USART1
//----------------------------------------------------------------------------------
#elif defined HAL_SPI_INTERFACE_USART1

void halSpiInit(uint8 clockrate) {
    // Keep peripheral in reset state
    U1CTL = SWRST;

    // 8-bit SPI Master 3-pin mode, with SMCLK as clock source
    U1CTL  |= CHAR + SYNC + MM;
    U1TCTL |= CKPH + SSEL1 + SSEL0 + STC;

    // Ignore clockrate argument for now, just use clock source/2
    U1BR0  = 0x02;
    U1BR1  = 0x00;
    U1MCTL = 0x00;

    // Enable SPI mode
    ME2 |= USPIE1;

    // Set up pins used by peripheral unit
    MCU_IO_PERIPHERAL(HAL_SPI_SOMI_PORT, HAL_SPI_SOMI_PIN);
    MCU_IO_PERIPHERAL(HAL_SPI_SIMO_PORT, HAL_SPI_SIMO_PIN);
    MCU_IO_PERIPHERAL(HAL_SPI_CLK_PORT,  HAL_SPI_CLK_PIN);
    MCU_IO_OUTPUT(HAL_SPI_CS_PORT, HAL_SPI_CS_PIN, 1);

    // Release for operation
    U1CTL &= ~SWRST;
}

//----------------------------------------------------------------------------------
//  USCIA0
//----------------------------------------------------------------------------------
#elif defined HAL_SPI_INTERFACE_USCIA0

void halSpiInit(uint8 clockrate) {
    // Keep peripheral in reset state
    UCA0CTL1 = UCSWRST;

    // 8-bit SPI Master 3-pin mode, with SMCLK as clock source
    UCA0CTL0  = UCCKPH + UCMSB + UCMST + UCSYNC;
    UCA0CTL1 |= UCSSEL1 + UCSSEL0;

    // Ignore clockrate argument for now, just use clock source/2
    UCA0BR0 = 0x02;
    UCA0BR1 = 0x00;

    // Set up pins used by peripheral unit
    MCU_IO_PERIPHERAL(HAL_SPI_SOMI_PORT, HAL_SPI_SOMI_PIN);
    MCU_IO_PERIPHERAL(HAL_SPI_SIMO_PORT, HAL_SPI_SIMO_PIN);
    MCU_IO_PERIPHERAL(HAL_SPI_CLK_PORT,  HAL_SPI_CLK_PIN);
    MCU_IO_OUTPUT(HAL_SPI_CS_PORT, HAL_SPI_CS_PIN, 1);

    // Release for operation
    UCA0CTL1 &= ~UCSWRST;
}

//----------------------------------------------------------------------------------
//  USCIB0
//----------------------------------------------------------------------------------
#elif defined HAL_SPI_INTERFACE_USCIB0

void halSpiInit(uint8 clockrate) {
    // Keep peripheral in reset state
    UCB0CTL1 = UCSWRST;

    // 8-bit SPI Master 3-pin mode, with SMCLK as clock source
    UCB0CTL0  = UCCKPH + UCMSB + UCMST + UCSYNC;
    UCB0CTL1 |= UCSSEL1 + UCSSEL0;

    // Ignore clockrate argument for now, just use clock source/2
    UCB0BR0 = 0x02;
    UCB0BR1 = 0x00;

    // Set up pins used by peripheral unit
    MCU_IO_PERIPHERAL(HAL_SPI_SOMI_PORT, HAL_SPI_SOMI_PIN);
    MCU_IO_PERIPHERAL(HAL_SPI_SIMO_PORT, HAL_SPI_SIMO_PIN);
    MCU_IO_PERIPHERAL(HAL_SPI_CLK_PORT,  HAL_SPI_CLK_PIN);
    MCU_IO_OUTPUT(HAL_SPI_CS_PORT, HAL_SPI_CS_PIN, 1);

    // Release for operation
    UCB0CTL1 &= ~UCSWRST;
}

//----------------------------------------------------------------------------------
//  USI
//----------------------------------------------------------------------------------
#elif defined HAL_SPI_INTERFACE_USI

void halSpiInit(uint8 clockrate) {
    // Keep peripheral in reset state
    USICTL0 = USISWRST;

    // 8-bit SPI Master 3-pin mode
    USICTL0 |= USIPE7 +  USIPE6 + USIPE5 + USIMST + USIOE;
    USICTL1  = USICKPH;

    // Use SMCLK as clock source, bit rate is clock source/2
    USICKCTL = USIDIV0 + USISSEL1;

    // Set up pins used by peripheral unit
    MCU_IO_PERIPHERAL(HAL_SPI_SOMI_PORT, HAL_SPI_SOMI_PIN);
    MCU_IO_PERIPHERAL(HAL_SPI_SIMO_PORT, HAL_SPI_SIMO_PIN);
    MCU_IO_PERIPHERAL(HAL_SPI_CLK_PORT,  HAL_SPI_CLK_PIN);
    MCU_IO_OUTPUT(HAL_SPI_CS_PORT, HAL_SPI_CS_PIN, 1);

    // Release for operation
    USICTL0 &= ~USISWRST;

    // Ensure SDO (SIMO) low
    USISRL = 0x00;
    USICNT = 1;
}

//----------------------------------------------------------------------------------
//  BITBANG
//----------------------------------------------------------------------------------
#elif defined HAL_SPI_INTERFACE_BITBANG

void halSpiInit(uint8 clockrate) {
    // Set up pins
    MCU_IO_INPUT(HAL_SPI_SOMI_PORT, HAL_SPI_SOMI_PIN);
    MCU_IO_OUTPUT(HAL_SPI_SIMO_PORT, HAL_SPI_SIMO_PIN, 0);
    MCU_IO_OUTPUT(HAL_SPI_CLK_PORT,  HAL_SPI_CLK_PIN, 0);
    MCU_IO_OUTPUT(HAL_SPI_CS_PORT, HAL_SPI_CS_PIN, 1);
}

uint8 halSpiBitbangOut(uint8 data) {
    uint8 i;
    uint8 rc = 0;

    for(i = 8; i > 0; i--)
    {
        // clock low
        MCU_IO_CLR(HAL_SPI_CLK_PORT, HAL_SPI_CLK_PIN);

        if (data & 0x80)
            MCU_IO_SET(HAL_SPI_SIMO_PORT, HAL_SPI_SIMO_PIN);
        else
            MCU_IO_CLR(HAL_SPI_SIMO_PORT, HAL_SPI_SIMO_PIN);
        data <<= 1;

        // clock high
        MCU_IO_SET(HAL_SPI_CLK_PORT, HAL_SPI_CLK_PIN);

        rc <<= 1;
        if (HAL_SPI_SOMI_VAL)
            rc |= 0x01;
    }

    // clock low
    MCU_IO_CLR(HAL_SPI_CLK_PORT, HAL_SPI_CLK_PIN);

    return(rc);
}

uint8 halSpiBitbangIn(void) {
    uint8 i;
    uint8 data = 0;

    for(i = 8; i > 0; i--)
    {
        // clock high
        MCU_IO_SET(HAL_SPI_CLK_PORT, HAL_SPI_CLK_PIN);

        data <<= 1;
        if (HAL_SPI_SOMI_VAL)
            data |= 0x01;

        // clock low
        MCU_IO_CLR(HAL_SPI_CLK_PORT, HAL_SPI_CLK_PIN);
    }

    return(data);
}
//----------------------------------------------------------------------------------
//  CC430 USCI
//----------------------------------------------------------------------------------
#elif defined HAL_SPI_INTERFACE_CC430USCI

void halSpiInit(uint8 clockrate) {
    // Keep peripheral in reset state
    UCB0CTL1 = UCSWRST;

    // 8-bit SPI Master 3-pin mode, with SMCLK as clock source
    UCB0CTL0  = UCCKPH + UCMSB + UCMST + UCSYNC;
    UCB0CTL1 |= UCSSEL1 + UCSSEL0;

    // Ignore clockrate argument for now, just use clock source/2
    UCB0BR0 = 0x02;
    UCB0BR1 = 0x00;

    // Set up pins used by peripheral unit
    MCU_IO_PERIPHERAL(HAL_SPI_SOMI_PORT, HAL_SPI_SOMI_PIN);
    MCU_IO_PERIPHERAL(HAL_SPI_SIMO_PORT, HAL_SPI_SIMO_PIN);
    MCU_IO_PERIPHERAL(HAL_SPI_CLK_PORT,  HAL_SPI_CLK_PIN);
    MCU_IO_OUTPUT(HAL_SPI_CS_PORT, HAL_SPI_CS_PIN, 1);

    // Release for operation
    UCB0CTL1 &= ~UCSWRST;
}

//----------------------------------------------------------------------------------
//  Unknown interface
//----------------------------------------------------------------------------------
#else
#error "You must select an SPI interface (USART/USCI/USI/BITBANG)"
#endif
