/*******************************************************************************
* Description:  This file contains functions that interface with CC1101 and 
* CC2500 series devices. 
*
* Version :  1.00, March 2011
* Comments: Initial Release Version
* Author  : Thomas Almholt, Texas Instruments
********************************************************************************
* Compiled with IAR Embedded Workbench v5.10.4.50168
*******************************************************************************/
#include "msp430.h"
#include "CCx_hal.h"
#include "CC2520_def.h"
#include "CC2520_spi.h"

/*******************************************************************************
* Initialize the radio hardware
* This particular setting sets up the Radio for 100kB/s GFSK modulation
******************************************************************************/
int radio_init(char RF_IDx) {
  
  // Configure SPI ports for each radio
  CC_SPISetup(RF_IDx);                    // Initialize SPI port
  
  // Perform a cold start of the CC radio in RF_CC430
  CC_PowerupReset(RF_IDx);          // Reset CCxxxx
  __delay_cycles(100000); 
  
  /****************************** RF settings SoC *******************************
  * TX power = 5 
  * Frequency = 2405 
  * IEEE 802.15.4 channel = 0x0B 
  * Modulated = true 
  *******************************************************************************/
  
  CC_SPIWriteReg(FREQCTRL ,     0x0B, RF_IDx);   // Controls the RF frequency
  CC_SPIWriteReg(TXPOWER  ,     0xF7, RF_IDx);   // Controls the output power
  CC_SPIWriteReg(MDMCTRL0 ,     0x85, RF_IDx);   // Controls modem
  CC_SPIWriteReg(MDMCTRL1 ,     0x14, RF_IDx);   // Controls modem
  CC_SPIWriteReg(RXCTRL   ,     0x3F, RF_IDx);   // Test/tuning of RX modules
  CC_SPIWriteReg(FSCTRL   ,     0x5A, RF_IDx);   // Test/tuning of synthesizer
  CC_SPIWriteReg(AGCCTRL1 ,     0x11, RF_IDx);   // Test/tuning of AGC
  CC_SPIWriteReg(ADCTEST0 ,     0x10, RF_IDx);   // Tune ADC performance.
  CC_SPIWriteReg(ADCTEST2 ,     0x03, RF_IDx);   // Tune ADC performance.
  CC_SPIWriteReg(EXTCLOCK ,     0x00, RF_IDx);   // Controls clock output
  CC_SPIWriteReg(FRMCTRL0 ,     0x40, RF_IDx);   // Frame handling
  CC_SPIWriteReg(FRMCTRL1 ,     0x00, RF_IDx);   // Frame handling
  CC_SPIWriteReg(MDMTEST0 ,     0x05, RF_IDx);   // Test register for modem
  
  CC_SPIReadReg(FSCTRL ,  RF_IDx);   // Test register for modem
  
  return 0;
}

/*******************************************************************************
* Prepare the radio with a packet to be sent
* Fill the FIFO with data, but do not initiate TX mode
******************************************************************************/
int radio_prepare(unsigned char *payload, unsigned short payload_len, char RF_IDx) {
  CC_SPIWriteBurstReg(INS_TXBUF, payload, payload_len, RF_IDx);  // Write TX data
  return 0;
}

/* Send the packet that has previously been prepared (used for exact timing)*/
int radio_transmit(char RF_IDx) {
   CC_SPIStrobe(INS_STXON, RF_IDx);           // Change state to TX, initiating
   return(0);
}

/* Send the packet that has previously been prepared (used for exact timing)*/
int radio_receive_on(char RF_IDx) {
   CC_SPIStrobe(INS_SRXON, RF_IDx);            // Change state to TX, initiating
   __delay_cycles(1000);                     // 1 ms delay for letting RX settle
  return(0);
}

/* Prepare & transmit a packet in same call (slightly worse timing jitter) */
int radio_send(unsigned char *payload, unsigned short payload_len, char RF_IDx) {
   CC_SPIWriteBurstReg(INS_TXBUF, payload, payload_len, RF_IDx); // Write TX data
   CC_SPIStrobe(INS_SRXON, RF_IDx);               // Change state to TX, initiating
   return(0);
}

/* Read a received packet into a buffer */
int radio_read(unsigned char *buf, unsigned short *buf_len, char RF_IDx) {
    char status;
    char pktLen;
    
    if (pktLen = CC_SPIReadStatus(RXFIFOCNT, RF_IDx)) {
    if (pktLen == *buf_len) {                       // If pktLen size <= rxBuffer
      CC_SPIReadBurstReg(INS_RXBUF, buf, pktLen, RF_IDx); // Pull data
      status  = CC_SPIReadReg(RSSISTAT , RF_IDx);   // read status register
      return (status);
    } else {                                         // Return CRC_OK bit
      *buf_len = pktLen;                             // Return the large size
      CC_SPIStrobe(INS_SFLUSHTX, RF_IDx);               // Flush the RX FIFO's
      return 0;                                      // Error
    }
  } else {
      return 0;                                      // Error
  }
}

/* Perform a Clear-Channel Assessment (CCA) to find out if channel is clear */
int radio_channel_clear(char RF_IDx) {
  int status;
  status = CC_SPIReadStatus(RSSISTAT, RF_IDx) & 0x40;
  return(status);
}

/* Wait for radio to become idle (currently receiving or transmitting) */
int radio_wait_for_idle(char RF_IDx) {
  if(RF_IDx==RF_SLOTA) {
    while (!(CC_GDO0_PxIN&CC_GDO0_PIN));
                                            // Wait GDO0 to go hi -> sync TX'ed
    while (CC_GDO0_PxIN&CC_GDO0_PIN);
                                            // Wait GDO0 to clear -> end of pkt
    CC_GDO0_PxIFG &= ~CC_GDO0_PIN;          // After pkt TX, this flag is set.
  } else {
    while (!(CC_GDO2_PxIN&CC_GDO2_PIN));
                                            // Wait GDO0 to go hi -> sync TX'ed
    while (CC_GDO2_PxIN&CC_GDO2_PIN);
                                            // Wait GDO0 to clear -> end of pkt
    CC_GDO2_PxIFG &= ~CC_GDO2_PIN;          // After pkt TX, this flag is set.
  }
  return(0);
}

/* Wait for radio to become idle (currently receiving or transmitting) */
int radio_is_busy(char RF_IDx) {
  if(RF_IDx==RF_SLOTA) {
    while (!(CC_GDO0_PxIN&CC_GDO0_PIN));
                                            // Wait GDO0 to go hi -> sync TX'ed
    while (CC_GDO0_PxIN&CC_GDO0_PIN);
                                            // Wait GDO0 to clear -> end of pkt
    CC_GDO0_PxIFG &= ~CC_GDO0_PIN;          // After pkt TX, this flag is set.
  } else {
    while (!(CC_GDO2_PxIN&CC_GDO2_PIN));
                                            // Wait GDO0 to go hi -> sync TX'ed
    while (CC_GDO2_PxIN&CC_GDO2_PIN);
                                            // Wait GDO0 to clear -> end of pkt
    CC_GDO2_PxIFG &= ~CC_GDO2_PIN;          // After pkt TX, this flag is set.
  }
  return(0);
}


/* Check if the radio driver has just received a packet */
int radio_pending_packet(char RF_IDx);

/* Change channel of radio, using predefined paramters */
int radio_set_channel(unsigned char rf_channel, char RF_IDx) {
    CC_SPIWriteReg(FREQCTRL, rf_channel, RF_IDx);
    return(rf_channel);
}

/* Idle the radio, used when leaving low power modes (below)*/
int radio_idle(char RF_IDx) {
  CC_SPIStrobe(STR_SRFOFF , RF_IDx);
  CC_SPIStrobe(INS_SFLUSHTX, RF_IDx);               // Flush the RX FIFO's
  CC_SPIStrobe(INS_SFLUSHRX, RF_IDx);               // Flush the TX FIFO's  
  return(0);
}

/* Put the radio into sleep mode */
int radio_sleep(char RF_IDx);


/*******************************************************************************
* @brief  ISR for Timer A0
* 
* @param  none
* 
* @return none
*******************************************************************************/

#ifdef MSP430_F5xx
  #pragma vector=TIMER0_A0_VECTOR     /* MSP430F5x */
  __interrupt void TIMER0_A0_ISR(void) {
    __bic_SR_register_on_exit(LPM3_bits);    
  }
#else
  #pragma vector=TIMERA0_VECTOR      /* MSP430F2x */
  __interrupt void TIMERA0_ISR(void) {
    __bic_SR_register_on_exit(LPM3_bits);   
  }
#endif

#pragma vector=PORT1_VECTOR
__interrupt void Radio_ISR(void) {
  
  if(CC_GDO2_PxIFG & CC_GDO2_PIN) {
    //__bic_SR_register_on_exit(LPM3_bits);    // Clear LPM0 bits from 0(SR)     
    // no operation.    
  }
  if(CC_GDO0_PxIFG & CC_GDO0_PIN) {
    rf_end_packet = 1;
    __bic_SR_register_on_exit(LPM3_bits);      // Clear LPM0 bits from 0(SR)     
  }
  CC_GDO0_PxIFG &= ~CC_GDO0_PIN;
  CC_GDO2_PxIFG &= ~CC_GDO2_PIN;
}




/* ****************************************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS, 
* EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF 
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY 
* OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND 
* NON-INFRINGEMENT OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD
* TO THE PROGRAM OR YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR 
* INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR 
* NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY 
* OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. EXCLUDED 
* DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION, 
* COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS,
* OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S AGGREGATE 
* LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE PROGRAM 
* EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted by Texas 
* Instruments is distributed as "freeware".  You may, only under TI's copyright
* in the Program, use and modify the Program without any charge or restriction.  
* You may distribute to third parties, provided that you transfer a copy of this
* license to the third party and the third party agrees to these terms by its 
* first use of the Program. You must reproduce the copyright notice and any 
* other legend of ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains copyrighted material, 
* trade secrets and other TI proprietary information and is protected by 
* copyright laws, international copyright treaties, and trade secret laws, as
* well as other intellectual property laws.  To protect TI's rights in the 
* Program, you agree not to decompile, reverse engineer, disassemble or 
* otherwise translate any object code versions of the Program to a human-
* readable form.  You agree that in no event will you alter, remove or destroy
* any copyright notice included in the Program.  TI reserves all rights not 
* specifically granted under this license. Except as specifically provided 
* herein, nothing in this agreement shall be construed as conferring by 
* implication, estoppel, or otherwise, upon you, any license or other right
* under any TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
*******************************************************************************/
