/**************************************************************************************************
 Filename:       oad_app.c

 Description:    Linux Host application for mac network processor

   Copyright (C) {YEAR} Texas Instruments Incorporated - http://www.ti.com/


   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

     Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.

     Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the
     distribution.

     Neither the name of Texas Instruments Incorporated nor the names of
     its contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

**************************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <getopt.h>
#include <pthread.h>
#include <time.h>
#include <sys/time.h>

#include "npi_lnx.h"

#include "oad_server.h"
#include "oad.h"

// Linux surrogate interface, including both RTIS and NPI
#include "npi_ipc_client.h"
#include "rti_lnx.h"
#include "oad_app.h"
#include "oad_app_main.h"
#include "timer.h"

#include "hal_rpc.h"
#include "npi_lnx_ipc_rpc.h"

// macros
#define PAUSE() { fflush(stdout); while (getchar()!='\n'); }

#ifdef __BIG_DEBUG__
#define debug_printf(fmt, ...) printf( fmt, ##__VA_ARGS__)
#else
#define debug_printf(fmt, ...)
#endif

// Application state variable
uint8 appState;
struct timeval curTime, startTime, prevTimeSend, prevTimeRec;
struct timeval curTime, prevTime;

// Pairing reference
uint8 destIdx;

uint8 appOadUseTOAD = FALSE;

// ch from console
char ch;
// string from console
char str[128];

// Toggle Timer Print on Server state variable
static uint8 toggleTimerPrintOnServer = FALSE;
static uint8 toggleBigDebugPrintOnServer = FALSE;

static uint8 recoveryPairingRef = 0x00;

//THREAD and MUTEX 
static pthread_mutex_t appThreadMutex;
static pthread_mutex_t appInitMutex;

static pthread_t AppThreadId;
static void *appThreadFunc(void *ptr);
static int appThreadTerminate;

static void appProcessEvents(uint16 events);
uint8 sbExec(uint8 *pBuf, int length);

uint8 OAD_app_threadId;

#define SB_DST_ADDR_DIV                    4

#define NAME_ELEMENT(element) [element] = #element

uint8 appSendDataState;

static void appInitSyncRes(void);

const char * const AppState_list[AP_STATE_READY + 1] = { //18 Application States
		[0 ... AP_STATE_READY] = NULL,
		NAME_ELEMENT(AP_STATE_INIT),
		NAME_ELEMENT(AP_STATE_INIT_OAD),
		NAME_ELEMENT(AP_STATE_OAD),
		NAME_ELEMENT(AP_STATE_RESET),
		NAME_ELEMENT(AP_STATE_READY)
};

appSendData_t  appSendData;

//RTi is used to read the RTI software version, all callback needs to be implemented:
void RTI_InitCnf( rStatus_t status ){};
void RTI_PairCnf( rStatus_t status, uint8 dstIndex, uint8 devType ){};
void RTI_PairAbortCnf( rStatus_t status ){};
 void RTI_UnpairCnf( rStatus_t status, uint8 dstIndex ){};
void RTI_UnpairInd( uint8 dstIndex ){};
void RTI_AllowPairCnf( rStatus_t status, uint8 dstIndex, uint8 devType ){};
void RTI_SendDataCnf( rStatus_t status ){};
void RTI_StandbyCnf( rStatus_t status ){};
void RTI_RxEnableCnf( rStatus_t status ){};
void RTI_EnableSleepCnf( rStatus_t status ){};
 void RTI_DisableSleepCnf( rStatus_t status ){};
void RTI_ResetInd( void ){};
void RTI_PairInd( rStatus_t status, uint8 dstIndex, uint8 devType ){};
void RTI_IrInd( uint8 irData ){};



int appInit(int mode, char threadId) 
{
	appThreadTerminate = 0;
	uint8 value[1];

	printf("Target device: CC%s\n", cc253x);

	//Initialize Mutex
	appInitSyncRes();

	OAD_app_threadId = threadId;

	printf("\n-------------------- START TOGGLE DEBUG TRACES on SERVER/DAEMON SIDE-------------------\n");
	npiMsgData_t pMsg;
	pMsg.len = 1;
	pMsg.subSys = RPC_SYS_SRV_CTRL | RPC_CMD_SREQ;
	pMsg.cmdId = NPI_LNX_CMD_ID_CTRL_BIG_DEBUG_PRINT_REQ;

	// Turn big debug OFF
	toggleBigDebugPrintOnServer = FALSE;

	pMsg.pData[0] = toggleBigDebugPrintOnServer;

	// send debug flag value
	NPI_SendSynchData( &pMsg );
	if (0 == pMsg.pData[0])
	{
		printf("__BIG_DEBUG_ACTIVE set to: 0x%.2X\n", toggleBigDebugPrintOnServer);
	}

	pMsg.cmdId = NPI_LNX_CMD_ID_CTRL_TIME_PRINT_REQ;

	// Turn timer debug OFF
	toggleTimerPrintOnServer = FALSE;

	pMsg.pData[0] = toggleTimerPrintOnServer;

	// send debug flag value
	NPI_SendSynchData( &pMsg );
	if (0 == pMsg.pData[0])
	{
		printf("__DEBUG_TIME_ACTIVE set to: 0x%.2X\n", toggleTimerPrintOnServer);
	}
	printf("\n-------------------- END TOGGLE DEBUG TRACES on SERVER/DAEMON SIDE -------------------\n\n");

	printf("\n-------------------- START SOFTWARE VERSION READING-------------------\n");
	if (RTI_SUCCESS != RTI_ReadItem(RTI_CONST_ITEM_SW_VERSION, 1, value))
	{
		fprintf(stderr, "Failed to read Software Version.\n");

		pMsg.cmdId = NPI_LNX_CMD_ID_CTRL_TIME_PRINT_REQ;

		// Turn timer debug OFF
		toggleTimerPrintOnServer = TRUE;

		pMsg.pData[0] = toggleTimerPrintOnServer;

		// send debug flag value
		NPI_SendSynchData( &pMsg );
		if (0 == pMsg.pData[0])
		{
			printf("__DEBUG_TIME_ACTIVE set to: 0x%.2X\n", toggleTimerPrintOnServer);
		}
		else
		{
			// Failed to set debug traces
			printf("Failed to set __DEBUG_TIME_ACTIVE to: 0x%.2X\n", toggleTimerPrintOnServer);
		}
	}
	else
	{
		printf("- Software Version = 0x%x\n", value[0]);
	}
	printf("-------------------- END SOFTWARE VERSION READING-------------------\n");

	if (pthread_create(&AppThreadId, NULL, appThreadFunc, NULL))
	{
		// thread creation failed
		printf("Failed to create app thread\n");
		return -1;
	}

	return 0;
}

static void appInitSyncRes(void)
{
	// initialize all mutexes
	if (pthread_mutex_init(&appThreadMutex, NULL))
	{
		printf("Fail To Initialize Mutex appThreadMutex\n");
		exit(-1);
	}

	if (pthread_mutex_init(&appInitMutex, NULL))
	{
		printf("Fail To Initialize Mutex appInitMutex\n");
		exit(-1);
	}
}

static void *appThreadFunc(void *ptr)
{
	//uint8 readbuf[128];
	//uint8 pollStatus = FALSE;
	uint16 events = 0;
	uint8 *pBuf;

	/* lock mutex in order not to lose signal */
	pthread_mutex_lock(&appThreadMutex);

	appState = AP_STATE_READY;

	printf("App Thread Started \n");

	//Display menu...
	DispMenuReady();

	/* thread loop */
	while (!appThreadTerminate) 
	{
		// Wait for event
		sem_wait(&event_mutex);

		events = timer_get_event(OAD_app_threadId);
		// Process events
		if (events != 0) {
			appProcessEvents(events);
			debug_printf("State: %s [0x%.2X]\n", AppState_list[appState], appState);
		}

		// Only process actions if there is a character available
		if (consoleInput.handle == MAIN_INPUT_READY)
		{
			ch = consoleInput.latestCh;
			strcpy(str, consoleInput.latestStr);

			if (appState == AP_STATE_OAD)
			{
				// Abort ongoing test for any key pressed.
				// Return to Send Data Init state
				appState = AP_STATE_READY;
			}
			// Other states
			else if (appState == AP_STATE_RESET)
			{

			}
			else if (ch == 'm')
			{
				// Display menu
				DispMenuReady();
			}
			else if (ch == '1')
			{
				if (oadBinFilePresent)
				{
					if ( appState == AP_STATE_READY)
					{
						printf("Entered OAD state\n");
						appState = AP_STATE_OAD;
						appOadUseTOAD = FALSE;
					}
					else if  ( ( appState == AP_STATE_OAD ) ||
							( appState == AP_STATE_TOAD_KO ) ||
							( appState == AP_STATE_TOAD_KO_NO_PAIRING ) )
					{
						printf("Exited OAD state\n");
						appState = AP_STATE_READY;
						// Stop timers
						timer_start_timerEx(OAD_app_threadId,
								OAD_APP_EVT_OAD_TIMEOUT,
								0);
						timer_start_timerEx(OAD_app_threadId,
								OAD_APP_EVT_TOAD_SAFE_NORMAL_TOAD_WINDOW,
								0);
						timer_start_timerEx(OAD_app_threadId,
								OAD_APP_EVT_OAD_KO,
								0);
					}
				}
				else
					printf("No binary file found\n");
			}
			else if (ch == '2')
			{
				if (oadBinFilePresent)
				{
					if ( appState == AP_STATE_READY)
					{
						printf("Upload file %s\n", oadImageFile);
						pBuf = malloc(oadImageLen);
						//rewind at the begining
						fseek(oadImageFd, 0L, SEEK_SET);
						fread(pBuf, 1, oadImageLen, oadImageFd);
						OAD_ServerInit(OAD_app_threadId, pBuf, oadImageLen);
						printf("Entered OAD state\n");
						appState = AP_STATE_OAD;
						appOadUseTOAD = TRUE;
						free(pBuf);

						// Start SAFE_NORMAL_TOAD_WINDOW timer, just in case
						// everything has failed on the Controller side
						timer_start_timerEx(OAD_app_threadId,
								OAD_APP_EVT_TOAD_SAFE_NORMAL_TOAD_WINDOW,
								10000);
					}
					else if  ( ( appState == AP_STATE_OAD ) ||
							( appState == AP_STATE_TOAD_KO ) ||
							( appState == AP_STATE_TOAD_KO_NO_PAIRING ) )
					{
						printf("Exited OAD state\n");
						appState = AP_STATE_READY;
						// Stop timers
						timer_start_timerEx(OAD_app_threadId,
								OAD_APP_EVT_OAD_TIMEOUT,
								0);
						timer_start_timerEx(OAD_app_threadId,
								OAD_APP_EVT_TOAD_SAFE_NORMAL_TOAD_WINDOW,
								0);
						timer_start_timerEx(OAD_app_threadId,
								OAD_APP_EVT_OAD_KO,
								0);
					}
				}
				else
					printf("No binary file found\n");
			}
			else if (ch == '4')
			{
				printf("\n-------------------- START SOFTWARE VERSION READING-------------------\n");

				uint8 value[1];
				if (RTI_SUCCESS != RTI_ReadItem(RTI_CONST_ITEM_SW_VERSION, 1, value))
				{
					fprintf(stderr, "Failed to read Software Version.\n");
				}
				else
				{
					printf("- Software Version = 0x%x\n", value[0]);
				}
				printf("-------------------- END SOFTWARE VERSION READING-------------------\n");
				DispMenuReady();
			}
			else if (ch == 'r')
			{
				// Return to Ready State
				appState = AP_STATE_READY;
				DispMenuReady();
			}
			else if (ch == 't')
			{
				npiMsgData_t pMsg;
				pMsg.len = 1;
				pMsg.subSys = RPC_SYS_SRV_CTRL | RPC_CMD_AREQ;
				pMsg.cmdId = NPI_LNX_CMD_ID_CTRL_TIME_PRINT_REQ;

				if (toggleTimerPrintOnServer == FALSE)
				{
					// Turn timer debug ON
					toggleTimerPrintOnServer = TRUE;
				}
				else
				{
					// Turn timer debug OFF
					toggleTimerPrintOnServer = FALSE;
				}

				pMsg.pData[0] = toggleTimerPrintOnServer;

				// send debug flag value
				NPI_SendAsynchData( &pMsg );
				if (0 == pMsg.pData[0])
				{
					printf("__DEBUG_TIME_ACTIVE set to: 0x%.2X\n", toggleTimerPrintOnServer);
				}
			}
			else if (ch == 'y')
			{
				npiMsgData_t pMsg;
				pMsg.len = 1;
				pMsg.subSys = RPC_SYS_SRV_CTRL | RPC_CMD_SREQ;
				pMsg.cmdId = NPI_LNX_CMD_ID_CTRL_BIG_DEBUG_PRINT_REQ;

				if (toggleBigDebugPrintOnServer == FALSE)
				{
					// Turn timer debug ON
					toggleBigDebugPrintOnServer = TRUE;
				}
				else
				{
					// Turn timer debug OFF
					toggleBigDebugPrintOnServer = FALSE;
				}

				pMsg.pData[0] = toggleBigDebugPrintOnServer;

				// send debug flag value
				NPI_SendSynchData( &pMsg );
				if (0 == pMsg.pData[0])
				{
					printf("__BIG_DEBUG_ACTIVE set to: 0x%.2X\n", toggleBigDebugPrintOnServer);
				}
			}
			else if (ch != '\n')
			{
				printf("unknown command %c (0x%.2X) \n", ch, ch);
				DispMenuReady();
			}

			// 'q' requires special attention, hence it is not part of if/else
			if (ch == 'q')
			{
				//Terminate Thread
				appThreadTerminate = 1;
				//Terminate Thread and exit everything
			}

			// Release handle at the end to indicate to main thread that character is processed
			consoleInput.handle = MAIN_INPUT_RELEASED;
		}

	}
	pthread_mutex_unlock(&appThreadMutex);

  return NULL;
}

/**************************************************************************************************
 *
 * @fn      RTI_ReceiveDataInd
 *
 * @brief   RTI receive data indication callback asynchronously initiated by
 *          another node. The client is expected to complete this function.
 *
 * input parameters
 *
 * @param   srcIndex:  Pairing table index.
 * @param   profileId: Profile identifier.
 * @param   vendorId:  Vendor identifier.
 * @param   rxLQI:     Link Quality Indication.
 * @param   rxFlags:   Receive flags.
 * @param   len:       Number of bytes to send.
 * @param   *pData:    Pointer to data to be sent.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 */
void RTI_ReceiveDataInd(uint8 srcIndex, uint8 profileId, uint16 vendorId,
		uint8 rxLQI, uint8 rxFlags, uint8 len, uint8 *pData) {
	int i;
	static uint8 lastSource = RTI_INVALID_PAIRING_REF;

	if ( (appState == AP_STATE_TOAD_KO) ||
			(appState == AP_STATE_TOAD_KO_NO_PAIRING) ||
			appState == AP_STATE_OAD)
	{
		if ((rxFlags & RTI_RX_FLAGS_VENDOR_SPECIFIC)
				&& (vendorId == RTI_VENDOR_TEXAS_INSTRUMENTS))
		{
			// TI vendor specific commands are profile neutral
			if (pData[0] == RTI_PROTOCOL_OAD)
			{
				OAD_ProcessDataInd(srcIndex, pData, len);
				// Must stop SAFE_NORMAL_TOAD_WINDOW timer
				timer_start_timerEx(OAD_app_threadId,
						OAD_APP_EVT_TOAD_SAFE_NORMAL_TOAD_WINDOW,
						0);
			}
			else if (pData[0] == RTI_PROTOCOL_POLL)
			{
				OAD_HandlePoll(srcIndex);
				// Extend SAFE_NORMAL_TOAD_WINDOW timer, in case
				// poll was invalid.
				timer_start_timerEx(OAD_app_threadId,
						OAD_APP_EVT_TOAD_SAFE_NORMAL_TOAD_WINDOW,
						5000);

				if (appState == AP_STATE_TOAD_KO_NO_PAIRING)
				{
					// If we received a message in this state it means
					// we should abort the ongoing AllowPair
					RTI_AllowPairAbortReq();
				}
			}
			// Now change state as we have recovered
			if ( (appState == AP_STATE_TOAD_KO) || (appState == AP_STATE_TOAD_KO_NO_PAIRING) )
			{
				// This means that the Controller has recovered, and we're no longer in
				// KO state. Hence, we can go back to OAD state.
				appState = AP_STATE_OAD;
			}
		}
	}
	else
	{
		if ((rxFlags & RTI_RX_FLAGS_VENDOR_SPECIFIC)
				&& (vendorId == RTI_VENDOR_TEXAS_INSTRUMENTS)
				&& (pData[0] == RTI_PROTOCOL_POLL) )
		{
				OAD_HandlePoll(srcIndex);
		}
		else if ((rxFlags & RTI_RX_FLAGS_VENDOR_SPECIFIC)
				&& (vendorId == RTI_VENDOR_TEXAS_INSTRUMENTS)
				&& (pData[0] == RTI_PROTOCOL_OAD)
				&& (pData[1] == (OAD_STATUS_CMD | OAD_RSP_BIT)))
		{
			printf("\n*****************************************\n");
			printf("Status Command received from 0x%.2X\n", srcIndex);
			printf("OAD Client State : \t\t0x%.2X\n", pData[2]);
			printf("Previous Sequence Number : \t0x%.4X\n", *(uint16 *)&pData[3]);
			printf("Active Image ID: \t\t0x%.8X\n", *(uint32 *)&pData[5]);
			printf("Downloaded Image ID: \t\t0x%.8X\n", *(uint32 *)&pData[9]);
			uint32 maxSize = pData[13] + pData[14] * 0x100 + pData[15] * 0x10000;
			printf("Max Image Size Allowed: \t%d bytes (%dkB)\n", maxSize, maxSize/0x400);
			printf("*****************************************\n");
		}
		else
		{
			if (srcIndex != lastSource)
				printf("\n");
			printf("Source Idx: %d, profileId %d, rxLQI %d (-%3d dBm)",
					srcIndex, profileId, rxLQI, (0xFF - rxLQI)/2);

			printf("\tData: ");
			for (i = 0; i < len; i++)
				printf("%d ", pData[i]);
			printf("\n");
		}
	}

	lastSource = srcIndex;

//	//For Debug Purpose!!!
//	if (error)
//		exit(-1);
}

/**************************************************************************************************
 *
 * @fn      appProcessEvents
 *
 * @brief   Process events function
 *
 * @param   events - 16 bit mask
 *
 * @return  void
 */
static void appProcessEvents(uint16 events)
{
	uint16 procEvents = 0;
	int mutexRet = 0;

	if (events & OAD_APP_EVT_OAD_OK)
	{
		printf("OAD successful, State: %s [0x%.2X]\n", AppState_list[appState], appState);
		appState = AP_STATE_READY;
		procEvents |= OAD_APP_EVT_OAD_OK;
		DispMenuReady();
	}
	if (events & OAD_APP_EVT_OAD_TIMEOUT)
	{
		if (appState != AP_STATE_READY)
		{
			printf("OAD Timed Out. State: %s [0x%.2X]\n", AppState_list[appState], appState);
			procEvents |= OAD_APP_EVT_OAD_TIMEOUT;
			if ( (appState == AP_STATE_OAD) && (appOadUseTOAD == TRUE) )
			{
				// Assume KO state for timeout:
				// 1. It can still recover if an OAD message is received
				// 2. We can timeout for recovery
				appState = AP_STATE_TOAD_KO;
				timer_start_timerEx(OAD_app_threadId,
						OAD_APP_EVT_TOAD_SAFE_NORMAL_TOAD_WINDOW,
						10000);
			}
			else
			{
				appState = AP_STATE_READY;
			}
		}
		else
		{
			printf("OAD Timed Out, but we're in state: %s [0x%.2X]\n", AppState_list[appState], appState);
			procEvents |= OAD_APP_EVT_OAD_TIMEOUT;
		}
	}
	if (events & OAD_APP_EVT_OAD_KO)
	{
		procEvents |= OAD_APP_EVT_OAD_KO;
		if (appOadUseTOAD == FALSE)
		{
			appState = AP_STATE_READY;
			printf("OAD KO. State: %s [0x%.2X]\n", AppState_list[appState], appState);
		}
		else
		{
			// Notify user of failed TOAD
			printf("TOAD unsuccessful, allowing retry\n");

			if (appState == AP_STATE_OAD)
			{
				// Need to go into a state
				appState = AP_STATE_TOAD_KO;
				// If we are in TOAD mode start a timer to account
				// for lost pairing entry on the client side
				// Placed here it is an initial kick-off timer.

				// Start timer greater than the period expected that a TOAD client will poll
				// The period should exceed worst case normal behavior. A very slow CRC may
				// take up to 20-30 seconds. This is the case where the TOAD has lost its
				// pairing entry.
				timer_start_timerEx(OAD_app_threadId,
						OAD_APP_EVT_TOAD_SAFE_NORMAL_TOAD_WINDOW,
						10000);
			}
			else
			{
				printf("Unexpected state: 0x%.2X\n", appState);
			}
		}
	}
	if (events & OAD_APP_EVT_TOAD_SAFE_NORMAL_TOAD_WINDOW)
	{
		if (appOadUseTOAD == TRUE)
		{
			printf("TOAD Safe Window Expired. State: %s [0x%.2X]\n", AppState_list[appState], appState);

			// This is the case where we have to call RTI_AllowPairReq() so
			// that the temporary pairing information can be exchanged.
			appState = AP_STATE_TOAD_KO_NO_PAIRING;

			// Allow reception of a special Discovery Request with the special Profile ID 0xD1 and the special User String
			// User String active; 1 Device (Television); 2/3 Profiles (ZRC / & ZID / & TOAD)
			uint8 pBuf[RTI_USER_STRING_LENGTH];

			// Only support TOAD Profile 0xD1
			pBuf[0] = RTI_PROFILE_TOAD;
			RTI_WriteItemEx(RTI_PROFILE_RTI, RTI_CP_ITEM_APPL_PROFILE_ID_LIST, RTI_MAX_NUM_PROFILE_IDS, pBuf);
			pBuf[0] = RTI_BUILD_APP_CAPABILITIES(1, 1, 2);
			RTI_WriteItemEx(RTI_PROFILE_RTI, RTI_CP_ITEM_APPL_CAPABILITIES, 1, pBuf);

			// Use first pairing entry as base for a temporary pairing to be used as
			// the recovery pairing entry.
			rcnNwkPairingEntry_t apTempPairingEntryBuf;
			apTempPairingEntryBuf.pairingRef = 0x00;// Use the first pairing as basis.

			(void)RTI_WriteItemEx(RTI_PROFILE_RTI, RTI_SA_ITEM_PT_CURRENT_ENTRY_INDEX, sizeof(rcnNwkPairingEntry_t), (uint8*)&apTempPairingEntryBuf.pairingRef);
			(void)RTI_ReadItemEx(RTI_PROFILE_RTI, RTI_SA_ITEM_PT_CURRENT_ENTRY, sizeof(rcnNwkPairingEntry_t), (uint8*)&apTempPairingEntryBuf);
			uint8 maxNumOfPairingEntries;
			(void)RTI_ReadItemEx(RTI_PROFILE_RTI, RTI_CONST_ITEM_MAX_PAIRING_TABLE_ENTRIES, 1, (uint8*)&maxNumOfPairingEntries);
			// Alternate between the two last pairing table entries
			recoveryPairingRef++;
			if ( recoveryPairingRef >= maxNumOfPairingEntries )
				recoveryPairingRef = maxNumOfPairingEntries - 2;
			apTempPairingEntryBuf.pairingRef = recoveryPairingRef;// Use the second pairing entry for this temporary pairing.
			apTempPairingEntryBuf.securityKeyValid = FALSE;// Disable security for this link
			apTempPairingEntryBuf.recFrameCounter = 0;// Reset received frame counter

			// Make sure we have updated SRC NWK Address
			(void)RTI_ReadItemEx(RTI_PROFILE_RTI, RTI_SA_ITEM_SHORT_ADDRESS, sizeof(apTempPairingEntryBuf.srcNwkAddress), (uint8*)&apTempPairingEntryBuf.srcNwkAddress);

			// Make sure we have updated Pan ID
			(void)RTI_ReadItemEx(RTI_PROFILE_RTI, RTI_SA_ITEM_PAN_ID, sizeof(apTempPairingEntryBuf.panId), (uint8*)&apTempPairingEntryBuf.panId);

			// To make sure we send a response using Short Address we need to say that the
			// recipient is a Target! This is a workaround to avoid modifying the RNP
			apTempPairingEntryBuf.recCapabilities = RTI_BUILD_NODE_CAPABILITIES(1, 1, 0, 1);

			// Use constant OAD_IMAGE_ID for now. Could be read from image directly
			pBuf[0] = BREAK_UINT32( OAD_IMAGE_ID, 0);
			pBuf[1] = BREAK_UINT32( OAD_IMAGE_ID, 1);
			pBuf[2] = BREAK_UINT32( OAD_IMAGE_ID, 2);
			pBuf[3] = BREAK_UINT32( OAD_IMAGE_ID, 3);
			uint16 randNwkAddr = (uint16)rand();
			pBuf[4] = BREAK_UINT32( randNwkAddr, 0);
			pBuf[5] = BREAK_UINT32( randNwkAddr, 1);
			apTempPairingEntryBuf.nwkAddress = randNwkAddr;

			pBuf[6] = BREAK_UINT32( apTempPairingEntryBuf.srcNwkAddress, 0);
			pBuf[7] = BREAK_UINT32( apTempPairingEntryBuf.srcNwkAddress, 1);

			(void)RTI_WriteItemEx(RTI_PROFILE_RTI, RTI_SA_ITEM_USER_STRING, RTI_USER_STRING_LENGTH, pBuf);
			// Apply these new network settings to the RNP
			(void)RTI_WriteItemEx(RTI_PROFILE_RTI, RTI_SA_ITEM_PT_CURRENT_ENTRY_INDEX, sizeof(rcnNwkPairingEntry_t), (uint8*)&apTempPairingEntryBuf.pairingRef);
			(void)RTI_WriteItemEx(RTI_PROFILE_RTI, RTI_SA_ITEM_PT_CURRENT_ENTRY, sizeof(rcnNwkPairingEntry_t), (uint8*)&apTempPairingEntryBuf);

			// Display temporary pairing entry.
			printf("*****************************************\n");
			printf("* Pairing Index: \t 	0x%.2X\n", apTempPairingEntryBuf.pairingRef);
			printf("* SRC NWK Address: \t 	0x%.4hX\n", apTempPairingEntryBuf.srcNwkAddress);
			printf("* Logical Channel: \t 	0x%.2hX\n", apTempPairingEntryBuf.logicalChannel); //((apTempPairingEntryBuf.logicalChannel >> 8) & 0x00FF));
			printf("* IEEE Address:  		%.2hX", (apTempPairingEntryBuf.ieeeAddress[SADDR_EXT_LEN - 1] & 0x00FF));
			int j;
			for (j = (SADDR_EXT_LEN - 2); j >= 0; j--)
			{
				printf(":%.2X", (apTempPairingEntryBuf.ieeeAddress[j] & 0x00FF));
			}
			printf("\n");
			printf("* PAN Id: \t \t \t0x%.4X\n", apTempPairingEntryBuf.panId);
			printf("* NWK Address: \t \t 	0x%.4hX\n", apTempPairingEntryBuf.nwkAddress);
			printf("* Rec Capabilities: \t 	0x%.2hX\n",
					(apTempPairingEntryBuf.recCapabilities & 0x00FF));
			printf("* Security Key Valid: \t \t0x%.2hX\n",
					(apTempPairingEntryBuf.securityKeyValid & 0x00FF));
			printf("* Vendor Identifier: \t \t0x%.4hX\n",
					apTempPairingEntryBuf.vendorIdentifier);
			printf("* Device Type List: \t 	[0x%.2hX, 0x%.2hX, 0x%.2hX]\n",
					(apTempPairingEntryBuf.devTypeList[0] & 0x00FF),
					(apTempPairingEntryBuf.devTypeList[1] & 0x00FF),
					(apTempPairingEntryBuf.devTypeList[2] & 0x00FF));
			printf("* Received Frame Counter: \t0x%.8X (%u)\n",
					apTempPairingEntryBuf.recFrameCounter, apTempPairingEntryBuf.recFrameCounter);
			printf("* Profiles Discovered: \t 	0x%.4X\n",
					apTempPairingEntryBuf.profileDiscs[0]);
			printf("*****************************************\n");

			// Activate the configuration setting
			//When Launching the Thread, Mutex is unlocked.
			if ( (mutexRet = pthread_mutex_trylock(&appInitMutex)) == EBUSY)
			{
				debug_printf("[MUTEX] appInit Mutex busy\n");
			}
			else
			{
				debug_printf("[MUTEX] appInit Lock status: %d\n", mutexRet);
			}
			RTI_InitReq();
		}
		// Mark this event as processed
		procEvents |= OAD_APP_EVT_TOAD_SAFE_NORMAL_TOAD_WINDOW;
	}

	// Clear event
	timer_clear_event(OAD_app_threadId, events);
}

