/**************************************************************************************************
 Filename:       oad_app_main.cpp

 Description:    Linux serial boot loader application (target node)

  Copyright (C) {YEAR} Texas Instruments Incorporated - http://www.ti.com/


   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

     Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.

     Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the
     distribution.

     Neither the name of Texas Instruments Incorporated nor the names of
     its contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

**************************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <poll.h>


// Linux surrogate interface
#include "npi_ipc_client.h"
#include "npi_lnx_error.h"


#include "oad_server.h"
#include "oad_app_main.h"
#include "timer.h"

#include "hal_rpc.h"
#define SB_DST_ADDR_DIV                    4

// macros
#define PAUSE() { fflush(stdout); while (getchar()!='\n'); }

#define NAME_ELEMENT(element) [element] = #element

struct pollfd fds[1];
const char *device = "";
char *cc253x = "";

char *oadImageFile = "./imageFile.bin";
int oadBinFilePresent = TRUE;
static int oadImagePresent = FALSE;
int oadImageLen;
FILE *oadImageFd;

enum 
{
	OAD_main_linux_threadId,
	OAD_App_threadId,
	OAD_main_threadId_tblSize
};

sem_t event_mutex;

static void print_usage(const char *prog) {
	printf("Usage: %s [-DsbdlHOLC3]\n", prog);
	puts(
			"  -D --device \t\tdevice to use (default /dev/spidev4.0). For Socket use format IPaddress:port\n"
			"  -d --cc253x device \tType of CC253x device. (2531 - 2533 - 2530, or special ZID Dongle: '2531ZID')\n"
			"  -b --oadimage  \t\tbinary image file to upload to remote device using the Over-the-Air Download protocol (default ./imagefile.bin)\n");
	exit(1);
}

static void parse_opts(int argc, char *argv[])
{
	while (1)
	{
		static const struct option lopts[] =
		{
			{ "device", 1, 0, 'D' },
			{ "cc253x", 1, 0, 'd' },
			{ "oadimage", 1, 0, 'b' },
			{ NULL, 0, 0, 0 },
		};
		int c;

		c = getopt_long(argc, argv, "D:d:b:", lopts, NULL);

		if (c == -1)
			break;

		switch (c)
		{
		case 'D':
			device = optarg;
			break;
		case 'd':
			cc253x = optarg;
			break;
		case 'b':
			oadImagePresent = TRUE;
			oadImageFile = optarg;
			break;
		default:
			print_usage(argv[0]);
			break;
		}
	}
}

int main(int argc, char **argv) 
{
	if (argc < 2)
	{
		print_usage(argv[0]);
		return -1;
	}

	printf("Starting...\n");

	// Return value for main
	int ret = 0;

	consoleInput.latestCh = ' ';
	consoleInput.handle = MAIN_INPUT_RELEASED;

	// setup filedescriptor for stdin
	fds[0].fd = fileno(stdin);
	fds[0].events = POLLIN;

	parse_opts(argc, argv);

	// Initialize shared semaphore. Must happen before program begins execution
	sem_init(&event_mutex,0,1);

	// Subscribe to  wanted Service

	// Initialize Network Processor Interface.
	if ((ret = NPI_ClientInit(device)) != NPI_LNX_SUCCESS)
	{
		fprintf(stderr, "Failed to start NPI library module, device; %s\n", device);
		print_usage(argv[0]);
		return ret;
	}

	if (oadImagePresent)
	{
		//openimage file
		oadImageFd = fopen(oadImageFile, "rb");
		if ((!oadImageFd) && (argv[2]))
		{
			printf("Error with upload binary file %s\n", oadImageFile);
			oadBinFilePresent = FALSE;
		}
	}
	else
		oadBinFilePresent = FALSE;

	if (oadBinFilePresent)
	{
		//get File lenght
		fseek(oadImageFd, 0, SEEK_END);
		oadImageLen = ftell(oadImageFd);
		//rewind at the begining
		fseek(oadImageFd, 0L, SEEK_SET);
		printf("binary image file length: %d\n", oadImageLen);
	}

	//Start MAC thread, management of RTI command in separate thread.
	if ((ret = appInit(0, OAD_App_threadId)) != 0)
	{
		return ret;
	}

	printf("Starting timer thread\n");

	//Start MAC timer thread, management of timer in separate thread.
	// No Timer for Mac for now...
	if ((ret = timer_init(OAD_main_threadId_tblSize)) != 0)
	{
		printf("Failed to start timer thread. Exiting...");
		return ret;
	}

	printf("\n------------------------ INITIALIZE OAD SERVER  ----------------------\n");
	printf("Upload file %s (size %d)\n", oadImageFile, oadImageLen);
	uint8 *pBuf = malloc(oadImageLen);
	//rewind at the begining
	fseek(oadImageFd, 0L, SEEK_SET);
	fread(pBuf, 1, oadImageLen, oadImageFd);
	OAD_ServerInit(OAD_App_threadId, pBuf, oadImageLen);
	free(pBuf);
	printf("------------------------ OAD SERVER INITIALIZED ----------------------\n");


	//first Menu will be display at the end of the RNP initialization.

	int pollRet;

	//MANAGE MAIN DISPLAY HERE
	// Equivalent to OSAL Main Loop
	while (1) 
	{
		//Check Display buffer, if not empty display one element

		// Check for input characters, timeout after 500ms
		pollRet = poll(fds, 1, 500);
		if ((pollRet == 1) && (consoleInput.handle == MAIN_INPUT_RELEASED))
		{
			fgets(consoleInput.latestStr, sizeof(consoleInput.latestStr), stdin);
			// Remove \n character from string
			char* p;
			if ( (p= strchr(consoleInput.latestStr, '\n')) != NULL)
				*p = '\0';
			consoleInput.latestCh = consoleInput.latestStr[0];
			if (consoleInput.latestCh == 'q')
			{
				ret = 0;
				break;
			}
			// Do not act on -1, . and new line (\n)
			if ( (consoleInput.latestCh != -1)
					&& (consoleInput.latestCh != '.')
					&& (consoleInput.latestCh != '\n') )
			{
				// Indicate to application thread that the input is ready
				consoleInput.handle = MAIN_INPUT_READY;
				// Release resources waiting for this event
				sem_post(&event_mutex);
			}
		}
	}

	// Destroy semaphores
	sem_destroy(&event_mutex);

	// Remember to terminate all threads.
	// [FLO] They are automatically terminated by exiting the application

	// Close Network Processor Connection
	printf("[APP]Close Socket Connection\n");
	NPI_ClientClose();

	return ret;
}

void DispMenuReady(void)
{
	printf("------------------------------------------------------\n");
	printf("Main MENU:\n");
	printf("q- exit\n");
	printf("1- Upload binary image to remote device \n");
	printf("2- Upload binary image to remote device, using TOAD \n");
	printf("4- Read Network Processor Software Version \n");
	printf("r- Return from OAD state\n");
	printf("t- Toggle __DEBUG_TIME_ACTIVE on Server\n");
	printf("y- Toggle __BIG_DEBUG on Server\n");
	printf("m- Show This Menu\n");
}

