/**************************************************************************************************
  Filename:       oad.h
  Revised:        $Date: 2012-01-20 13:08:00 -0800 (fr, 20 jan 2012) $
  Revision:       $Revision: 166 $

  Description:    Header for RemoTI OAD.

  Copyright 2008-2011 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED �AS IS� WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/

#ifndef _OAD_H
#define _OAD_H

#include "hal_types.h"
#include "rti_lnx_constants.h"

#if defined _WIN3 || defined unix
#define NEAR_FUNC
#define XDATA
#elif defined __ICC430__
#define NEAR_FUNC
#define XDATA
#else
#define NEAR_FUNC __near_func
#define XDATA __xdata
#endif

/*********************************************************************
 * Macros
 */
#define OAD_GET_BASE(i)             ((i==OAD_OP_IMAGE)? OAD_OP_BASE:OAD_DL_BASE)

// Converts byte aligned address to word aligned
#define OAD_WORD_ALIGN(a)           (((int32)(a))>>2)
#define OAD_BYTE_ALIGN(a)           (((int32)(a))<<2)

/*********************************************************************
 * Constants
 */

// OAD Boot Status Values
#define OAD_BOOT_OP_STATE                   1
#define OAD_COPY_DL_STATE                   2

// OAD Nonvolatile Memory Actions
#define OAD_READ                            0
#define OAD_WRITE                           1

// OAD Image Types
#define OAD_OP_IMAGE                        0
#define OAD_DL_IMAGE                        1

#if defined __ICC430__
// Flash Parameter for MSP430:
#define OAD_PAGESIZE                        0x200
#define OAD_PAGE_SHIFT                      9
#else
// Flash Parameter for 8051 SoC: Minimum of 0x800 since OAD bootloader needs that much at low flash.
// Note that rounding will waste 0x400 and high flash on parts with 1K flash page size.
#define OAD_PAGESIZE                        0x800
#define OAD_PAGE_SHIFT                      11
#endif

#if defined ( CC2533F96OAD )
#if ((HAL_FLASH_PAGE_SIZE % 2) != 0)
#error Formula breaks because of odd boundary.
#endif
#if (!defined FEATURE_TOAD) && (!defined FEATURE_TOAD_RC)
#define OAD_STORAGE_SIZE         ((HAL_FLASH_PAGE_CNT * 1024UL - \
                                 ((HAL_NV_PAGE_CNT + 1) * HAL_FLASH_PAGE_SIZE) - OAD_PAGESIZE) / 2)
#define OAD_DL_BASE     (OAD_STORAGE_SIZE + OAD_PAGESIZE)
#else //FEATURE_TOAD
// Size of TOAD must match the allocated space in the linker file. Currently the TOAD
// span two areas: 
//-D_CODE_BEG_PART1=0x0000
//-D_CODE_END_PART1=0x2FFF
//-D_CODE_BEG_PART2=0x18000
//-D_CODE_END_PART2=0x1CFFF
// This amounts to 12K + 20K = 32K
#define TOAD_SIZE                (16 * HAL_FLASH_PAGE_SIZE)
#define OAD_STORAGE_SIZE         (HAL_FLASH_PAGE_CNT * 1024UL - \
                                 ((HAL_NV_PAGE_CNT + 1) * HAL_FLASH_PAGE_SIZE) - TOAD_SIZE)
#define OAD_DL_BASE          0x3000
#define OAD_DL_END_PART1     0x7FFF
//#define OAD_DL_SIZE_PART1    0x5000   //(0x8000 - 0x3000)
//#define OAD_DL_BASE_PART2    0xD000
//#define OAD_DL_END_PART2     0xFFFF
//#define OAD_DL_SIZE_PART2    0x3000   //(0x10000 - 0x0D000)
//#define OAD_DL_BASE_PART3    0x10000
//#define OAD_DL_SIZE_PART3    ( 0x8000 - ((HAL_NV_PAGE_CNT + 1) * HAL_FLASH_PAGE_SIZE))
#define OAD_DL_END           ((HAL_FLASH_PAGE_CNT * 1024UL - \
                              ((HAL_NV_PAGE_CNT + 1) * HAL_FLASH_PAGE_SIZE)) - 1)
#define OAD_DL_SKIP_SIZE       0x5000
#endif //FEATURE_TOAD

#define TOAD_PAGE_SIZE         (HAL_FLASH_PAGE_SIZE / HAL_FLASH_WORD_SIZE)

#else

#if defined ( CC2530F256OAD )
// part with 256K flash
  #define OAD_DL_BASE                       0x20000
  #define OAD_STORAGE_SIZE                  (0x20000 - 2*OAD_PAGESIZE)
#elif defined __ICC430__
// host MSP430
  #define OAD_DL_BASE                       0x27600
  // Total flash size minus OAD image download start address and one page for
  // power-cycle CRC
  #define OAD_STORAGE_SIZE                  (0x37000 - OAD_DL_BASE)
  // Store the image CRC at the last 4 bytes of the downloadable area
  #define OAD_IMAGE_CRC_SIZE                4
  #define OAD_IMAGE_CRC_OFFSET              (OAD_STORAGE_SIZE - OAD_IMAGE_CRC_SIZE)
#else
// part with 128K flash: either CC2430F128 or CC2530F128
  #define OAD_DL_BASE                       0x10000
  #define OAD_STORAGE_SIZE                  (0x10000 - 2*OAD_PAGESIZE)
#endif

#if !defined _WIN32 && !defined unix
#if ((OAD_DL_BASE % HAL_FLASH_PAGE_SIZE) != 0)
#error Base address not on page boundary
#endif
#endif
#endif

#if (!defined FEATURE_TOAD) && (!defined FEATURE_TOAD_RC)
#define OAD_OP_BASE                         0x00800
#else
// There's no difference between OP and DL image for TOAD, and hence their base is the same
#define OAD_OP_BASE                         0x3000
#endif // FEATURE_TOAD_RC
#define OAD_DL_HEADER_OFFSET                0x100

// OAD Preamble constants
#ifndef OAD_PREAMBLE_OFFSET
#define OAD_PREAMBLE_OFFSET                 0xC0
#endif //OAD_PREAMBLE_OFFSET
#define OAD_PRE_LEN_POS                     (OAD_PREAMBLE_OFFSET + 0)
#define OAD_PRE_EN_DL_POS                   (OAD_PREAMBLE_OFFSET + 4)
#define OAD_PRE_BOOT_OP_POS                 (OAD_PREAMBLE_OFFSET + 8)

// Size of data sent with each data command
// Note: The OAD_PAGESIZE modulo OAD_DL_DATA_LEN must be 0
#define OAD_DL_DATA_LEN                     64


// Magic numbers used in preamble to indicate valid OAD image
#define OAD_MAGIC                           ((uint32)0xF8F81234)
#define OAD_ENABLE_DL                       ((uint32)0x0ADEAB1E)
#define OAD_BOOT_OP                         ((uint32)0x0AD0B007)
#define TOAD_BOOT_OP                        ((uint32)0x70ADB007)

// Offset to OAD packet from start of MSDU in MAC data indication
#define OAD_DATA_OFFSET                     2

// OAD Packet command codes
#define OAD_STATUS_CMD                      0
#define OAD_START_CMD                       1
#define OAD_DATA_CMD                        2
#define OAD_ENABLE_CMD                      3
#define OAD_CANCEL_CMD                      4

// Indicates packet is response to a command
#define OAD_RSP_BIT                         0x80

// OAD Packet structure
#define OAD_PROTOCOL_BYTE                   0
#define OAD_CMD_BYTE                        1
#define OAD_STATUS_BYTE                     2

// Length of OAD response packets
#ifdef FEATURE_TOAD_RC
#define OAD_STATUS_RSP_LEN                  17
#else
#define OAD_STATUS_RSP_LEN                  16
#endif
#define OAD_START_RSP_LEN                   3
#define OAD_DATA_RSP_LEN                    5
#define OAD_ENABLE_RSP_LEN                  3
#define OAD_CANCEL_RSP_LEN                  3

// Length of OAD request packets
#define OAD_STATUS_REQ_LEN                  2
#ifndef FEATURE_OAD_EXT
#define OAD_START_REQ_LEN                   4
#else
#define OAD_START_REQ_LEN		            5	// One extra byte for ImageType
#endif //FEATURE_OAD_EXT
#define OAD_DATA_REQ_LEN                    (OAD_DL_DATA_LEN+6)
#define OAD_ENABLE_REQ_LEN                  10
#define OAD_CANCEL_REQ_LEN                  2

// OAD Status Codes
#define OAD_SUCCESS                         0
#define OAD_FAILURE                         1
#define OAD_ALREADY_STARTED                 2
#define OAD_NOT_STARTED                     3
#define OAD_DL_RESEND_DATA                  4
#define OAD_DL_NOT_COMPLETE                 5
#define OAD_BAD_CRC                         6
#define OAD_INVALID_FILE                    7
#define OAD_NO_RESPOSNE                     8
#define OAD_CANCELED                        9
#define OAD_NO_RESOURCES                    10
#define OAD_FILESYSTEM_ERROR                11
#define OAD_IMAGE_TOO_BIG                   12
#define OAD_INCOMPATIBLE_IMAGE              13

// OAD Profile
#define OAD_PROFILE_ID                      RTI_PROFILE_TI
#define OAD_VENDOR_ID                       RTI_VENDOR_TEXAS_INSTRUMENTS
#define OAD_TX_OPTIONS                      (RTI_TX_OPTION_ACKNOWLEDGED | RTI_TX_OPTION_VENDOR_SPECIFIC)

// OAD Image ID
#ifndef OAD_IMAGE_ID
#define OAD_IMAGE_ID                        0x2533F096
#endif

#define TOAD_RX_OPTION_DISC_RESPONSE        0x40  // !! Must not cover 3 LSBs

#define TOAD_USER_STRING_OAD_IMAGE_ID_OFFSET        0
#define TOAD_USER_STRING_ASSIGNED_NWK_ADDR_OFFSET   4
#define TOAD_USER_STRING_NWK_ADDR_OFFSET            6
#define TOAD_USER_STRING_PAN_ID_OFFSET             10
#define TOAD_USER_STRING_IEEE_ADDR_OFFSET          12
#define TOAD_USER_STRING_EXT_LENGTH                (TOAD_USER_STRING_IEEE_ADDR_OFFSET + SADDR_EXT_LEN + 1)

#ifdef FEATURE_OAD_EXT
// OAD Image Type identifier	- Type identifier to be appended to Start Request cmd
//				- Only used if OAD is not exclusively for updating the SoC image,
//				  i.e. OAD protocol could now support a multiple of different images
#define	OAD_TI_SOC_RNP			            0x00  // TI System-on-Chip RNP image
#define	OAD_RAW_BIN			                0x11  // Raw binary
// NOTE:  Possible confusion here with the image type distinction OP / DL, which is also
//        called OAD Image Type.
#endif


#define TOAD_STATUS_LENGTH 17
/*********************************************************************
 * Typedefs
 */

// Void function data type
typedef NEAR_FUNC void func_t(void);

// Image Preamble
// To be compatible with MS and unix native target
// declare pragma for structure packing
#if defined(_MSC_VER) || defined(unix) || (defined(__ICC430__) && (__ICC430__==1))
#pragma pack(1)
#endif
typedef struct
{
  uint32 length;
  uint32 enable_dl;
  uint32 boot_op;
  uint32 magic;
  uint32 imgID;
  func_t *startAddress;
} preamble_t;

#if defined(_MSC_VER) || defined(unix) || (defined(__ICC430__) && (__ICC430__==1))
#pragma pack()
#endif

/*********************************************************************
 * External Function Declarations
 */
#ifdef __cplusplus
extern "C"
{
#endif

#ifndef __ICC430__
// Flash assembly routines
extern NEAR_FUNC uint8 GetCodeByte(uint32 address);
extern NEAR_FUNC void flashErasePage(uint8, uint8 XDATA *);
extern NEAR_FUNC void halFlashDmaTrigger(void);
#endif //__ICC430__

// OAD CRC Entrypoint
uint8 OAD_CheckCode(uint32 imgCRC, uint32 imageLen, uint8 imageType);

// OAD Flash Entrypoint
extern void  OAD_GetPreamble(uint8 itype, preamble_t *info);
extern uint8 OAD_GetStatus(void);
extern void  OAD_SetStatus(uint8 status);
#ifdef __ICC430__
extern void  OAD_ReadFlash(uint32 address, uint8 *buf, uint16 len);
void OAD_WriteFlash(uint32 address, uint8 *buf, uint8 len);
#else
extern void  OAD_ReadFlash(uint32 page, uint8 XDATA *buf, uint16 len);
void OAD_WriteFlash(uint16 address, uint8 XDATA *buf, uint8 len);
#endif //__ICC430__
void OAD_EraseFlash(uint32 address);

#ifdef __ICC430__
/*********************************************************************
 * Declarations, Typedefs, Defines taken from hal_oad.h to make it work for the MSP430F5438
 */
#define HAL_OAD_XNV_IS_INT         TRUE
#define HAL_OAD_DL_OSET           0x27600

#endif //__ICC430__

#ifdef __cplusplus
}
#endif

#endif // _OAD_H
