/**************************************************************************************************
 Filename:       oad_server.c
 Revised:        $Date: 2012-01-27 14:33:06 -0800 (fr, 27 jan 2012) $
 Revision:       $Revision: 184 $

 Description:    This file contains a sample OAD Server application


 Copyright 2008-2009 Texas Instruments Incorporated. All rights reserved.

 IMPORTANT: Your use of this Software is limited to those specific rights
 granted under the terms of a software license agreement between the user
 who downloaded the software, his/her employer (which must be your employer)
 and Texas Instruments Incorporated (the "License").  You may not use this
 Software unless you agree to abide by the terms of the License. The License
 limits your use, and you acknowledge, that the Software may not be modified,
 copied or distributed unless embedded on a Texas Instruments microcontroller
 or used solely and exclusively in conjunction with a Texas Instruments radio
 frequency transceiver, which is integrated into your product.  Other than for
 the foregoing purpose, you may not use, reproduce, copy, prepare derivative
 works of, modify, distribute, perform, display or sell this Software and/or
 its documentation for any purpose.

 YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
 PROVIDED �AS IS� WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
 INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
 NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
 TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
 NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
 LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
 INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
 OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
 OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
 (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

 Should you have any questions regarding your right to use this Software,
 contact Texas Instruments Incorporated at www.TI.com.
 **************************************************************************************************/

#if defined unix
#include "rti_lnx.h"
#include "oad_app_main.h"
#include "timer.h"
#else
#include "rti.h"
#endif
#include "oad.h"
#include "oad_app.h"
#include "oad.h"
#include "oad_server.h"
#include "oad_client.h"

#if defined unix
#include <stdlib.h>
#include <string.h>
uint8 *pOADimageBuffer;
#endif // unix
typedef struct {
	uint8 state;
	uint8 index;
	uint16 seqNum;
	uint16 address;
	uint16 imageLen;
	uint16 imageID;
	uint32 crc;
	uint8 xferStatus;
	uint32 pXferData;
	uint32 baseAddress;
	uint32 preambleAddress;
	uint32 binSize;
	uint8 buf[OAD_DL_DATA_LEN];
} OAD_SrvInfo_t;

OAD_SrvInfo_t OAD_SrvInfo;

uint8 OAD_SrvAncestorThreadId;

static void OAD_SendCommand(uint8 index, uint8 cmd, uint8 *pData, uint8 len);
static void OAD_SendCancel(uint8 index);
static void OAD_SendStatusRequest(uint8 index);
static void OAD_StartTransfer(uint8 index, uint16 imageID, uint8 status);
static void OAD_SendStart(uint8 index, uint16 imageLen);
static void OAD_SendEnable(uint8 index, uint32 imageId, uint32 CRC);
static uint8 *OAD_GetNextDataSet(OAD_SrvInfo_t *pInfo);
static void OAD_SendData(uint8 index, uint16 seqNum, uint16 address,
		uint8 *data);
static uint32 crc32Value(uint16 inCRC);
static void OAD_XferProcessCommand(uint8 status, uint8 Command);

//extern void _ltoa(uint32 num, uint8 *buf, uint8 radix);
//extern void setEventFlag(volatile uint16 *event_flag, uint16 event);
//extern void clearEventFlag(volatile uint16 *event_flag, uint16 event);
//extern volatile uint16 apEvtFlags;
/**************************************************************************************************
 *
 * @fn          OAD_ServerInit
 *
 * @brief       Processes data indication from a remote device
 *
 * @param       srcIndex - Address of device that sent the packet
 *              pData - pointer to data to send
 *              len - Length of pData in bytes
 *
 * @return      none
 */
#if defined unix
void OAD_ServerInit(uint8 threadId, uint8 *pBuf, int imageSize)
#else
void OAD_ServerInit(uint8 threadId)
#endif // unix
{
	OAD_SrvAncestorThreadId = threadId;
	if (oadImageFd != NULL)
		printf("OAD Ready\n");
	else {
		printf("No image... --> returning\n");
		return;
	}

#if defined unix
	// Copy the file to buffer. Remember to free it!
	if (pOADimageBuffer != NULL)
	free(pOADimageBuffer);
	pOADimageBuffer = (uint8*)malloc(imageSize);
	memcpy(pOADimageBuffer, pBuf, imageSize);
#endif // unix
	printf("Push OAD button on RC ('LIVE' on black 'Yellow' on white)\n");

	OAD_SrvInfo.state = OAD_SERVER_IDLE_STATE;
	OAD_SrvInfo.imageLen = OAD_WORD_ALIGN(oadImageLen);
	OAD_SrvInfo.binSize = oadImageLen;
#if defined unix
	OAD_SrvInfo.baseAddress = 0;
#else
	OAD_SrvInfo.baseAddress = oadImageLen;
#endif // unix
	OAD_SrvInfo.preambleAddress = OAD_PREAMBLE_ADDR;

#if defined OAD_DEBUG
	preamble_t preamble;

	memcpy((uint8*) &preamble,
			(uint8*) &pOADimageBuffer[OAD_SrvInfo.preambleAddress], 20);

	printf("preambleAddress(offset)\t = %.4X\n", OAD_SrvInfo.preambleAddress);
	printf("preamble.length \t = %.8X @ %.8X (%.8X @ %.8X)\n", preamble.length, (unsigned int)&preamble.length, (unsigned int)pOADimageBuffer[OAD_SrvInfo.preambleAddress], (unsigned int)pOADimageBuffer[OAD_SrvInfo.preambleAddress]);
	printf("preamble.enable_dl \t = %.8X @ %.8X (%.8X @ %.8X)\n", preamble.enable_dl, (unsigned int)&preamble.enable_dl, ((unsigned int*)pOADimageBuffer)[(OAD_SrvInfo.preambleAddress/4) + 1], (unsigned int)&((unsigned int*)pOADimageBuffer)[(OAD_SrvInfo.preambleAddress/4) + 1]);
	printf("preamble.boot_op \t = %.8X @ %.8X (%.8X @ %.8X)\n", preamble.boot_op, (unsigned int)&preamble.boot_op, ((unsigned int*)pOADimageBuffer)[(OAD_SrvInfo.preambleAddress/4) + 2], (unsigned int)&((unsigned int*)pOADimageBuffer)[(OAD_SrvInfo.preambleAddress/4) + 2]);
	printf("preamble.magic \t \t = %.8X @ %.8X (%.8X @ %.8X)\n", preamble.magic, (unsigned int)&preamble.magic, ((unsigned int*)pOADimageBuffer)[(OAD_SrvInfo.preambleAddress/4) + 3], (unsigned int)&((unsigned int*)pOADimageBuffer)[(OAD_SrvInfo.preambleAddress/4) + 3]);
	printf("preamble.imgID \t \t = %.8X @ %.8X (%.8X @ %.8X)\n", preamble.imgID, (unsigned int)&preamble.imgID, ((unsigned int*)pOADimageBuffer)[(OAD_SrvInfo.preambleAddress/4) + 4], (unsigned int)&((unsigned int*)pOADimageBuffer)[(OAD_SrvInfo.preambleAddress/4) + 4]);
#endif //defined OAD_DEBUG
	return;
}

/**************************************************************************************************
 *
 * @fn          OAD_ProcessDataInd
 *
 * @brief       Processes data indication from a remote device
 *
 * @param       srcIndex - Address of device that sent the packet
 *              pData - pointer to data to send
 *              len - Length of pData in bytes
 *
 * @return      none
 */
void OAD_ProcessDataInd(uint8 srcIndex, uint8 *pData, uint8 len) {
	uint8 cmd;
	uint16 tempSeq;
	uint32 storageSize;
	uint32 activeID;
	preamble_t preamble;
	static uint8 TOAD_reboot;

	// Verify this is a response from the target of the OAD transfer
	if (srcIndex == OAD_SrvInfo.index) {
		if (pData[OAD_CMD_BYTE] & OAD_RSP_BIT) {
			cmd = pData[OAD_CMD_BYTE] & ~OAD_RSP_BIT;

			switch (cmd) {
			case OAD_STATUS_CMD:
				activeID = BUILD_UINT32(pData[5], pData[6], pData[7], pData[8]);
				storageSize = BUILD_UINT32(pData[13], pData[14], pData[15], 0);
				// Check if active image ID suits the downloadable image
				// This can be different as image ID is used differently from vendor to vendor.
				// In this example, 16 most significant bits are used to identify
				// compatible hardware platform.
				if (OAD_SrvInfo.binSize > storageSize)
					printf("bin is too big !!\n");

				// check if we have a TOAD RC communicating with us:
				if ((1 == pData[16]) && (TOAD_STATUS_LENGTH == len)) {
					// TOAD device found, mean it will reboot after start frame send if successful
					TOAD_reboot = TRUE;
				} else
					TOAD_reboot = FALSE;

#if defined unix
				memcpy((uint8*) &preamble,
						(uint8*) &pOADimageBuffer[OAD_SrvInfo.preambleAddress], 20);
#else
				HalXNVRead(OAD_SrvInfo.preambleAddress, (uint8*) &preamble, 20);
#endif // unix
#if defined OAD_DEBUG
				printf("preamble.imgID & 0xFFFF0000 \t = %.8X\n", preamble.imgID & 0xFFFF0000);
				printf("activeID & 0xFFFF0000 \t \t = %.8X\n", activeID & 0xFFFF0000);
#endif //defined OAD_DEBUG
				if ( ((preamble.imgID & 0xFFFF0000) == (activeID & 0xFFFF0000))
					|| ( (activeID & 0xFFFF0000) == 0xFFFF0000) )
				{
					// Start downloading
					printf("Poll received\n");
					if ((activeID & 0xFFFF0000) == 0xFFFF0000)
					{
						// The received activeID indicates that the TOAD may have erased the first page
						// and reset before writing the preamble. This could be caused by unfortunate,
						// and very unlikely powercycling at the small window where it is possible. It
						// could also be that the server broke down before sending the preamble bytes,
						// and therefore extended the vulnerable window.
						printf("Image ID with high bytes 0xFFFF. Download probably aborted prematurely before this attempt.\n");
					}
					// Status byte is the state of the OAD Client
					if (pData[OAD_STATUS_BYTE] == OAD_CLIENT_DL_STATE)
					{
						// CRC may fail if a reset caused this; hence allow CRC to catch up.
						OAD_SrvInfo.seqNum = 0xffff;
						uint16 requestedBlock = (BUILD_UINT16(pData[OAD_STATUS_BYTE + 1], pData[OAD_STATUS_BYTE + 2]));
						// Verify that the requestedBlock is within the boundary
						if ((requestedBlock * OAD_WORD_ALIGN(OAD_DL_DATA_LEN)) > OAD_SrvInfo.imageLen)
						{
							// If requestedBlock == 0xFFFF it means that the client is requesting the first packet,
							// This is not necessarily an erroneous case.
							if (requestedBlock != 0xFFFF)
								printf("ERR: Requested block out of bounds. Block: 0x%.4X out of 0x%.4X\n",
										requestedBlock,
										OAD_SrvInfo.imageLen/OAD_WORD_ALIGN(OAD_DL_DATA_LEN));
							// Start over from beginning.
							printf("Starting from the beginning\n");
						}
						else
						{
							for (tempSeq = 0; tempSeq <= requestedBlock; tempSeq++)
							{
								OAD_GetNextDataSet((OAD_SrvInfo_t*) &OAD_SrvInfo);
							}
						}
						// The previous loop has set seqNum to the next expected packet
					}
					OAD_StartTransfer(srcIndex, 0, pData[OAD_STATUS_BYTE]);
					if (TRUE == TOAD_reboot)
					{
						printf("TOAD Found ! Controller will now reboot...\n");
						OAD_SrvInfo.state = OAD_SERVER_IDLE_STATE;
					}
				}
				else
				{
					printf("OAD Incompatible\n");
				}
				break;

			case OAD_START_CMD:
				if (OAD_SrvInfo.state == OAD_SERVER_XFER_STATE) {
					OAD_XferProcessCommand(pData[OAD_STATUS_BYTE],
							OAD_DATA_CMD);
				} else {
					//start failed
					printf("Start Failed\n");
					if (OAD_IMAGE_TOO_BIG == pData[OAD_STATUS_BYTE])
						printf("bin too big\n");

					timer_start_timerEx(OAD_SrvAncestorThreadId,
							OAD_APP_EVT_OAD_KO, 3000);
					TOAD_reboot = FALSE;
				}
				break;

			case OAD_DATA_CMD:
				if (OAD_SrvInfo.state == OAD_SERVER_XFER_STATE) {
					// Currently the client will only report a failure if the sequence
					// number does not match the expected value.  This should not
					// happen.  We do not correct for this.  Retransmit max times and
					// timeout.
					if ((pData[OAD_STATUS_BYTE] == OAD_SUCCESS
							|| pData[OAD_STATUS_BYTE] == OAD_DL_RESEND_DATA)
							&& ((uint16) pData[OAD_STATUS_BYTE + 1]
									| ((uint16) pData[OAD_STATUS_BYTE + 2] << 8))
									== OAD_SrvInfo.seqNum) {
#if defined unix
						int numOfBlocks = OAD_BYTE_ALIGN(OAD_SrvInfo.imageLen)/(OAD_DL_DATA_LEN);
						double percentDone = ((double)(OAD_SrvInfo.seqNum + 2)/numOfBlocks)*100;
						printf("OAD progress %4.2f\%%(%d/%d)", percentDone, OAD_SrvInfo.seqNum + 1, (numOfBlocks - 1));
						if ( (OAD_SrvInfo.seqNum + 1) == (numOfBlocks - 1) )
						{
							printf("\n");
						}
						else
						{
							printf("\r");
						}
#endif //defined unix
						OAD_XferProcessCommand(pData[OAD_STATUS_BYTE],
								OAD_DATA_CMD);
					}
				} else if (OAD_SrvInfo.state == OAD_SERVER_ENABLE_STATE)
					OAD_XferProcessCommand(pData[OAD_STATUS_BYTE],
							OAD_ENABLE_CMD);

				break;

			case OAD_ENABLE_CMD:
				if (OAD_SrvInfo.state == OAD_SERVER_IDLE_STATE) {
					TOAD_reboot = FALSE;
					// Disable timeout timer
					timer_start_timerEx(OAD_SrvAncestorThreadId,
							OAD_APP_EVT_OAD_TIMEOUT, 0);
					switch (pData[OAD_STATUS_BYTE]) {
					case OAD_SUCCESS:
						// Set event to notify application
						timer_set_event(OAD_SrvAncestorThreadId,
								OAD_APP_EVT_OAD_OK);
						break;
					case OAD_DL_NOT_COMPLETE:
						printf("OAD incomplete\n");
						timer_set_event(OAD_SrvAncestorThreadId,
								OAD_APP_EVT_OAD_KO);
						break;
					case OAD_BAD_CRC:
						printf("OAD bad CRC\n");
						timer_set_event(OAD_SrvAncestorThreadId,
								OAD_APP_EVT_OAD_KO);
						break;
					default:
						printf("OAD failed\n");
						timer_set_event(OAD_SrvAncestorThreadId,
								OAD_APP_EVT_OAD_KO);
						break;
					}
				}
				break;

			default:
				break;
			}
		}
	}
}

/**************************************************************************************************
 *
 * @fn          OAD_SendCommand
 *
 * @brief       Sends a command to a remote device
 *
 * @param       index - Address of device to send cmd to
 *              cmd - ID of command to be sent
 *              pData - pointer to data to send
 *              len - Length of pData in bytes
 *
 * @return      none
 */
static void OAD_SendCommand(uint8 index, uint8 cmd, uint8 *pData, uint8 len) {
	// Set protocol identifier
	pData[OAD_PROTOCOL_BYTE] = RTI_PROTOCOL_OAD;

	// Set the command
	pData[OAD_CMD_BYTE] = cmd;

	// Send the command
	RTI_SendDataReq(index, OAD_PROFILE_ID, OAD_VENDOR_ID, OAD_TX_OPTIONS, len,
			pData);
}

/**************************************************************************************************
 *
 * @fn          OAD_SendStatusRequest
 *
 * @brief       Sends a status request command message
 *
 * @param       index - Address of device to send cmd to
 *
 * @return      none
 */
static void OAD_SendStatusRequest(uint8 index) {
	uint8 buf[OAD_STATUS_REQ_LEN];

	// Set index so that it is verified with the response
	OAD_SrvInfo.index = index;

	// Send the command
	OAD_SendCommand(index, OAD_STATUS_CMD, buf, OAD_STATUS_REQ_LEN);
}

/**************************************************************************************************
 *
 * @fn          OAD_SendStart
 *
 * @brief       Sends a start download command message
 *
 * @param       index - Address of device to send cmd to
 *              imageLen - length of image in number of 32 bit words
 *
 * @return      none
 */
static void OAD_SendStart(uint8 index, uint16 imageLen) {
	uint8 buf[OAD_START_REQ_LEN];
	uint8 *pMsg = buf + 2;

	// Insert the image length
	*pMsg++ = LO_UINT16(imageLen);
	*pMsg++ = HI_UINT16(imageLen);

	// Send the command
	OAD_SendCommand(index, OAD_START_CMD, buf, OAD_START_REQ_LEN);
}

/**************************************************************************************************
 *
 * @fn          OAD_SendData
 *
 * @brief       Sends a data packet
 *
 * @param       index - Address of device to send cmd to
 *              seqNum - sequence of data packet
 *              address - address of data packet
 *              data - pointer to data
 *
 * @return      none
 */
static void OAD_SendData(uint8 index, uint16 seqNum, uint16 address,
		uint8 *data) {
	uint8 buf[OAD_DATA_REQ_LEN];
	uint8 *pMsg = buf + 2;
	uint8 i;

	// Insert the sequence number
	*pMsg++ = LO_UINT16(seqNum);
	*pMsg++ = HI_UINT16(seqNum);

	// Insert the address
	*pMsg++ = LO_UINT16(address);
	*pMsg++ = HI_UINT16(address);

	// Insert the data
	for (i = 0; i < OAD_DL_DATA_LEN; i++)
		*pMsg++ = data[i];

	// Send the command
	OAD_SendCommand(index, OAD_DATA_CMD, buf, OAD_DATA_REQ_LEN);
}

/**************************************************************************************************
 *
 * @fn          OAD_SendEnable
 *
 * @brief       Sends an enable command message
 *
 * @param       index - Address of device to send cmd to
 *              imageId - ID of image to enable
 *              CRC - CRC of image to enable
 *
 * @return      none
 */
static void OAD_SendEnable(uint8 index, uint32 imageId, uint32 CRC) {
	uint8 buf[OAD_ENABLE_REQ_LEN];
	uint8 *pMsg = buf + 2;

	// Insert the image length
	*pMsg++ = BREAK_UINT32(imageId, 0);
	*pMsg++ = BREAK_UINT32(imageId, 1);
	*pMsg++ = BREAK_UINT32(imageId, 2);
	*pMsg++ = BREAK_UINT32(imageId, 3);

	// Insert the image CRC
	*pMsg++ = BREAK_UINT32(CRC, 0);
	*pMsg++ = BREAK_UINT32(CRC, 1);
	*pMsg++ = BREAK_UINT32(CRC, 2);
	*pMsg++ = BREAK_UINT32(CRC, 3);

	// Send the command
	OAD_SendCommand(index, OAD_ENABLE_CMD, buf, OAD_ENABLE_REQ_LEN);
}

/**************************************************************************************************
 *
 * @fn          OAD_SendCancel
 *
 * @brief       Sends an cancel download command message
 *
 * @param       index - Address of device to cancel cmd to
 *
 * @return      none
 */
static void OAD_SendCancel(uint8 index) {
	uint8 buf[OAD_CANCEL_REQ_LEN];

	// Send the command
	OAD_SendCommand(index, OAD_CANCEL_CMD, buf, OAD_ENABLE_REQ_LEN);
}

/**************************************************************************************************
 *
 * @fn          OAD_GetNextDataSet
 *
 * @brief       Returns a pointer to the next set of data to transfer to the client.
 *
 * @param       pInfo - Pointer to the OAD Server Info for this transfer
 *              dataPacket - array of bytes OAD_DL_DATA_LEN long to hold next data packet
 *
 * @return      none
 */
static uint8 *OAD_GetNextDataSet(OAD_SrvInfo_t *pInfo) {
	uint8 length = 0;
	// Increment the sequence number
	pInfo->seqNum++;

	// Update address to match sequence number
	pInfo->address = pInfo->seqNum * OAD_WORD_ALIGN(OAD_DL_DATA_LEN);

	// Update address to match sequence number
	pInfo->pXferData = OAD_SrvInfo.baseAddress
			+ (pInfo->seqNum * OAD_DL_DATA_LEN);

	// Increment the address
	if (pInfo->seqNum == 0) {
		pInfo->crc = 0xFFFFFFFF;
	} else if (pInfo->seqNum % 10 == 0) {
		asm("NOP");
	}

	// Read next bytes from the Flash
#if defined unix
	memcpy(pInfo->buf,
			(uint8*) &pOADimageBuffer[pInfo->pXferData], OAD_DL_DATA_LEN);
#if defined OAD_DEBUG
	printf("pInfo->seqNum \t \t= %.4X (%d) \n", pInfo->seqNum, pInfo->seqNum);
	printf("pInfo->pXferData \t= %.8X (%d) \n", pInfo->pXferData, pInfo->pXferData);
	printf("pInfo->buf \t \t= %.8X @ %.8X\n", ((unsigned int*)pInfo->buf)[0], (unsigned int)pInfo->buf);
	printf("pOADimageBuffer \t= %.8X @ %.8X\n", pOADimageBuffer[pInfo->pXferData], (unsigned int)&pOADimageBuffer[pInfo->pXferData]);
#endif //defined OAD_DEBUG
#else
	HalXNVRead(pInfo->pXferData, pInfo->buf, OAD_DL_DATA_LEN);
#endif // unix
	//add length in the preamble
	if (pInfo->address == OAD_WORD_ALIGN(OAD_PREAMBLE_OFFSET)) {
		uint32 preambleLen = OAD_BYTE_ALIGN(pInfo->imageLen);

		pInfo->buf[0] = BREAK_UINT32(preambleLen, 0);
		pInfo->buf[1] = BREAK_UINT32(preambleLen, 1);
		pInfo->buf[2] = BREAK_UINT32(preambleLen, 2);
		pInfo->buf[3] = BREAK_UINT32(preambleLen, 3);

#if defined OAD_DEBUG
		printf("pInfo->seqNum \t \t= %.4X (%d) \n", pInfo->seqNum,
				pInfo->seqNum);
		printf("pInfo->pXferData \t= %.8X (%d) \n", pInfo->pXferData,
				pInfo->pXferData);
		printf("pInfo->buf \t \t= %.8X @ %.8X\n",
				((unsigned int*) pInfo->buf)[0], (unsigned int) pInfo->buf);
		printf("pOADimageBuffer \t= %.8X @ %.8X\n",
				pOADimageBuffer[pInfo->pXferData],
				(unsigned int) &pOADimageBuffer[pInfo->pXferData]);
#endif //defined OAD_DEBUG

	}

	//Update CRC
	// Calculate the CRC over the data buffer
	{
		uint32 temp1, temp2, crc;
		uint8 i;

		//check length,
		if ((OAD_BYTE_ALIGN(pInfo->imageLen)
				- (OAD_DL_DATA_LEN * (pInfo->seqNum))) < OAD_DL_DATA_LEN)
			length = (OAD_BYTE_ALIGN(pInfo->imageLen)
					- (OAD_DL_DATA_LEN * (pInfo->seqNum)));
		else
			length = OAD_DL_DATA_LEN;

		crc = pInfo->crc;

		for (i = 0; i < length; i++) {
			temp1 = (crc >> 8) & 0x00FFFFFFL;
			temp2 = crc32Value(((uint16) crc ^ pInfo->buf[i]) & 0xFF);
			crc = temp1 ^ temp2;
		}

		pInfo->crc = crc;
		// XOR CRC with all bits on
		//This Need To be done at the end    pInfo->crc = crc ^ 0xFFFFFFFF;

	}

	return pInfo->buf;
}

/**************************************************************************************************
 *
 * @fn          OAD_XferProcessCommand
 *
 * @brief       Function that manages the transfer of an image to a target device
 *
 * @param       pVoid - Pointer to the OAD Server Info for this transfer
 *
 * @return      none
 */

static void OAD_XferProcessCommand(uint8 status, uint8 Command) {
	OAD_SrvInfo_t *pInfo = (OAD_SrvInfo_t*) &OAD_SrvInfo;
	uint8 *pDataSet;

	if (status == OAD_SUCCESS) {
		//following test is to not change pInfo->state when start command is send.
		switch (Command) {
		case OAD_START_CMD:
			// Send start command to target with image length
			printf("Sending Start...\n");
			OAD_SendStart(pInfo->index, pInfo->imageLen);
			timer_start_timerEx(OAD_SrvAncestorThreadId,
					OAD_APP_EVT_OAD_TIMEOUT, 3000);

			//if status success, Next State is Data transfert
			pInfo->state = OAD_SERVER_XFER_STATE;
			break;

		case OAD_DATA_CMD:
			// Get next data packet
			pDataSet = OAD_GetNextDataSet(pInfo);
			// Send the data
			OAD_SendData(pInfo->index, pInfo->seqNum, pInfo->address, pDataSet);
			timer_start_timerEx(OAD_SrvAncestorThreadId,
					OAD_APP_EVT_OAD_TIMEOUT, 3000);
			//Display Info
			//          if ((pInfo->seqNum % 400) == 0)
			//          {
			//            char String[16]="data Block: ";
			//            HalLcdDisplayPercentBar("bin uploading...", (100*((uint32)pInfo->seqNum))/(pInfo->imageLen/OAD_WORD_ALIGN(OAD_DL_DATA_LEN)), HAL_LCD_LINE_4);
			//            HalLcdWriteStringValue(String, pInfo->seqNum, 10, HAL_LCD_LINE_6);
			//          }
			// See if this is the last transfer
			if (pInfo->address + OAD_WORD_ALIGN(OAD_DL_DATA_LEN)
					>= pInfo->imageLen)
				//if status success, Next State is Enable req
				pInfo->state = OAD_SERVER_ENABLE_STATE;
			break;

		case OAD_ENABLE_CMD:
			// Send a enable to target with image crc
			printf("Sending Enable... Allow client to verify the image\n");
			pInfo->crc ^= 0xFFFFFFFF;
			OAD_SendEnable(pInfo->index, pInfo->imageID, pInfo->crc);
			timer_start_timerEx(OAD_SrvAncestorThreadId,
					OAD_APP_EVT_OAD_TIMEOUT, 20000);

			// Indicate we are waiting for a response
			//if status success, go back to idle
			pInfo->state = OAD_SERVER_IDLE_STATE;
			break;

		default:
			break;
		}
	} else {
		//Previous command Failed...
		printf("Prev. Cmd failed...\n");
		OAD_SrvInfo.state = OAD_SERVER_IDLE_STATE;
		OAD_SrvInfo.imageLen = OAD_WORD_ALIGN(OAD_SrvInfo.binSize);
		timer_set_event(OAD_SrvAncestorThreadId, OAD_APP_EVT_OAD_KO);
	}

}
/**************************************************************************************************
 *
 * @fn          OAD_StartTransfer
 *
 * @brief       Called to start an image transfer to a remote device
 *
 * @param       index - Address of device to send cmd to
 *              pFilename - name and path of file to send
 *
 * @return      none
 */
static void OAD_StartTransfer(uint8 index, uint16 imageID, uint8 status) {
	if (OAD_SrvInfo.state == OAD_SERVER_IDLE_STATE) {
		// Status byte is the state of the OAD Client
		if (status == OAD_CLIENT_DL_STATE) {
			// Set status to success
			status = OAD_SUCCESS;
			// Move to the transfer state
			OAD_SrvInfo.index = index;
			OAD_SrvInfo.state = OAD_SERVER_XFER_STATE;
			OAD_SrvInfo.imageID = imageID;
			// Already increased to expected packet

			OAD_XferProcessCommand(status, OAD_DATA_CMD);
		} else {
			// Move to the start state
			OAD_SrvInfo.index = index;
			OAD_SrvInfo.state = OAD_SERVER_START_STATE;
			OAD_SrvInfo.imageID = imageID;
			OAD_SrvInfo.seqNum = 0xffff;

			OAD_XferProcessCommand(status, OAD_START_CMD);
		}
	} else {
		printf("ERR: OAD already started...\n");
		OAD_SrvInfo.state = OAD_SERVER_IDLE_STATE;
		OAD_SrvInfo.imageLen = OAD_WORD_ALIGN(OAD_SrvInfo.binSize);
		// Disable timeout timer, could happen in case of TOAD
		timer_start_timerEx(OAD_SrvAncestorThreadId, OAD_APP_EVT_OAD_KO, 0);
		timer_set_event(OAD_SrvAncestorThreadId, OAD_APP_EVT_OAD_KO);
	}

}

/**************************************************************************************************
 *
 * @fn          OAD_HandlePoll
 *
 * @brief       Handles a received poll command
 *
 * @param       none
 *
 * @return      none
 */
void OAD_HandlePoll(uint8 destIdx) {
	if ( (destIdx == OAD_SrvInfo.index) &&
			(OAD_SrvInfo.state != OAD_SERVER_IDLE_STATE) )
	{
		// Since the Poll came from the Client we are serving at
		// the moment it is best practice to reset.
		OAD_SrvInfo.state = OAD_SERVER_IDLE_STATE;
		printf("Poll received from Client who is currently being served, this caused a state change in the OAD\n");
	}
	if (OAD_SrvInfo.state == OAD_SERVER_IDLE_STATE) {
		if ((BinFile1) || (BinFile2))
			OAD_SendStatusRequest(destIdx);
		else
			OAD_SendCancel(destIdx);
	}
	else
	{
		printf("Poll received from Client who is not currently being served\n");
	}
}

/*********************************************************************
 * @fn      CheckCode
 *
 * @brief   Do all code sanity checks. 
 *
 * @param   imgCRC - Expected CRC
 *          imageLen - Size on bytes of download image
 *          imageType - Used when FEATURE_OAD_HOST is defined to allow a
 *                      binary image with no preamble magic.
 *
 * @return  OAD_SUCCESS, OAD_NO_RESOURCES or OAD_BAD_CRC
 */
uint8 OAD_CheckCode(uint32 imgCRC, uint32 imageLen, uint8 imageType) {
#define OAD_CRC_BUF_LEN 64
	preamble_t preamble;
	// start with all bits on. common practice to guard against initial sequence of
	// 0's (even though we don't have this case)
	uint32 ulCRC = 0xFFFFFFFF;
	uint32 ulTemp1, ulTemp2;
	uint32 baseAddr = OAD_SrvInfo.baseAddress;
	uint8 pCrcBuf[OAD_CRC_BUF_LEN]; // Don't need dynamic memory due to small call stack.
	(void) imageType; // Only used by Host to allow a binary type in 2-step OAD.
	uint8 buf[16];
	uint32 preambleLen = imageLen;

	if (pCrcBuf == NULL)
		return OAD_NO_RESOURCES;

	// Read next bytes from the Flash
#if defined unix
	memcpy((uint8*) &preamble,
			(uint8*) &pOADimageBuffer[OAD_PREAMBLE_OFFSET],
			sizeof(preamble_t));
#else
	HalXNVRead(OAD_SrvInfo.baseAddress + OAD_PREAMBLE_OFFSET,
			(uint8*) &preamble, sizeof(preamble_t));
#endif // unix
	if (preamble.magic == OAD_MAGIC) {
		while (imageLen) {
			if (baseAddr % OAD_CRC_BUF_LEN == 0) {
#if defined unix
				memcpy(pCrcBuf,
						(uint8*) &pOADimageBuffer[baseAddr],
						OAD_CRC_BUF_LEN);
#else
				HalXNVRead(baseAddr, pCrcBuf, OAD_CRC_BUF_LEN);
#endif // unix
			}

			if ((baseAddr - OAD_SrvInfo.baseAddress) == OAD_PREAMBLE_OFFSET) {

				pCrcBuf[0] = BREAK_UINT32(preambleLen, 0);
				pCrcBuf[1] = BREAK_UINT32(preambleLen, 1);
				pCrcBuf[2] = BREAK_UINT32(preambleLen, 2);
				pCrcBuf[3] = BREAK_UINT32(preambleLen, 3);
			}

			ulTemp1 = (ulCRC >> 8) & 0x00FFFFFFL;
			ulTemp2 = crc32Value(
					((uint16) ulCRC ^ pCrcBuf[baseAddr % OAD_CRC_BUF_LEN])
							& 0xFF);
			ulCRC = ulTemp1 ^ ulTemp2;

			baseAddr++;
			imageLen--;
		}

		ulCRC ^= 0xFFFFFFFF; // XOR with all bits on. this is a common practice
	}

	buf[0] = 'C';
	buf[1] = 'R';
	buf[2] = 'C';
	buf[3] = ':';
	buf[4] = '0';
	buf[5] = 'x';
#if defined unix
	sprintf((char*)&buf[6], "%.2X",(unsigned int)ulCRC);
#else
	_ltoa(ulCRC, &buf[6], 16);
#endif // defined unix
	printf("%s\n", buf);

	return (ulCRC == imgCRC) ? OAD_SUCCESS : OAD_BAD_CRC;
}

/*********************************************************************
 * @fn      crc32Value
 *
 * @brief   Calculate CRC value for 16-bit word. 
 *
 * @param   inCRC - input
 *
 * @return  CRC value
 */
#define CRC32_POLYNOMIAL      ((uint32)0xEDB88320)  // reversed version or 802.3 polynomial 0x04C11DB7
static uint32 crc32Value(uint16 inCRC) {
	int16 j;
	uint32 ulCRC = inCRC;

	// for this byte (inCRC)...
	for (j = 8; j; j--) {
		// lsb on? yes -- shift right and XOR with poly. else just shift
		if (ulCRC & 1)
			ulCRC = (ulCRC >> 1) ^ CRC32_POLYNOMIAL;
		else
			ulCRC >>= 1;
	}

	return ulCRC;
}

