#ifndef _PHY_H
#define _PHY_H

#include "typedefs.h"
#include "g3_phy.h"
/******************************************************************************
* FILE PURPOSE: Public header file module PLC PHY
*******************************************************************************
*
* FILE NAME: phy.h
*
* DESCRIPTION:
*       This file contains public definitions, data structures and function
*       prototypes for PLC PHY.
*
*  Copyright (c) 2009 Texas Instruments Inc.
*  All Rights Reserved This program is the confidential and proprietary
*  product of Texas Instruments Inc.  Any Unauthorized use, reproduction or
*  transfer of this program is strictly prohibited.
*
* HISTORY:
*
* 02/27/2009 S.Yim    Written
* 11/30/2009 S.Yim    Updated PHY API
*
*
* LIST OF FUNCTION PROTOTYPES:
*
******************************************************************************/

/*****************************************************************************/
/* Definitions                                                               */
/*****************************************************************************/
#define PHY_MAX_CB_PARMS     8

/* PHY status */
typedef enum
{
  PHY_STAT_SUCCESS                = 0,
  PHY_STAT_FAILURE                = 1,
  PHY_STAT_PREAMBLE_NOT_DETECTED  = 2,
  PHY_STAT_HEADER_CRC_FAILED      = 3,
  PHY_STAT_PAYLOAD_CRC_FAILED     = 4,
  PHY_STAT_ILLEGAL_PARAMETERS     = 5,
  PHY_STAT_ILLEGAL_OPERATIONS     = 6,
  PHY_STAT_UNKNOWN_ID             = 7,
  PHY_STAT_TX_LATE                = 8,
  PHY_STAT_INVALID_LEN            = 9,
  PHY_STAT_INVALID_SCH            = 10,
  PHY_STAT_INVALID_LEV            = 11,
  PHY_STAT_BUF_OVRUN              = 12,
  PHY_STAT_BUSY                   = 13,
  PHY_STAT_CMD_IN_PLACE           = 14,
  PHY_STAT_NOT_IN_SYNC            = 15,
  PHY_STAT_RX_BUF_OVERRUN         = 16,
  PHY_STAT_SYNCP_NOT_READY        = 17
}PHY_status_t;

/* PHY callback events */
typedef enum
{
  PHY_EV_RX_START_DONE        = 0,
  PHY_EV_TX_PPDU_DONE         = 1,
  PHY_EV_RX_PPDU_DONE         = 2,
  PHY_EV_TX_SUSPEND_DONE      = 3,
  PHY_EV_RX_SUSPEND_DONE      = 4,
  PHY_EV_TX_TESTMODE_DONE     = 5,
  PHY_EV_TX_ACK_DONE          = 6,
  PHY_EV_RX_ACK_DONE          = 7,
  PHY_EV_RX_BIT_START         = 8,
  PHY_EV_TX_PPDU_START        = 9,
  PHY_EV_RX_FCH_DONE          = 10,
  PHY_EV_RX_PKT_RCV_DONE      = 11,
  PHY_EV_TX_ERROR             = 12

}PHY_ev_t;

/* PHY band mode */
typedef enum
{
  PHY_MODE_CEN_A                  = 0,  
  PHY_MODE_CEN_B                  = 1,
  PHY_MODE_CEN_BC                 = 2,
  PHY_MODE_CEN_BCD                = 3,
  PHY_MODE_CEN_ABC                = 4,   // combined ABC band mode
  PHY_MODE_FCC                    = 5,   // FCC band must be greater than Cenelec band
  PHY_MODE_FCC_SHORT_FCH          = 6,   // Shortened FCH
  PHY_NUM_MODES                   = 7
}PHY_band_mode_t;


/*****************************************************************************/
/* Data Structures                                                           */
/*****************************************************************************/
/* Tx PPDU done callback data structure */ 
typedef struct
{
  UINT32       ppduAddr;          // Tx PPDU address passed by caller
  UINT32       timeStamp;         // timestamp in 10us

}PHY_cbTxPpdu_t;

/* Rx channel acquired callback data structure */ 
typedef struct
{
  SINT16       rssi;              // rssi in dBm
  UINT16       nSymbols;          // number of symbols for ppdu

}PHY_cbRxSync_t;

/* Rx PPDU callback data structure */ 
typedef struct
{
  UINT32       ppduInfoAddr;       // Rx PPDU info address (see PHY_rxPpdu_t in phy_rx.h)

}PHY_cbRxPpdu_t;

/* Rx ACK frame callback data structure */ 
typedef struct
{
   UINT16       ack;      // 1 = ACK, 0 = NAK
   UINT16       fcs;      // received FCS (signature for the ACK/NAK frame)

}PHY_cbRxAck_t;

/* Rx FCH callback data structure */ 
typedef struct
{
   UINT16       num_symbols;  // 695us* num_symbols is the duration of the packet payload
   UINT16       dt;           // delimiter type
   UINT32       time;         // time stamp of the last symbol of FCH
}PHY_cbRxFch_t;

/* Rx PPDU timing callback data structure */ 
typedef struct
{
   UINT32       time;         // time stamp of the last symbol of Data packet
}PHY_cbRxPpduTiming_t;

/* Tx ACK frame callback data structure */ 
typedef struct
{
  //UINT32       ackFrmAddr;       // Tx ACK frame address 
  UINT32       timeStamp;         // timestamp in 10us
}PHY_cbTxAck_t;

/* Tx PPDU prepare stage */ 
typedef struct
{
  UINT16       stage;        
}PHY_cbTxStage_t;

/* PHY callbacks data structure */
typedef struct
{
  PHY_status_t         status;     // callback status  
  union
  {
    PHY_cbTxPpdu_t     txPpdu;     // PHY Tx ppdu done callback
    PHY_cbRxSync_t     rxSync;     // PHY Rx channel acq callback
    PHY_cbRxPpdu_t     rxPpdu;     // PHY Rx ppdu callback
	PHY_cbRxPpduTiming_t   rxPpduTiming; // PHY Rx ppdu timing
    PHY_cbRxFch_t      rxFch;      // PHY Rx FCH information
    PHY_cbTxAck_t      txAck;      // PHY Tx ACK frame done callback (G3 only)
    PHY_cbRxAck_t      rxAck;      // PHY RX ACK frame callback (G3 only)
	PHY_cbTxStage_t    txStage;    // PHY TX encoding stage (G3 only)
  }cbParms;

}PHY_cbData_t;

/* Callback functions */
typedef void (*PHY_cbFunc_t)(PHY_ev_t eventID, PHY_cbData_t *cbData_p);


/* PHY band information data structure */ 
typedef struct
{
  UINT16   default_first_tone_idx; // default first tone index in a band configuration
  UINT16   num_subbands;      // number of subbands in tone map
  UINT16   num_tones_in_sb;   // number of tones in a subband
  UINT16   num_fch_ibits;     // number of FCH information bits
  UINT16   num_fch_payloadbits; // number of FCH payload bits (with CRC)
  UINT16   num_fch_ubits;     // number of FCH total used bits (with padding)
  UINT16   num_fch_cbits;     // number of FCH coded bits (2x fch_ubits)
  UINT16   num_nve_bf_tones;  // number of NVE frequency average tones
  UINT16   num_prmb_symbs;    // number of preamble symbols
}PHY_band_params_t;

typedef struct
{
    UINT16 band;
    UINT16 num_tones;
    UINT16 first_tone_idx;
    UINT16 last_tone_idx;
    UINT16 num_used_sb;       // number of used subbands
    UINT16   num_nve_bf_groups; // number of NVE frequency average groups
    UINT16   num_fch_symbs;    // number of FCH symbols
	UINT16   coh_pilot_offset;  // coherent modulation first pilot offset
    PHY_band_params_t mode;
}PHY_handle_t;


/*****************************************************************************/
/* External variables                                                        */
/*****************************************************************************/
extern PHY_handle_t  PHY_handle_s;
extern ComplexShortVCU  PHY_syncp_freq[G3_PHY_MAX_NUM_TONES_IN_SYMB];
extern ComplexShortVCU  PHY_rx_syncp_freq[G3_PHY_MAX_NUM_TONES_IN_SYMB];

/*****************************************************************************/
/* Function Prototypes                                                       */
/*****************************************************************************/
const char *PHY_getLibVersion(void);
void PHY_init(void);

#endif // _PHY_H






