/*****************************************************************************
* FILE PURPOSE: Header file for PHY TX
*******************************************************************************
*
* FILE NAME: phy_tx.h
*
* DESCRIPTION:
*       Header file for PHY TX.
*
*  Copyright (c) 2009 Texas Instruments Inc.
*  All Rights Reserved This program is the confidential and proprietary
*  product of Texas Instruments Inc.  Any Unauthorized use, reproduction or
*  transfer of this program is strictly prohibited.
*
* HISTORY:
*
* 04/21/2009 fum    Written
* 11/30/2009 S.Yim  Updated Get/Set/CB API with data structure
*
******************************************************************************/
#ifndef _PHY_TX_H
#define _PHY_TX_H

#include <typedefs.h>
#include <phy.h>

/******************************************************************************
 * Definitions
 *****************************************************************************/
typedef enum
{
  PHY_TX_SET_TONEMASK       = 0,        // Set Tonemask
  PHY_TX_SET_BLKILV         = 1,        // Set Block Interleaver
  PHY_TX_SET_COH            = 2,        // Set Coherent modulation
  PHY_TX_SET_TXZC           = 3,        // Set packet transmission with zero crossing
  PHY_TX_SET_BANDSEL        = 4,        // Set band selection (Cenelec or FCC)
  PHY_TX_SET_TESTMOD        = 5,        // Set test mode Modulation
  PHY_TX_SET_TESTLVL        = 6,        // Set test mode transmission level
  PHY_TX_SETTUNING          = 7,        // Set tuning

  NUM_PHY_TX_SETPARMS       = 8        

}PHY_tx_setType_t;

typedef enum
{
  PHY_TX_GET_TONEMASK       = 0,        // Get Tonemask
  PHY_TX_GET_BLKILV         = 1,        // Get Block Interleaver
  PHY_TX_GET_ZCT            = 2,        // Get zero-crossing time (not supported yet)
  PHY_TX_GET_TIME           = 3,
  PHY_TX_GET_NUMINFOTONES   = 4,        // get number of information tones based on tone mask and tone map
  PHY_TX_GET_NUMFCHSYMBS    = 5,        // get PHY FCH symbols
  PHY_TX_GET_COH            = 6,        // get coherent modulation 
  PHY_TX_GET_TXZC           = 7,        // get packet transmission with zero crossing
  PHY_TX_GET_BANDSEL        = 8,        // get band selection
  PHY_TX_GET_TESTMOD        = 9,        // get test mode Modulation
  PHY_TX_GET_TESTLVL        = 10,       // get test mode transmission level
  PHY_TX_GETTUNING          = 11,       // Get tuning

  NUM_PHY_TX_GETPARMS       = 12  

}PHY_tx_getType_t;

/******************************************************************************
 * Data Structures
 *****************************************************************************/
/* TX PPDU parameter structure */
typedef struct
{
  UINT16              *ppdu_p;       // Pointer to PPDU payload
  UINT16              length;           // Length of PPDU payload in bytes
  UINT16              level;            // TX level, 0x00-0x20
  UINT16              mcs;              // TX modulation & coding scheme
  UINT32              toneMap;          // Tone Map b31-b0
  UINT16              txGain[2];        // TX Gain
                                        // word 0: b0-3:   number of gain step for 20-30 kHz
                                        //         b4-7:   number of gain step for 10-20 kHz
                                        //         b8-11:  number of gain step for 40-50 kHz
                                        //         b12-15: number of gain step for 30-40 kHz
                                        // word 1: b0-3:   number of gain step for 60-70 kHz
                                        //         b4-7:   number of gain step for 50-60 kHz
                                        //         b8-11:  number of gain step for 80-90 kHz
                                        //         b12-15: number of gain step for 70-80 kHz

  UINT16              dt;               // delimiter type
                                        // 0 = start of frame with no response expected  
                                        // 1 = start of frame with response expected  
  UINT16              rpt;              // 0 - normal TX; 1 - re-transmit
  UINT32              txTime;           // TX time (in 10us)
}PHY_tx_ppdu_t;

typedef struct
{
  UINT16              ack;              // 0-nack; 1-ack
  UINT16              fcs;              // fcs
  UINT16              level;            // TX level
}PHY_tx_ack_t;

typedef struct
{
  UINT32              phyTxPpduCnt;        // number of phy tx packets
}PHY_tx_stat_t;

/* Set data structure */
typedef struct
{
  UINT16 nTones_start;
  UINT16 toneMask[G3_PHY_TONEMASK_WORDS]; // leave this as 6 for now even though it starts from second word
}PHY_txToneMask_t;

typedef union
{
//  UINT16       toneMask[G3_PHY_TONEMASK_WORDS];
  PHY_txToneMask_t toneMask;
  UINT16       blkIlv;
  UINT16       cohMod;
  UINT16       zc;
  UINT16       bandsel;
  UINT16       testMod;
  UINT16       testLvl;
  SINT16       tuningLevel;      // -60 (-6dB) to 30 (3dB) in 0.1dB resolution
}PHY_txSetData_t;

typedef struct
{
  UINT32   lastZcaTime;    // last zero crossing time in 10us (20-bits)
  UINT16   zcaPeriod;      // half-cycle period
  UINT32   lastZcbTime;    // last zero crossing time in 10us (20-bits)
}PHY_zcTime_t;

/* Get data structure */
typedef union
{
  PHY_zcTime_t zcTime;
//  UINT16       toneMask[G3_PHY_TONEMASK_WORDS];
  PHY_txToneMask_t toneMask;
  UINT16       blkIlv;
  UINT16       cohMod;
  UINT32       currTime;
  UINT32       numInfoTones;
  UINT16       numFCHSymbs;
  UINT16       zc;
  UINT16       bandsel;
  UINT16       testMod;
  UINT16       testLvl;
  SINT16       tuningLevel;      // -60 (-6dB) to 30 (3dB) in 0.1dB resolution
}PHY_txGetData_t;

/* PHY TX QUE message */
typedef struct
{
  UINT16  stage;
}PHY_txMsg_t;
/******************************************************************************
 * Externals
 *****************************************************************************/


/******************************************************************************
 * Prototypes
 *****************************************************************************/
PHY_status_t PHY_txInit(void);
PHY_status_t PHY_txPpdu(PHY_tx_ppdu_t *ppdu, PHY_cbFunc_t cb_p);
PHY_status_t PHY_txPreparePpdu(PHY_tx_ppdu_t *ppdu, PHY_cbFunc_t cb_p);
PHY_status_t PHY_txStart(void);
PHY_status_t PHY_txSet(UINT16 setType, PHY_txSetData_t *setParms_p);
PHY_status_t PHY_txGet(UINT16 getType, PHY_txGetData_t *getParms_p);
PHY_status_t PHY_txGetStats(PHY_tx_stat_t *getParms_p);

PHY_status_t PHY_txSmRun(UINT16 mode);

PHY_status_t PHY_txPpduStartCbReg(PHY_cbFunc_t cb_p);

PHY_status_t PHY_txAck(PHY_tx_ack_t *ack_p, PHY_cbFunc_t cb_p);
PHY_status_t PHY_txPrepareAck(PHY_tx_ack_t *ack_p, PHY_cbFunc_t cb_p);
PHY_status_t PHY_txRpt(PHY_cbFunc_t cb_p);

#endif

