// TI File $Revision::1$
//###########################################################################
//  FILE: DSP2834x_AT59BV802D.c 
//  
//  TITLE: XINTF driver for ATMEL external flash memory AT59BV802D.
//
//  NOTE: This file implements various functions used to interface to the AT59BV802D external
//  flash memory through the XINTF.  These functions include those defined in 
//  DSP2834x_ExtFlashAPI.h, which allows the SDFlash application to program the 
//  external flash memory device in place through the DSP and JTAG.    
//
//###########################################################################
//    Ver | dd mmm yyyy | Who | Description of changes
// =======|=============|=====|==============================================
//   1.00 | 28 Jul 2009 | TI  | Initial Release
//###########################################################################
#include "DSP28x_Project.h"
#include "DSP2834x_MAX29GL640EL.h"

//This function erases all sectors on the device.  Pass in an
//address in an unlocked sector.  
Uint16 FL_ChipErase(Uint16 *addr) 
{
//	Uint16 testData;
	Uint16 data1, data2;
	//initiate chip erase
	*FL_CmdLoc1 = 0xAA;
	*FL_CmdLoc2 = 0x55;	
	*FL_CmdLoc1 = 0x80;
	*FL_CmdLoc1 = 0xAA;  
	*FL_CmdLoc2 = 0x55; 
	*FL_CmdLoc1 = 0x10; 	
	
	//poll until erase is complete or erase operation fails	
	do
	{
	    //read data and check if erase is complete
		//testData = *addr;
		data1 = *addr;
		data2 = *addr;
		if( ( data1 & 0x0040) == ( data2 & 0x0040) )
		{
			break;
		}		
	}while((data1 & 0x0020) == 0x0000); //continue to check data if not timed out

	//verify that erase occured

	if(*addr == 0xFFFF)
	{
		//erase operation success
		return STATUS_SUCCESS;
	}
	//erase operation failed
	return STATUS_FAIL_ERASE;
}

//This function erases the sectors in the device specified by the mask.
//A value of 0xFFFFFFFF will cause all sectors to be erased.
Uint16 FL_SectorErase(Uint32 mask)
{
	Uint16 status;		
	
	status = STATUS_SUCCESS;

	//check each bit in the mask; if it is set, erase the corresponding sector. 
	if(mask & FL_SBA0 ){status |= FL_EraseSingleSector(FL_SA0 );}
	if(mask & FL_SBA1 ){status |= FL_EraseSingleSector(FL_SA1 );}
	if(mask & FL_SBA2 ){status |= FL_EraseSingleSector(FL_SA2 );}
	if(mask & FL_SBA3 ){status |= FL_EraseSingleSector(FL_SA3 );}
	if(mask & FL_SBA4 ){status |= FL_EraseSingleSector(FL_SA4 );}
	if(mask & FL_SBA5 ){status |= FL_EraseSingleSector(FL_SA5 );}
	if(mask & FL_SBA6 ){status |= FL_EraseSingleSector(FL_SA6 );}
	if(mask & FL_SBA7 ){status |= FL_EraseSingleSector(FL_SA7 );}
	if(mask & FL_SBA8 ){status |= FL_EraseSingleSector(FL_SA8 );}
	if(mask & FL_SBA9 ){status |= FL_EraseSingleSector(FL_SA9 );}
	if(mask & FL_SBA10){status |= FL_EraseSingleSector(FL_SA10);}
	if(mask & FL_SBA11){status |= FL_EraseSingleSector(FL_SA11);}
	if(mask & FL_SBA12){status |= FL_EraseSingleSector(FL_SA12);}
	if(mask & FL_SBA13){status |= FL_EraseSingleSector(FL_SA13);}
	if(mask & FL_SBA14){status |= FL_EraseSingleSector(FL_SA14);}
	if(mask & FL_SBA15){status |= FL_EraseSingleSector(FL_SA15);}
	if(mask & FL_SBA16){status |= FL_EraseSingleSector(FL_SA16);}
	if(mask & FL_SBA17){status |= FL_EraseSingleSector(FL_SA17);}
	if(mask & FL_SBA18){status |= FL_EraseSingleSector(FL_SA18);}
	if(mask & FL_SBA19){status |= FL_EraseSingleSector(FL_SA19);}
	if(mask & FL_SBA20){status |= FL_EraseSingleSector(FL_SA20);}
	if(mask & FL_SBA21){status |= FL_EraseSingleSector(FL_SA21);}
	if(mask & FL_SBA22){status |= FL_EraseSingleSector(FL_SA22);}	

	if(mask & FL_SBA23){status |= FL_EraseSingleSector(FL_SA23);}
	if(mask & FL_SBA24){status |= FL_EraseSingleSector(FL_SA24);}
	if(mask & FL_SBA25){status |= FL_EraseSingleSector(FL_SA25);}
	if(mask & FL_SBA26){status |= FL_EraseSingleSector(FL_SA26);}
	if(mask & FL_SBA27){status |= FL_EraseSingleSector(FL_SA27);}
	if(mask & FL_SBA28){status |= FL_EraseSingleSector(FL_SA28);}
	if(mask & FL_SBA29){status |= FL_EraseSingleSector(FL_SA29);}
	if(mask & FL_SBA30){status |= FL_EraseSingleSector(FL_SA30);}
	if(mask & FL_SBA31){status |= FL_EraseSingleSector(FL_SA31);}
	
	return status;	 
}

//This function erases a single sector of memory.  Pass in an address
//corresponding to any word in the sector to be erased. 
Uint16 FL_EraseSingleSector(Uint16 *addr)
{
	//Uint16 testData;
    Uint16 data1,data2;

	//initiate sector erase
	*FL_CmdLoc1 = 0xAA;
	*FL_CmdLoc2 = 0x55;
	*FL_CmdLoc1 = 0x80;
	*FL_CmdLoc1 = 0xAA;  
	*FL_CmdLoc2 = 0x55; 
	*addr = 0x30; 
	
	//poll until erase is complete or erase operation fails	
	do
	{
	    //read data and check if erase is complete
		//testData = *addr;
		data1 = *addr;
		data2 = *addr;
		if( ( data1 & 0x0040) == ( data2 & 0x0040) )
		{
			break;
		}		
	}while((data1 & 0x0020) == 0x0000); //continue to check data if not timed out
	
	//verify that erase occured
	if(*addr == 0xFFFF)
	{
		//erase operation success
		return STATUS_SUCCESS;
	}
	//erase operation failed
	return STATUS_FAIL_ERASE;
}

//This function writes a single word of data to the memory.  Pass in the
//address and the data, the function will return after the write is complete.
Uint16 FL_WriteWord(Uint16 *addr, Uint16 data)
{	
	Uint16 oldData;
	Uint16 newData1,newData2;

	//read previous data
	oldData = *addr;

	//enter single program mode
	*FL_CmdLoc1 = 0xAA;
	*FL_CmdLoc2 = 0x55;
	*FL_CmdLoc1 = 0xA0;

	//write data to desired address
	*addr = data;
	
	do
	{
		newData1 = *addr;
		newData2 = *addr;
		//Check if bit in data is toggling
		if((newData1 & 0x0040)  == (newData2 & 0x0040))
		{
			break; //bit did not toggle: write has completed
		}
	}while((newData1 & 0x0020) == 0); //continue to check data if not timed out
	
	//verify correct or incorrect data
	if(*addr == (data & oldData))
	{
		//program operation success
		return STATUS_SUCCESS;
	}
	//program operation failed
	return STATUS_FAIL_PROGRAM;	
}

//This function writes an array of data to the device.  Pass in a pointer to the
//first address to be written to.  If the operation fails, the pointer will be 
//pointing to the address that failed. Otherwise the pointer will point to the
//first address after the block of data that was just written. Also pass in the 
//location of the data buffer and the length of the buffer. 
Uint16 FL_WriteWordArray(Uint16 **addr, Uint16 *data, Uint16 length)
{
	Uint32 iter;

	//for each element in the data buffer
	for(iter = 0; iter < length; iter++)
	{
		//copy the word in the data buffer to the device
		if(FL_WriteWord(*addr,*data) != STATUS_SUCCESS)
		{
			return STATUS_FAIL_PROGRAM; //operation failed
		}
		//move to the next word
		(*addr)++;
		data++;
	}
	return STATUS_SUCCESS; //operation succeeded
}

//This function verifies an array of data in the device.  Pass in a pointer to the
//first address to be verified.  If the operation fails, the pointer will be 
//pointing to the address that failed. Otherwise the pointer will point to the
//first address after the block of data that was just verified. Also pass in the 
//location of the data buffer and the length of the buffer.  
Uint16 FL_VerifyWordArray(Uint16 **addr, Uint16 *data, Uint16 length)
{
	Uint16 iter;

	//for each word in the data buffer
	for(iter = 0; iter < length; iter++)
	{
		//compare data in buffer and data in device
		if(**addr != *data)
		{
			return STATUS_FAIL_VERIFY; //data is not the same
		}		
		//move to next word
		(*addr)++;
		data++;
	}
	return STATUS_SUCCESS; //all data is verified to be correct
}

//This function locks a sector.  This prevents writes to and 
//erases of the sector until the device is power cycled.  Pass
//in any address in the sector to be locked.
void FL_LockSector(Uint16 *addr)
{
	*FL_CmdLoc1 = 0xAA;
	*FL_CmdLoc2 = 0x55;	
	*FL_CmdLoc1 = 0x80;
	*FL_CmdLoc1 = 0xAA;  
	*FL_CmdLoc2 = 0x55; 
	*addr = 0x60; 
}

//This function intializes the DSP to interface to the external 
//memory device.
void FL_Init(void)
{
	//Redirect FL_Init function based on desired zone
	//#if FL_ZONE==6
		FL_InitXintfZone6();
	//#endif
	//#if FL_ZONE==7
	//	FL_InitXintfZone7();
	//#endif
}

//This function initializes the external memory to interface through
//XINTF zone 6
void FL_InitXintfZone6(void)
{
	EALLOW;
    //Make sure the XINTF clock is enabled
	SysCtrlRegs.PCLKCR3.bit.XINTFENCLK = 1;

    //Steup gpio muxing for x16 XINTF data bus
	GpioCtrlRegs.GPCMUX1.bit.GPIO64 = 3; //XD15
	GpioCtrlRegs.GPCMUX1.bit.GPIO65 = 3; //XD14
	GpioCtrlRegs.GPCMUX1.bit.GPIO66 = 3; //XD13
	GpioCtrlRegs.GPCMUX1.bit.GPIO67 = 3; //XD12
	GpioCtrlRegs.GPCMUX1.bit.GPIO68 = 3; //XD11
	GpioCtrlRegs.GPCMUX1.bit.GPIO69 = 3; //XD10
	GpioCtrlRegs.GPCMUX1.bit.GPIO70 = 3; //XD9
	GpioCtrlRegs.GPCMUX1.bit.GPIO71 = 3; //XD8
	GpioCtrlRegs.GPCMUX1.bit.GPIO72 = 3; //XD7
	GpioCtrlRegs.GPCMUX1.bit.GPIO73 = 3; //XD6
	GpioCtrlRegs.GPCMUX1.bit.GPIO74 = 3; //XD5
	GpioCtrlRegs.GPCMUX1.bit.GPIO75 = 3; //XD4
	GpioCtrlRegs.GPCMUX1.bit.GPIO76 = 3; //XD3
	GpioCtrlRegs.GPCMUX1.bit.GPIO77 = 3; //XD2
	GpioCtrlRegs.GPCMUX1.bit.GPIO78 = 3; //XD1
	GpioCtrlRegs.GPCMUX1.bit.GPIO79 = 3; //XD0

	//Steup gpio muxing for XINTF
	GpioCtrlRegs.GPAMUX2.bit.GPIO28 = 3; //XZCS6_L
//	GpioCtrlRegs.GPBMUX1.bit.GPIO34 = 3; //XREADY

//	GpioCtrlRegs.GPBMUX1.bit.GPIO35 = 3; //XR_H_W_L	  CCB  XRDpin is used as read signal

	GpioCtrlRegs.GPBMUX1.bit.GPIO38 = 3; //XZWE0_L

	//Steup gpio muxing for XINTF x20 addr bus	
	GpioCtrlRegs.GPAMUX2.bit.GPIO29 = 3; //XA19
	GpioCtrlRegs.GPAMUX2.bit.GPIO30 = 3; //XA18
	GpioCtrlRegs.GPAMUX2.bit.GPIO31 = 3; //XA17	 
	GpioCtrlRegs.GPBMUX1.bit.GPIO39 = 3; //XA16
	GpioCtrlRegs.GPCMUX2.bit.GPIO87 = 3; //XA15
	GpioCtrlRegs.GPCMUX2.bit.GPIO86 = 3; //XA14
	GpioCtrlRegs.GPCMUX2.bit.GPIO85 = 3; //XA13
	GpioCtrlRegs.GPCMUX2.bit.GPIO84 = 3; //XA12
	GpioCtrlRegs.GPCMUX2.bit.GPIO83 = 3; //XA11
	GpioCtrlRegs.GPCMUX2.bit.GPIO82 = 3; //XA10
	GpioCtrlRegs.GPCMUX2.bit.GPIO81 = 3; //XA9
	GpioCtrlRegs.GPCMUX2.bit.GPIO80 = 3; //XA8	
	GpioCtrlRegs.GPBMUX1.bit.GPIO47 = 3; //XA7
	GpioCtrlRegs.GPBMUX1.bit.GPIO46 = 3; //XA6
	GpioCtrlRegs.GPBMUX1.bit.GPIO45 = 3; //XA5
	GpioCtrlRegs.GPBMUX1.bit.GPIO44 = 3; //XA4
	GpioCtrlRegs.GPBMUX1.bit.GPIO43 = 3; //XA3
	GpioCtrlRegs.GPBMUX1.bit.GPIO42 = 3; //XA2
	GpioCtrlRegs.GPBMUX1.bit.GPIO41 = 3; //XA1
	GpioCtrlRegs.GPBMUX1.bit.GPIO40 = 3; //XA0   

    //set address 20 21 as gpio 
    GpioCtrlRegs.GPBMUX1.bit.GPIO35 = 0; // Gpio as Gpio
   	GpioCtrlRegs.GPBMUX2.bit.GPIO63 = 0;
    GpioCtrlRegs.GPBDIR.bit.GPIO35 = 1; // Gpio as Output
	GpioCtrlRegs.GPBDIR.bit.GPIO63 = 1;
  //  GpioDataRegs.GPBCLEAR.bit.GPIO35 = 1; // Clear Gpio
  //  GpioDataRegs.GPBCLEAR.bit.GPIO63 = 1;
    GpioDataRegs.GPBSET.bit.GPIO35 = 1; // set 1 to Gpio
    GpioDataRegs.GPBSET.bit.GPIO63 = 1;


	//Disable XINTF pin pull-ups
	GpioCtrlRegs.GPAPUD.all |= 0xF0008FFF;
	GpioCtrlRegs.GPBPUD.all |= 0xFFFFFFF8;
	GpioCtrlRegs.GPCPUD.all |= 0x00FFFFFF;
    
	//XINTF config
    XintfRegs.XINTCNF2.bit.XTIMCLK = 1; // XTIMCLK = SYSCLKOUT/2  
    XintfRegs.XINTCNF2.bit.CLKOFF = 0; // XCLKOUT is enabled  
    XintfRegs.XINTCNF2.bit.WRBUFF = 0; // Buffer up to 3 writes     
    XintfRegs.XINTCNF2.bit.CLKMODE = 1; // XCLKOUT = XTIMCLK/2    
    XintfRegs.XINTCNF2.bit.BY4CLKMODE = 1; // XCLKOUT = XTIMCLK/4 (SYSCLKOUT/8)

    // Zone 6 config    
    XintfRegs.XTIMING6.bit.X2TIMING = 1; //Double wait states
	XintfRegs.XTIMING6.bit.XSIZE = 3; //16 bit mode
	XintfRegs.XTIMING6.bit.READYMODE = 1; //Asynchronous XREADY mode
	XintfRegs.XTIMING6.bit.USEREADY = 0; //Ignore XREADY
	XintfRegs.XTIMING6.bit.XRDLEAD = 3; //14 cycles
	XintfRegs.XTIMING6.bit.XRDACTIVE = 3; //14 cycles 
	XintfRegs.XTIMING6.bit.XRDTRAIL = 3; //14 cycles
	XintfRegs.XTIMING6.bit.XWRLEAD = 1; //6 cycles
	XintfRegs.XTIMING6.bit.XWRACTIVE = 3; //14 cycles
	XintfRegs.XTIMING6.bit.XWRTRAIL = 2; //10 cycles    

   //Force a pipeline flush to ensure that the write to
   //the last register configured occurs before returning.
   asm(" RPT #7 || NOP");

   EDIS;
}

//This function initializes the external memory to interface through
//XINTF zone 7
void FL_InitXintfZone7(void)
{
	EALLOW;
    //Make sure the XINTF clock is enabled
	SysCtrlRegs.PCLKCR3.bit.XINTFENCLK = 1;

    //Steup gpio muxing for x16 XINTF data bus
	GpioCtrlRegs.GPCMUX1.bit.GPIO64 = 3; //XD15
	GpioCtrlRegs.GPCMUX1.bit.GPIO65 = 3; //XD14
	GpioCtrlRegs.GPCMUX1.bit.GPIO66 = 3; //XD13
	GpioCtrlRegs.GPCMUX1.bit.GPIO67 = 3; //XD12
	GpioCtrlRegs.GPCMUX1.bit.GPIO68 = 3; //XD11
	GpioCtrlRegs.GPCMUX1.bit.GPIO69 = 3; //XD10
	GpioCtrlRegs.GPCMUX1.bit.GPIO70 = 3; //XD9
	GpioCtrlRegs.GPCMUX1.bit.GPIO71 = 3; //XD8
	GpioCtrlRegs.GPCMUX1.bit.GPIO72 = 3; //XD7
	GpioCtrlRegs.GPCMUX1.bit.GPIO73 = 3; //XD6
	GpioCtrlRegs.GPCMUX1.bit.GPIO74 = 3; //XD5
	GpioCtrlRegs.GPCMUX1.bit.GPIO75 = 3; //XD4
	GpioCtrlRegs.GPCMUX1.bit.GPIO76 = 3; //XD3
	GpioCtrlRegs.GPCMUX1.bit.GPIO77 = 3; //XD2
	GpioCtrlRegs.GPCMUX1.bit.GPIO78 = 3; //XD1
	GpioCtrlRegs.GPCMUX1.bit.GPIO79 = 3; //XD0

	//Steup gpio muxing for XINTF	
//	GpioCtrlRegs.GPBMUX1.bit.GPIO34 = 3; //XREADY
	GpioCtrlRegs.GPBMUX1.bit.GPIO35 = 3; //XR_H_W_L	
	GpioCtrlRegs.GPBMUX1.bit.GPIO37 = 3; //XZCS7_L
	GpioCtrlRegs.GPBMUX1.bit.GPIO38 = 3; //XZWE0_L

	//Steup gpio muxing for XINTF x20 addr bus	
	GpioCtrlRegs.GPAMUX2.bit.GPIO29 = 3; //XA19
	GpioCtrlRegs.GPAMUX2.bit.GPIO30 = 3; //XA18
	GpioCtrlRegs.GPAMUX2.bit.GPIO31 = 3; //XA17	 
	GpioCtrlRegs.GPBMUX1.bit.GPIO39 = 3; //XA16
	GpioCtrlRegs.GPBMUX1.bit.GPIO40 = 3; //XA15
	GpioCtrlRegs.GPBMUX1.bit.GPIO41 = 3; //XA14
	GpioCtrlRegs.GPBMUX1.bit.GPIO42 = 3; //XA13
	GpioCtrlRegs.GPBMUX1.bit.GPIO43 = 3; //XA12
	GpioCtrlRegs.GPBMUX1.bit.GPIO44 = 3; //XA11
	GpioCtrlRegs.GPBMUX1.bit.GPIO45 = 3; //XA10
	GpioCtrlRegs.GPBMUX1.bit.GPIO46 = 3; //XA9
	GpioCtrlRegs.GPBMUX1.bit.GPIO47 = 3; //XA8	
	GpioCtrlRegs.GPCMUX2.bit.GPIO80 = 3; //XA7
	GpioCtrlRegs.GPCMUX2.bit.GPIO81 = 3; //XA6
	GpioCtrlRegs.GPCMUX2.bit.GPIO82 = 3; //XA5
	GpioCtrlRegs.GPCMUX2.bit.GPIO83 = 3; //XA4
	GpioCtrlRegs.GPCMUX2.bit.GPIO84 = 3; //XA3
	GpioCtrlRegs.GPCMUX2.bit.GPIO85 = 3; //XA2
	GpioCtrlRegs.GPCMUX2.bit.GPIO86 = 3; //XA1
	GpioCtrlRegs.GPCMUX2.bit.GPIO87 = 3; //XA0     

	//Disable XINTF pull-ups
	GpioCtrlRegs.GPAPUD.all |= 0xF0008FFF;
	GpioCtrlRegs.GPBPUD.all |= 0xFFFFFFF8;
	GpioCtrlRegs.GPCPUD.all |= 0x00FFFFFF;
    
	//XINTF config
    XintfRegs.XINTCNF2.bit.XTIMCLK = 1; // XTIMCLK = SYSCLKOUT/2  
    XintfRegs.XINTCNF2.bit.CLKOFF = 0; // XCLKOUT is enabled  
    XintfRegs.XINTCNF2.bit.WRBUFF = 0; // Buffer up to 3 writes     
    XintfRegs.XINTCNF2.bit.CLKMODE = 1; // XCLKOUT = XTIMCLK/2    
    XintfRegs.XINTCNF2.bit.BY4CLKMODE = 1; // XCLKOUT = XTIMCLK/4 (SYSCLKOUT/8)

    // Zone 7 config    
    XintfRegs.XTIMING7.bit.X2TIMING = 1; //Double wait states
	XintfRegs.XTIMING7.bit.XSIZE = 3; //16 bit mode
	XintfRegs.XTIMING7.bit.READYMODE = 1; //Asynchronous XREADY mode
	XintfRegs.XTIMING7.bit.USEREADY = 0; //Ignore XREADY
	XintfRegs.XTIMING7.bit.XRDLEAD = 3; //14 cycles
	XintfRegs.XTIMING7.bit.XRDACTIVE = 3; //14 cycles 
	XintfRegs.XTIMING7.bit.XRDTRAIL = 3; //14 cycles
	XintfRegs.XTIMING7.bit.XWRLEAD = 1; //6 cycles
	XintfRegs.XTIMING7.bit.XWRACTIVE = 3; //14 cycles
	XintfRegs.XTIMING7.bit.XWRTRAIL = 2; //10 cycles    

    //Force a pipeline flush to ensure that the write to
    //the last register configured occurs before returning.
    asm(" RPT #7 || NOP");

    EDIS;
}
