/*****************************************************************************
* FILE PURPOSE: Header file for PHY AFE
*******************************************************************************
*
* FILE NAME: phy_afe.h
*
* DESCRIPTION:
*       Header file for HAL AFE.
*
*  Copyright (c) 2009 Texas Instruments Inc.
*  All Rights Reserved This program is the confidential and proprietary
*  product of Texas Instruments Inc.  Any Unauthorized use, reproduction or
*  transfer of this program is strictly prohibited.
*
* HISTORY:
*
* 03/05/2009 fum    Written
* 03/23/2009 S.Yim  Rx+AFE integration
* 10/22/2009 M.Fu   Change Sync block len to 256
*
******************************************************************************/
#ifndef _HAL_AFE_H
#define _HAL_AFE_H

#include <typedefs.h>

/******************************************************************************
 * Definitions
 *****************************************************************************/
/* Sampling and PWM frequencies */
#define HAL_AFE_KHZ_250    250
#define HAL_AFE_KHZ_400    400
#define HAL_AFE_KHZ_500    500
#define HAL_AFE_KHZ_800    800
#define HAL_AFE_KHZ_1000   1000
#define HAL_AFE_KHZ_1200   1200
#define HAL_AFE_KHZ_1500   1500
#define HAL_AFE_KHZ_2000   2000

/* Timer cfg */
#define HAL_AFE_TIMER_INT_ENABLE     0x1
#define HAL_AFE_TIMER_INT_DISABLE    0x0
#define HAL_AFE_TIMER_1_SHOT         0x2
#define HAL_AFE_TIMER_1_SHOT_SHIFT     1

/* HAL status */
typedef enum
{
  HAL_STAT_SUCCESS = 0,     // HAL success
  HAL_STAT_FAILURE = 1      // HAL failure
}HAL_status_t;

/* HAL_AFE set code */
typedef enum
{
  AFE_TX_CFG     = 0,        // TX configure
  AFE_TX_START   = 1,        // TX start
  AFE_TX_STOP    = 2,        // TX stop
  AFE_TX_RECFG   = 3,        // Reconfigure TX
  AFE_TX_DMASIZE = 4,        // Set TX DMA size
  AFE_TX_DMAADDR = 5,        // Set TX DMA address
  AFE_TX_T0CFG   = 6,        // Configure CPU timer 0
  AFE_TX_T0START = 7,        // Start CPU timer 0
  AFE_TX_T0STOP  = 8,        // Stop CPU timer 0

  AFE_RX_START   = 9,        // RX start
  AFE_RX_STOP    = 10,       // RX stop
  AFE_RX_RECFG   = 11,       // RX reconfigure
  AFE_RX_SETGAIN = 12,       // Set AGC gain
  AFE_RX_UPDATEGAIN = 13,    // Update AGC gain
  AFE_RX_CANCEL_GAIN = 14,   // Cancel gain update

#ifndef F2803X
  AFE_SEL_ECAP    = 15,
  AFE_SEL_XINT1   = 16,
  AFE_SEL_XINT2   = 17,

  NUM_AFE_SETPARMS = 18
#else
  NUM_AFE_SETPARMS = 15
#endif
}HAL_afe_setCode_t;

typedef enum
{
   HAL_AFE_ECAP1 = 1,
   HAL_AFE_ECAP2 = 2,
   HAL_AFE_ECAP3 = 3,
   HAL_AFE_ECAP4 = 4,
   HAL_AFE_ECAP5 = 5,
   HAL_AFE_ECAP6 = 6
}HAL_afe_ecapSel_t;

/* Callback functions */
typedef void (*HAL_cbFunc_t)(UINT32);

/* HAL_AFE get code */
typedef enum
{
  AFE_RX_DMATIME   = 0,        // RX DMA time
  AFE_ZCTIME       = 1,        // ZC time
  AFE_RX_DMAIDX    = 2,        // RX DMA index
  AFE_RND_NUM      = 3,        // random number generator
#if defined(F2803X) && defined(HB)
  AFE_RX_BUFPTR_SZ = 4,
  NUM_AFE_GETPARMS = 5
#else
  NUM_AFE_GETPARMS = 4
#endif
}HAL_afe_getCode_t;

/*****************************************************************************/
/* Data Structures                                                           */
/*****************************************************************************/
/* AFE RX set parameters */
typedef struct
{
  UINT16 size[2];            // Sizes of RX ping-pong buffers
  UINT16 *dstBufAddr[2];     // DMA desination ping-pong buffer addresses
  UINT16 dstBufIdx;          // Destination buffer index. Always point to the readable buffer
}HAL_afe_rxSetParms_t;

/* AFE ADC offset adjust parameters */
typedef struct
{
  UINT16 size;               // Size of data buffer needs offset adjustment
  SINT16 *data_p;             // Pointer to the data buffer
}HAL_afe_adcAdjParms_t;

/* AFE TX set parameters */
typedef struct
{
  SINT16 *txBuf_p;            // Pointer to the TX buffer
  UINT16 txSize;             // Size of TX buffer
  UINT32 t0PrdInUs;          // CPU timer 0 period (in us)
  UINT16 t0Flags;            // bit 0: CPU timer 0 interrupt enabled (1) or disabled (1)
                             // bit 1: 1 shot = 1, 0 = continuous
                             // This is enabled for txTime timeout, but disabled for TX DMA
  HAL_cbFunc_t cb_p;
  UINT32 cb_param;
}HAL_afe_txSetParms_t;

typedef struct
{
  UINT32 zcaTime;
  UINT32 zcbTime;
}HAL_afe_zcTime_t;

/* AFE Profile */
typedef struct
{
  UINT16 rx_fs_kHz;
  UINT16 tx_fs_kHz;
  UINT16 tx_pwm_kHz;
  UINT16 band;         //0-A; 1-BCD
}HAL_afe_prfParms_t;

#if defined(F2803X) && defined(HB)
/* RX Buffer Data Pointer & size, the previous just completed */
typedef struct
{
  UINT16 *buf_p;
  UINT32 size;
}HAL_afe_rxBufPtrSz_t;
#endif

typedef struct
{
  UINT16 ch2;
  UINT16 ch4;
}HAL_afe_adcReslts_t;
/******************************************************************************
 * Externals
 *****************************************************************************/
extern HAL_afe_txSetParms_t HAL_afe_txSetParms_s;
/******************************************************************************
 * Prototypes
 *****************************************************************************/
HAL_status_t HAL_afeInit(HAL_afe_prfParms_t *setParms_p);
HAL_status_t HAL_afeSet(HAL_afe_setCode_t setCode, void *setParms_s);
HAL_status_t HAL_afeGet(HAL_afe_getCode_t getCode, void *getParms_p);
HAL_status_t HAL_afeTxInit(void);
HAL_status_t HAL_afeRxInit(void);
HAL_status_t HAL_afeRxDmaCh1IntFunc(void);
HAL_status_t HAL_afeTxDmaCh2IntFunc(void);
HAL_status_t HAL_cpuTint0Func(void);
HAL_status_t HAL_afeEcap1IntFunc(void);
HAL_status_t HAL_afeEcap3IntFunc(void);
HAL_status_t HAL_afeEcap4IntFunc(void);
HAL_status_t HAL_haltLPM(void);
HAL_status_t HAL_xint1Func(void);
HAL_status_t HAL_xint2Func(void);

//HAL_status_t HAL_afe031Init(void);
HAL_status_t HAL_afeDacCnv(int16 *in_p, UINT16 len, int16 lev, UINT16 clip, int16 *out_p);
HAL_status_t HAL_afeDacCnvInP(int16 *data_p, Uint16 len);

const char  *HAL_getLibVersion(void);

#ifdef F2803X
void HAL_pwm6TimerCfg(void);
void HAL_pwm6TimerTimeout(UINT32 timeoutInUs);
UINT16 HAL_pwm6TimerTimeoutChk(void);
void HAL_pwm6TimerTimeoutClr(void);
HAL_status_t HAL_afe_bandSelect(UINT16 band);
void HAL_afe_adcReadReslts(HAL_afe_adcReslts_t *adcval_p);
#endif

HAL_status_t HAL_afe_setTXAttn(UINT16 attn);
UINT16 HAL_afe_getTXAttn(void);

UINT16 HAL_interrupt_disable(void);
void HAL_interrupt_restore(UINT16);

#endif

