//===========================================================================
//    Purpose:
//         This CSC provides an interface to Initialize, Write and Read a 
//    GPIO pin.
//===========================================================================

#include "gpio.h"
#include "gpio_private.h"

#define GPIO_ODD_BANK_PIN_ADDER (16u)
#define MAX_GPIO                (144u)

GpioDevice gpio_device;
#pragma DATA_SECTION(gpio_device, "gpio")
//Overlap the GPIO memory section

//===========================================================================
//Private Function Declarations
//===========================================================================
static GpioRegisters*       get_gpio_register_block           (GpioBank bank);
static uint32               get_gpio_register_bit             (GpioBank bank, GpioPin pin);
static bool                 bank_or_pin_out_of_range          (GpioBank bank, GpioPin pin);
static void                 write_bit                         (volatile uint32 *register_value, uint32 mask, GpioValue value);
static uint8                index_of_bank_and_pin             (GpioBank bank, GpioPin pin);
/////////////////////////////////////////////////////////
///////This variable is not actually a problem///////////
static const                bool GPIO_INITIALIZED = true;
static                      bool gpio_initialized_array[MAX_GPIO] = {false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false,
                                                                     false, false, false, false, false, false};
static                      GpioObject gpio_object_array[MAX_GPIO];

//===========================================================================
//    Purpose:
//         Creates a handle for the specified bank and pin for later use
//         and sets the associated direction bit
//===========================================================================
GpioHandle gpio_initialize(GpioBank bank, GpioPin pin, GpioDirection direction)
{
    GpioHandle handle;
    
    if(bank_or_pin_out_of_range(bank, pin))
    {
        handle = 0u;
    }
    else
    {
        handle = index_of_bank_and_pin(bank, pin);
        if(gpio_initialized_array[handle - 1u] == GPIO_INITIALIZED)
        {
            handle = 0u;
        }
        else
        {
            gpio_initialized_array[handle - 1u]            = GPIO_INITIALIZED;
            gpio_object_array[handle - 1u].gpio_registers  = get_gpio_register_block(bank);
            gpio_object_array[handle - 1u].mask            = get_gpio_register_bit(bank, pin);
            
            write_bit(&(gpio_object_array[handle - 1u].gpio_registers->DIR), gpio_object_array[handle - 1u].mask, (GpioValue)direction);
        }
    }
    return handle;
}
//===========================================================================
//    Purpose:
//         Performs a write to the GPIO bank and pin specified by the handle
//===========================================================================
GpioErrorCode gpio_write(const GpioHandle gpio_write_handle, GpioValue value)
{
    GpioErrorCode code;

    if((gpio_write_handle > MAX_GPIO) || (gpio_write_handle <= 0u))
    {
        code = GPIO_ERROR;
    }
    else if(gpio_initialized_array[gpio_write_handle - 1u] != GPIO_INITIALIZED)
    {
        code = GPIO_ERROR;
    }
    else if((gpio_object_array[gpio_write_handle - 1u].gpio_registers->DIR & gpio_object_array[gpio_write_handle - 1u].mask) == gpio_object_array[gpio_write_handle - 1u].mask)
    {
        code = GPIO_ERROR;
    }
    else if(value == GPIO_OUTPUT_HIGH)
    {
        gpio_object_array[gpio_write_handle - 1u].gpio_registers->SET_DATA = gpio_object_array[gpio_write_handle - 1u].mask;
        code = GPIO_NO_ERROR;
    }
    else
    {
        gpio_object_array[gpio_write_handle - 1u].gpio_registers->CLR_DATA = gpio_object_array[gpio_write_handle - 1u].mask;
        code = GPIO_NO_ERROR;
    }
    return code;
}
//===========================================================================
//    Purpose:
//         Performs a read of the GPIO bank and pin specified by the handle
//===========================================================================
GpioErrorCode gpio_read(const GpioHandle gpio_read_handle, GpioValue *value)
{
    GpioErrorCode code = GPIO_NO_ERROR;
    //[CPIO-SDD1361]
    if((gpio_read_handle > MAX_GPIO) || (gpio_read_handle <= 0u))
    {
        code = GPIO_ERROR;
    }
    else
    {
        //[CPIO-SDD1159]
        *value = (GpioValue)((gpio_object_array[gpio_read_handle - 1u].gpio_registers->IN_DATA & gpio_object_array[gpio_read_handle - 1u].mask) == gpio_object_array[gpio_read_handle - 1u].mask);
    }
    return code;
}
//===========================================================================
//    Purpose:
//         Return a pointer to the register block from the given bank.
//===========================================================================
static GpioRegisters* get_gpio_register_block(GpioBank bank)
{
////////////////////////////////////////////////////////
////This is the first problem variable//////////////////
    static GpioRegisters* const gpio_block_look_up[9u] =
    {
        &gpio_device.block_0_1,
        &gpio_device.block_0_1,
        &gpio_device.block_2_3,
        &gpio_device.block_2_3,
        &gpio_device.block_4_5,
        &gpio_device.block_4_5,
        &gpio_device.block_6_7,
        &gpio_device.block_6_7,
        &gpio_device.block_8
    };
    
    return (gpio_block_look_up[bank]);
}
//===========================================================================
//    Purpose:
//         The pins are in a range of 0-15 but the register bits are 0-31.
//         The odd banks (1, 3, 5, 7) share the same bank with the pin
//         offset by 16.  This returns the bit mask for the bit based on the
//         input pin.
//===========================================================================
static uint32 get_gpio_register_bit(GpioBank bank, GpioPin pin)
{
//////////////////////////////////////////////////////////
////This is the second problem variable///////////////////
    static const uint32 bit_mask[32u] =
    {
        0x00000001u, 0x00000002u, 0x00000004u, 0x00000008u,
        0x00000010u, 0x00000020u, 0x00000040u, 0x00000080u,
        0x00000100u, 0x00000200u, 0x00000400u, 0x00000800u,
        0x00001000u, 0x00002000u, 0x00004000u, 0x00008000u,
        0x00010000u, 0x00020000u, 0x00040000u, 0x00080000u,
        0x00100000u, 0x00200000u, 0x00400000u, 0x00800000u,
        0x01000000u, 0x02000000u, 0x04000000u, 0x08000000u,
        0x10000000u, 0x20000000u, 0x40000000u, 0x80000000u
    };
    uint16 pin_offset = (uint16)pin;
    
    if(((uint8)bank % 2u) == 1u)
    {
        pin_offset += GPIO_ODD_BANK_PIN_ADDER;
    }
    return(bit_mask[pin_offset]);
}
//===========================================================================
//    Purpose:
//         Checks if either the bank or the pin is out of valid range.  Return
//    true if it is and false otherwise.
//===========================================================================
static bool bank_or_pin_out_of_range(GpioBank bank, GpioPin pin)
{
    bool out_of_range = false;
//This range check is defensive programming
    if((bank > GPIO_BANK8) || (pin > GPIO_PIN15) || (bank < GPIO_BANK0) || (pin < GPIO_PIN0))
    {
        out_of_range = true;
    }
    return out_of_range;
}
//===========================================================================
//    Purpose:
//         Set or clear the bits in the register_value based on the mask.
//===========================================================================
static void write_bit(volatile uint32 *register_value, uint32 mask, GpioValue value)
{
    if(value == GPIO_OUTPUT_HIGH)
    {
        (*register_value |= mask);
    }
    else
    {
        (*register_value &= ~mask);
    }
}
//===========================================================================
//    Purpose:
//         Return the index to a given GPIO pin based on the input bank and pin.
//===========================================================================
static uint8 index_of_bank_and_pin(GpioBank bank, GpioPin pin)
{
    return(((uint8)bank * 16u) + (uint8)pin + 1u);
}
