//===========================================================================
//    Purpose: Implementation of the LED output
//      Two color LEDs are connected indirectly through the EMIFA 0x63000004
//      Writing a 1 turns off a color
//      Writing a 0 turns on a color
//      Because of how it is routed after the EMIFA,
//       reading the input has no effect and does not reflect the output state
//  LED1 - bit 1 is red, bit 2 is green (both on for orange)
//  LED2 - bit 3 is red, bit 4 is green (both on for orange)
//===========================================================================
#include "led.h"

//Keep track of the state of the LEDs internally, since we cannot read it
static unsigned short led_state=0xFFFF;//default to both off

/////////////////////////////////////////
//Set the LED color to the selected value
/////////////////////////////////////////
void Led_Set(LedNumber led, LedColor color)
{
  unsigned short* output_ptr = (unsigned short*)0x63000004u;
  unsigned short red_mask = 0x02u;
  unsigned short green_mask;
  if(led == LED2)
  { //shift upwards for control of the second LED
    red_mask = red_mask << 2;
  }
  //The green bit is always one higher than the red bit
  green_mask = red_mask << 1;
  switch(color)
  { //Determine the pattern to change for the two bits of interest
    case LED_OFF:
      //Turn off both
      led_state |= red_mask;
      led_state |= green_mask;
      break;
    case LED_RED:
      //Turn on RED
      led_state &= ~red_mask;
      //Turn off green
      led_state |= green_mask;
      break;
    case LED_GREEN:
      //Turn off RED
      led_state |= red_mask;
      //Turn off green
      led_state &= ~green_mask;
      break;
    case LED_ORANGE:
    default:
      //Turn on both
      led_state &= ~red_mask;
      led_state &= ~green_mask;
      break;
  }
  *output_ptr = led_state;
}

/////////////////////////////////////////
//Cycle the LED color through four values
/////////////////////////////////////////
void Led_Cycle(LedNumber led)
{ //Keep track of the cycle color since we cannot read the outputs
  //Start at different colors because we can
  static LedColor led1_cycle_color = LED_RED;
  static LedColor led2_cycle_color = LED_GREEN;
  //Point to the color holder of interest
  LedColor *theColor;
  if(led == LED1)
  {
    theColor = &led1_cycle_color;
  }
  else
  {
    theColor = &led2_cycle_color;
  }
  //Update to the next enumerated color
  *theColor += 1;
  if((*theColor) > 3)
  { //Limit to the valid enumeration values
    *theColor = LED_OFF;
  }
  Led_Set(led,*theColor);
}
