//===========================================================================
//    Purpose: Custom implementation of the OMAP timer interface.
//    TI document sprufm5d - timer setup
//    TI document sprugm7e - PLL mapping to timers (p. 86, Table 8-1. System PLLC Output Clocks)
//===========================================================================
#include "timer.h"

//Timer register offsets
#define TIMER_TIM12 (4u)
#define TIMER_TIM34 (5u)
#define TIMER_PRD12 (6u)
#define TIMER_PRD34 (7u)
#define TIMER_TCR   (8u)
#define TIMER_TGCR  (9u)

//Timer Control Register (TCR) bit field masks
#define ENAMODE12_CLR (0xFFFFFF7Fu)//zero in disabled or single run
#define ENAMODE12_RUN (0x00000040u)//low for disabled, high for single run

//Timer Global Control Register (TGCR) bit field masks
//TIMMMODE is a two bit field
#define TIMMODE_64 (0xFFFFFFF3u)//64-bit timer when cleared
#define TIM1234RS (0x00000003u)//Timer 3:4 and 1:2 active low reset

//===================
//Timer CSC Variables
//===================
static volatile unsigned int* timer_0=(uint32*)0x01C20000;// 128B of registers
static volatile unsigned int* timer_1 =(uint32*)0x01C21000;//128B of registers

//============================================================================
//Purpose: Initialize both timers to operate in 64-bit mode.  These timers are
//  taken out of reset, but disabled.
//  Timer 0 / 1 - cycle source PLL0_AUXCLK (OSCIN) (16.777MHz)
//[MLVB-SDD2818]
//============================================================================
void Timer_Initialize(void)
{
  //Reset the timer
  timer_0[TIMER_TGCR] &= (~TIM1234RS);//Start at reset
  timer_1[TIMER_TGCR] &= (~TIM1234RS);//Start at reset
  
  //1. Select 64-bit mode, TIMMODE = 00b
  timer_0[TIMER_TGCR] &= TIMMODE_64;
  timer_1[TIMER_TGCR] &= TIMMODE_64;
  
  //Only ever using a single bit for single runs.  Set the other low
  timer_0[TIMER_TCR] &= ENAMODE12_CLR;
  timer_1[TIMER_TCR] &= ENAMODE12_CLR;
  
  //2. Remove from reset
  timer_0[TIMER_TGCR] |= TIM1234RS;
  timer_1[TIMER_TGCR] |= TIM1234RS;
}

//============================================================================
//Purpose: Start one of the 64-bit timers with the desired period (in clock
//  cycles).
//  Timer 0 / 1 - cycle source PLL0_AUXCLK (OSCIN) (16.777MHz)
//[MLVB-SDD2818]
//============================================================================
void Timer_Start(TimerEnum timer, unsigned long long period_cycles)
{
  if(timer == TIMER_0)
  { //TIMER 0
    //Disable the timer
    timer_0[TIMER_TCR] &= (~ENAMODE12_RUN);
    
    //Remove from reset
    timer_0[TIMER_TGCR] |= TIM1234RS;
      
    //Clear the count
    timer_0[TIMER_TIM12] = 1u;
    timer_0[TIMER_TIM34] = 0u;
    
    //Set the period
    timer_0[TIMER_PRD12] = (unsigned int)period_cycles;
    timer_0[TIMER_PRD34] = (unsigned int)(period_cycles >> 32u);
        
    //Enable the timer for a single run
    timer_0[TIMER_TCR] |= ENAMODE12_RUN;
  }
  else
  { //TIMER 1
    //Disable the timer
    timer_1[TIMER_TCR] &= (~ENAMODE12_RUN);
    
    //Remove from reset
    timer_1[TIMER_TGCR] |= TIM1234RS;
    
    //Clear the count
    timer_1[TIMER_TIM12] = 1u;
    timer_1[TIMER_TIM34] = 0u;
    
    //Set the period
    timer_1[TIMER_PRD12] = (unsigned int)period_cycles;
    timer_1[TIMER_PRD34] = (unsigned int)(period_cycles >> 32u);
    
    //Enable the timer for a single run
    timer_1[TIMER_TCR] |= ENAMODE12_RUN;
  }
}

//============================================================================
//Purpose: Disable the indicated timer.  The current count value is maintained
//============================================================================
void Timer_Stop(TimerEnum timer)
{
  if(timer == TIMER_0)
  { //TIMER 0
    timer_0[TIMER_TCR] &= (~ENAMODE12_RUN);
  }
  else
  { //TIMER 1
    timer_1[TIMER_TCR] &= (~ENAMODE12_RUN);
  }
}

//============================================================================
//Purpose: Return the count value of the 64-bit timer.
//Note: Due to the limited scope of the MLVB, the 64-bit timer should never
//  reach a value above 32 bits during normal operations.  The upper 32 bits
//  are latched by the OMAP when the lower bits are read.
//ASSUMPTION: The Timer value of interest will never exceed 32-bits in size,
//  so the upper 32-bits are not used.
//[MLVB-SDD2818]
//============================================================================
unsigned long long Timer_GetTime(TimerEnum timer)
{
  unsigned int timer_half_word;
  unsigned long long timer_full_word;
  if(timer == TIMER_0)
  { //TIMER 0
    timer_full_word = timer_0[TIMER_TIM12];
    timer_half_word = timer_0[TIMER_TIM34];
    timer_full_word |= (((unsigned long long)timer_half_word) << 32u);
  }
  else
  { //TIMER 1
    timer_full_word = timer_1[TIMER_TIM12];
    timer_half_word = timer_1[TIMER_TIM34];
    timer_full_word |= (((unsigned long long)timer_half_word) << 32u);
  }
  return timer_full_word;
}

//============================================================================
//Purpose: Determine if TIMER 0 or TIMER 1 is currently enabled.
//============================================================================
bool Timer_IsRunning(TimerEnum timer)
{
  unsigned int enabled;
  if(timer == TIMER_0){
    enabled = (timer_0[TIMER_TCR] & ENAMODE12_RUN);
  }else{
    enabled = (timer_1[TIMER_TCR] & ENAMODE12_RUN);
  }
  return(enabled!=0u)&&(Timer_GetTime(timer)!=0u);
}

