/** @file   SPI.c
 *  @brief  Polling based SPI driver
 *  @author LR
 *  @date   Feb 2011
 *  @note   History:
 *          Date        Author      Comment
 *          02/2011     LR          File created and added to GIT
 *  Built with IAR Embedded Workbench Version: 5.20
 *
 */
/* --COPYRIGHT--,BSD
 * Copyright (c) 2011, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/


//
// Include files
//
#include "RFID_hardware.h"


//
// Public functions
//
/******************************************************************************
 * @brief   Initializes the SPI module
 *
 * @return  none
 *****************************************************************************/
void USCISPIset(void)
{

  UCB0CTL1 |= UCSWRST;                      // Disable USCI first
  #ifdef USE_TRF7970
  UCB0CTL0 |= UCMSB + UCMST + UCSYNC;  // 3-pin, 8-bit SPI master, Clock Pol/Phase = 0
  #else
  UCB0CTL0 |= UCCKPH + UCMSB + UCMST + UCSYNC;  // 3-pin, 8-bit SPI master
  #endif
  UCB0CTL1 |= UCSSEL_2;                     // SMCLK

  UCB0BR0 = SPIBAUD0;
  UCB0BR1 = SPIBAUD1;
  // P3.1,3.2,3.3 UCB0SIMO,UCB0SOMI,UCBOCLK option select
  SPI_PSEL |= (SPI_MOSI_PIN + SPI_MISO_PIN + SPI_CLK_PIN);

  SLAVESELECT_OUTPUT();                     // P3.0 - Slave Select
  SLAVESELECT_HIGH();                       // Slave Select - inactive ( high)

  UCB0CTL1 &= ~UCSWRST;                     // **Initialize USCI state machine**
}

/******************************************************************************
 * @brief   Waits for the bus to be idle
 *
 * @return  none
 *****************************************************************************/
void USCISPIWaitBus(void)
{
    while(UCSPISTAT & UCBUSY)    // Ensure USCI is not busy
        ;
}


/******************************************************************************
 * @brief   Transfers data to the SPI slave (TX and RX in same transfer)
 *
 * @param data_out Data being sent
 *
 * @return  uint8_t with received data
 * @note This function polls for bus activity. CS is not handled here
 *****************************************************************************/
uint8_t USCISPITransfer(uint8_t data_out)
{
    while (!(UCSPIIFG & UCB0TXIFG))   // USCI_B0 TX buffer ready?
        ;
        UCSPITXBUF = data_out;             // Previous data to TX, RX

     USCISPIWaitBus();

     return UCSPIRXBUF;
}
