/** @file   nfc.c
 *  @brief  NFC communication driver
 *  @author LR
 *  @date   Feb 2011
 *  @note   History:
 *          Date        Author      Comment
 *          02/2011     LR          File created and added to GIT
 *  Built with IAR Embedded Workbench Version: 5.20
 *
 */
/* --COPYRIGHT--,BSD
 * Copyright (c) 2011, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//
// Include files
//
#include "RFID_hardware.h"
#include "nfc.h"
#include "rfid_hal.h"
#include "Android_P2P.h"

//
// Macro definitions
//
/*! Enable to check timings using GPIOs */
//#define CHECK_TIMINGS

//
// Local global functions
//
/*! Target NFC ID */
static const uint8_t TargetCID[10] = TARGET_CID;
/*! Current state in State machine*/
NFC_state_t NFC_State;

/*! Buffer with received data */
uint8_t buffer[BUF_LENGTH/2];
/*! Buffer with data to be transmitted  (Max TX buf + command + 2xlen) */
uint8_t tx_buf[FIFO_LEN+3];
/*! Buffer pointer */
static uint8_t * buf_ptr;
/*! Interrupt, error, flow flags */
uint8_t u8InterruptFlags;

//
// Imported variables
//
/*! Pointer to BSL transmission buffer */
extern char* BSL430_SendBuffer;
/*! Pointer to BSL transmission buffer header */
extern char *BSL430_SendBuffHeader;
//
// Local prototypes
//
static void NFC_Target_InitLoop(void);
int8_t NFC_ProcessReceivedData(uint8_t * buffer);
static int8_t NFCSend(uint8_t  *pbuf);

//
// Public functions
//
/******************************************************************************
 * @brief   Initializes the NFC state machine
 *
 * @return  none
 *****************************************************************************/
void NFC_Init(void)
{
    NFC_State = POWER_OFF_STATE;
}

/******************************************************************************
 * @brief   NCF Target State machine
 *  Main NFC state machine handling the communication. It has 5 states:
 *  POWER_OFF_STATE: Default after reset
 *  SENSE_STATE: Target is waiting for an Initiator
 *  PRESELECTED_STATE: Target received an SDD
 *  SELECTED_STATE: Target exchanged ATR_REQ and ATR_RES and can exchange data now
 *  SEND_ACK_STATE: Packet was received and acknowledge is being sent
 *
 * @return    DATA_RECEIVED         A packet has been received and can be processed
 *             RX_PACKET_ONGOING     Reception in progress
 *             RET_ERR               Error during reception
 *****************************************************************************/
int8_t NFCTarget_statemachine(void)
{
    int8_t ret = RX_PACKET_ONGOING;

    switch (NFC_State)
    {
        case POWER_OFF_STATE:
        case SENSE_STATE:
            NFC_Target_InitLoop();
            buf_ptr = &buffer[0];
            NFC_State = SENSE_STATE;
            u8InterruptFlags = INT_BUSY;

            Radio_Wait_IRQ();
            if (u8InterruptFlags&INT_CRITICAL_ERROR)
                NFC_State = POWER_OFF_STATE;

            if ((u8InterruptFlags &
                (INT_RX_COMPLETE | INT_CRITICAL_ERROR | INT_COMM_ERROR))
                ==  INT_RX_COMPLETE)
                ret = NFC_ProcessReceivedData(&buffer[0]); // Check if the message was a ATR_REQ

        break;
       case PRESELECTED_STATE:
        case SELECTED_STATE:
 	     // Now wait until the part is selected
            // restart the pointer and wait for a message
            buf_ptr = &buffer[0];
            u8InterruptFlags = INT_BUSY;
            do{
                Radio_Wait_IRQ();
            }while (u8InterruptFlags & INT_BUSY);

            if ((u8InterruptFlags &
                (INT_RX_COMPLETE | INT_CRITICAL_ERROR | INT_COMM_ERROR))
                ==  INT_RX_COMPLETE)
                ret = NFC_ProcessReceivedData(&buffer[0]); // Check if the message was a ATR_REQ
            else if ((u8InterruptFlags&INT_CRITICAL_ERROR))
                NFC_State = POWER_OFF_STATE;    // Timeout Error
            // After a message is received, send the response packet
            // Including additional data from BSL API
            if (ret == DATA_RECEIVED)
                NFC_State = SENDING_ACK_STATE;
        break;
        case SENDING_ACK_STATE:
            NFCSend(&buffer[0]);
 	        NFC_State = SELECTED_STATE;
        break;
    }

    return ret;
}

/******************************************************************************
 * @brief  RF Field change handler
 *  Called when the TRF7970 detected an RF Field change
 *  It puts the state machine in idle(Power Off/ Sense) state
 *
 * @return    None
 *****************************************************************************/
void NFC_IRQ_RF(void)
{
    uint8_t  command;

    //RADIO_DIRECTCOMMAND(Reset, &command);
    // Reset FIFO with an extra dummy clock after last byte has been read out
	command = 0x00;
	RADIO_WRITERAW(Reset|BIT7, &command,1);
    ResetDecoders();
    u8InterruptFlags = INT_CLEAR;
    if (NFC_State != SENSE_STATE)
    {
        NFC_State = POWER_OFF_STATE;
        u8InterruptFlags = INT_DISCONNECT;
    }
}

/******************************************************************************
 * @brief  Resets the TRF7970 decoders
 *
 * @return    None
 *****************************************************************************/
void ResetDecoders(void)
{
	    RADIO_DIRECTCOMMAND(StopDecoders, buffer); RADIO_DIRECTCOMMAND(RunDecoders, buffer);
}

/******************************************************************************
 * @brief  RX handler
 *  Called when the TRF7970 detects a data Reception
 *  It puts the data in buffers which will be passed to upper layers
 *
 * @param Fifo_Int 1: Interrupted by FIFO, 0: RX complete
 *
 * @return    None
 *****************************************************************************/
void  NFC_IRQ_RX(uint8_t Fifo_Int)
{
    uint8_t Reg;
    uint8_t command[3];
    uint8_t mode;
    extern uint8_t buffer[];

	RADIO_READSINGLE(NFCTargetProtocol, &Reg);
	mode = Reg&0x3F; // Get the Current NFC mode

	RADIO_READSINGLE(FIFOStatus, &Reg); 		//determine the number of bytes left in FIFO
	Reg = 0x7F & Reg;

	// Get all bytes from FIFO and put them in the buffer
	RADIO_READCONT(FIFO, buf_ptr, Reg);

	buf_ptr += Reg; // Increment pointer register

	if (mode == 0x13)
	{
		// We are waiting for NFC-F @ 424

		// Clear the busy flag
		if (!Fifo_Int)
		{
			//RADIO_DIRECTCOMMAND(Reset, &Reg);
			// Reset FIFO with an extra dummy clock
			Reg = 0x00;
			RADIO_WRITERAW(Reset|BIT7, &Reg,1);
			RADIO_READSINGLE(IRQStatus, command);
			u8InterruptFlags = INT_RX_COMPLETE;
		}
		else
		{
			//Check if FIFO is empty
			RADIO_READSINGLE(FIFOStatus, &Reg);
			Reg = 0x7F & Reg;
			if (Reg == 0x00)
			{
				// If FIFO is empty, signal completion
				//RADIO_DIRECTCOMMAND(Reset, &Reg);
				// Reset FIFO with an extra dummy clock
				Reg = 0x00;
				RADIO_WRITERAW(Reset|BIT7, &Reg,1);
				RADIO_READSINGLE(IRQStatus, command);
				u8InterruptFlags = INT_RX_COMPLETE;
			}
		}
	}
}

/******************************************************************************
 * @brief  TX handler
 *  Called when the TRF7970 detects a Transmission complete
 *
 * @return    None
 *****************************************************************************/
void  NFC_IRQ_TX(void)
{
    uint8_t Reg;
    //reset the FIFO after last byte has been read out
    //RADIO_DIRECTCOMMAND(Reset, &Reg);
    // Reset FIFO with an extra dummy clock
	Reg = 0x00;
	RADIO_WRITERAW(Reset|BIT7, &Reg,1);
    u8InterruptFlags = INT_TX_COMPLETE;
}

/******************************************************************************
 * @brief  Protocol Error handler
 *  Called when the TRF7970 detects a protocol error
 *
 * @return    None
 *****************************************************************************/
void NFC_IRQ_COLL_AVOID_FINISH(void)
{
    u8InterruptFlags = INT_CLEAR;
}


//
// Private functions
//
/******************************************************************************
 * @brief   Initializes the radio at the beggining of state machine
 * Puts the radio in target passive mode @ 106kbps with the default NFC ID
 *
 * @return  none
 *****************************************************************************/
static void NFC_Target_InitLoop(void)
{
    uint8_t i;

    OOK_DIR_IN();

    //buffer[0] = 0x00; //OLD
    buffer[0] = 0x80; //direct command idle
    RADIO_WRITERAW(SoftInit|BIT7, &buffer[0],1);

    buffer[0] = 0x21;
    RADIO_WRITESINGLE(ModulatorControl, buffer);

    buffer[0] = 0x23;			// 424kbps  NFC Passive Target mode
    RADIO_WRITESINGLE(ISOControl, buffer);

    buffer[0] = 0x21;
    RADIO_WRITESINGLE(ChipStateControl, buffer);

    // Set regulator level
    buffer[0] = 0x01;
    RADIO_WRITESINGLE(RegulatorControl, buffer);

    buffer[0] = BIT2|BIT1;
    RADIO_WRITESINGLE(NFCTargetLevel, buffer);

    buffer[0] = BIT0;//0x01;
    RADIO_WRITESINGLE(NFCLowDetection, buffer);

    for (i=0; i < 10; i++)
        buffer[i] = TargetCID[i];
    RADIO_WRITECONT(NFCID, &buffer[0], 10);

    buffer[0] = 0x0f;
    RADIO_WRITESINGLE(FIFOIRQLevel, buffer);

    buffer[0] = 0x1C;
    RADIO_WRITESINGLE(RXSpecialSettings, buffer);

    //IRQ status register address
    RADIO_READCONT(IRQStatus, buffer,2);

    //RADIO_DIRECTCOMMAND(Reset, buffer);
    // Reset FIFO with an extra dummy clock
	buffer[0] = 0x00;
	RADIO_WRITERAW(Reset|BIT7, &buffer[0],1);

    ResetDecoders();
}

/******************************************************************************
 * @brief   Processes the raw received data
 *  This limited NFC implementation can only handle ATR_REQ and DEP_REQ
 *
 * @param buffer pointer to data buffer
 * @return    DATA_RECEIVED         A packet has been received and can be processed
 *             RX_PACKET_ONGOING     Reception in progress
 *             RET_ERR               Error during reception
 *****************************************************************************/
int8_t NFC_ProcessReceivedData(uint8_t * data)
{
	uint8_t * resp;
    uint16_t command;
    uint8_t i;
    int8_t ret = RX_PACKET_ONGOING;
    uint8_t response[50];
    uint8_t slots;
    uint8_t DSAP, PTYPE, SSAP;

    command = (data[1]<<8) + (data[2]);

    switch (command)
    {
        case SENSF_REQ:
        /* SENSF_REQ:
         | 0     | 1  | 2:3   |  4  |  5  |
         | LEN:6 | 00 | FFFF  | RC  | TSN |
          SENSF_RES:
         | 0        | 1:2  | 3:12   |  13  |  14 | 15  | 16  |17   | 18+     |

      */
        slots = data[5];
        if(data[4] == 0) //RC = 0
        	response[0] = 18; // Length
        else //RC = 1
        	response[0] = 20; // Length
		response[1] = 0x01;
		for(i = 0; i < 8; i++){
			response[2 + i] = TargetCID[i];
		}
		response[10] = 0xc0;
		response[11] = 0xc1;
		response[12] = 0xc2;
		response[13] = 0xc3;
		response[14] = 0xc4;
		response[15] = 0xc5;
		response[16] = 0xc6;
		response[17] = 0xc7;
		if(data[4] == 1) //If RC = 1, provide system info
		{
			response[18] = 0x0F;
			response[19] = 0xAB;
		}

        __delay_cycles(19200);   //2.4ms


        for(i = 0; i < (slots + 1); i++){
            if(i == 0){			//send the reply in the calculated timeslot
                {
                    NFCSend(&response[0]);
                    P1DIR |= BIT0;
                    P1OUT ^= BIT0;
                     NFC_State = PRESELECTED_STATE; // If Ok, move to selected state
                    break;
                }
            }
            __delay_cycles(9600);  //1.2ms
        }

        break;
        case ATR_REQ:
       /* ATR_REQ:
         | 0        | 1:2  | 3:12   |  13  |  14 | 15  | 16  | 17+     |
         | LEN:17+n | D400 | NFCIDi | DIDi | BSi | BRi | PPi | Gi[0:n] |
          ATR_RES:
         | 0        | 1:2  | 3:12   |  13  |  14 | 15  | 16  |17   | 18+     |
         | LEN:17   | D501 | NFCIDt | DIDt | BSt | BRt | TO  | PPt | Gi[0:n] |
      */
            response[0] = 31; // Length
            // ATR_RES command
            response[1] = (ATR_RES&0xFF00)>>8;
            response[2] = (ATR_RES&0x00FF);

            //if ((data[3] != TargetCID[0]) || (data[4] != TargetCID[1]))
               // return RET_ERR;
            for(i = 0; i < 8; i++)
            {
            	if(data[3 + i] != TargetCID[i])
            		return RET_ERR;
            }

            for(i = 0; i < 10; i++){
				response[3 + i] = TargetCID[i];
			}
           response[13] = 0x00;            // DIDt is same as DIDi
           response[14] = data[14];            // BSt is same as BSi
           response[15] = data[15];            // BRt is same as BRi
            // TO
           response[16] = TIMEOUT_TARGET;
           response[17] = 0x32;     // PPt is same as PPi without Gi

			response[18] = 0x46;    // LLCP Magic #
			response[19] = 0x66;
			response[20] = 0x6D;

			response[21] = 0x01;    // Ver 1.0
			response[22] = 0x01;
			response[23] = 0x10;

			response[24] = 0x03;    // Services
			response[25] = 0x02;
			response[26] = 0xFF;
			response[27] = 0xFF;

			response[28] = 0x04;
			response[29] = 0x01;
			response[30] = 0x96;

			__delay_cycles(19200);   //2.4ms


        for(i = 0; i < (slots + 1); i++){
            if(i == 0){			//send the reply in the calculated timeslot
                {
                    NFCSend(&response[0]);
                    P1DIR |= BIT0;
                    P1OUT ^= BIT0;
                     NFC_State = SELECTED_STATE; // If Ok, move to selected state
                    break;
                }
            }
            __delay_cycles(9600);
        }

        break;
        case DEP_REQ:
       /* DEP_REQ:
         | 0        | 1:2  | 3    |
         | LEN      | D406 | PFB  |
          LLCP:
         | 4:5                | 6     | 7       | 8:22            |
         | DSAP, PTYPE, SSAP  | Type  | Length  | Value           |
         | 0521               | 06    | 0F      | Service Name    |
         | SDP, CONNECT, 21   | SN    | n       | com.android.npp |


          DEP_RES:
         | 0     | 1:2  | 3   |
         | LEN   | D507 | PFB |
          LLCP:
         | 4:5                | 6     | 7       | 8:22            |
         | DSAP, PTYPE, SSAP  | Type  | Length  | Value           |
         | 05A1               | 06    | 0F      | Service Name    |
         | SDP, CC, 21   | SN    | n       | com.android.npp |
      */
        	// Check PFB  type
            if ((buffer[3]&0xE0) == 0x00) //information PDU
            {

              DSAP = (buffer[4] & 0xFC) >> 2; //DSAP = first 6 bits of first PDU byte
              SSAP = (buffer[5] & 0x3F); //SSAP = last 6 bits of second PDU byte
              PTYPE = ((buffer[4] & 0x03) << 2) | ((buffer[5] & 0xC0) >> 6);//4 bits between DSAP and SSAP

              //if(buffer[4] == 0x05 && buffer[5] == 0x21) //Receive CONNECT, reply CC
              if(PTYPE == 0x04 && DSAP != 0x04)  //Receive CONNECT (not SNEP), reply CC
              {
            	resp = Android_CONNECT_received(buffer, DSAP, SSAP);
              }
              //else if (buffer[4] == 0 && buffer[5] == 0) //receive SYMM, reply CONNECT
              else if (DSAP == 0 & PTYPE == 0 && SSAP == 0) //receive SYMM, reply CONNECT
              {
                resp = Android_SYMM_received(buffer);
              }
              //else if( buffer[4] == 0x85 && buffer[5] == 0x90 ) //receive CC, reply I
              else if (PTYPE == 0x06) //receive CC, reply I
              {
                resp = Android_CC_received(buffer);
              }
              //else if(buffer[4] == 0x43 && buffer[5] == 0x21) //receive I, reply SYMM
              else if(PTYPE == 0x0C) //receive I, reply SYMM
              {
            	resp = Android_I_received(buffer);
              }
              //else if(buffer[4] == 0x11 && buffer[5] == 0x20) //phone is attempting SNEP connection, reply DM reason 3
              else if(PTYPE == 0x04 && DSAP == 0x04) //phone is attempting SNEP connection, reply DM reason 3
              {
                resp = Android_SNEP_received(buffer);
              }
              else if(PTYPE == 0x0D) //receive RR, reply DISC
			  {
			    resp = Android_RR_received(buffer);
			  }
              else{
                __no_operation();
              }


              for(i = 0; i < (slots + 1); i++){
                    if(i == 0){			//send the reply in the calculated timeslot
                        {
                            NFCSend(resp);
                            P1DIR |= BIT0;
                            P1OUT ^= BIT0;
                             //NFC_State = SELECTED_STATE; // If Ok, move to selected state
                            break;
                        }
                    }
                    __delay_cycles(9600);
                }
            }
            else{
            	if(buffer[3] == 0x80) //ATN PDU received, send ATN RES
            	{
            		//Android_ATN_received(buffer);
            	}
              __no_operation();
            }
            // else ignore
        break;
        default:
        __no_operation();
        break;
    }

    return ret;
}

/******************************************************************************
 * @brief  Sends data to the Initiator
 *  An initiator is supposed to be waiting for a passive target response here
 *  No Collision Avoidance implemented
 *
 * @param pbuf Pointer to buffer with data being sent
 *
 * @return RET_OK: Transmission successful, RET_ERR: Error during TX
 *****************************************************************************/
static int8_t NFCSend(uint8_t  *pbuf)
{
	uint8_t total_length, length, i, Register[3];
	int8_t ret = RET_OK;

    u8InterruptFlags = INT_BUSY;

	//NFCstate = *pbuf;	//NFCstate global variable is the main transmit counter
	total_length = *pbuf;
    length = total_length;
	if(length > FIFO_LEN)
		length = FIFO_LEN;

    // First, clear FIFO with a reset
    tx_buf[0] = TransmitCRC | BIT7;  // TX command
    tx_buf[1] = TXLenghtByte1 | BIT5;   // Write cont TXLength
    tx_buf[2] = (uint8_t)(total_length >> 4);   // buf[3] = LEN LSn
    tx_buf[3] = (uint8_t)(total_length << 4);   // buf[4] = LEN MSn

    for (i=0; i < length ; i++)
         tx_buf[i+4] = pbuf[i];

    RADIO_WRITERAW(Reset | BIT7, tx_buf, length+4);

    // Adjust counters
	total_length -= length;
	pbuf += length;

    do{
        // Wait for TX interrupt
        u8InterruptFlags = INT_BUSY;
        do{
            Radio_Wait_IRQ();
        }while(u8InterruptFlags & INT_BUSY);

        if ((u8InterruptFlags & INT_TX_COMPLETE) &&  (total_length > 0))
        {
            if (total_length > FIFO_MORE)
                length = FIFO_MORE;
            else
                length = total_length;

            // Send next packet
            for (i=0; i < length ; i++)
                tx_buf[i] = pbuf[i];
            RADIO_WRITECONT(FIFO, &tx_buf[0], length);

            total_length -= length;
	        pbuf += length;
        }
        else if (u8InterruptFlags & INT_CRITICAL_ERROR)
        {
            ret = RET_ERR;
	    	buffer[0] = 0x01;
            RADIO_WRITESINGLE(ChipStateControl, buffer);
            total_length = 0;   // exit loop
        }

    }while(total_length > 0);

    RADIO_READCONT(IRQStatus, Register,2);

	return(ret);
}
