/** @file   rfid_hal.c
 *  @brief  Driver interfacing with TRF79xx radio
 *  @author LR
 *  @date   Feb 2011
 *  @note   History:
 *          Date        Author      Comment
 *          02/2011     LR          File created and added to GIT
 *  Built with IAR Embedded Workbench Version: 5.20
 *
 */
/* --COPYRIGHT--,BSD
 * Copyright (c) 2011, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//
// Include files
//
#include "RFID_hardware.h"
#include "rfid_hal.h"
#include "nfc.h"

//
// Public Functions
//
/******************************************************************************
 * @brief   Sets the default settings for the RFID radio
 * Performs a soft reset and wait for completion
 *
 * @return  none
 *****************************************************************************/
void Radio_InitialSettings(void)
{
    unsigned char  command[2];

    RADIO_DIRECTCOMMAND(SoftInit, &command[0]);
    DELAY_MS(20);
}

/******************************************************************************
 * @brief   Sends a command/data to the radio
 *  The expected formats are:
 *  single Write:
 *  Format: Start-AdrX-Datax-Stop
 *  Continuous Write:
 *  Format: Start-AdrX-Datax-Data(x+1)-Data(x+2)-...-Data(x+n)-Stop
 *  Single Read:
 *  Format: Start-AdrX-Datax-Stop
 *      Buffer Before:   X
 *      Buffer After:    DataX
 *  Continous Read:
 *  Format: Start-AdrX-Datax-Data(x+1)-Data(x+2)-...-Data(x+n)-Stop
 *      Buffer Before:   XXXXXXX
 *      Buffer After:    DataX-Data(x+1)-Data(x+n)
 *  Command:
 *  Start-CmdX-optional Data-Stop
 *
 * @param mode_addr Start register address and mode:
 *                  BIT7: Command Control bit
 *                  BIT6: R/W
 *                  BIT5: Continuous mode
 *                  BIT4:BIT0 Address/command
 * @param pbuf Pointer to the buffer with data and where data will be returned
 * @param length Lenght of the data in buffer
 *
 * @return  none
 * @note this function is mostly called through Macros defined in rfid_hal.h
 *****************************************************************************/
void Radio_Transfer(uint8_t mode_addr, uint8_t *pbuf, uint8_t length)
{
    uint16_t i;
    uint8_t *txbuf, *rxbuf;

    for (i=length; i > 0; i--)
    {
        if (!(mode_addr & BIT6))    // If writing, readjust buff
           pbuf[i] = pbuf[i-1];
    }
    pbuf[0] = mode_addr;

    rxbuf = txbuf = pbuf; // Initialize pointer

    SLAVESELECT_LOW(); //Start SPI Mode

    (void) USCISPITransfer(*txbuf++);
    USCISPIWaitBus();

    while(length > 0)
    {
        *rxbuf++ = USCISPITransfer(*txbuf++);
        length--;
    }
    USCISPIWaitBus();
    SLAVESELECT_HIGH(); //Stop SPI Mode

}


/******************************************************************************
 * @brief Waits for a radio IRQ
 * This function will lock the mcu while waiting for IRQ. Note that the
 * microcontroller is not expected to do anything else in BSL
 *
 * @return  none
 *****************************************************************************/
void Radio_Wait_IRQ(void)
{
    uint8_t Register[4];

    // Wait for an IRQ forever
    while (!IRQ_IS_SET())
        ;

    do
    {
         RADIO_READCONT(IRQStatus, &Register[0],2);    /* IRQ status register address */
         Radio_Process_IRQ(Register[0]);

    }while (IRQ_IS_SET());

    return;
}

/******************************************************************************
 * @brief Process a TRF797x IRQ
 * Called when an IRQ is detected, calls the appropiate function from NFC layer
 *
 * @param IRQ_Stat value read from IRQ register
 *
 * @return  none
 *****************************************************************************/
void Radio_Process_IRQ(uint8_t IRQ_Stat)
{

     if (IRQ_Stat & BIT2)
           NFC_IRQ_RF();
     else if (IRQ_Stat & BIT5)
    	 __no_operation();
    else if (IRQ_Stat & BIT7)
    {
         NFC_IRQ_TX();
    }
    else if (IRQ_Stat & BIT6)
    {
        // We are waiting for a FeliCa 424
        // Reception detected, pass if FIFO RX or end of RX
         NFC_IRQ_RX(IRQ_Stat & BIT5);
        //NFC_ProcessReceivedData(&buffer[0]);
    }
    else if (IRQ_Stat & BIT1)
    {
         NFC_IRQ_COLL_AVOID_FINISH();
    }
}

