/** @file   rfid_hal.h
 *  @brief  Header file for driver interfacing with TRF79xx radio
 *  @author LR
 *  @date   Feb 2011
 *  @note   History:
 *          Date        Author      Comment
 *          02/2011     LR          File created and added to GIT
 *  Built with IAR Embedded Workbench Version: 5.20
 *
 */
/* --COPYRIGHT--,BSD
 * Copyright (c) 2011, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/

#ifndef __RFID_HAL_H
#define __RFID_HAL_H


//
// Macro definitions
//
/*! No flags set */
#define INT_CLEAR           0
/*! Module is busy waiting for IRQ */
#define INT_BUSY            BIT0
/*! RX detected and in buffer */
#define INT_RX_COMPLETE     BIT1
/*! Critical error during RFID processing */
#define INT_CRITICAL_ERROR  BIT2
/*! Communication error (protocol error) */
#define INT_COMM_ERROR      BIT7
/*! TX Completed */
#define INT_TX_COMPLETE     BIT3
/*! RF Field not detected */
#define INT_DISCONNECT      BIT6

/*! Buffer length (Max Number of characters in a frame) */
#define BUF_LENGTH 300
/*! TRF7970 FIFO Lentgh */
#define FIFO_LEN	50
/*! TRF7970 FIFO size when adding more data */
#define FIFO_MORE	26

//
// TRF7970 Register addresses
//
#define ChipStateControl        0x00
#define ISOControl              0x01
#define ISO14443Boptions        0x02
#define ISO14443Aoptions        0x03
#define TXtimerEPChigh          0x04
#define TXtimerEPClow           0x05
#define TXPulseLenghtControl    0x06
#define RXNoResponseWaitTime    0x07
#define RXWaitTime              0x08
#define ModulatorControl        0x09
#define RXSpecialSettings       0x0A
#define RegulatorControl        0x0B
#define IRQStatus               0x0C
#define IRQMask                 0x0D
#define CollisionPosition       0x0E
#define RSSILevels              0x0F
#define RAMStartAddress		    0x10
#define NFCLowDetection		    0x16
#define FIFOIRQLevel		    0x14
#define NFCID                   0x17
#define NFCTargetLevel          0x18
#define NFCTargetProtocol       0x19
#define TestSetting1            0x1A
#define TestSetting2            0x1B
#define FIFOStatus              0x1C
#define TXLenghtByte1           0x1D
#define TXLenghtByte2           0x1E
#define FIFO                    0x1F

//
// TRF7970 Commands
//
#define Idle                    0x00
#define SoftInit                0x03
#define InitialRFCollision      0x04
#define ResponseRFCollisionN    0x05
#define ResponseRFCollision0    0x06
#define  Reset                  0x0F
#define TransmitNoCRC           0x10
#define TransmitCRC             0x11
#define DelayTransmitNoCRC      0x12
#define DelayTransmitCRC        0x13
#define TransmitNextSlot        0x14
#define CloseSlotSequence       0x15
#define StopDecoders            0x16
#define RunDecoders             0x17
#define ChectInternalRF         0x18
#define CheckExternalRF         0x19
#define AdjustGain              0x1A

//
// MACRO-like functions
//
/******************************************************************************
 * @brief Writes a single register
 * @param a Register address
 * @param b Pointer to Buffer with data
 * @return  none
 *****************************************************************************/
#define RADIO_WRITESINGLE(a, b)     Radio_Transfer((a&0x1F), b, 1)
/******************************************************************************
 * @brief Reads a single register
 * @param a Register address
 * @param b Pointer to Buffer with data to send and where data will be stored
 * @return  none
 *****************************************************************************/
#define RADIO_READSINGLE(a, b)      Radio_Transfer(((a&0x1F)|BIT6), b, 1)
/******************************************************************************
 * @brief Writes continous registers
 * @param a Base Register address
 * @param b Pointer to Buffer with data
 * @param l lentgh
 * @return  none
 *****************************************************************************/
#define RADIO_WRITECONT(a, b, l)    Radio_Transfer(((a&0x1F)|BIT5), b, l)
/******************************************************************************
 * @brief Reads continous registers
 * @param a Register address
 * @param b Pointer to Buffer with data to send and where data will be stored
 * @param l lentgh
 * @return  none
 *****************************************************************************/
#define RADIO_READCONT(a, b, l)     Radio_Transfer(((a&0x1F)|(BIT5|BIT6)), b, l)
/******************************************************************************
 * @brief Sends a command to the radio
 * @param a Register address
 * @param b Pointer to Buffer with additional data
 * @return  none
 *****************************************************************************/
#define RADIO_DIRECTCOMMAND(a, b)    Radio_Transfer(((a&0x1F)|BIT7), b, 0)
/******************************************************************************
 * @brief Performs a raw transfer
 * @param a Register address including mode and format
 * @param b Pointer to Buffer with data to send and where data will be stored
 * @param l lentgh
 * @return  none
 *****************************************************************************/
#define RADIO_WRITERAW(a, b, l)     Radio_Transfer(a, b, l)

//
// Function prototypes
//
void Radio_InitialSettings(void);
void Radio_Transfer(uint8_t mode_addr, uint8_t *pbuf, uint8_t length);
void Radio_Wait_IRQ(void);
void Radio_Process_IRQ(uint8_t IRQ_Stat);

#endif      // RFID_HAL_H
