#!/usr/bin/python

import wx
from math import fmod
from numpy import mod, matrix
from pyauto.coherentsamp import CoherentSamp
from pyauto.const import NUM_SAMPLES_CHOICES, GPIB_TYPE_CHOICES, DATA_FORMAT_CHOICES, DATA_COLUMNS_CHOICES, \
                         DATA_WINDOW_CHOICES, WV5_DATA_FILE, FIGURES, \
                         FFTYLIM, FFTTITLE, FFTXLABEL, FFTYLABEL, \
                         HISTTITLE, HISTXLABEL, HISTYLABEL, \
                         DNLTITLE, DNLXLABEL, DNLYLABEL, INLTITLE, INLXLABEL, INLYLABEL
from pyauto.hardware_com import write_siggen
from pyauto.data_analysis import adc_spectrum, adc_2tspectrum, compute_dnl_inl_from_histogram, compute_histogram_from_data, \
                                 compute_afft_from_data
from pyauto.file_io import print_status, read_data_xy, read_data_y, save_xy_file, print_traceback_if_debug, message_alert

from ctypes import windll, pointer, POINTER, byref, c_ulong, c_int, c_uint, c_double, c_char_p, \
                   create_string_buffer, Structure

import os
os.environ['PATH'] = os.path.curdir+'\\wavevision5' + ';' + os.environ['PATH'] # Add folder to path to find DLL
wv5dll = windll.wvdll
user32 = windll.user32

class CoherencyTool(wx.Frame):
    def __init__(self, *args, **kwargs):
        mFrame = wx.Frame.__init__(self, *args, **kwargs)
        self.fill_panel()
        self.clk_freq_input.SetValue('155')
        self.sig_freq_input.SetValue('70')
        self.sig_amp_input.SetValue('0')
        self.num_samples_options.SetValue('32768')
        self.gpib_address_input.SetValue('20')
        self.gpib_type_combo.SetValue(GPIB_TYPE_CHOICES[0])
        self.write_coh_check.SetValue(True)
        self.find_coh_freq(wx.Event)

    def fill_panel(self):
        self.panel_label = wx.StaticText(self, label="Write a coherent signal to the SigGen:")
        font = wx.Font(9, wx.NORMAL, wx.NORMAL, wx.BOLD)
        self.panel_label.SetFont(font)
        self.clk_freq_label     = wx.StaticText(self, label="Clock Frequency (MHz):")
        self.clk_freq_input     = wx.TextCtrl(self, size=(60,20))
        self.sig_freq_label     = wx.StaticText(self, label="Signal Frequency (MHz):")
        self.sig_freq_input     = wx.TextCtrl(self, size=(60,20))
        self.sig_amp_label      = wx.StaticText(self, label="Signal Amplitude (dBm):")
        self.sig_amp_input      = wx.TextCtrl(self, size=(60,20))
        self.num_samples_label  = wx.StaticText(self, -1, "# of Samples ")
        num_samples_choices     = NUM_SAMPLES_CHOICES
        self.num_samples_options = wx.ComboBox(self, -1, value=num_samples_choices[3], choices=num_samples_choices, \
                style=wx.CB_READONLY)
        self.coh_freq_label     = wx.StaticText(self, label="Coherent Signal Frequency (MHz):")
        self.coh_freq_input     = wx.TextCtrl(self, size=(60,20), style=wx.TE_READONLY)
        self.gpib_address_label = wx.StaticText(self, label="GPIB address: ")
        self.gpib_address_input = wx.TextCtrl(self, size=(30,21))
        self.gpib_type_combo = wx.ComboBox(self, -1, value=GPIB_TYPE_CHOICES[0], choices=GPIB_TYPE_CHOICES, \
                style=wx.CB_READONLY)
        self.gpib_write_button  = wx.Button(self, -1, "GPIB Write")
        self.write_coh_check    = wx.CheckBox(self, -1, "Write Coherent Frequency?")

        ### Grid Bag Sizer to Organize Panel
        self.top_sizer = wx.GridBagSizer(vgap=5,hgap=5)
        self.top_sizer.Add(self.panel_label,         (1,1), (1,3), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.clk_freq_label,      (2,1), (1,2), flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.clk_freq_input,      (2,3),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.sig_freq_label,      (3,1), (1,2), flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.sig_freq_input,      (3,3),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.sig_amp_label,       (4,1), (1,2), flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.sig_amp_input,       (4,3),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.num_samples_label,   (5,1), (1,2), flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.num_samples_options, (5,3),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.coh_freq_label,      (6,1), (1,2), flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.coh_freq_input,      (6,3),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.gpib_address_label,  (7,1),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.gpib_address_input,  (7,2),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.gpib_type_combo,     (7,3),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.write_coh_check,     (8,1), (1,2), flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.gpib_write_button,   (8,3),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.AddGrowableRow(9)
        self.top_sizer.AddGrowableCol(4)
        self.SetSizer(self.top_sizer)

        ### Event Bindings
        self.Bind(wx.EVT_BUTTON, self.gpib_write_button_func, id = self.gpib_write_button.GetId())
        self.Bind(wx.EVT_TEXT, self.find_coh_freq, id = self.clk_freq_input.GetId())
        self.Bind(wx.EVT_TEXT, self.find_coh_freq, id = self.sig_freq_input.GetId())
        self.Bind(wx.EVT_COMBOBOX, self.find_coh_freq, id = self.num_samples_options.GetId())

    def gpib_write_button_func(self, event):
        amp = float(self.sig_amp_input.GetValue())
        gpib_addr = int(self.gpib_address_input.GetValue())
        gpib_type = self.gpib_type_combo.GetValue()
        if self.write_coh_check.GetValue() == True:
            freq = 1e6*float(self.coh_freq_input.GetValue())
        else:
            freq = 1e6*float(self.sig_freq_input.GetValue())

        try:
            gpib_write_mask = {'FREQ':    freq, \
                               'AMP':     amp, \
                               'GPIB':    gpib_addr, \
                               'GENTYPE': gpib_type}
            write_siggen(gpib_write_mask)
        except:
            alert = wx.MessageDialog(self, "Error Writing to GIPB", style = wx.OK)
            response = alert.ShowModal()
            alert.Destroy()
        return


    def find_coh_freq(self, event):
        if (self.clk_freq_input.GetValue() == '') or (self.sig_freq_input.GetValue() == ''):
            return
        try:
            freq_clk_Hz = float(self.clk_freq_input.GetValue())*1e6
            freq_sig_Hz = float(self.sig_freq_input.GetValue())*1e6
            N =           int(self.num_samples_options.GetValue())
            freq_sig_coh_Hz = CoherentSamp().setCohSampFreq(freq_sig_Hz, freq_clk_Hz, N)
        except:
            print_status('Illegal data type')
        else:
            self.coh_freq_input.SetValue(`freq_sig_coh_Hz/1e6`)
        return


### Create a class for the FFT control Panel to encapsulate its functionality and layout
class AnalysisTools(wx.Frame):

    def __init__(self, *args, **kwargs):
        mFrame = wx.Frame.__init__(self, *args, **kwargs)
        self.fill_panel()
        self.set_input_mask({'BITS':'14','WINDOW':DATA_WINDOW_CHOICES[2],'HARM':'7','IMD':'3',
                             'DCEXCL':'6','FUNDEXCL':'6','HARMEXCL':'3',
                             'SAMP':NUM_SAMPLES_CHOICES[5],
                             'FORMAT':DATA_FORMAT_CHOICES[0],'COLUMNS':DATA_COLUMNS_CHOICES[0],
                             'FILE':WV5_DATA_FILE, 'FIG':FIGURES[0]})
        self.figurePanel = wx.GetApp().frame.mPanel.figure_panel

    def fill_panel(self):
        ### Create Panel Objects
        self.panel_label = wx.StaticText(self, -1, "Choose data file, select settings, and plot analysis:")
        font = wx.Font(9, wx.NORMAL, wx.NORMAL, wx.BOLD)
        self.panel_label.SetFont(font)
        self.num_bits_label      = wx.StaticText(self, -1, "# ADC Bits: ")
        self.num_bits_input      = wx.TextCtrl(self, size=(50,20))
        self.data_window_label   = wx.StaticText(self, -1, "Data Window: ")
        self.data_window_options = wx.ComboBox(self, -1, value=DATA_WINDOW_CHOICES[0], choices=DATA_WINDOW_CHOICES, \
                style=wx.CB_READONLY)
        self.harmonics_label     = wx.StaticText(self, -1, "# of Harmonics: ")
        self.harmonics_input     = wx.TextCtrl(self, size=(50,20))
        self.intermods_label     = wx.StaticText(self, -1, "Order of Intermods: ")
        self.intermods_input     = wx.TextCtrl(self, size=(50,20))

        self.num_samples_label   = wx.StaticText(self, -1, "# of Samples ")
        self.num_samples_options = wx.ComboBox(self, -1, value=NUM_SAMPLES_CHOICES[0], choices=NUM_SAMPLES_CHOICES, \
                style=wx.CB_READONLY)
        self.data_format_label   = wx.StaticText(self, -1, "Data Format ")
        self.data_format_options = wx.ComboBox(self, -1, value=DATA_FORMAT_CHOICES[0], choices=DATA_FORMAT_CHOICES, \
                style=wx.CB_READONLY)
        self.data_columns_label  = wx.StaticText(self, -1, "Data Columns ")
        self.data_columns_options = wx.ComboBox(self, -1, value=DATA_COLUMNS_CHOICES[0], choices=DATA_COLUMNS_CHOICES, \
                style=wx.CB_READONLY)
        self.dc_excl_label       = wx.StaticText(self, -1, "DC Excl Bins: ")
        self.dc_excl_input       = wx.TextCtrl(self, size=(50,20))
        self.fund_excl_label     = wx.StaticText(self, -1, "Fund Excl Bins: ")
        self.fund_excl_input     = wx.TextCtrl(self, size=(50,20))
        self.harm_excl_label     = wx.StaticText(self, -1, "Harm Excl Bins: ")
        self.harm_excl_input     = wx.TextCtrl(self, size=(50,20))
        self.file_label = wx.StaticText(self, -1, "File: ")
        self.file_input = wx.TextCtrl(self, size=(185,20))
        self.browse_file_button  = wx.Button(self, -1, "Browse")
        self.figures_label       = wx.StaticText(self, -1, "Plot to: ")
        self.figures_options     = wx.ComboBox(self, -1, value=FIGURES[0], choices=FIGURES, \
                style=wx.CB_READONLY)
        self.plot_data_XY_button = wx.Button(self, -1, "Plot Data XY")
        self.plot_data_Y_button  = wx.Button(self, -1, "Plot Data Y")
        self.data2fft_button     = wx.Button(self, -1, "Data --> FFT")
        self.data22tfft_button   = wx.Button(self, -1, "Data --> 2T FFT")
        self.data2hist_button    = wx.Button(self, -1, "Data --> HIST")
        self.data2afft_button    = wx.Button(self, -1, "Data --> AFFT")
        self.data2dnlinl_button  = wx.Button(self, -1, "Data --> DNLINL")

        ### Grid Bag Sizer to Organize Panel
        self.top_sizer = wx.GridBagSizer(vgap=5,hgap=5)
        self.top_sizer.Add(self.panel_label,         (0,1), (1,8), flag=wx.ALIGN_LEFT|wx.EXPAND, border=10)
        self.top_sizer.Add(self.num_bits_label,      (1,1), flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.num_bits_input,      (1,2), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.data_window_label,   (2,1), flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.data_window_options, (2,2), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.harmonics_label,     (3,1), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.harmonics_input,     (3,2), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.intermods_label,     (4,1), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.intermods_input,     (4,2), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.dc_excl_label,       (5,1), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.dc_excl_input,       (5,2), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.fund_excl_label,     (6,1), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.fund_excl_input,     (6,2), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.harm_excl_label,     (7,1), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.harm_excl_input,     (7,2), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.num_samples_label,   (8,1), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.num_samples_options, (8,2), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.data_format_label,   (9,1), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.data_format_options, (9,2), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.data_columns_label,  (10,1), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.data_columns_options,(10,2), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.file_label,          (1,4), flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.file_input,          (1,5), (1,3), flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.browse_file_button,  (1,8), flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.figures_label,       (2,4), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.figures_options,     (2,5), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.plot_data_XY_button, (3,5), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.plot_data_Y_button,  (4,5), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.data2fft_button,     (3,6), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.data22tfft_button,   (4,6), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.data2afft_button,    (5,6), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.data2hist_button,    (6,6), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.data2dnlinl_button,  (7,6), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.AddGrowableRow(10)
        self.top_sizer.AddGrowableCol(10)
        self.SetSizer(self.top_sizer)

        ### Event Bindings
        self.Bind(wx.EVT_BUTTON,   self.browse_file_button_bind,  id = self.browse_file_button.GetId())
        self.Bind(wx.EVT_BUTTON,   self.plot_data_XY_button_bind, id = self.plot_data_XY_button.GetId())
        self.Bind(wx.EVT_BUTTON,   self.plot_data_Y_button_bind,  id = self.plot_data_Y_button.GetId())
        self.Bind(wx.EVT_BUTTON,   self.data2fft_button_bind,     id = self.data2fft_button.GetId())
        self.Bind(wx.EVT_BUTTON,   self.data22tfft_button_bind,   id = self.data22tfft_button.GetId())
        self.Bind(wx.EVT_BUTTON,   self.data2afft_button_bind,    id = self.data2afft_button.GetId())
        self.Bind(wx.EVT_BUTTON,   self.data2hist_button_bind,    id = self.data2hist_button.GetId())
        self.Bind(wx.EVT_BUTTON,   self.data2dnlinl_button_bind,  id = self.data2dnlinl_button.GetId())

    ### Binding methods
    def plot_data_XY_button_bind(self, event):
        print_status('Plotting Data...')
        analysis_mask = self.get_input_mask()
        file = self.file_input.GetValue()
        fig = self.figures_options.GetValue()
        self.load_plot_data_XY(file, fig, analysis_mask)

    def plot_data_Y_button_bind(self, event):
        print_status('Plotting Data...')
        analysis_mask = self.get_input_mask()
        file = self.file_input.GetValue()
        fig = self.figures_options.GetValue()
        self.load_plot_data_Y(file, fig, analysis_mask)

    def data2fft_button_bind(self, event):
        print_status('Loading file, Calculating FFT, and Plotting...')
        analysis_mask = self.get_input_mask()
        file = self.file_input.GetValue()
        fig = self.figures_options.GetValue()
        self.load_analyze_plot_fft(file, fig, analysis_mask)

    def data22tfft_button_bind(self, event):
        print_status('Loading file, Calculating 2-tone FFT, and Plotting...')
        analysis_mask = self.get_input_mask()
        file = self.file_input.GetValue()
        fig = self.figures_options.GetValue()
        self.load_analyze_plot_2tfft(file, fig, analysis_mask)

    def data2afft_button_bind(self, event):
        print_status('Calculating and Plotting AFFT...')
        analysis_mask = self.get_input_mask()
        file = self.file_input.GetValue()
        fig = self.figures_options.GetValue()
        self.load_analyze_plot_afft(file, fig, analysis_mask)

    def data2hist_button_bind(self, event):
        print_status('Calculating and Plotting Histogram...')
        analysis_mask = self.get_input_mask()
        file = self.file_input.GetValue()
        fig = self.figures_options.GetValue()
        self.load_analyze_plot_hist(file, fig, analysis_mask)

    def data2dnlinl_button_bind(self, event):
        print_status('Calculating and Plotting DNL...')
        analysis_mask = self.get_input_mask()
        file = self.file_input.GetValue()
        self.load_analyze_plot_dnlinl(file, analysis_mask)

    def browse_file_button_bind(self, event):
        dlg = wx.FileDialog(self, "Choose a file:")
        dlg.SetPath(self.file_input.GetValue())
        if dlg.ShowModal() == wx.ID_OK:
            self.file_input.SetValue(dlg.GetPath())
        dlg.Destroy()

    ### Load/Analysis/Plot methods
    def load_plot_data_XY(self, file, fig, analysis_mask):
        try:
            print_status('Opening file '+file)
            sample,value = read_data_xy(file)
        except:
            print_status('*** Error reading file ***')
            raise
            return False

        ## Change from 2's Complement to binary if requested
        if analysis_mask['FORMAT'] == DATA_FORMAT_CHOICES[1]:
            self.convertTo2sComplement(value, int(analysis_mask['BITS']))

        yspan = (max(value)+min(value))/2
        self.figurePanel.plot_xy(fid     = fig, \
                                 x       = sample, \
                                 y       = value, \
                                 xylim   = [sample[0], sample[len(sample)-1], min(value)-0.2*yspan, max(value)+0.2*yspan], \
                                 title   = 'XY Plot', \
                                 xlabel  = 'X', \
                                 ylabel  = 'Y', \
                                 info    = '')
        return True

    def load_plot_data_Y(self, file, fig, analysis_mask):
        try:
            print_status('Opening file '+file)
            value = read_data_y(file)
            sample = range(0,len(value))
        except:
            print_status('*** Error reading file ***')
            raise
            return False

        ## Change from 2's Complement to binary if requested
        if analysis_mask['FORMAT'] == DATA_FORMAT_CHOICES[1]:
            self.convertTo2sComplement(value, int(analysis_mask['BITS']))

        yspan = (max(value)+min(value))/2
        self.figurePanel.plot_xy(fid     = fig, \
                                 x       = range(0,len(value)), \
                                 y       = value, \
                                 xylim   = [sample[0], sample[len(sample)-1], min(value)-0.2*yspan, max(value)+0.2*yspan], \
                                 title   = 'Y Plot', \
                                 xlabel  = 'Index', \
                                 ylabel  = 'Y', \
                                 info    = '')

        return True

    def load_analyze_plot_fft(self, file, fig, analysis_mask):
        sample = None
        value = None
        try:
            if analysis_mask['COLUMNS'] == DATA_COLUMNS_CHOICES[0]:
                #Extract 2-column data
                print_status('Reading 2-column data')
                sample,value = read_data_xy(file)
            elif analysis_mask['COLUMNS'] == DATA_COLUMNS_CHOICES[1]:
                #Extract 1-column data
                print_status('Reading 1-column data')
                value = read_data_y(file)
                sample = range(0,len(value))
            else:
                print_status('ERROR: Unreadable File')
        except:
            print_status('*** Error reading file ***')
            raise
            return False

        if int(analysis_mask['SAMP']) > len(sample):
            print_status('ERROR: Not enough samples ('+`len(sample)`+') in data file')
            return False
        else:
            sample = sample[0:int(analysis_mask['SAMP'])] # Use only specified nuber of samples
            value = value[0:int(analysis_mask['SAMP'])]

        ## Change from 2's Complement to binary if requested
        if analysis_mask['FORMAT'] == DATA_FORMAT_CHOICES[1]:
            self.convertTo2sComplement(value, int(analysis_mask['BITS']))
        else:
            pass

        results = adc_spectrum(value,
                               int(analysis_mask['BITS']), \
                               analysis_mask['WINDOW'], \
                               int(analysis_mask['HARM']), \
                               int(analysis_mask['DCEXCL']), \
                               int(analysis_mask['FUNDEXCL']), \
                               int(analysis_mask['HARMEXCL']))

        self.figurePanel.plot_xy(fid     = fig, \
                                 x       = range(0,len(results['XmagdBPlot'])), \
                                 y       = results['XmagdBPlot'], \
                                 xylim   = [0,len(results['XmagdBPlot'])/2.0,FFTYLIM[0],FFTYLIM[1]], \
                                 title   = FFTTITLE, \
                                 xlabel  = FFTXLABEL, \
                                 ylabel  = FFTYLABEL, \
                                 info    = results['info'])
        return True


    def load_analyze_plot_2tfft(self, file, fig, analysis_mask):
        sample = None
        value = None
        try:
            if analysis_mask['COLUMNS'] == DATA_COLUMNS_CHOICES[0]:
                #Extract 2-column data
                print_status('Reading 2-column data')
                sample,value = read_data_xy(file)
            elif analysis_mask['COLUMNS'] == DATA_COLUMNS_CHOICES[1]:
                #Extract 1-column data
                print_status('Reading 1-column data')
                value = read_data_y(file)
                sample = range(0,len(value))
            else:
                print_status('ERROR: Unreadable File')
        except:
            print_status('*** Error reading file ***')
            raise
            return False

        if int(analysis_mask['SAMP']) > len(sample):
            print_status('ERROR: Not enough samples ('+`len(sample)`+') in data file')
            return False
        else:
            sample = sample[0:int(analysis_mask['SAMP'])] # Use only specified nuber of samples
            value = value[0:int(analysis_mask['SAMP'])]

        ## Change from 2's Complement to binary if requested
        if analysis_mask['FORMAT'] == DATA_FORMAT_CHOICES[1]:
            self.convertTo2sComplement(value, int(analysis_mask['BITS']))
        else:
            pass

        results = adc_2tspectrum(value,
                               int(analysis_mask['BITS']), \
                               analysis_mask['WINDOW'], \
                               int(analysis_mask['HARM']), \
                               int(analysis_mask['IMD']), \
                               int(analysis_mask['DCEXCL']), \
                               int(analysis_mask['FUNDEXCL']), \
                               int(analysis_mask['HARMEXCL']))

        self.figurePanel.plot_xy(fid     = fig, \
                                 x       = range(0,len(results['XmagdBPlot'])), \
                                 y       = results['XmagdBPlot'], \
                                 xylim   = [0,len(results['XmagdBPlot'])/2.0,FFTYLIM[0],FFTYLIM[1]], \
                                 title   = FFTTITLE, \
                                 xlabel  = FFTXLABEL, \
                                 ylabel  = FFTYLABEL, \
                                 info    = results['info'])
        #self.figurePanel.plot_xy(fid     = fig, \
        #                         x       = range(0,len(results['temp'])), \
        #                         y       = results['temp'], \
        #                         xylim   = [0,len(results['temp'])/2.0, -0.5, 1.5], \
        #                         title   = FFTTITLE, \
        #                         xlabel  = FFTXLABEL, \
        #                         ylabel  = FFTYLABEL, \
        #                         info    = results['info'])
        return True


    def load_analyze_plot_afft(self, file, fig, analysis_mask):
        sample = None
        value = None
        try:
            if analysis_mask['COLUMNS'] == DATA_COLUMNS_CHOICES[0]:
                #Extract 2-column data
                sample,value = read_data_xy(file)
            elif analysis_mask['COLUMNS'] == DATA_COLUMNS_CHOICES[1]:
                #Extract 1-column data
                sample,value = read_data_y(file)
            else:
                pass
        except:
            print_status('*** Error reading file ***')
            raise
            return False

        if fmod(len(sample),int(analysis_mask['SAMP'])) <> 0:
            print_status('ERROR: Incorrect number of samples ('+`len(sample)`+') in data file')
            return False
        else:
            pass

        ## Change from 2's Complement to binary if requested
        if analysis_mask['FORMAT'] == DATA_FORMAT_CHOICES[1]:
            self.convertTo2sComplement(value, int(analysis_mask['BITS']))

        bins, ave_fft = compute_afft_from_data(sample, value, analysis_mask)
        save_xy_file(bins, ave_fft, file+'_afft')
        self.figurePanel.plot_xy(fid     = fig, \
                                 x       = bins, \
                                 y       = ave_fft, \
                                 xylim   = [0,len(bins)/2.0,FFTYLIM[0],FFTYLIM[1]], \
                                 title   = FFTTITLE, \
                                 xlabel  = FFTXLABEL, \
                                 ylabel  = FFTYLABEL, \
                                 info    = '')
        return True

    def load_analyze_plot_hist(self, file, fig, analysis_mask):
        try:
            sample,value = read_data_xy(file)
        except:
            print_status('*** Error reading file ***')
            return False

        ## Change from 2's Complement to binary if requested
        if analysis_mask['FORMAT'] == DATA_FORMAT_CHOICES[1]:
            self.convertTo2sComplement(value, int(analysis_mask['BITS']))

        for i in range(0,len(value)):
            value[i] = int(value[i])
        bits = int(analysis_mask['BITS'])
        code,code_count = compute_histogram_from_data(value,bits)
        save_xy_file(code, code_count, file+'_hist')
        self.figurePanel.plot_xy(fid     = fig, \
                                 x       = code, \
                                 y       = code_count, \
                                 xylim   = [code[0], code[len(code)-1], -10, max(code_count)+10], \
                                 title   = HISTTITLE, \
                                 xlabel  = HISTXLABEL, \
                                 ylabel  = HISTYLABEL, \
                                 info    = '')

        return True

    def load_analyze_plot_dnlinl(self, file, analysis_mask):
        try:
            sample,value = read_data_xy(file)
        except:
            print_status('*** Error reading file ***')
            return False

        ## Change from 2's Complement to binary if requested
        if analysis_mask['FORMAT'] == DATA_FORMAT_CHOICES[1]:
            self.convertTo2sComplement(value, int(analysis_mask['BITS']))

        for i in range(0,len(value)):
            value[i] = int(value[i])
        bits = int(analysis_mask['BITS'])
        try:
            code,code_count = compute_histogram_from_data(value,bits)
            dnl,inl = compute_dnl_inl_from_histogram(code_count)
        except:
            print_traceback_if_debug()
            print_status('ERROR: Histogram, INL, or DNL not computed')
            return False

        save_xy_file(code, code_count, file+'_dnl')
        self.figurePanel.plot_xy(fid    = FIGURES[0], \
                                 x      = range(0,len(dnl)), \
                                 y      = dnl, \
                                 xylim  = [code[0], code[len(code)-1], min(dnl)-0.2, max(dnl)+0.2], \
                                 title  = DNLTITLE, \
                                 xlabel = DNLXLABEL, \
                                 ylabel = DNLYLABEL, \
                                 info   = '')
        save_xy_file(code, code_count, file+'_inl')
        self.figurePanel.plot_xy(fid    = FIGURES[1], \
                                 x      = range(0,len(inl)), \
                                 y      = inl, \
                                 xylim  = [code[0], code[len(code)-1], min(inl)-0.2, max(inl)+0.2], \
                                 title  = INLTITLE, \
                                 xlabel = INLXLABEL, \
                                 ylabel = INLYLABEL, \
                                 info   = '')
        return True

    def convertTo2sComplement(self, x, num_bits):
        for i in range(0,len(x)):
            if x[i] > pow(2.0,(num_bits-1))-1:
                x[i] = (x[i]-pow(2.0,(num_bits-1)))
            else:
                x[i] = (x[i]+pow(2.0,(num_bits-1)))
        return x

    ### Input Mask get/set methods
    def get_input_mask(self):
        bits = self.num_bits_input.GetValue()
        window = self.data_window_options.GetValue()
        harm = self.harmonics_input.GetValue()
        imd = self.intermods_input.GetValue()
        dcexcl = self.dc_excl_input.GetValue()
        fundexcl = self.fund_excl_input.GetValue()
        harmexcl = self.harm_excl_input.GetValue()
        samp = self.num_samples_options.GetValue()
        format = self.data_format_options.GetValue()
        columns = self.data_columns_options.GetValue()
        file = self.file_input.GetValue()
        fig = self.figures_options.GetValue()
        return {'BITS':bits,'WINDOW':window,'HARM':harm,'IMD':imd,'SAMP':samp, \
                'DCEXCL':dcexcl,'FUNDEXCL':fundexcl,'HARMEXCL':harmexcl, \
                'FORMAT':format,'COLUMNS':columns,'FILE':file,'FIG':fig}

    def set_input_mask(self, mask):
        if 'BITS'     in mask: self.num_bits_input.SetValue(mask['BITS'])
        if 'WINDOW'   in mask: self.data_window_options.SetValue(mask['WINDOW'])
        if 'HARM'     in mask: self.harmonics_input.SetValue(mask['HARM'])
        if 'IMD'      in mask: self.intermods_input.SetValue(mask['IMD'])
        if 'DCEXCL'   in mask: self.dc_excl_input.SetValue(mask['DCEXCL'])
        if 'FUNDEXCL' in mask: self.fund_excl_input.SetValue(mask['FUNDEXCL'])
        if 'HARMEXCL' in mask: self.harm_excl_input.SetValue(mask['HARMEXCL'])
        if 'SAMP'     in mask: self.num_samples_options.SetValue(mask['SAMP'])
        if 'FORMAT'   in mask: self.data_format_options.SetValue(mask['FORMAT'])
        if 'COLUMNS'  in mask: self.data_columns_options.SetValue(mask['COLUMNS'])
        if 'FILE'     in mask: self.file_input.SetValue(mask['FILE'])
        if 'FIG'      in mask: self.figures_options.SetValue(mask['FIG'])
        return

class SPITool(wx.Frame):
    def __init__(self, *args, **kwargs):
        mFrame = wx.Frame.__init__(self, *args, **kwargs)
        self.fill_panel()
        self.spi_dev_input.SetValue('0')
        self.spi_addrw_input.SetValue('7')
        self.spi_addr_input.SetValue('0000000')
        self.spi_dataw_input.SetValue('8')
        self.spi_data_input.SetValue('00000111')

    def fill_panel(self):
        self.panel_label = wx.StaticText(self, label="Write Bytes to a SPI address:")
        font = wx.Font(9, wx.NORMAL, wx.NORMAL, wx.BOLD)
        self.panel_label.SetFont(font)
        self.spi_dev_label    = wx.StaticText(self, label="SPI Device (0-3):")
        self.spi_dev_input    = wx.TextCtrl(self, size=(30,20))
        self.spi_addrw_label  = wx.StaticText(self, label="SPI Addr Width:")
        self.spi_addrw_input  = wx.TextCtrl(self, size=(30,20))
        self.spi_addr_label   = wx.StaticText(self, label="SPI Addr (bin):")
        self.spi_addr_input   = wx.TextCtrl(self, size=(100,20))
        self.spi_dataw_label  = wx.StaticText(self, label="SPI Data width:")
        self.spi_dataw_input  = wx.TextCtrl(self, size=(30,20))
        self.spi_data_label   = wx.StaticText(self, label="SPI Data (bin):")
        self.spi_data_input   = wx.TextCtrl(self, size=(100,20))
        self.spi_bbang_check  = wx.CheckBox(self, -1, "Bit-Bang")
        self.spi_scsb_check   = wx.CheckBox(self, -1, "Invert SCSb polarity")
        self.spi_write_button = wx.Button(self, -1, "SPI Write")
        self.spi_read_button  = wx.Button(self, -1, "SPI Read")
        self.spi_return_label = wx.StaticText(self, label="Read Return:")
        self.spi_return_input = wx.TextCtrl(self, size=(100,20), style=wx.TE_READONLY)

        ### Grid Bag Sizer to Organize Panel
        self.top_sizer = wx.GridBagSizer(vgap=5,hgap=5)
        self.top_sizer.Add(self.panel_label,         (1,1), (1,3), flag=wx.ALIGN_LEFT|wx.EXPAND)
        self.top_sizer.Add(self.spi_addrw_label,     (2,1),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_addrw_input,     (2,2),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_addr_label,      (2,3),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_addr_input,      (2,4),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_dataw_label,     (3,1),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_dataw_input,     (3,2),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_data_label,      (3,3),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_data_input,      (3,4),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_dev_label,       (4,1),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_dev_input,       (4,2),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_bbang_check,     (2,5),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_scsb_check,      (3,5),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_write_button,    (5,1),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_read_button,     (6,1),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_return_label,    (6,3),        flag=wx.ALIGN_RIGHT|wx.EXPAND)
        self.top_sizer.Add(self.spi_return_input,    (6,4),        flag=wx.ALIGN_RIGHT|wx.EXPAND)

        self.top_sizer.AddGrowableRow(7)
        self.top_sizer.AddGrowableCol(4)
        self.SetSizer(self.top_sizer)

        ### Event Bindings
        self.Bind(wx.EVT_BUTTON, self.spi_write_button_bind, id = self.spi_write_button.GetId())
        self.Bind(wx.EVT_BUTTON, self.spi_read_button_bind, id = self.spi_read_button.GetId())

    def spi_write_button_bind(self, event):
        self.write_to_spi()
        return
    def spi_read_button_bind(self, event):
        self.read_from_spi()
        return

    def write_to_spi(self):
        print_status('Writing to SPI...')
        dev      = int(self.spi_dev_input.GetValue())   # device number
        addr_w   = int(self.spi_addrw_input.GetValue()) # Address bit width
        addr_bin = self.spi_addr_input.GetValue()       # Address binary string
        data_w   = int(self.spi_dataw_input.GetValue()) # Data bit width
        data_bin = self.spi_data_input.GetValue()       # Data binary string
        print_status('SPI command: 0-'+addr_bin+'-'+data_bin)

        ## Create SPI data sequence (prepend with 0 for write command)
        cmd_seq = '0' + addr_bin + data_bin
        cmd_seq_w = addr_w + data_w + 1

        ## Check that address and data are valid
        if len(addr_bin) <> addr_w:
            print_status('SPI ERROR: Address length is not correct')
            return
        if len(data_bin) <> data_w:
            print_status('SPI ERROR: Address length is not correct')
            return

        ## Create usable/shorthand vars for API calls
        C0 = c_ulong(0)
        C1 = c_ulong(1)
        C8 = c_ulong(8)
        C16 = c_ulong(16)
        SPIMANREG = c_ulong(53281)
        SPIINTCONFIGREG = c_ulong(53282)
        CYPRESSACCESS = c_ulong(16777216)
        SPIACCESS = c_ulong(16777217)

        ## Check if SPI command should be bit-bang or standard API call
        if self.spi_bbang_check.GetValue() == True:

            # Check Polarity of the SCSb line (invert if checkbox = True)
            SCSB_pol = int(self.spi_scsb_check.GetValue())

            # initialize SPI Manual Control Register variable, initialize bit fields and binary to integer xform matrix
            SDO  = 0
            SDI  = 0
            SCLK = 0 # start clock low
            SCSB = 1^SCSB_pol # start with chip select disabled (apply SCSB polarity)
            OBEN = 0 # start output buffer off
            bin2int = matrix([[16],[8],[4],[2],[1]]) # binary to integer transform matrix
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)

            # Configure SPI Interface Configuration Register (0xD022) for Manual mode
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIINTCONFIGREG, C8, C0)
            if (rtn == False):
                print_status('Error configuring SPI interface register')
                return False

            # Enable the output buffer (OBEN)
            OBEN = 1
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))
            if (rtn == False):
                print_status('Error configuring SPI manual control register')
                return False

            # Assert SCLK (SCS high)
            SCLK = 1
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))

            # Enable the device (SCS low and SCLK high)
            SCSB = 0^SCSB_pol
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))

            for i in range(0, cmd_seq_w):
                # Assert the SDI line and deassert SCLK
                SDI = int(cmd_seq[i])
                SCLK = 0
                reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
                rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))
                # Assert SCLK
                SCLK = 1
                reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
                rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))

            # Deassert SCLK
            SCLK = 0
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))

            # Disable the device (SCS high)
            SCSB = 1^SCSB_pol
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))

            # Disable the output buffer
            OBEN = 0
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))

        ## Perform SPI command using standard API call
        else:

            # Convert binary addr value to integer number
            addr_int = 0
            for i in range(0,addr_w):
                addr_int = addr_int + int(addr_bin[addr_w-1-i]) * 2**i

            # Convert binary data value to integer number
            data_int = 0
            for i in range(0,data_w):
                data_int = data_int + int(data_bin[data_w-1-i]) * 2**i

            # Configure SPI Interface Configuration Register (0xD022) for CLK/64 mode
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIINTCONFIGREG, C8, C16)
            if (rtn == False):
                print_status('Error configuring SPI interface register')
                return False

            # Set SPI addr HIGH register (0xD023 - 53283)
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, SPIACCESS, C0, c_ulong(dev), C8, c_ulong(addr_int), C8, c_ulong(data_int))
            if (rtn == False):
                print_status('Error writing to SPI - Check that Hardware has been Reset')
                return False

        return

    def read_from_spi(self):
        print_status('Reading from SPI...')
        dev      = int(self.spi_dev_input.GetValue())
        addr_w   = int(self.spi_addrw_input.GetValue())
        addr_bin = self.spi_addr_input.GetValue()
        data_w   = int(self.spi_dataw_input.GetValue())
        print_status('SPI command: 1-'+addr_bin+'-'+'x'*data_w)

        cmd_seq = '1' + addr_bin # create SPI data sequence (prepend with 1 for read command)
        cmd_seq_w = addr_w + 1

        ## Check that address and data are valid
        if len(addr_bin) <> addr_w:
            print_status('SPI ERROR: Address length is not correct')
            return

        ## Create usable/shorthand vars for API calls
        C0 = c_ulong(0)
        C8 = c_ulong(8)
        C16 = c_ulong(16)
        SPIMANREG = c_ulong(53281)
        SPIINTCONFIGREG = c_ulong(53282)
        CYPRESSACCESS = c_ulong(16777216)
        SPIACCESS = c_ulong(16777217)

        ## Initialize register return values
        reg_return = ''
        spi_man_reg_return = c_ulong(0)

        ## Check if SPI command should be bit-bang or standard API call
        if self.spi_bbang_check.GetValue() == True:

            # Check Polarity of the SCSb line (invert if checkbox = True)
            SCSB_pol = int(self.spi_scsb_check.GetValue())

            # initialize SPI Manual Control Register variable, initialize bit fields and binary to integer xform matrix
            SDO  = 0
            SDI  = 0
            SCLK = 0 # start clock low
            SCSB = 1^SCSB_pol # start chip select disabled
            OBEN = 0 # start output buffer off
            bin2int = matrix([[16],[8],[4],[2],[1]])
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)

            # Configure SPI Interface Configuration Register (0xD022) for Manual mode
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIINTCONFIGREG, C8, C0)
            if (rtn == False):
                print_status('Error configuring SPI interface register')
                return False

            # Enable the output buffer (OBEN)
            OBEN = 1
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))
            if (rtn == False):
                print_status('Error configuring SPI manual control register')
                return False

            # Assert SCLK (SCS high)
            SCLK = 1
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))

            # Enable the device (SCS low and SCLK high)
            SCSB = 0^SCSB_pol
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))

            # Send address
            for i in range(0, cmd_seq_w):
                # Assert the SDI line and deassert SCLK
                SDI = int(cmd_seq[i])
                SCLK = 0
                reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
                rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))
                # Assert SCLK
                SCLK = 1
                reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
                rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))

            # Read SDO line
            for i in range(0, data_w):
                # Deassert SCLK
                SCLK = 0
                reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
                rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))
                # Assert SCLK
                SCLK = 1
                reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
                rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))
                # Read SDO value from SPI manual register
                rtn = wv5dll.WvNewestHardwareRead(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, byref(spi_man_reg_return))
                # Get SDO and store char to reg_return
                SDO = mod(spi_man_reg_return.value,2) # SDO is LSB of returned reg value
                reg_return = reg_return + str(SDO)

            # Deassert SCLK
            SCLK = 0
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))

            # Disable the device (SCS high)
            SCSB = 1^SCSB_pol
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))

            # Disable the output buffer
            OBEN = 0
            reg_value = int(matrix([OBEN, SCSB, SCLK, SDI, SDO])*bin2int)
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIMANREG, C8, c_ulong(reg_value))

            # Set text box to returned register binary string
            self.spi_return_input.SetValue(reg_return)

        ## Perform SPI command using standard API call
        else:

            reg_return_int = c_ulong(0)

            # Convert binary addr value to integer number
            addr_int = 0
            for i in range(0,addr_w):
                addr_int = addr_int + int(addr_bin[addr_w-1-i]) * 2**i

            # Configure SPI Interface Configuration Register (0xD022) for CLK/64 mode
            rtn = wv5dll.WvNewestHardwareWrite(C0, C0, CYPRESSACCESS, C0, C0, C16, SPIINTCONFIGREG, C8, C16)
            if (rtn == False):
                print_status('Error configuring SPI interface register')
                return False

            # SPI read command
            rtn = wv5dll.WvNewestHardwareRead(C0, C0, SPIACCESS, C0, c_ulong(dev), C8, c_ulong(addr_int), C8, byref(reg_return_int))
            if (rtn == False):
                print_status('Error reading from SPI - Check that Hardware has been Reset')
                return False

            # Convert returned reg value from integer to binary string
            for i in range(0,data_w):
                bit = mod(reg_return_int.value,2**(data_w-1-i)) # get binary bit value
                reg_return = reg_return + str(bit)              # Append the bit to the reg_return bit string
                reg_return_int = c_ulong(reg_return_int.value - bit * 2**(data_w-1-i)) # subtract the value of the bit from the integer value

            # Set text box to returned register binary string
            self.spi_return_input.SetValue(reg_return)

        return

