/******************************************************************************
* FILE PURPOSE: Public header file
*******************************************************************************
*
* FILE NAME: mp_sbfm.h
*
* DESCRIPTION:
*       This file contains public definitions, data structures and function
*       prototypes for CM3 CPU
*
*  Copyright (c) 2009 Texas Instruments Inc.
*  All Rights Reserved This program is the confidential and proprietary
*  product of Texas Instruments Inc.  Any Unauthorized use, reproduction or
*  transfer of this program is strictly prohibited.
*
* HISTORY:
*
* 08/16/2007 R.Liang           initial version
* 04/16/2008 R.Liang           this file only contain the MP related
*                              APIs, all BD related APIs are moved to
*                              single buffer API
* 05/04/2008 R.Liang           merge the single core and multi core buffer 
*                              management APIs into one APIs
* 09/10/2008 R.Liang          increase the buf_ref cnt to uint16
* 11/11/2008 R.Liang		  ported to DSP/BIOS
* 03/05/2009 R.Liang	      ported to F28335
*							  Note F28335 don't have uint8 data type
*
* LIST OF FUNCTION PROTOTYPES:
*  buffer management APIs for MP implementation
*
*
******************************************************************************/

#ifndef  __MP_SBFM_H__
#define  __MP_SBFM_H__

/* include the default data type */
#include <typedefs.h>
#include <proj_assert.h>
#include <spinlock.h>
#include <string.h>

#if defined (CM3)

#define BYTES_PER_STORE_IN_BF     (1)
#define DATA_RAM_BASE_ADDR             0x20000000
#else
#define _F28335_CPU_

#define BYTES_PER_STORE_IN_BF     (2)

#endif
// for big endian case
// High byte is coming first

#define MAX_NUM_POOL                   (4)

#define OFFSET_TYPE_LOCAL_REF                      (1)
#define OFFSET_TYPE_REMOTE_REF                     (0)

#define PD_ATTACHED                                1
#define PD_NOT_ATTACHED                            0
#define RD_LEN_NO_USED                             0

#define BD_OWNER_SW                                1
#define BD_OWNER_HW                                0

#define EOP_LAST_BLOCK                             1

/* error code */
typedef enum
{
   SBM_SUCCESS =0,
   SBM_OUT_OF_HANDLER,
   SBM_OUT_OF_MEMORY,
   SBM_INVALID_PARAMETER,
   SBM_NULL_POINTER,
   SBM_DAT_OFFSET,
   SBM_ERR_BD,
   SBM_ERR_UNDERFLOW,
   SBM_FAILURE
} SBM_STATUS_t;

/* smart DMA BD definition */

typedef struct _SMART_BUF_DESC_s
{
   struct _SMART_BUF_DESC_s *pnext;       /* Pointer to the next buffer descriptor */

   /* note 
    F28335 only support the bit filed up to 16 bit

    A bit field of type integer is signed. Bit fields are 
    packed into words beginning at the low-order bits, and
    do not cross word boundaries. Therefore, bit fields are 
    limited to a maximum size of 16 bits, regardless
     of what size is used in the C source. (ANSI/ISO 6.5.2.1, K&R A8.3)
   */

   /* 2nd word */
   //UINT32 offset:24;                      /* buffer offset */
   UINT32 offset_L:16;        /* low 16 bits */
   UINT32 offset_H:8;         /* high 8 bit */

   UINT32 pd_flag:1;                      /* PD flag */
   //UINT32 rd_flag:2;                    /* RD flag */
   UINT32 reserved1_3:2;
   UINT32 reserved1_1:1;
   UINT32 offset_type:1;
   UINT32 flags:1;                        /* EOP flag */
   UINT32 reserved1_2:1;
   UINT32 owner:1;

   /* 3rd word */
   UINT32 len:16;                /* use the same name to match old BD */
   UINT32 dat_offset:10;
   UINT32 bit_en_end:3;
   UINT32 bit_en_start:3;

} SBD_s,*PSBD_s,**PPSBD_s;


#define MAC_MSG_BLOCK_SIZE             256


#define MP_BUF_MAGIC             0xdeedbeef
#define BUF_BD_RATIO             (3)
#define BUF_REF_CNT_BYTE         (2)
#define BD_TRACK_BYTE            (2)
#define BUF_TRACK_BYTE           (2)
#define SBD_SIZE_BYTE            (6)

/* MP pool handler
   Note : the pool handler and pool buffer must be located in the shared buffer
   because they are accessed by MP.
*/

typedef struct _MP_smart_pool_handler_s
{
   /* MP implementation specific */
   UINT32 magic;           /* magic number to indicate if pool is inited */   
   UINT16 spinlock_num;     /* which spinlock is used to access the shared memeory */
   UINT16 ref_cnt;         /* reference count to indicate how mant time are created */

   UINT16 valid;      /* 0 invalid, 1 valid */
   UINT16 pool_type;  /* 0 : fixed-size data block pool
                        1 : linear buffer pool
                     */
                     
   /* pool creating parameters */
   UINT32 pool_addr;       /* pool start address */
   UINT32 pool_size;       /* pool size in bytes */
   
   /* internal data structure */
   UINT32 actual_size;     /* number bytes can be used (aligned in word) */

   UINT16 block_size;      /* buffer block size */   


   UINT16 block_size_shift;/* shift bit to simulate the division */   
   UINT16 offset_type;     /* buffer offset type */   
   
   UINT16 num_buf;         /* number buffer */
   UINT16 num_bd;          /* number BD */
   
   /* buf and BD start address */
      
   UINT32 buf_base_addr;      /* buf array start address */   
   SBD_s *bd_array;           /* BD array list */
   UINT16 *free_bd_list_array;   
   UINT16 *free_buf_list_array;  
   UINT16 *ref_cnt_array;      /* ref cnt array */

   /*  BD and BUF track */
   UINT16 head_free_bd;
   UINT16 tail_free_bd;
   UINT16 num_free_bd;

   UINT16 head_free_buf;
   UINT16 tail_free_buf;
   UINT16 num_free_buf;


   /* linear buffer pool */
   UINT16 * bd_state_array;   /* for linear pool */  
   
   UINT32 total_free_bytes;
   UINT32 total_used_bytes;   /* for linear pool */ 

   /* debug info */
   UINT32 high_water_mark_block;
   UINT32 high_water_mark_bd;
   UINT32 high_water_mark_bytes;  /* high watermark in the pool */
#if 0     
   /* the folloing fields are used for linear buffer pool */
   

   /* in the linear buffer pool, one buffer is represented by one BD */
   
   
   //UINT16 num_block;       /* number of block */
   
   //UINT32 data_base_addr;
   UINT32 bd_state_addr;   /* for linear pool */   
   //UINT32 used_block;
   UINT32 high_water_mark_bytes;  /* high watermark in the pool */
       
#endif
} MP_SPOOL_HND_s;


/* module handler */
typedef struct _MP_sbm_handler_s
{
   MP_SPOOL_HND_s pool_handler[MAX_NUM_POOL];   
   UINT32 BM_base_address;
   UINT16 bm_inited;
} MP_SBM_HND_s;


/* global data */
extern MP_SBM_HND_s MP_SBM_hnd;



/*===================================================================================
   Memory access functions
  ===================================================================================
*/  
#ifdef _F28335_CPU_
UINT16 BF_GET_BYTE(UINT32 octet_addr);
void   BF_PUT_BYTE(UINT32 octet_addr,UINT16 byte);

UINT16 BF_GET_WORD(UINT32 octet_addr);
void   BF_PUT_WORD(UINT32 octet_addr,UINT16 word);

UINT32 BF_GET_LONG(UINT32 octet_addr);
void   BF_PUT_LONG(UINT32 octet_addr,UINT32 data);

void BF_MEMSET(uint32 octet_addr,uint16 byte,uint16 len);
void BF_MEMCPY(uint32 dst_offset,uint32 src_offset,uint16 len);
#endif

/*===================================================================================
   FUNCTION APIS
  ===================================================================================
*/  

void MP_SBM_init(UINT32 dma_base);
void MP_SBM_start(void);

SBM_STATUS_t MP_SBM_pool_reset(MP_SPOOL_HND_s *pHnd);
MP_SPOOL_HND_s * MP_SBM_pool_create(UINT32 pool_addr,UINT32 pool_size,UINT16 block_size,UINT16 offset_type,
         UINT16 lock_num);
MP_SPOOL_HND_s * MP_SBM_pool_create_1(UINT32 buf_pool_addr,UINT32 buf_pool_size,UINT16 block_size,
   UINT32 bd_pool_addr,UINT32 bd_pool_size,UINT16 lock_num);
SBM_STATUS_t MP_SBM_pool_delete(MP_SPOOL_HND_s *pHnd);


SBD_s * MP_SBM_pool_malloc(MP_SPOOL_HND_s *pHnd,UINT32 byte,UINT16 data_offset);
SBM_STATUS_t MP_SBM_pool_free(MP_SPOOL_HND_s *pHnd,SBD_s *pbuf);

UINT32 MP_SBM_pool_get_free_bytes(const MP_SPOOL_HND_s *pHnd);
UINT16 MP_SBM_pool_get_block_size(const MP_SPOOL_HND_s *pHnd);


/* linear pool related APIs */
SBM_STATUS_t MP_SBM_lpool_reset(MP_SPOOL_HND_s *pHnd);
MP_SPOOL_HND_s * MP_SBM_lpool_create(UINT32 pool_addr,UINT32 pool_size,UINT16 num_bd,
         UINT16 lock_num);
SBM_STATUS_t MP_SBM_lpool_delete(MP_SPOOL_HND_s *pHnd);
SBD_s * MP_SBM_lpool_malloc(MP_SPOOL_HND_s *pHnd,UINT16 bd_index,UINT32 byte);
SBM_STATUS_t MP_SBM_lpool_free(MP_SPOOL_HND_s *pHnd,SBD_s *pbuf);
SBM_STATUS_t MP_SBM_lpool_set_used_bytes(MP_SPOOL_HND_s *pHnd,UINT32 byte);


UINT16 MP_SBM_Get_Buf_Ref_Cnt(MP_SPOOL_HND_s *pHnd,SBD_s *pbuf);
UINT16 MP_SBM_Inc_Buf_Ref_Cnt(MP_SPOOL_HND_s *pHnd,SBD_s *pbuf);
UINT16 MP_SBM_Is_Buf_Cloned(MP_SPOOL_HND_s *pHnd,SBD_s *pbuf);
SBD_s * MP_SBM_pool_alloc_BD(MP_SPOOL_HND_s *pHnd);
SBM_STATUS_t MP_SBM_pool_free_BD(MP_SPOOL_HND_s *pHnd,SBD_s *pbd);
SBD_s * MP_SBM_Clone_Buf(MP_SPOOL_HND_s *pHnd,SBD_s *pbuf);
SBD_s *MP_SBM_Clone_Buf_Chain(MP_SPOOL_HND_s *pHnd, SBD_s *psbd);

/* =============================================================================

   common APIs for single core and multi core

   =============================================================================
*/  
UINT32 SBM_get_baseAddr(void);
void SBM_setup_BD(SBD_s *pbd,UINT32 addr);
SBM_STATUS_t SBM_Adj_offset(SBD_s * pbd,UINT16 data_offset);

#define SBM_get_addr  SBM_get_buf_addr
UINT32 SBM_get_data_addr(const SBD_s *pbd);
UINT32 SBM_get_buf_addr(const SBD_s *pbd);
SBM_STATUS_t SBM_Inc_Data_offset(SBD_s * pbd,UINT16 data_offset);
SBM_STATUS_t SBM_Dec_Data_offset(SBD_s * pbd,UINT16 data_offset);
void SBM_Terminate_BD(SBD_s *pbd);
void SBM_reset_owner(SBD_s *pbd);

UINT32 SBM_convert2Addr(const UINT32 offset,UINT16 offset_type);
UINT32 SBM_convert2Offset(const UINT32 addr,UINT16 offset_type);

SBM_STATUS_t SBM_append_bd(SBD_s *pprev,SBD_s *pcur);
SBD_s * SBM_get_tail(SBD_s *pbd);

SBM_STATUS_t SBM_get_addr_in_packet(SBD_s *pbuf,UINT16 byte_posi,
      UINT32 *offset,UINT16 *bd_len);


SBD_s * SBM_get_posi_in_packet(SBD_s *pbuf,UINT16 byte_posi,
         UINT32 *addr,UINT16 *bd_len,PSBD_s * pprevBD);

UINT32 SBM_get_size_in_BD_chain(SBD_s *pbuf);

UINT16 SBM_copy_buffer(SBD_s *pbd, UINT16 offset, UINT16 *pbuf, UINT16 len);
UINT16 SBM_move_buffer(SBD_s *pbd, UINT16 offset, UINT16 *pbuf, UINT16 len);
UINT16 SBM_set_buffer(SBD_s *pbd, UINT16 offset, UINT16 *pbuf, UINT16 len);
void SBM_memset_buffer(SBD_s *pbd, UINT16 offset, UINT16 c, UINT16 len);
UINT16 SBM_copy_bd(SBD_s *psbd_src, UINT16 offset_src, SBD_s *psbd_dst, UINT16 offset_dst);

UINT16 SBM_set_buffer2(SBD_s *pbd, UINT16 offset, UINT16 *pbuf, UINT16 len,
  UINT16 pbuf_offset);
void SBM_put_long_bigend(SBD_s *pbd, UINT16 offset, UINT32 data);
UINT32 SBM_get_long_bigend(SBD_s *pbd, UINT16 offset);

SBD_s * MP_SBM_pool_malloc_single_buff(MP_SPOOL_HND_s *pHnd);
SBD_s * SBM_copy_bd_chain(MP_SPOOL_HND_s *pHnd, SBD_s *pInput);

SBD_s *MP_SBM_dup_pkt(MP_SPOOL_HND_s *pHnd,SBD_s *pkt_p);

#endif
