//###########################################################################
//
// FILE:   PLC_AppBoot.c
//
// TITLE:  PLC Application Boot Loader
// ASSUMPTIONS:
//
//    This program requires the DSP2833x header files.
// R.Liang This is general PLC boot loader
//###########################################################################

#include <typedefs.h>   
#include <DSP28x_Project.h>     

#include <crc32.h>
#include <boot_api.h>

#ifdef F2806X

// platform CPU depend
/*---- example include file ---------------------------------------------------*/
#include "F2806x_CpuTimers.h"
#else

#ifdef F28M35X
#include "F28M35X_CpuTimers.h"
#else

#include "DSP2833x_CpuTimers.h"
#endif

#endif


/* Following symbols should be referred to linker file */
extern Uint16 SBD_API_LoadStart;
extern Uint16 SBD_API_LoadEnd;
extern Uint16 SBD_API_RunStart;

extern Uint16 PROJ_ASSERT_API_LoadStart;
extern Uint16 PROJ_ASSERT_API_LoadEnd;
extern Uint16 PROJ_ASSERT_API_RunStart;

extern Uint16 secureRamFuncs_loadstart;
extern Uint16 secureRamFuncs_loadend;
extern Uint16 secureRamFuncs_runstart;

extern Uint16 RamfuncsLoadStart;
extern Uint16 RamfuncsLoadEnd;
extern Uint16 RamfuncsRunStart;

extern Uint16 CRC_API_LoadStart;
extern Uint16 CRC_API_LoadEnd;
extern Uint16 CRC_API_RunStart;

extern Uint16 CSL_API_LoadStart;
extern Uint16 CSL_API_LoadEnd;
extern Uint16 CSL_API_RunStart;

extern Uint16 econst_LoadStart;
extern Uint16 econst_LoadEnd;
extern Uint16 econst_RunStart;

#if defined( F2806X) || defined (F28M35X)

extern Uint16 phyRamFuncs_loadstart;
extern Uint16 phyRamFuncs_loadend;
extern Uint16 phyRamFuncs_runstart;

#else

extern Uint16 CRC32_TABLE_LoadStart;
extern Uint16 CRC32_TABLE_LoadEnd;
extern Uint16 CRC32_TABLE_RunStart;

#endif


#pragma CODE_SECTION(HWI_disable, "ramfuncs");
UINT32 HWI_disable(void)
{
  asm(" setc INTM");
  return 0;
}

#pragma CODE_SECTION(HWI_restore, "ramfuncs");
void HWI_restore(UINT32 intr_reg)
{
  asm(" clrc INTM");
}

/******************************************************************************
* FUNCTION NAME: F28x_init
*
* DESCRIPTION:   This function initializes the F28335 device
*
* Input Parameters:    
* Output Parameters:  
*
******************************************************************************/
void F28x_init(void)
{
  DINT;         // Global Disable all Interrupts
  IER = 0x0000;        // Disable CPU interrupts
  IFR = 0x0000;        // Clear all CPU interrupt flags


#ifdef F2806X
  // Run the device calibration routine to trim the internal
  // oscillators to 10Mhz.
  EALLOW;
  SysCtrlRegs.PCLKCR0.bit.ADCENCLK = 1; 

  SysCtrlRegs.PCLKCR0.bit.ADCENCLK = 0;
  EDIS; 
#endif

// Step 1. Initialize System Control:
// PLL, WatchDog, enable Peripheral Clocks
// This example function is found in the DSP2833x_SysCtrl.c file.
  InitSysCtrl();

#ifdef F28M35X

#else

#ifndef F2806X  
  EALLOW;
  SysCtrlRegs.MAPCNF.bit.MAPEPWM = 1;
  EDIS;
#endif

#endif 
  // Step 1a. Reset peripherals
  DMAInitialize();

  //  InitCpuTimers();
  CpuTimer0.RegsAddr = &CpuTimer0Regs;
  CpuTimer1.RegsAddr = &CpuTimer1Regs;


// Specific clock setting for this example:
   //EALLOW;
   //SysCtrlRegs.HISPCP.all = ADC_MODCLK;	// HSPCLK = SYSCLKOUT/ADC_MODCLK
   //EDIS;

// Step 2. Initialize GPIO:
// This example function is found in the DSP2833x_Gpio.c file and
// illustrates how to set the GPIO to it's default state.
// InitGpio();  // Skipped for this example

// Step 3. Clear all interrupts and initialize PIE vector table:
// Disable CPU interrupts
   DINT;

// Initialize the PIE control registers to their default state.
// The default state is all PIE interrupts disabled and flags
// are cleared.
// This function is found in the DSP2833x_PieCtrl.c file.
   InitPieCtrl();

// Disable CPU interrupts and clear all CPU interrupt flags:
   IER = 0x0000;
   IFR = 0x0000;

// Initialize the PIE vector table with pointers to the shell Interrupt
// Service Routines (ISR).
// This will populate the entire table, even if the interrupt
// is not used in this example.  This is useful for debug purposes.
// The shell ISR routines are found in DSP2833x_DefaultIsr.c.
// This function is found in DSP2833x_PieVect.c.
   InitPieVectTable();

}

/*------------------------------------------------------------------
  Simple memory copy routine to move code out of flash into SARAM
-----------------------------------------------------------------*/

void Example_MemCopy(Uint16 *SourceAddr, Uint16* SourceEndAddr, Uint16* DestAddr)
{
    while(SourceAddr < SourceEndAddr)
    { 
       *DestAddr++ = *SourceAddr++;
    }
    return;
}

void App_LoadCodeData_RAM(void)
{
  // Copy the Flash API functions to SARAM
  // F2806x Flash API is in the ROM

  // We must also copy required user interface functions to RAM. 
  Example_MemCopy(&secureRamFuncs_loadstart, &secureRamFuncs_loadend, &secureRamFuncs_runstart);

  // We must also copy required user interface functions to RAM. 
  Example_MemCopy(&RamfuncsLoadStart, &RamfuncsLoadEnd, &RamfuncsRunStart);

  // copy the CRC drive function to SARAM
  Example_MemCopy(&CRC_API_LoadStart, &CRC_API_LoadEnd, &CRC_API_RunStart);

  Example_MemCopy(&econst_LoadStart, &econst_LoadEnd, &econst_RunStart);
  
#if defined( F2806X) || defined(F28M35X)    

  // copy the PPHY funcfunction to SARAM
  Example_MemCopy(&phyRamFuncs_loadstart, &phyRamFuncs_loadend, &phyRamFuncs_runstart);      
  
  
#else
  // Copy the CRC32 API functions to SARAM
  Example_MemCopy(&CRC32_TABLE_LoadStart, &CRC32_TABLE_LoadEnd, &CRC32_TABLE_RunStart);


#endif
}

UINT16 AppBoot_ReadConfig(void)
{
  Uint16 kk,option;

  EALLOW;
  
  // config GPIO as input 
  // 
  // this part can be customized
  
  GpioCtrlRegs.GPAMUX1.bit.GPIO4 = 0;  // 0=GPIO,  1=ECAP1,  2=Resv,  3=Resv
  GpioCtrlRegs.GPADIR.bit.GPIO4  = 0;          // 1=OUTput,  0=INput 
#ifdef F28M35X
  // no PUD field
#else
  
  GpioCtrlRegs.GPAPUD.bit.GPIO4  = 0;          // 1-Pull-up disable; 0-pull-up enable 
#endif

  EDIS;

  for (kk = 0; kk < 10000; kk++)
  {
   ;
  }

  option = GpioDataRegs.GPADAT.bit.GPIO4;
  
  return option;
}

void main(void)
{
  UINT16 option,crc;

  F28x_init();

  App_LoadCodeData_RAM();
  
  // Call Flash Initialization to setup flash waitstates
  // This function must reside in RAM
  // InitFlash code in CSL.lib
  InitFlash();   // Call the flash wrapper init function

#ifdef F2806X
  // For 2806x, we don't have BIT
  option = 1;
#else
  // R.Liang
  // please uncomment out this if you need BIT

  // option= AppBoot_ReadConfig();
  option=1;
#endif
  // here is new boot load option
  // 1. read GPIO bin, if set, go DFU mode
  // 2. GPIO =0, default
  // 3. check CRC32, if the CRC32 failure go to DFU
  // 4. otherwise, go to autoboot
  
  if (option==0)
  {
    Jump_Start_BIT();
  }
  else if (option==1)
  {
    // run the CRC32 checking
    crc = Flash_Check_Firmware_CRC32();

#ifdef F28M35X
    crc = 1;
#endif 
    if (crc==0)
    { // CRC is bad
      Jump_Start_DFU();
    }
    else
    { // CRC is good
      JUMP_Start_PLC_App();
    }
  }
  else
  { // user want to go DFU
    Jump_Start_DFU();
  }
  for (;;)
  {

  }


}

