/*******************************************************************
 *
 *    DESCRIPTION:
 *       This function contains buffer management functions
 *
 *  Copyright (c) 2004 Texas Instruments Inc.
 *  All Rights Reserved This program is the confidential and proprietary
 *  product of Texas Instruments Inc.  Any Unauthorized use, reproduction or
 *  transfer of this program is strictly prohibited.
 *
 *    AUTHOR:
 *
 *    HISTORY:
 *
 *  11/28/05    : RL. initial outline of API
 *  01/14/06    : RL. check against the coding standard  
 *  07/31/06    : RL added pool creation
 *  02/05/07    : RL changed the free BD search method
 *  04/15/07    : RL changed the module name into bfm
 *  08/16/07    : RL ported to Cortex M3
 *  04/18/08    : RL optimized for LTE
 *  06/16/08    : RL PKTM code review, changed the uint-X to UINT-X
 *  06/17/08    : RL change local variables to UINT32 to save space
 *  09/05/08    : RL optimized the malloc function when the allocated buffer
                    size can be fited into single buffer.
                    SBM_get_addr is obsolted. It is replaced by SBM_get_data_addr
 *  09/10/08    : RL change the buf ref count to 1 base.
 *  11/11/08    : RL ported to DSP/BIOS
 *  01/21/09    : RL fixed several compiling warnings
                  also added the new buffer pool creating function  
 *  01/25/10    : RL remove the OOPS and assert when error occurs
 *  01/15/2012  : RL port to COncerto CM3
 *******************************************************************/

/** include files **/
#include <mp_sbfm.h>

/** local definitions **/

/** default settings **/

/** external functions **/

/** external data **/

/** internal functions **/
/*
SBM_STATUS_t MP_SBM_pool_reset(MP_SPOOL_HND_s *pHnd)    
*/

/** public data **/
MP_SBM_HND_s MP_SBM_hnd;

/** private data **/

/** public functions **/

/** private functions **/
SBD_s * __MP_SBM_pool_malloc_one_buf(MP_SPOOL_HND_s *pHnd,UINT32 byte);


#define BUFFER_DBG


inline UINT32 __MP_SBM_GET_OFFSET(const SBD_s *pbd)
{
  UINT32 addr;
        
  // RLiang
  // need to promote the pbd->offset_H to UINT32     
  addr = ((UINT32)(pbd->offset_H))<<16;  
  
  addr += pbd->offset_L;
  
  return addr;
}

inline void __MP_SBM_SET_OFFSET(SBD_s *pBD,UINT32 offset) 
{
  pBD->offset_H = (offset>>16) & 0xFF;
  pBD->offset_L = (offset) & 0xFFFF;
}

/******************************************************************************
 * FUNCTION NAME: BF_GET_BYTE
 *
 * DESCRIPTION:  
 *	This is memory access function. It will get one byte from the specifef 
 *	octet address.
 *
 * Return Value: byte in offset (low 8 bit)
 *               
 * intput : octet_addr: octet (byte address)
 *    
 *
 * Output Parameters:
 *
 ******************************************************************************/
UINT16 BF_GET_BYTE(UINT32 octet_addr)
{
#if defined (CM3)
  UINT16 data;
  UINT8 *paddr;

  paddr = (UINT8*) octet_addr;
  data = *paddr;

  return data;
#else

  UINT32 phy_addr;
  UINT16 offset,data;

  // convert the octet addr to PHY addre and offset
  phy_addr = (octet_addr>>1);
  offset   = (octet_addr & 0x01);
  
  data = __byte((int *)phy_addr,offset);
  
  return data;
#endif

}

/******************************************************************************
 * FUNCTION NAME: BF_PUT_BYTE
 *
 * DESCRIPTION:  
 *	This is memory access function. It will write one byte to the specifef 
 *	octet address. It is read/modify/write procedure.
 *
 * Return Value: no
 *               
 *
 * intput :
 *    
 *
 * Output Parameters:
 *
 ******************************************************************************/
void BF_PUT_BYTE(UINT32 octet_addr,UINT16 byte)
{
#if defined (CM3)
  UINT8 *paddr;

  paddr = (UINT8*) octet_addr;
  *paddr= (byte & 0xFF);

#else
  UINT32 phy_addr;
  UINT16 offset;

  // convert the octet addr to PHY addre and offset
  phy_addr = (octet_addr>>1);
  offset   = (octet_addr & 0x01);
  
  __byte((int *)phy_addr,offset) = byte;
#endif

}

/******************************************************************************
 * FUNCTION NAME: BF_GET_WORD
 *
 * DESCRIPTION:  
 *	This is memory access function. It will get one word from the specifef 
 *	octet address.
 *
 * Return Value: word in offset (2 bytes)
 *               
 * intput : octet_addr: octet (byte address)
 *    
 *
 * Output Parameters:
 *
 ******************************************************************************/
UINT16 BF_GET_WORD(UINT32 octet_addr)
{
#if defined (CM3)
  UINT16 *paddr,data;

  paddr = (UINT16*) octet_addr;
  data = *paddr;

  return data;
#else

  UINT32 phy_addr;
  UINT16 offset,data,data1;

  // convert the octet addr to PHY addre and offset
  phy_addr = (octet_addr>>1);
  offset   = (octet_addr & 0x01);
  
  data = __byte((int *)phy_addr,offset);
  
  data1= __byte((int *)phy_addr,(offset+1));
  
  data |= (data1<<8);
  
  return data;
#endif
}

/******************************************************************************
 * FUNCTION NAME: BF_PUT_WORD
 *
 * DESCRIPTION:  
 *	This is memory access function. It will write one word to the specifef 
 *	octet address. It is read/modify/write procedure.
 *
 * Return Value: no
 *               
 *
 * intput :
 *    
 *
 * Output Parameters:
 *
 ******************************************************************************/
void BF_PUT_WORD(UINT32 octet_addr,UINT16 word)
{
#if defined (CM3)
  UINT16 *paddr;

  paddr = (UINT16*) octet_addr;
  *paddr= word;;

#else
  UINT32 phy_addr;
  UINT16 offset;

  // convert the octet addr to PHY addre and offset
  phy_addr = (octet_addr>>1);
  offset   = (octet_addr & 0x01);
  
  __byte((int *)phy_addr,offset) = (word);
  
  __byte((int *)phy_addr,(offset+1))= (word>>8);
#endif    
}

/******************************************************************************
 * FUNCTION NAME: BF_GET_LONG
 *
 * DESCRIPTION:  
 *	This is memory access function. It will get one long word from the specifef 
 *	octet address.
 *
 * Return Value: long word (4 bytes)
 *               
 * intput : octet_addr: octet (byte address)
 *    
 *
 * Output Parameters:
 *
 ******************************************************************************/
UINT32 BF_GET_LONG(UINT32 octet_addr)
{
#if defined (CM3)
  UINT32 *paddr,data;

  paddr = (UINT32 *)octet_addr;
  data = *paddr;
  
  return data;
#else
  UINT32 phy_addr;
  UINT16 offset;
  SINT16 *pdata;
  UINT32 long_data,data;
  // convert the octet addr to PHY addre and offset
  phy_addr = (octet_addr>>1);
  pdata =(SINT16 *)phy_addr;
  offset   = (octet_addr & 0x01);
  
  data = __byte(pdata,offset);
  long_data = data;
  
  data= __byte(pdata,(offset+1));
  long_data |=(data<<8);
  
  data= __byte(pdata,(offset+2));
  long_data |=(data<<16);

  data= __byte(pdata,(offset+3));
  long_data |=(data<<24); 
  
  return long_data;
#endif  
}

/******************************************************************************
 * FUNCTION NAME: BF_PUT_LONG
 *
 * DESCRIPTION:  
 *	This is memory access function. It will write one long word to the specifef 
 *	octet address. It is read/modify/write procedure.
 *
 * Return Value: no
 *               
 *
 * intput :
 *    
 *
 * Output Parameters:
 *
 ******************************************************************************/
void BF_PUT_LONG(UINT32 octet_addr,UINT32 data)
{
#if defined (CM3)
  UINT32 *paddr;

  paddr = (UINT32 *)octet_addr;
  *paddr = data;
#else

  UINT32 phy_addr;
  UINT16 offset;
  SINT16 *pdata;
  
  // convert the octet addr to PHY addre and offset
  phy_addr = (octet_addr>>1);
  pdata =(SINT16 *)phy_addr;
  
  offset   = (octet_addr & 0x01);
  
  __byte(pdata,offset) 	= (data );
  offset++;
  data >>=8;
  __byte(pdata,offset) 	= (data );
  data >>=8;
  offset++;
  __byte(pdata,offset) 	= (data );
  offset++;
  data >>=8;
  __byte(pdata,offset) 	= (data );
#endif  
}

/******************************************************************************
 * FUNCTION NAME: BF_MEMSET
 *
 * DESCRIPTION:  
 *	This is memory access function. It will mimic the standard C memset
 *	function.
 * Return Value: 
 *               
 * intput : octet_addr: octet (byte address)
 *    
 *
 * Output Parameters:
 *
 ******************************************************************************/
void BF_MEMSET(uint32 octet_addr,uint16 byte,uint16 len)
{
#if defined (CM3)
  UINT8 *paddr;

  paddr = (UINT8 *)octet_addr;
  memset(paddr,byte,len);
#else
  uint16 i;
  uint32 temp;
  
  //call put_byte function for all octet addr
  temp = octet_addr;
  
  for (i=0;i<len;i++)
  {
    BF_PUT_BYTE(temp++,byte);
  }
#endif  
}

/******************************************************************************
 * FUNCTION NAME: BF_MEMCPY
 *
 * DESCRIPTION:  
 *	This is memory access function. It will mimic the standard C memcpy
 *	function.
 * Return Value: 
 *               
 * intput : octet_addr: octet (byte address)
 *    
 *
 * Output Parameters:
 *
 ******************************************************************************/
void BF_MEMCPY(uint32 dst_offset,uint32 src_offset,uint16 len)
{
#if defined (CM3)
  UINT8 * pdst,*psrc;

  pdst = (UINT8 *)dst_offset;
  psrc = (UINT8 *)src_offset;

  memcpy(pdst,psrc,len);
#else
  uint16 *src_addr,*dst_addr;
  uint16 i,loop_cnt;
  uint16 src_data;
  
  if  ( ( ((dst_offset & 0x1 )==0) && ( (src_offset & 0x01)==0))  ||
        ( ((dst_offset & 0x1 )==1) && ( (src_offset & 0x01)==1)) )
  {
    if ( ((dst_offset & 0x1 )==1) && ( (src_offset & 0x01)==1))
    {
      src_data = BF_GET_BYTE(src_offset++);
      BF_PUT_BYTE(dst_offset++,src_data);

      len -=1;
    }
  
    // both dst and src offset are even
    // read 2 bytes at one time
    loop_cnt = len>>1;

    src_addr = (UINT16 *) (src_offset>>1);
    dst_addr = (UINT16 *) (dst_offset>>1);
    for (i=0;i<loop_cnt;i++)
    {
      *dst_addr++ = *src_addr++;
    }
     
     if (len!=(loop_cnt<<1))
     {  // there is one extra byte       
        src_data = BF_GET_BYTE(src_offset+len-1);
        BF_PUT_BYTE(dst_offset+len-1,src_data);
     }
  }
  else 
  {
    for(i=0;i<len;i++)
    {
      src_data = BF_GET_BYTE(src_offset++);
      BF_PUT_BYTE(dst_offset++,src_data);
    }

  }

#endif    
}
/*
 *  FUNCTION: MP_SBM_init
 *
 *  PARAMETERS:
 *      input       
 *          dma_base    : Smart DMA base address
 *      
 *      output      : no
 *  DESCRIPTION:
 *      This function initialize the BM module's internal structure. It is
 *      called before all other BM APIs. It is onlu called one time.
 *
 *  RETURNS: 
 *      no
 *
 *  Note: Only support the single core. MP_core support is TBD
 */
void MP_SBM_init(UINT32 dma_base)
{      
    /* this will be change if Data ram change       
    */    
    memset(MP_SBM_hnd.pool_handler,0,sizeof(MP_SBM_hnd.pool_handler));

    /* set up the smart DMA base address */        
    MP_SBM_hnd.BM_base_address = dma_base;
    MP_SBM_hnd.bm_inited=1;   /* mark the BM module is inited */  
}

/*
 *  FUNCTION: MP_SBM_start
 *
 *  PARAMETERS:
 *      input   no
 *      output  no
 *
 *  DESCRIPTION:
 *      This function will start the smart buffer manager
 *
 *  RETURNS: 
 *  
 *  Note: right now, this function is empty. later we might fill
 *  this functions
 */

void MP_SBM_start(void)
{   

}

/*
 *  FUNCTION: MP_SBM_pool_reset
 *
 *  PARAMETERS:
 *      input:  pHnd        pool handler
 *
 *  DESCRIPTION:
 *      This function will reclaim all data block and BDs. it is internal
 *      function. The caller must make sure there are no buffers are used
 *      by any module otherwise the behavior is undefined.
 *
 *      This is the BIG free function. In stead of calling free function
 *      for each BD, this function call will free all BDs.
 *
 *      Note: this function is called by MP pool create, the lock is
 *      avaiable,  so we don't need to get lock
 *
 *  RETURNS: 
 *       status value. 0 when it is sucessful, otherwise there are some errors 
 */
SBM_STATUS_t MP_SBM_pool_reset(MP_SPOOL_HND_s *pHnd)
{
   UINT16 i,num_block;         
   SBD_s *pbd;
       
   /* reset the free BD array */
   num_block = pHnd->num_bd;      
   for (i=0;i<num_block;i++) 
   {
      pbd = &(pHnd->bd_array[i]);      
      pHnd->free_bd_list_array[i]=i;
      /* reset bd to zero */
      memset(pbd,0,sizeof(SBD_s));      
   }
   
   pHnd->head_free_bd =0;    
   pHnd->tail_free_bd = num_block-1;
   pHnd->num_free_bd = num_block;
   
   /* reset the free Buf array */
   num_block = pHnd->num_buf;      
   for (i=0;i<num_block;i++) 
   {      
      pHnd->free_buf_list_array[i] = i;      
      pHnd->ref_cnt_array[i]=0;
   }
      
   pHnd->head_free_buf =0;    
   pHnd->tail_free_buf = num_block-1;
   pHnd->num_free_buf = num_block;

#ifdef BUFFER_DBG
   /* actual size for data block */
   pHnd->actual_size = num_block * pHnd->block_size;      
   pHnd->high_water_mark_block =0;      
   pHnd->high_water_mark_bd =0; 
#endif

   return SBM_SUCCESS;
}

/*
 *  FUNCTION: MP_SBM_pool_create
 *
 *  PARAMETERS:
 *      input       : 
 *       pool_addr      : the staring  address of memory pool
 *       pool_size      : total bytes of memory pool
 *       block_size     : data block size in bytes
 *        
 *     output      : no
 *
 *  DESCRIPTION:
 *  This function creates the fixed-size memory pool. Each data block in the
 *  pool has one BD and state. The BD is 8 byte and state is one byte. The BDs 
 *  are managed as linear array to faciltate the search. 
 *
 *  The BD is also word aligned because of hardware DMA requeirement. The data
 *  block is also word aligned.
 *
 *  Note if the block size is not 2's power, we gave the error
 *
 *  RETURNS: 
 *      pool handle of the pool memory. Whenever reference to the pool, the
 *      pool handle is used. If the returend pool handler is NULL, some erorrs
 *      occured in BM modules (run out of handler).
 */

MP_SPOOL_HND_s * MP_SBM_pool_create(UINT32 pool_addr,UINT32 pool_size,UINT16 block_size,
   UINT16 offset_type,UINT16 lock_num)
{
   UINT16 i,num_block,temp,shift_bit;
   MP_SPOOL_HND_s *pHnd;
   UINT32 addr,actual_size,found;   
   UINT32 interrupt_saved;
   
   /* make sure the block size is power of 2 
      otherwise it is error
   */
   found=0;
   temp =0x1;
   for (i=0;i<=15;i++)
   {
      temp = 0x1<<i;
      
      if (block_size == temp)
      {
         shift_bit = i;
         found=1;
         break;
      }

      temp <<=1;
   }

   if (found==0)
   {  /* the block size is not power 2s */
      return NULL;
   }
   
   interrupt_saved = spin_lock(lock_num);   
   
   /* serach the pool handle to see if the pool is created */
   for (i=0;i<MAX_NUM_POOL;i++)
   {
      pHnd = &MP_SBM_hnd.pool_handler[i];
      if (  (pHnd->magic == MP_BUF_MAGIC) &&
            (pHnd->valid ==1) &&
            (pHnd->pool_addr == pool_addr) &&
            (pHnd->pool_size == pool_size) &&
            (pHnd->block_size == block_size) &&
            (pHnd->spinlock_num == lock_num)
         )
      {  /* found the matched handler */
         pHnd->ref_cnt++;
         spin_unlock(lock_num,interrupt_saved);         
         pHnd = &MP_SBM_hnd.pool_handler[i];

         return pHnd;
      }      
   }

   /* handler is not found, need to create one */
   /* try to find the free handler from the BM module */
   found =0;
   for (i=0;i<MAX_NUM_POOL;i++)
   {
      pHnd = &MP_SBM_hnd.pool_handler[i];
      
      if ( (pHnd->magic == MP_BUF_MAGIC) &&
           (pHnd->valid ==1) ) 
      {
         continue;
      }

      found =1; 
      break;    
   }

   if (found==0)
   {
      /* there are no more handlers in the BM module */
      spin_unlock(lock_num,interrupt_saved);
      
      return NULL;
   }
   
   
   /* here we found out free handle */
   pHnd->magic = MP_BUF_MAGIC;
   pHnd->ref_cnt++;
   pHnd->valid =1;
   pHnd->pool_type =0;  /* fix size data block pool */
   
   /* save the spin lock number */
   pHnd->spinlock_num = lock_num;

   /* save the parameter for reference    	
   
   */
   pHnd->pool_addr   = pool_addr;
   pHnd->pool_size   = pool_size;
   pHnd->block_size  = block_size;
   pHnd->block_size_shift = shift_bit;
   
   pHnd->offset_type = offset_type;
   
   /* align the address to 4 bytes boundary for BD */
   addr = ((pool_addr+3)>>2)<<2;

   /* we need to make sure the state end in word 
      the extra 8 byte is used for word alignment
   */
   actual_size = pool_size - (addr-pool_addr)-8;   

   actual_size = pool_size/BYTES_PER_STORE_IN_BF - (addr-pool_addr)-8;  

   /* memory layout is
      BD                (ratio*N block)
      BUFs              (N block)         
      track_free_BD     (ratio*N *2)	sizeof(UINT16)
      track_free_Buf     (N*2)    		sizeof(UINT16)   
      Ref_Cnt           (N: per BUF) 	sizeof(UINT16)

      Size information:         
      BD:						sizeof (SBD_s)
        BD_TRACK (UINT16)		sizeof (UINT16)
        BUF_TRACK (UINT16)	sizeof (UINT16)
   */  

   /* num of block */
   temp = sizeof(SBD_s)* BUF_BD_RATIO + block_size/BYTES_PER_STORE_IN_BF + 
          BUF_BD_RATIO*sizeof(UINT16) + /* BD tracking */
          sizeof(UINT16) +  /* buf tracking */
          sizeof(UINT16) ;  /* ref count tracking */
           
   num_block = actual_size /temp;

   /* set up the toatl buf and BD */
   pHnd->num_buf = num_block;  
   pHnd->num_bd = BUF_BD_RATIO*num_block;
   
   /* setup BD base address, round to word aligned */      
   pHnd->bd_array = (SBD_s *) addr;
   
   /* set up buf base address */
   pHnd->buf_base_addr = (UINT32 ) ( (UINT32)pHnd->bd_array + ((UINT32)pHnd->num_bd)*sizeof(SBD_s));
   
   /* set up the free BD array address */   
   pHnd->free_bd_list_array = (UINT16 *)((UINT32)pHnd->buf_base_addr + num_block *block_size/BYTES_PER_STORE_IN_BF);
   
   /* set up the free buf array address */   
   pHnd->free_buf_list_array = (UINT16 *) ((UINT32)pHnd->free_bd_list_array + 
                               (UINT32)pHnd->num_bd *sizeof(UINT16));
   
   /* set up the ref_cnt array address */
   pHnd->ref_cnt_array = (UINT16 *) ( (UINT32)(pHnd->free_buf_list_array) + num_block * sizeof(UINT16));
   
   /* reset the BD, BUF tracking and ref_cnt */
   MP_SBM_pool_reset(pHnd);     

   spin_unlock(lock_num,interrupt_saved);
   
   return pHnd;
}


MP_SPOOL_HND_s * MP_SBM_pool_create_1(UINT32 buf_pool_addr,UINT32 buf_pool_size,UINT16 block_size,
   UINT32 bd_pool_addr,UINT32 bd_pool_size,UINT16 lock_num)
{
   UINT32 i,num_block,temp,shift_bit;
   MP_SPOOL_HND_s *pHnd;
   UINT32 addr,actual_size,found;   
   UINT32 interrupt_saved;   
   
   /* make sure the block size is power of 2 
      otherwise it is error
   */
   found=0;
   temp =0x1;
   for (i=0;i<=15;i++)
   {
      temp = 0x1<<i;
      
      if (block_size == temp)
      {
         shift_bit = i;
         found=1;
         break;
      }

      temp <<=1;
   }

   if (found==0)
   {  /* the block size is not power 2s */
      return NULL;
   }
   
   interrupt_saved = spin_lock(lock_num);   
   
   /* serach the pool handle to see if the pool is created */
   for (i=0;i<MAX_NUM_POOL;i++)
   {
      pHnd = &MP_SBM_hnd.pool_handler[i];
      if (  (pHnd->magic == MP_BUF_MAGIC) &&
            (pHnd->valid ==1) &&
            (pHnd->pool_addr == buf_pool_addr) &&
            (pHnd->pool_size == buf_pool_size) &&
            (pHnd->block_size == block_size) &&
            (pHnd->spinlock_num == lock_num)
         )
      {  /* found the matched handler */
         pHnd->ref_cnt++;
         spin_unlock(lock_num,interrupt_saved);         
         pHnd = &MP_SBM_hnd.pool_handler[i];

         return pHnd;
      }      
   }

   /* handler is not found, need to create one */
   /* try to find the free handler from the BM module */
   found =0;
   for (i=0;i<MAX_NUM_POOL;i++)
   {
      
      if ( (pHnd->magic == MP_BUF_MAGIC) &&
           (pHnd->valid ==1) ) 
      {
         continue;
      }

      found =1; 
      break;    
   }

   if (found==0)
   {
      /* there are no more handlers in the BM module */
      spin_unlock(lock_num,interrupt_saved);
      
      return NULL;
   }
      
   /* here we found out free handle */
   pHnd->magic = MP_BUF_MAGIC;
   pHnd->ref_cnt++;
   pHnd->valid =1;
   pHnd->pool_type =0;  /* fix size data block pool */
   
   /* save the spin lock number */
   pHnd->spinlock_num = lock_num;

   /* save the parameter for reference */
   pHnd->pool_addr   = buf_pool_addr;
   pHnd->pool_size   = buf_pool_size;
   pHnd->block_size  = block_size;
   pHnd->block_size_shift = shift_bit;
   
   pHnd->offset_type = OFFSET_TYPE_REMOTE_REF;   /* default remote type */
   
   /* align the address to 4 bytes boundary for BD */
   addr = ((buf_pool_addr+3)>>2)<<2;

   /* we need to make sure the state end in word 
      the extra 8 byte is used for word alignment
   */
   actual_size = buf_pool_size - (addr-buf_pool_addr)-8;   

   /* compute the number of buffers in the buf_pool */
   num_block = actual_size/block_size;
   pHnd->num_buf = num_block; 

   /* set up buf base address */
   pHnd->buf_base_addr = (UINT32 ) addr;

   /* try to figure out how many BDs can be stored in the bd_pool 
      assume the bd_pool_address is word aligned
      Memory layout:

      BD            : (12 bytes * N)
      free_bd_list  : (2  bytes * N)
      free_buf_list : (2 bytes * num_buf )
      ref_cnt_array : (2 bytes * num_buf )

   */

   temp = bd_pool_size/BYTES_PER_STORE_IN_BF - (sizeof(UINT16) + sizeof(UINT16)) * num_block;

   num_block = temp / (sizeof(UINT16) + sizeof(SBD_s));
   pHnd->num_bd = num_block;

   /* set up BD array base address */
   pHnd->bd_array = (SBD_s *)bd_pool_addr;

   /* setup free bd list array */
   pHnd->free_bd_list_array = (UINT16 *)( (UINT32) pHnd->bd_array + num_block *sizeof(SBD_s));

   /* set up the free buf list array */
   pHnd->free_buf_list_array = (UINT16 *) ((UINT32) pHnd->free_bd_list_array + num_block * sizeof(UINT16));
   
   /* set up the ref_cnt array */
   pHnd->ref_cnt_array = (UINT16 *) ( (UINT32) pHnd->free_buf_list_array + (UINT32)pHnd->num_buf * sizeof(UINT16));
             
   /* reset the BD, BUF tracking and ref_cnt */
   MP_SBM_pool_reset(pHnd);     

   spin_unlock(lock_num,interrupt_saved);
   
   return pHnd;
}


/*
 *  FUNCTION: MP_SBM_pool_delete
 *
 *  PARAMETERS:
 *      input   : 
 *          pHnd:       pointer of pool handler created by MP_SBM_pool_create
 *      output  : no
 *  
 *  DESCRIPTION:
 *      This function removes or deletes the memory pool from BM module.
 *
 *  RETURNS: 
 *      status (0 for success, otherwise there are errors)
 */

SBM_STATUS_t MP_SBM_pool_delete(MP_SPOOL_HND_s *pHnd)
{
   UINT32 i;
   MP_SPOOL_HND_s *pTemp;
   UINT32 interrupt_saved,lock_num;
   
   lock_num = pHnd->spinlock_num;
   
   interrupt_saved = spin_lock(lock_num);   
   
   for (i=0;i<MAX_NUM_POOL;i++)
   {
      pTemp= &MP_SBM_hnd.pool_handler[i];
      if (pTemp==pHnd)
      {
         MP_SBM_hnd.pool_handler[i].ref_cnt--;
         if (MP_SBM_hnd.pool_handler[i].ref_cnt==0)
         {  /* it is time to delete */            
            pHnd->valid = 0;

            MP_SBM_hnd.pool_handler[i].magic = 0;            
         }
         spin_unlock(lock_num,interrupt_saved);
         return SBM_SUCCESS;
      }
   }

   spin_unlock(lock_num,interrupt_saved);
   return SBM_OUT_OF_HANDLER;
}

/*
 *  FUNCTION: __MP_SBM_pool_malloc_one_buf
 *
 *  PARAMETERS:
 *      input:
 *          pHnd:   pointer of pool handler
 *          byte:   request number of bytes (less than block size)
 *      output: no
 *
 *  DESCRIPTION:
 *      this is optimized malloc function for only single data buffer
 *       
 *  RETURNS: 
 *      pointer ot allocated BD. NULL if there are errors
 */

SBD_s * __MP_SBM_pool_malloc_one_buf(MP_SPOOL_HND_s *pHnd,UINT32 byte)
{   
   UINT32 buf_addr,buf_idx;
   SBD_s *phead;
   UINT32 interrupt_saved,lock_num;          
#ifdef BUFFER_DBG  
   UINT32 num_bk;
#endif

   lock_num = pHnd->spinlock_num;
   interrupt_saved = spin_lock(lock_num);      

   /* some error chcking 
      make sure we have enough BDs and buffers
   */

   if ( (pHnd->num_free_buf == 0) || (pHnd->num_free_bd == 0) )
   {
      /* Restore interrupts.  */
      spin_unlock(lock_num,interrupt_saved);
      //PROJ_OOPS;
      return NULL;
   }                 
   
   /* get the free (available ) bd from bd_free_list_array 
      this is two step reference
      1. get the index first from bd_free_list_arry;
      2. get the actual BD from BD_array
   */          

   phead = &(pHnd->bd_array[pHnd->free_bd_list_array[pHnd->head_free_bd++]]);

   /* handler the index wrap around */
   if (pHnd->head_free_bd >= pHnd->num_bd)
       pHnd->head_free_bd =0;

   buf_idx = pHnd->free_buf_list_array[pHnd->head_free_buf++];

   if (pHnd->ref_cnt_array[buf_idx])
   {
      spin_unlock(lock_num,interrupt_saved);
      //PROJ_OOPS;
      return NULL;
   }
   /* set buf ref cnt to 1 */
   pHnd->ref_cnt_array[buf_idx] = 1;
   
   /* get the free buf from free_buf_list array */                 
   buf_addr = buf_idx*pHnd->block_size/BYTES_PER_STORE_IN_BF + pHnd->buf_base_addr ;

   /* handler the index wrap around */ 

   if (pHnd->head_free_buf>=pHnd->num_buf)
       pHnd->head_free_buf =0;        

   /* 
      update the BD fields
      len      : 
      offset   :
      other fields are reset during init or free 
      move the free function 
   */

   phead->len   = byte;
   // convert to octet address
   buf_addr =BYTES_PER_STORE_IN_BF * buf_addr;
   
   __MP_SBM_SET_OFFSET(phead,(buf_addr - MP_SBM_hnd.BM_base_address));

   /* note the default ref_cnt is zero, it mean the buffer is referenced no more
      than 1 times.

      no operation is required in here
      during the init and free, the buf ref count is reset to zero
   */
           
   phead->flags = EOP_LAST_BLOCK;  /* this is last block in the BD chain */
   phead->pnext = NULL;

   /* update the new bd head index */     
   pHnd->num_free_bd -=1;   

   /* update the new buf head index */
   pHnd->num_free_buf -=1;   

#ifdef BUFFER_DBG   
   /* number of used blocks */
   num_bk = pHnd->num_buf - pHnd->num_free_buf;
   if (pHnd->high_water_mark_block < num_bk )
      pHnd->high_water_mark_block = num_bk;  
      
   num_bk = pHnd->num_bd - pHnd->num_free_bd;
   if (pHnd->high_water_mark_bd < num_bk )
      pHnd->high_water_mark_bd = num_bk;  
#endif
 
   /* Restore interrupts.  */
   spin_unlock(lock_num,interrupt_saved);
   
   return phead;
}


/*
 *  FUNCTION: MP_SBM_pool_malloc
 *
 *  PARAMETERS:
 *      input      
 *          pHnd     : pool handler
 *          byte     : required bytes for the specified pool
 *      output      : no
 *
 *  DESCRIPTION:
 *      This function will allocate the required bytes from the memory poll. 
 *      The returned data is the BD pointer to linked list frame buffer. 
 *
 *      Note : the following restriction is applied.
 *      1. no more data_offset per each data buffer even though we pass the 
 *         parameter data_offset. It is for backward compatible. Later, 
 *         I will remove this parameter
 *      2. use shift to implement the division
 *      3. buffer reference count is zero
 *      4. BD reset is moved to free function. During malloc time, the BD 
 *         is reset to zero by init or free function
 *  RETURNS: 
 *      pointer of header of BD chain. NULL if there are errors
 *
 */

SBD_s * MP_SBM_pool_malloc(MP_SPOOL_HND_s *pHnd,UINT32 byte,UINT16 data_offset)
{   
   UINT32 i,num_bk,remainder,block_size;
   UINT32 buf_addr;
   UINT32 bd_head_idx,buf_head_idx,buf_idx;  
   SBD_s *pcur,*pprev,*phead;
   UINT32 interrupt_saved,lock_num;          

   (void)data_offset;      /* avoid compiler warning */
   
   /* find out the number of frame buffers to hold the required bytes 
      Xailin suggest to use mask and shift, don't use div
      The block size is power of 2.
      
      num_bk = byte / actual_blk_size;
   */
   block_size = pHnd->block_size;  

   if (byte<=block_size)
   {
      phead = __MP_SBM_pool_malloc_one_buf(pHnd,byte);
      return phead;
   }
   
   //num_bk = byte /block_size;
   num_bk = byte>>(pHnd->block_size_shift);
   remainder = byte - (num_bk<<pHnd->block_size_shift); 
#if 0
   num_bk = byte /block_size;
   remainder = byte - (num_bk*block_size); 
#endif
   if ( remainder )
      num_bk++;   
   
   lock_num = pHnd->spinlock_num;
   interrupt_saved = spin_lock(lock_num);      
   
   /* some error chcking 
      make sure we have enough BDs and buffers
   */
   if ( (pHnd->num_free_buf<num_bk) || (pHnd->num_free_bd<num_bk) )
   {
      /* Restore interrupts.  */
      spin_unlock(lock_num,interrupt_saved);
      //PROJ_OOPS;
      return NULL;
   }

   pprev =0;   /* start to find the BD in the free BD array */                  

   bd_head_idx    = pHnd->head_free_bd;
   buf_head_idx   = pHnd->head_free_buf;      
   
   for (i=0;i<num_bk;i++)
   {
      /* get the free (available ) bd from bd_free_list_array 
         this is two step reference
         1. get the index first from bd_free_list_arry;
         2. get the actual BD from BD_array
      */          

      pcur = & (pHnd->bd_array[pHnd->free_bd_list_array[bd_head_idx++]]);
      
      /* handler the index wrap around */
      if (bd_head_idx>=pHnd->num_bd)
         bd_head_idx =0;

      buf_idx = pHnd->free_buf_list_array[buf_head_idx++];

      if (pHnd->ref_cnt_array[buf_idx])
      {
         //PROJ_OOPS;
         spin_unlock(lock_num,interrupt_saved);
         return NULL;
      }
      /* set the buf ref count to 1 */
      pHnd->ref_cnt_array[buf_idx] = 1;
      
      /* get the free buf from free_buf_list array */                    
      buf_addr = buf_idx*block_size/BYTES_PER_STORE_IN_BF + pHnd->buf_base_addr ;
      
      /* handler the index wrap around */ 
      if (buf_head_idx>=pHnd->num_buf)
         buf_head_idx =0;        
         
      /* update the BD fields
         len      : 
         offset   :
         other fields are reset during init or free

         move the free function 
      */
 
      pcur->len   = block_size;
      buf_addr = buf_addr * BYTES_PER_STORE_IN_BF;

      __MP_SBM_SET_OFFSET(pcur,(buf_addr - MP_SBM_hnd.BM_base_address));

      /* note the default ref_cnt is zero, it mean the buffer is referenced no more
         than 1 times.

         no operation is required in here
         during the init and free, the buf ref count is reset to zero
      */
         
      /* link the prevoius's next pointer to current */
      if (pprev!=0)
      {         
         pprev->pnext = pcur;
      }
      else
      {  /* this is header of linked list */
         phead = pcur;
      }
      /* update the pprev */
      pprev = pcur;     
      
   }

   /* terminate the last block */
   if (remainder)
      pcur->len = remainder;

   pcur->flags = EOP_LAST_BLOCK;  /* this is last block in the BD chain */
   pcur->pnext = NULL;
   
   /* update the new bd head index */
   pHnd->head_free_bd = bd_head_idx;      
   pHnd->num_free_bd -=num_bk;   
   
   /* update the new buf head index */
   pHnd->head_free_buf = buf_head_idx;      
   pHnd->num_free_buf -=num_bk;   

#ifdef BUFFER_DBG   
   /* number of used blocks */
   num_bk = pHnd->num_buf - pHnd->num_free_buf;
   if (pHnd->high_water_mark_block < num_bk )
      pHnd->high_water_mark_block = num_bk;  
      
   num_bk = pHnd->num_bd - pHnd->num_free_bd;
   if (pHnd->high_water_mark_bd < num_bk )
      pHnd->high_water_mark_bd = num_bk;  
         
#endif
   /* Restore interrupts.  */
   spin_unlock(lock_num,interrupt_saved);
   
   return phead;
}

/*
 *  FUNCTION: MP_SBM_pool_free
 *
 *  PARAMETERS:
 *      input       
 *          pHnd     : the pointer to pool handler
 *          pbuf     : the header BD of linked list BD
 *      output:     no
 *
 *  DESCRIPTION:
 *      This function will free the memory buffer to the specified memory pool
 *  RETURNS: 
 *      status
 */

SBM_STATUS_t MP_SBM_pool_free(MP_SPOOL_HND_s *pHnd,SBD_s *pbuf)
{   
   UINT32 bd_tail_idx,buf_tail_idx,idx; 
   SBD_s *pbd;       
   UINT32 interrupt_saved,lock_num;  
   UINT32 buf_ref_cnt;
   UINT32 buf_addr;   
   UINT32 shift_bit;      
   UINT32 *pbd_data;
   
   if (pbuf==NULL)
   {      
      return SBM_NULL_POINTER;
   }             
   
   lock_num = pHnd->spinlock_num;
   interrupt_saved = spin_lock(lock_num);  
         
   pbd = (SBD_s *)(pbuf);      

   bd_tail_idx    = pHnd->tail_free_bd;
   buf_tail_idx   = pHnd->tail_free_buf;
   
   // this is byte-wide shift
   shift_bit  = pHnd->block_size_shift;
#if defined (CM3)

#else
   //convert to word-wide shift
   shift_bit -=1;
#endif   
   do
   {
      /* compute the buffer index */

      buf_addr = __MP_SBM_GET_OFFSET(pbd) + MP_SBM_hnd.BM_base_address;  
      
      buf_addr = buf_addr / BYTES_PER_STORE_IN_BF;
      
      /* don't use divide, use mask and shift 
         idx = (buf_addr - buf_base_addr)/block_size;
      */
      idx = (buf_addr - pHnd->buf_base_addr) >> (shift_bit);

      /* error checkcing */
      if (idx>=pHnd->num_buf)
      {
         //PROJ_OOPS;
         /* error */
         spin_unlock(lock_num,interrupt_saved);
         return SBM_ERR_BD;
      }

      /* get the ref cnt for this buffer */      
      buf_ref_cnt = pHnd->ref_cnt_array[idx];

      if (buf_ref_cnt==0)
      {
         //PROJ_OOPS;
         spin_unlock(lock_num,interrupt_saved);
         return SBM_ERR_BD;
      }
      
      if (buf_ref_cnt==1)
      {  /* this buffer is reference only once, we can free it */

         /* handler the index wrap around */
         buf_tail_idx++;
         if (buf_tail_idx>=pHnd->num_buf)
            buf_tail_idx =0;

         /* save the buffer index to to free list */         
         pHnd->free_buf_list_array[buf_tail_idx]=idx;
         /* update the stat */             
         pHnd->num_free_buf++;
      }

      pHnd->ref_cnt_array[idx]--;
      
      /* start to free the BD */
      idx = ((UINT32) pbd- (UINT32) (pHnd->bd_array))/sizeof(SBD_s);
      if (idx>=pHnd->num_bd)
      {
         //PROJ_OOPS;
         /* error */
         spin_unlock(lock_num,interrupt_saved);
         return SBM_ERR_BD;
      }

      /* handler the index wrap around */
      bd_tail_idx++;
      if (bd_tail_idx>=pHnd->num_bd)
         bd_tail_idx = 0;

      /* update the free list */      
      pHnd->free_bd_list_array[bd_tail_idx] = idx;
      
      /* update the stat */
      pHnd->num_free_bd++;

      /* avoid GHS compiling errors */
      pbd_data = (UINT32 *)((UINT32)(pbd));  
      
      pbd = pbd->pnext;       /* traverse the BD linked list chain */

      /* reset the bd data in 2nd and 3rd words
         malloc don't need to reset these fields         
      */      
      pbd_data[1]=0;
      pbd_data[2]=0;
           
   } while (pbd);

   /* update the tail of free BD */
   pHnd->tail_free_bd = bd_tail_idx;
   pHnd->tail_free_buf= buf_tail_idx;
   
   /* Restore interrupts.  */
   spin_unlock(lock_num,interrupt_saved);

   return SBM_SUCCESS;
}


/*
 *  FUNCTION: MP_SBM_Get_Buf_Ref_Cnt
 *
 *  PARAMETERS:
 *      input       
 *          pHnd     : the pointer to pool handler
 *          pbuf     : the header BD of linked list BD
 *      output:     no
 *
 *  DESCRIPTION:
 *      this function returns the buffer reference counts
 *
 *  RETURNS: 
 *      reference count of BD buffer
 */

UINT16 MP_SBM_Get_Buf_Ref_Cnt(MP_SPOOL_HND_s *pHnd,SBD_s *pbuf)
{   
   UINT32 idx;
   UINT32 buf_addr;       
   SBD_s *pbd;          
   UINT16 shift;
   
   if (pbuf==NULL)
   {      
      return 0;
   }
            
   pbd = (SBD_s *)(pbuf);

   /* compute the buffer index */

   buf_addr = __MP_SBM_GET_OFFSET(pbd) + MP_SBM_hnd.BM_base_address;  

   buf_addr = buf_addr/BYTES_PER_STORE_IN_BF;
   
   /* don't use divide, use mask and shift 

      idx = (buf_addr - buf_base_addr)/block_size;
   */
   shift = pHnd->block_size_shift;
#if defined (CM3)

#else
   shift -=1; //word aligned
#endif
   idx = (buf_addr - pHnd->buf_base_addr) >> shift;

   /* error checkcing */
   if (idx>=pHnd->num_buf)
   {
      //PROJ_OOPS;
      /* error */      
      return 0;
   }
   
   return pHnd->ref_cnt_array[idx];
   
}

/*
 *  FUNCTION: MP_SBM_Inc_Buf_Ref_Cnt
 *
 *  PARAMETERS:
 *      input       
 *          pHnd     : the pointer to pool handler
 *          pbuf     : the header BD of linked list BD
 *      output:     no
 *
 *  DESCRIPTION:
 *      this function increases the buffer reference counts by one
 *
 *  RETURNS: 
 *      reference count of BD buffer
 */
UINT16 MP_SBM_Inc_Buf_Ref_Cnt(MP_SPOOL_HND_s *pHnd,SBD_s *pbuf)
{   
   UINT32 idx;
   UINT32 buf_addr;       
   SBD_s *pbd;          
   UINT32 interrupt_saved,lock_num;  
   UINT16 shift;
   
   if (pbuf==NULL)
   {      
      return 0;
   }
     
   pbd = (SBD_s *)(pbuf);

   /* compute the buffer index */

   buf_addr = __MP_SBM_GET_OFFSET(pbd) + MP_SBM_hnd.BM_base_address;  

   buf_addr = buf_addr /BYTES_PER_STORE_IN_BF;
   
   /* don't use divide, use mask and shift 

      idx = (buf_addr - buf_base_addr)/block_size;
   */
   shift = pHnd->block_size_shift;
#if defined (CM3)

#else
   shift -=1; //word aligned
#endif
   idx = (buf_addr - pHnd->buf_base_addr) >> shift;
   
   /* error checkcing */
   if (idx>=pHnd->num_buf)
   {
      //PROJ_OOPS;
      /* error */      
      return 0;
   }

   lock_num = pHnd->spinlock_num;
   interrupt_saved = spin_lock(lock_num);   

   pHnd->ref_cnt_array[idx]++;
   
   /* Restore interrupts.  */
   spin_unlock(lock_num,interrupt_saved);

   
   return pHnd->ref_cnt_array[idx];
}

/*
 *  FUNCTION: MP_SBM_Is_Buf_Cloned
 *
 *  PARAMETERS:
 *      input       
 *          pHnd     : the pointer to pool handler
 *          pbuf     : the header BD of linked list BD
 *      output:     no
 *
 *  DESCRIPTION:
 *      this function checks if the BD is cloned. When the BD buffer
 *      reference count is more than one, it is cloned.
 *
 *  RETURNS: 
 *      reference count of BD buffer
 */
UINT16 MP_SBM_Is_Buf_Cloned(MP_SPOOL_HND_s *pHnd,SBD_s *pbuf)
{   
   UINT16 buf_ref_cnt;
   
   buf_ref_cnt = MP_SBM_Get_Buf_Ref_Cnt(pHnd,pbuf);   

   return buf_ref_cnt;
}


/*
 *  FUNCTION: MP_SBM_pool_alloc_BD
 *
 *  PARAMETERS:
 *      input       
 *          pHnd     : the pointer to pool handler *         
 *      output:     no
 *
 *  DESCRIPTION:
 *      this function will allocate one BD data structure from pool handler
 *
 *  RETURNS: 
 *      pointer of BD data structure
 */

SBD_s * MP_SBM_pool_alloc_BD(MP_SPOOL_HND_s *pHnd)
{ 
   UINT32 bd_head_idx;   
   SBD_s *pcur;
   UINT32 interrupt_saved,lock_num;   
        
   lock_num = pHnd->spinlock_num;
   interrupt_saved = spin_lock(lock_num);      

   /* some erro checking */
   if (pHnd->num_free_bd==0)
   {  /* there is no enough BD blocks */ 
   
      /* Restore interrupts.  */
      spin_unlock(lock_num,interrupt_saved);
      return NULL;
   }   
   bd_head_idx    = pHnd->head_free_bd;
   
   pcur = &(pHnd->bd_array[pHnd->free_bd_list_array[bd_head_idx++]]);
   /* check for wrap around */
   if (bd_head_idx>=pHnd->num_bd)
      bd_head_idx =0;

   /* update the new bd head index */
   pHnd->head_free_bd = bd_head_idx;      
   pHnd->num_free_bd--;
   
   pcur->flags = EOP_LAST_BLOCK;  /* this is last block in the BD chain */
   pcur->pnext = NULL;

   /* Restore interrupts.  */
   spin_unlock(lock_num,interrupt_saved);
   
   return pcur;

}

/*
 *  FUNCTION: MP_SBM_pool_free_BD
 *
 *  PARAMETERS:
 *      input       
 *          pHnd     : the pointer to pool handler 
 *          pbd      : the pointer of BD 
 *      output:     no
 *
 *  DESCRIPTION:
 *      this function will free one BD data structure to pool handler
 *
 *  RETURNS: 
 *      pointer of BD data structure
 */
SBM_STATUS_t MP_SBM_pool_free_BD(MP_SPOOL_HND_s *pHnd,SBD_s *pbd)
{     
   UINT32 bd_tail_idx,idx;        
   UINT32 interrupt_saved,lock_num;     
   UINT32 *pbd_data;
   
   if (pbd==NULL)
   {      
      return SBM_NULL_POINTER;
   }             
   
   lock_num = pHnd->spinlock_num;
   interrupt_saved = spin_lock(lock_num);  
            
   bd_tail_idx    = pHnd->tail_free_bd;      

   /* start to free the BD */
   idx = ((UINT32) pbd- (UINT32) (pHnd->bd_array))/sizeof(SBD_s);
   if (idx>=pHnd->num_bd)
   {
      //PROJ_OOPS;
      /* error */
      spin_unlock(lock_num,interrupt_saved);
      return SBM_ERR_BD;
   }

   /* handler the index wrap around */
   bd_tail_idx++;
   if (bd_tail_idx>=pHnd->num_bd)
      bd_tail_idx = 0;

   /* update the free list */      
   pHnd->free_bd_list_array[bd_tail_idx] = idx;
   
   /* update the stat */
   pHnd->num_free_bd++;

   /* avoid GHS compiling errors */
   pbd_data = (UINT32 *)((UINT32)(pbd));  
     
   /* reset the bd data in 2nd and 3rd words
      malloc don't need to reset these fields         
   */      
   pbd_data[1]=0;
   pbd_data[2]=0;
             
   /* update the tail of free BD */
   pHnd->tail_free_bd = bd_tail_idx;   
   
   /* Restore interrupts.  */
   spin_unlock(lock_num,interrupt_saved);

   return SBM_SUCCESS;

}


/*
 *  FUNCTION: MP_SBM_Clone_Buf
 *
 *  PARAMETERS:
 *      input       
 *          pHnd     : the pointer to pool handler 
 *          pbuf     : the pointer of BD buffer
 *      output:     no
 *
 *  DESCRIPTION:
 *      this function will clone one BD data sturcture to pbuf. 2nd and 3th
 *      word in BD are copied from pbuf BD.
 *
 *  RETURNS: 
 *      pointer of BD data structure
 */
SBD_s * MP_SBM_Clone_Buf(MP_SPOOL_HND_s *pHnd,SBD_s *pbuf)
{   
   SBD_s *pbd;     
   UINT32 *dst,*src;
   
   pbd = MP_SBM_pool_alloc_BD(pHnd);
   if (pbd)
   {

     /* update the reference conunt */
     MP_SBM_Inc_Buf_Ref_Cnt(pHnd,pbuf);

     /* copy src' bd to dst's bd */
     src = (UINT32 *)( (UINT32)pbuf );
     dst = (UINT32 *)( (UINT32)pbd  );

  //   *dst++ = *src++;  /* first word */   
     dst[1] = src[1];  /* second word */
     dst[2] = src[2];    /* thrid word */
   }

   return pbd;
}

/*
 *  FUNCTION: MP_SBM_Clone_Buf_Chain
 *
 *  PARAMETERS:
 *      input       
 *          pHnd     : the pointer to pool handler 
 *          pbuf     : the pointer of BD buffer
 *      output:     no
 *
 *  DESCRIPTION:
 *      this function will clone a BD data structure chain to pbuf.
 *
 *  RETURNS: 
 *      pointer of BD data structure
 */
SBD_s *MP_SBM_Clone_Buf_Chain(MP_SPOOL_HND_s *pHnd, SBD_s *psbd)
{   
   SBD_s *psbd_new;
   SBD_s *psbd_prev;
   SBD_s *psbd_temp;

  for (psbd_prev = psbd_new = (SBD_s *) NULL; psbd; psbd = psbd->pnext)
  {
    /* allocate a new SBD link */
    psbd_temp = MP_SBM_pool_alloc_BD(pHnd);
    if (!psbd_temp)
    {
      if (psbd_new)
      {
        MP_SBM_pool_free(pHnd, psbd_new);
        psbd_new = (SBD_s *) NULL;
        break;
      }
    }

    /* copy offsets, lengths, etc. */
    memcpy(psbd_temp, psbd, sizeof(*psbd));

    /* preset next pointer */
    psbd_temp->pnext = (SBD_s *) NULL;

    /* update the reference count */
    MP_SBM_Inc_Buf_Ref_Cnt(pHnd, psbd_temp);

    /* update next SBD_s link */
    if (psbd_prev)
    {
      psbd_prev->pnext = psbd_temp;
    }
    psbd_prev = psbd_temp;

    /* set return SBD_s */
    if (!psbd_new)
    {
      psbd_new = psbd_temp;
    }
  }

   return psbd_new;
}


/*===========================================================================
  Function     : BM_pool_get_free_bytes
  Description  :
   This function is helper function. It get the total free bytes in the 
   specified memory pool.

  Parameters:
   input       
      pHnd     : the pointer to pool handler      
      
   output      : no

   return value
               : total free byte in the specified pool
      
 ============================================================================*/
UINT32 MP_SBM_pool_get_free_bytes(const MP_SPOOL_HND_s *pHnd)
{
   return (pHnd->num_free_buf * pHnd->block_size);   
}

/*===========================================================================
  Function     : MP_SBM_pool_get_block_size
  Description  :
   This function is helper function. It get the block size of specified 
   memory pool. This size is specifed during the pool creation

  Parameters:
   input       
      pHnd     : the pointer to pool handler      
      
   output      : no

   return value
               : block size of data block
      
 ============================================================================*/
UINT16 MP_SBM_pool_get_block_size(const MP_SPOOL_HND_s *pHnd)
{
   return pHnd->block_size;
}

/*===========================================================================
  Function     : BM_pool_reset
  Description  :
   This function will reclaim all data block and BDs

   Note: The caller must make sure there are no buffers are used by any module
   otherwise the behavior is undefined.

   This is BIG free function.
  Parameters:
   input       
      pHnd     : pool handler                                
   
   return value
      status value. 0 when it is sucessful, otherwise there are some errors 
      
 ============================================================================*/
SBM_STATUS_t __MP_SBM_lpool_reset(MP_SPOOL_HND_s *pHnd)
{   
   UINT32 i,num_bd;   
   SBD_s *pbd;
   UINT32 *pbd_data;
       
   /* num of block is equal to number of BD */    
   num_bd = pHnd->num_bd;           
   
   /* init the BD */
   for (i=0;i<num_bd;i++)
   {    
      pbd = &(pHnd->bd_array[i]);   
      
      /* don't use the memset */      
      pbd_data = (UINT32*)( (UINT32)pbd);
      *pbd_data++ = 0;
      *pbd_data++ = 0;
      *pbd_data = 0;

      pbd->flags = EOP_LAST_BLOCK;  /* single BD for burst */ 
              
   }
   
   pHnd->total_used_bytes =0;     
   pHnd->num_free_bd = pHnd->num_bd;
   return SBM_SUCCESS;

}


/*===========================================================================
  Function     : SBM_lpool_create
  Description  :
   This function will create a linear memory pool. Each block is variable
   size. 

   
  Parameters:
   input       
      pool_addr      : the pool staring address 
      pool_size      : total bytes in this pool
      num_bd         : maximum BDs in this pool
                                  
   output      : none
   
   return value
      pool handler when it is sucessful
      
   
 ============================================================================*/
MP_SPOOL_HND_s * MP_SBM_lpool_create(UINT32 pool_addr,UINT32 pool_size,
   UINT16 num_bd,UINT16 lock_num)
{

   UINT32 i;
   MP_SPOOL_HND_s *pHnd;
   UINT32 addr,actual_size,found;
   UINT32 interrupt_saved;
      
   interrupt_saved = spin_lock(lock_num);   
   
   /* serach the pool handle to see if the pool is created */
   for (i=0;i<MAX_NUM_POOL;i++)
   {
      pHnd = &MP_SBM_hnd.pool_handler[i];
      if ( (pHnd->magic == MP_BUF_MAGIC) &&
            (pHnd->valid ==1) &&
            (pHnd->pool_addr == pool_addr) &&
            (pHnd->pool_size == pool_size) &&  
            (pHnd->num_bd    == num_bd) &&
            (pHnd->spinlock_num == lock_num)
         )
      {  /* found the matched handler */
         pHnd->ref_cnt++;
         spin_unlock(lock_num,interrupt_saved);
         return pHnd;
      }      
   }

   /* handler is not found, need to create one */
   /* try to find the free handler from the BM module */
   found =0;
   for (i=0;i<MAX_NUM_POOL;i++)
   {
      pHnd = &MP_SBM_hnd.pool_handler[i];
      
      if ( (pHnd->magic == MP_BUF_MAGIC) &&
           (pHnd->valid ==1) ) 
      {
         continue;
      }

      found =1; 
      break;    
   }

   if (found==0)
   {
      /* there are no more handlers in the BM module */
      spin_unlock(lock_num,interrupt_saved);
      
      return NULL;
   }

   /* here we found out free handle */
   pHnd->magic = MP_BUF_MAGIC;
   pHnd->ref_cnt++;
   pHnd->valid =1;
   pHnd->pool_type =1;  /* linear pooll */
   
   /* save the spin lock number */
   pHnd->spinlock_num = lock_num;   

   /* save the user input parameters */
   pHnd->pool_addr = pool_addr;
   pHnd->pool_size = pool_size;
   /* num of block is equal to number of BD 
      user need to specify the number of BD, each buf is 
      represented by one BD
   */   
   pHnd->num_bd = num_bd;   
      
   /* align the BD address to 4 bytes boundary */
   addr = ((pool_addr+3)>>2)<<2;
   
   /* setup BD base address, round to word aligned */      
   pHnd->bd_array = (SBD_s *) addr;   
      
   /* set up the state base address 
      in the linear buffer, we don't need linked list to track which
      buffer is free beacuse in each frame we need to reuse the
      buffer. we use the bd_state_addr to track

   */
   pHnd->free_bd_list_array = (UINT16 *)(addr + num_bd *sizeof(SBD_s));

   /* setup the data buffer base address */
   pHnd->buf_base_addr =  ((UINT32)pHnd->free_bd_list_array) + num_bd*sizeof(UINT16);   
   
   actual_size = pool_size - (addr-pool_addr);   
   /* compute the total free data buffer in the linear pool 
      number of (BD + bd_state)
   */
   
   pHnd->total_free_bytes = actual_size - num_bd *(sizeof(SBD_s) +sizeof(UINT16));
   
   /* reset the BD */
   __MP_SBM_lpool_reset(pHnd);

#ifdef BUFFER_DBG  
   /* high water mark */
   pHnd->high_water_mark_bytes=0;
#endif
   spin_unlock(lock_num,interrupt_saved);
   return pHnd;
   
}

/*===========================================================================
  Function     : BM_lpool_delete
  Description  :
   This function will delete the linear pool from the BM module

   
  Parameters:
   input       
      pHnd        : linear pool handler created by BM_lpool_create
      
   output      : none
   
   return value
      0 when it is sucessful
      
   
 ============================================================================*/
SBM_STATUS_t MP_SBM_lpool_delete(MP_SPOOL_HND_s *pHnd)
{
   /* just call the pool delete function 
      if later, we need to more feature, we can re-write this function 
   */
    return MP_SBM_pool_delete(pHnd);   
}


/*===========================================================================
  Function     : BM_lpool_malloc
  Description  :
   This function will allocate the required bytes from the linear memory poll. 
   The returned value is single BD. 

  Parameters:
   input      
      pHnd     : the pointer of linear pool handler
      bd_index : BD index whixh is in range 0 -- (max-1)      
      byte     : required bytes for this BD.
      
   output      : no

   return value
      the pointer of BD If there is no enough memory
      in the poll, the returned pointer is NULL

 ============================================================================*/
SBD_s * MP_SBM_lpool_malloc(MP_SPOOL_HND_s *pHnd,UINT16 bd_index,UINT32 byte)
{     

   UINT32 available_byte,offset;
   SBD_s *pbuf;   
   UINT32 interrupt_saved,lock_num;   
             
   lock_num = pHnd->spinlock_num;
   interrupt_saved = spin_lock(lock_num);                      
   
   available_byte = pHnd->total_free_bytes - pHnd->total_used_bytes; 
   /* error check first */
   if ( (byte>available_byte) || 
        (bd_index>=pHnd->num_bd) )
   {      
      spin_unlock(lock_num,interrupt_saved);
      return NULL;
   }

   /*
      This is used for linear buffer, call this function by passing 
      burst index and request bytes

      return only single BD for linear memory

   */
   /* start to find the free buffer blocks in memory pool */   
   pbuf = (SBD_s *)&(pHnd->bd_array[bd_index]);
   
   /* since the free or reset function will set the BD content to default value 
      we don't need to any init here
   */   
   
   pbuf->len = byte;
   offset = pHnd->buf_base_addr + pHnd->total_used_bytes;
   
   offset = offset * BYTES_PER_STORE_IN_BF;
   
   __MP_SBM_SET_OFFSET(pbuf,(offset - MP_SBM_hnd.BM_base_address));
  
   /* update the pool tracking variables */
   pHnd->total_used_bytes += byte;     /* update the used bytes */
   pHnd->num_free_bd--;   

   /* mark this bd as used */
   pHnd->free_bd_list_array[bd_index]=1;
     
   spin_unlock(lock_num,interrupt_saved);

   return pbuf;
  
}

/*===========================================================================
  Function     : BM_lpool_free
  Description  :
   This function returns the memory to the linear pool.

  Parameters:
   input      
      pHnd     : the pointer of linear pool handler
      pbuf     : the pointer of BD created by BM_lppol_malloc
      
   output      : no

   return value
      0 when it is sucessful.
 ============================================================================*/
SBM_STATUS_t MP_SBM_lpool_free(MP_SPOOL_HND_s *pHnd,SBD_s *pbuf)
{
   UINT32 idx;  
   UINT32 interrupt_saved,lock_num;         
   UINT32 *pbd_data;
   
   if (pbuf==NULL)
   {      
      return SBM_NULL_POINTER;
   }
      
   lock_num = pHnd->spinlock_num;
   interrupt_saved = spin_lock(lock_num);             
  
   idx = ((UINT32) pbuf- (UINT32) (pHnd->bd_array))/sizeof(SBD_s);      

   pHnd->free_bd_list_array[idx]=1;

   pHnd->total_used_bytes -= pbuf->len;     /* update the used bytes */
   
   /* post reset the content of BD to default */
   pbd_data = (UINT32 *)((UINT32)pbuf);
   *pbd_data++ =0;
   *pbd_data++ =0;
   *pbd_data =0;
       
   pbuf->flags = EOP_LAST_BLOCK;  /* this is last block in the BD chain */
   pHnd->num_free_bd++; 
   
   spin_unlock(lock_num,interrupt_saved);   

   return SBM_SUCCESS;
  
}

/*===========================================================================
  Function     : BM_lpool_set_used_bytes
  Description  :
   This function set the used byte field in the linear pool. 

   Note: The used bytes in the linear pool are tracked by BM_lpool_create. This
   function will override the value. Be cautious when using this function.

  Parameters:
   input      
      pHnd     : the pointer of linear pool handler       
      byte     : total used byte in the linear pool.
      
   output      : no

   return value
      the pointer of BD If there is no enough memory
      in the poll, the returned pointer is NULL

 ============================================================================*/
SBM_STATUS_t MP_SBM_lpool_set_used_bytes(MP_SPOOL_HND_s *pHnd,UINT32 byte)
{ 
   UINT32 interrupt_saved,lock_num;

   lock_num = pHnd->spinlock_num;
   interrupt_saved = spin_lock(lock_num);     
   
   /* update the pool status */   
   pHnd->total_used_bytes = byte;     /* update the used bytes */   
   
   spin_unlock(lock_num,interrupt_saved);

   return SBM_SUCCESS;
  
}

/*===========================================================================
  Function     : MP_SBM_lpool_reset
  Description  :
   This function will reclaim all data block and BDs

   Note: The caller must make sure there are no buffers are used by any module
   otherwise the behavior is undefined.

   This is BIG free function.
  Parameters:
   input       
      pHnd     : pool handler                                
   
   return value
      status value. 0 when it is sucessful, otherwise there are some errors 
      
 ============================================================================*/
SBM_STATUS_t MP_SBM_lpool_reset(MP_SPOOL_HND_s *pHnd)
{
   UINT32 i,num_bd;   
   SBD_s *pbd;
   UINT32 interrupt_saved,lock_num;

   lock_num = pHnd->spinlock_num;
   interrupt_saved = spin_lock(lock_num);          

   num_bd = pHnd->num_bd;  
   
   /* init the BD */
   for (i=0;i<num_bd;i++)
   {
      pbd = &(pHnd->bd_array[i]);   
      memset(pbd,0,sizeof(SBD_s));
      pbd->flags = 1;      /* single BD for burst */            
   }
   
   pHnd->total_used_bytes =0;    
   
   spin_unlock(lock_num,interrupt_saved);
   return SBM_SUCCESS;

}

/* =========================================================================
   Common function APIS for single and multi core 
   =========================================================================
*/

/*===========================================================================
  Function     : SBM_get_baseAddr
  Description  :
   This function is helper function. it gets the base address of BM module.
   into offset.

  Parameters:
   input       : no      
      
   output      : no

   return value
               : base address of BM module
      
 ============================================================================*/
UINT32 SBM_get_baseAddr(void)
{
   return MP_SBM_hnd.BM_base_address;
}
/*===========================================================================
  Function     : SBM_setup_BD
  Description  :
   This function will set up the BD address offset and EOP to 1

  Parameters:
   input       
      offset   : physical address of BD  
      
   output      : no

   return value
               : offset of BD's buffer
      
 ============================================================================*/
void SBM_setup_BD(SBD_s *pbd,UINT32 addr)
{
   memset(pbd,0,sizeof(SBD_s));

   __MP_SBM_SET_OFFSET(pbd,SBM_convert2Offset((UINT32)(addr),OFFSET_TYPE_REMOTE_REF));
   
   pbd->flags =1;
}

/*===========================================================================
  Function     : SBM_Adj_offset
  Description  :
   This function will adjust the BD's buffer offset based on the offset type
   and data offset
  Parameters:
   input      
      pHnd     : pool handler
      pbd      : pointer to BD
   output      : no

   return value

 ============================================================================*/
SBM_STATUS_t SBM_Adj_offset(SBD_s * pbd,UINT16 data_offset)
{   
   /* error checking */
   if (data_offset>=pbd->len)
   {
      //PROJ_OOPS;
      return SBM_DAT_OFFSET;
   }
   pbd->dat_offset += data_offset;  
   pbd->len -=data_offset;      

   return SBM_SUCCESS;
}


/*===========================================================================
  Function     : SBM_get_data_addr
  Description  :
   This function is helper function. it will get the buffer data 
   physical address of BD.

  Parameters:
   input       
      pbd      : the pointer of BD     
      
   output      : no

   return value
               : physical address of BD's buffer
   
 ============================================================================*/
UINT32 SBM_get_data_addr(const SBD_s *pbd)
{
    
   if (pbd->offset_type == OFFSET_TYPE_REMOTE_REF)

   {
      return  ( __MP_SBM_GET_OFFSET(pbd) + pbd->dat_offset+MP_SBM_hnd.BM_base_address);  
   }    
   else
     return (__MP_SBM_GET_OFFSET(pbd) + (UINT32)pbd);    
}

/*===========================================================================
  Function     : SBM_get_buf_addr
  Description  :
   This function is helper function. it will get the buffer starting address
   of BD.

  Parameters:
   input       
      pbd      : the pointer of BD     
      
   output      : no

   return value
               : physical address of BD's buffer
   
 ============================================================================*/
UINT32 SBM_get_buf_addr(const SBD_s *pbd)
{
   if (pbd->offset_type == OFFSET_TYPE_REMOTE_REF)
     return (__MP_SBM_GET_OFFSET(pbd)+MP_SBM_hnd.BM_base_address);
   else
     return (__MP_SBM_GET_OFFSET(pbd) + (UINT32)pbd);
}


/*===========================================================================
  Function     : SBM_Inc_Data_offset
  Description  :
   This function is helper function. it will increase the data offset in the
   BD.

  Parameters:
   input       
      pbd      : the pointer of BD     
      
   output      : no

   return value
               : 0 success, otherwise there are errors
   
 ============================================================================*/
SBM_STATUS_t SBM_Inc_Data_offset(SBD_s * pbd,UINT16 data_offset)
{
   /* error checking */
   if (data_offset>pbd->len)
   {
      //PROJ_OOPS;
      return SBM_DAT_OFFSET;
   }
   pbd->dat_offset += data_offset;
   pbd->len -=data_offset;      

   return SBM_SUCCESS;
}

/*===========================================================================
  Function     : SBM_Dec_Data_offset
  Description  :
   This function is helper function. it will decrease the data offset in the
   BD.

  Parameters:
   input       
      pbd      : the pointer of BD     
      
   output      : no

   return value
               : 0 success, otherwise there are errors
   
 ============================================================================*/
SBM_STATUS_t SBM_Dec_Data_offset(SBD_s * pbd,UINT16 data_offset)
{
   /* error checking */
   if (data_offset>pbd->dat_offset)
   {
      //PROJ_OOPS;
      return SBM_DAT_OFFSET;
   }
   pbd->dat_offset -= data_offset;
   pbd->len +=data_offset;      

   return SBM_SUCCESS;
}

/*===========================================================================
  Function     : SBM_Terminate_BD
  Description  :
   This function is helper function. it will terminate the BD by setting 
   flags to EOP_LAST_BLOCK

  Parameters:
   input       
      pbd      : the pointer of BD     
      
   output      : no

   return value
               : nono
   
 ============================================================================*/
void SBM_Terminate_BD(SBD_s *pbd)
{
   pbd->pnext = NULL;
   pbd->flags = EOP_LAST_BLOCK;
}

/*===========================================================================
  Function     : SBM_reset_owner
  Description  :
   This function is helper function. it will reset the owner bit to zero.
   It indicates the software owns this buffers

  Parameters:
   input       
      pbd      : the pointer of BD     
      
   output      : no

   return value
               : nono
   
 ============================================================================*/
void SBM_reset_owner(SBD_s *pbd)
{  
   while (pbd)
   {
      pbd->owner =0;
      pbd = pbd->pnext;
   }        
}

/*===========================================================================
  Function     : SBM_convert2Addr
  Description  :
   This function is helper function. it converts the offset into real physical
   address.

  Parameters:
   input       
      offset   : offset of BD   
      
   output      : no

   return value
               : physical address of BD's buffer
      
 ============================================================================*/
UINT32 SBM_convert2Addr(const UINT32 offset,UINT16 offset_type)
{
   if (offset_type==OFFSET_TYPE_REMOTE_REF)
      return (offset+MP_SBM_hnd.BM_base_address);
   else
   {
      //PROJ_OOPS;
      return 0;
   }
}

/*===========================================================================
  Function     : SBM_convert2Offset
  Description  :
   This function is helper function. it converts the real physical address
   into offset.

  Parameters:
   input       
      offset   : physical address of BD  
      
   output      : no

   return value
               : offset of BD's buffer
      
 ============================================================================*/
UINT32 SBM_convert2Offset(const UINT32 addr,UINT16 offset_type)
{
   if (offset_type==OFFSET_TYPE_REMOTE_REF)
      return (addr-MP_SBM_hnd.BM_base_address);
   else
   {
      //PROJ_OOPS;
      return 0;
   }
}

/*===========================================================================
  Function     : SBM_append_bd
  Description  :
   This function chains two BD together. the previous BD (prev) can be linked 
   list BD. It will walk through the previous BD to find the tail BD, then 
   chain the BD together.   

  Parameters:
   input       : 
      pprev    : the pointer of first head BD of linked list BD.
      pcur     : the pointer of second BD which will chain to the first BD
      
   output      : no

   return value
      status   : 0 when it is sucessful, otherwise there are some errors.

Note the pcur BD is EOP terminated      
 ============================================================================*/
SBM_STATUS_t SBM_append_bd(SBD_s *pprev,SBD_s *pcur)
{   
   SBD_s *pbd;
               
   /* transverse through the linked list */
   pbd = pprev;

   while (pbd->pnext)
   {
       pbd = pbd->pnext;
   }

   pbd->pnext =  pcur;
   pbd->flags=0;  /* mark this buffer as not end of buffer */   
   
   return SBM_SUCCESS;
}

/*===========================================================================
  Function     : SBM_get_tail
  Description  :
   This function walks through the BD linked list to find the tail BD.

  Parameters:
   input       : 
      pbd      : the pointer of head BD of linked list BD.
      
   return value
      status   : the pointer to tail BD
      
 ============================================================================*/
SBD_s * SBM_get_tail(SBD_s *pbd)
{  
   while (pbd->pnext)
   {
       pbd = pbd->pnext;
   }
     
   return pbd;

}

/*===========================================================================
  Function     : SBM_get_addr_in_packet
  Description  :
   This function will figure out the physical address based on the offset
   in the linked list

   Note: it is time comsuming process because it has to figure out which 
   buffer the byte position belongs to.
   
   
  Parameters:
   input       : 
      pbuf        : pointer to packet linked list buffer
      byte_posi   : byte position (offset) in the linked list buffer
                 
                 
   output      : 
      offset    : the pointer of physical offset
      len       : the pointer of length
   
   returned    : 32 bit wide address
      
   
 ============================================================================*/
SBM_STATUS_t SBM_get_addr_in_packet(SBD_s *pbuf,UINT16 byte_posi,
      UINT32 *offset,UINT16 *bd_len)
{
   SBM_STATUS_t status;
   UINT32 len,byte_offset;
   SBD_s *pbd;    

   /* we need to tranverse the linked list buffer */         
   byte_offset = 0;
   
   *offset =0;
   /* transverse through the linked list */
   pbd = (SBD_s *) (pbuf);
   status = SBM_FAILURE;
   do 
   {
      /* current BD's LEN */
      len = pbd->len;      

      if ( (byte_posi>=byte_offset) && (byte_posi<(byte_offset + len)))
      {  /* data is in current BD */

         /* try to get the bytes */   
         *offset =  (__MP_SBM_GET_OFFSET(pbd) + pbd->dat_offset + byte_posi - byte_offset) +
                  MP_SBM_hnd.BM_base_address;        

         *bd_len = len + byte_offset - byte_posi; /* remaining BD len */
         status = SBM_SUCCESS;
         break;
      }
      pbd = pbd->pnext;      
      byte_offset +=len;
            
   } while (pbd);         
      
   return status;
}


/*===========================================================================
  Function     : SBM_get_posi_in_packet
  Description  :
   This function will return the BD in the posi of packet

   Note: it is time comsuming process because it has to figure out which 
   buffer the byte position belongs to.
   
   
  Parameters:
   input       : 
      pbuf        : pointer to packet linked list buffer
      byte_posi   : byte position (offset) in the linked list buffer
                 
                 
   output      : 
      offset    : the pointer of physical offset
      len       : the pointer of length
   
   returned    : 32 bit wide address
      
   
 ============================================================================*/
SBD_s * SBM_get_posi_in_packet(SBD_s *pbuf,UINT16 byte_posi,
         UINT32 *addr,UINT16 *bd_len,PSBD_s * pprevBD)
{      
   UINT32 len,byte_offset,relative_len;
   SBD_s *pbd,*pprev;    
   
   /* we need to tranverse the linked list buffer */         
   byte_offset = 0;
   
   *addr =0;
   /* transverse through the linked list */
   pbd = (SBD_s *) (pbuf);
   pprev = NULL;
   
   do 
   {
      /* current BD's LEN */
      len = pbd->len;      

      //if ( (byte_posi>=byte_offset) && (byte_posi<(byte_offset + len)))
      if ( byte_posi<(byte_offset + len) )
      {  /* data is in current BD */

         /* try to get the bytes */
         relative_len = byte_posi - byte_offset;

         *addr =  (__MP_SBM_GET_OFFSET(pbd) + pbd->dat_offset+relative_len) +
                  MP_SBM_hnd.BM_base_address;

         *bd_len = len - relative_len; /* remaining BD len */
         
         break;
      }
      pprev = pbd;
      pbd = pbd->pnext;      
      byte_offset +=len;
            
   } while (pbd);         

   *pprevBD = pprev;
   return pbd;
}


/*===========================================================================
  Function     : SBM_get_size_in_BD_chain
  Description  :
   This function will get the total data size in the BD chain 

  Parameters:
   input             
      pbuf     : the header BD of linked list BD
      
   output      : no

   return value: total data size in the BD chain
               : 
      
 ============================================================================*/
UINT32 SBM_get_size_in_BD_chain(SBD_s *pbuf)
{
   UINT32 total_size=0;
   SBD_s *pbd;  
      
   if (pbuf==NULL)
   {
      return 0;
   }
   pbd = pbuf;
   
   while (pbd)
   {
      total_size += pbd->len;
      /* for next BD */
      pbd = pbd->pnext;
   }
   
   return total_size;
}

/*===========================================================================
  Function     : SBM_copy_buffer
  Description  :
   quick and dirty copy the SBD_s into a flat buffer.
   need to optimize it to take advantage of BD knowledge so we are not
   walking the chain with each iteration by calling SBM_get_addr_in_packet().

  Parameters:
   input             
      pbd      : the BD chain to copy from
      offset   : zero base offset into the BD buffer to start copying from
      pbuf     : pointer to buffer to copy into
      len      : count of bytes to copy

   output      : pbuf filled with data copied from BD

   return value: count of bytes copied
      
 ============================================================================*/
UINT16 SBM_copy_buffer(SBD_s *pbd, UINT16 offset, UINT16 *pbuf, UINT16 len)
{
  UINT32 addr;
  UINT16 bd_len;
  UINT16 len_remain;
  SBM_STATUS_t status;
#if defined (CM3)
  UINT32 byte_pbuf = ((UINT32)(pbuf));
#else
  UINT32 byte_pbuf = ((UINT32)(pbuf)) << 1;
#endif

  for (len_remain = len; len_remain != 0; )
  {
    status = SBM_get_addr_in_packet(pbd, offset, &addr, &bd_len);
    if (status != SBM_SUCCESS)
    {
      break;
    }

    if (len_remain < bd_len)
    {
      bd_len = len_remain;
    }
    
    //memcpy(pbuf, (const void *) addr, bd_len);
    BF_MEMCPY(byte_pbuf, addr, bd_len);
    //pbuf += bd_len;
    byte_pbuf += bd_len;
    offset += bd_len;
    len_remain -= bd_len;
  }

  return (len - len_remain);

}

/*===========================================================================
  Function     : SBM_move_buffer
  Description  :
   quick and dirty move the SBD_s into a flat buffer.
   need to optimize it to take advantage of BD knowledge so we are not
   walking the chain with each iteration by calling SBM_get_addr_in_packet().

  Parameters:
   input             
      pbd      : the BD chain to copy from
      offset   : zero base offset into the BD buffer to start copying from
      pbuf     : pointer to buffer to copy into
      len      : count of bytes to copy

   output      : pbuf filled with data copied from BD

   return value: count of bytes copied
      
 ============================================================================*/
UINT16 SBM_move_buffer(SBD_s *pbd, UINT16 offset, UINT16 *pbuf, UINT16 len)
{
  UINT32 addr;
  UINT16 bd_len;
  UINT16 len_remain;
  UINT16 offset_tmp;
  PSBD_s prevBD;
  SBD_s *pbd_tmp;
  SBM_STATUS_t status;

  /* first pass to copy into the buffer */
  for (len_remain = len, offset_tmp = offset; len_remain != 0 && pbd; )
  {
    status = SBM_get_addr_in_packet(pbd, offset_tmp, &addr, &bd_len);
    if (status != SBM_SUCCESS)
    {
      break;
    }

    if (len_remain < bd_len)
    {
      bd_len = len_remain;
    }

    memcpy(pbuf, (const void *) addr, bd_len);

    pbuf += bd_len;
    offset_tmp += bd_len;
    len_remain -= bd_len;
  }

  /* second pass to adjust the data offsets (the "move") */
  for (len_remain = len, offset_tmp = offset; len_remain != 0 && pbd; )
  {
    pbd_tmp = SBM_get_posi_in_packet(pbd, offset_tmp, &addr, &bd_len, &prevBD);
    if (bd_len > len_remain)
    {
      bd_len = len_remain;
    }
    SBM_Inc_Data_offset(pbd_tmp, bd_len);

    offset_tmp += bd_len;
    len_remain -= bd_len;
  }
  return (len - len_remain);
}

/*===========================================================================
  Function     : SBM_set_buffer2
  Description  :
   quick and dirty method to copy a buffer into a SBD_s chain.
   need to optimize it to take advantage of BD knowledge so we are not
   walking the chain with each iteration by calling SBM_get_addr_in_packet().

  Parameters:
   input             
      pbd      : the BD chain to copy from
      offset   : zero base offset into the BD buffer to start copying from
      pbuf     : pointer to the flat buffer
      len      : count of bytes to set
      buf_off  : byte offset into the pbuf 

   output      : pbuf filled with data copied from BD

   return value: count of bytes copied
   Note: it is different from SBM_set_buffer, as it takes one extra
   parameter, so that the starting byte offset of the flat buffer can
   also be specified.
      
 ============================================================================*/
UINT16 SBM_set_buffer2(SBD_s *pbd, UINT16 offset, UINT16 *pbuf, UINT16 len,
  UINT16 pbuf_offset)
{
  UINT32 addr;
  UINT16 bd_len;
  UINT16 len_remain;
  SBM_STATUS_t status;
#if defined (CM3)
  UINT32 byte_pbuf = (((UINT32)(pbuf))     ) + pbuf_offset;
#else
  UINT32 byte_pbuf = (((UINT32)(pbuf)) << 1) + pbuf_offset;
#endif

  for (len_remain = len; len_remain != 0; )
  {
    status = SBM_get_addr_in_packet(pbd, offset, &addr, &bd_len);
    if (status != SBM_SUCCESS)
    {
      break;
    }

    if (len_remain < bd_len)
    {
      bd_len = len_remain;
    }
    //memcpy((void *) addr, pbuf, bd_len);
    BF_MEMCPY(addr, byte_pbuf, bd_len);
    offset += bd_len;
    len_remain -= bd_len;
    //pbuf += bd_len;
    byte_pbuf += bd_len;
  }

  return (len - len_remain);
}


/*===========================================================================
  Function     : SBM_set_buffer
  Description  :
   quick and dirty method to copy a buffer into a SBD_s chain.
   need to optimize it to take advantage of BD knowledge so we are not
   walking the chain with each iteration by calling SBM_get_addr_in_packet().

  Parameters:
   input             
      pbd      : the BD chain to copy from
      offset   : zero base offset into the BD buffer to start copying from
      c        : value to set buffer to
      len      : count of bytes to set

   output      : pbuf filled with data copied from BD

   return value: count of bytes copied
      
 ============================================================================*/
UINT16 SBM_set_buffer(SBD_s *pbd, UINT16 offset, UINT16 *pbuf, UINT16 len)
{
  UINT32 addr;
  UINT16 bd_len;
  UINT16 len_remain;
  SBM_STATUS_t status;
#if defined (CM3)
  UINT32 byte_pbuf = ((UINT32)(pbuf)) ;
#else
  UINT32 byte_pbuf = ((UINT32)(pbuf)) << 1;
#endif

  for (len_remain = len; len_remain != 0; )
  {
    status = SBM_get_addr_in_packet(pbd, offset, &addr, &bd_len);
    if (status != SBM_SUCCESS)
    {
      break;
    }

    if (len_remain < bd_len)
    {
      bd_len = len_remain;
    }
    //memcpy((void *) addr, pbuf, bd_len);
    BF_MEMCPY(addr, byte_pbuf, bd_len);
    offset += bd_len;
    len_remain -= bd_len;
    //pbuf += bd_len;
    byte_pbuf += bd_len;
  }

  return (len - len_remain);
}

/*===========================================================================
  Function     : SBM_memset_buffer
  Description  :
   quick and dirty memset.
   need to optimize it to take advantage of BD knowledge so we are not
   walking the chain with each iteration by calling SBM_get_addr_in_packet().

  Parameters:
   input             
      pbd      : the BD chain to copy from
      offset   : zero base offset into the BD buffer to start copying from
      c        : value to set buffer to
      len      : count of bytes to set

   output      : pbuf filled with data copied from BD

   return value: count of bytes copied
      
 ============================================================================*/
void SBM_memset_buffer(SBD_s *pbd, UINT16 offset, UINT16 c, UINT16 len)
{
  UINT32 addr;
  UINT16 bd_len;
  UINT16 len_remain;
  SBM_STATUS_t status;

  for (len_remain = len; len_remain != 0; )
  {
    status = SBM_get_addr_in_packet(pbd, offset, &addr, &bd_len);
    if (status != SBM_SUCCESS)
    {
      break;
    }

    if (len_remain < bd_len)
    {
      bd_len = len_remain;
    }
    //memset((void *) addr, c, bd_len);
    BF_MEMSET(addr, c, bd_len);
    offset += bd_len;
    len_remain -= bd_len;
  }
}

/*===========================================================================
  Function     : SBM_copy_bd
  Description  :
   quick and dirty copy the SBD_s into another SBD.
   need to optimize it to take advantage of BD knowledge so we are not
   walking the chain with each iteration by calling SBM_get_addr_in_packet().

  Parameters:
   input             
      psbd_src    : the BD chain to copy from
      offset_src  : zero base offset into the BD buffer to start copying from
      psbd_dest   : the BD chain to copy to
      offset_src  : zero base offset into the BD buffer to start copying to

   output         : psbd_dst filled with data copied from psbd_src

   return value: count of bytes copied
      
 ============================================================================*/
UINT16 SBM_copy_bd(SBD_s *psbd_src, UINT16 offset_src, SBD_s *psbd_dst, UINT16 offset_dst)
{
  UINT32 addr;
  UINT16 bd_len;
  UINT16 ui16;
  UINT16 src_len;
  UINT16 len_remain;
  SBM_STATUS_t status;

  src_len = SBM_get_size_in_BD_chain(psbd_src);
  if (src_len <= offset_src)
  {
    return 0;
  }

  for (len_remain = src_len - offset_src; len_remain != 0; )
  {
    status = SBM_get_addr_in_packet(psbd_src, offset_src, &addr, &bd_len);
    if (status != SBM_SUCCESS)
    {
      break;
    }

    if (len_remain < bd_len)
    {
      bd_len = len_remain;
    }
 
    ui16 = SBM_copy_buffer(psbd_dst, offset_dst, (UINT16 *) addr, bd_len);
    len_remain -= ui16;

    if (ui16 != bd_len)
    {
      break;
    }

    offset_src += ui16;
    offset_dst += ui16;
  }

  return (src_len - len_remain);
}


/*===========================================================================
  Function     : SBM_put_long_bigend
  Description  :
   quick and dirty method to set a long value into a SBD_s chain.
   need to optimize it to take advantage of BD knowledge so we are not
   walking the chain with each iteration by calling SBM_get_addr_in_packet().

  Parameters:
   input             
      pbd      : the BD chain to copy from
      offset   : byte offset into the BD chain 
      data     : the value to be set

   output      : 

   return value: none
   NOTE: this API will put the bytes in a big edian way, significant 
   byte will be put in low octet address, so it is different from 
   the BF_PUT_LONG
      
===========================================================================*/
void SBM_put_long_bigend(SBD_s *pbd, UINT16 offset, UINT32 data)
{
  UINT32 addr;
  UINT16 remain_len;
  SBM_STATUS_t status;

  status = SBM_get_addr_in_packet(pbd, offset, &addr, &remain_len);
  if (status != SBM_SUCCESS)
  {
    //TODO: report error 
    proj_assert(0);
  }

  if (remain_len >= 4) {
    //BF_PUT_LONG(addr, data);
    BF_PUT_BYTE(addr, (uint16)((data >> 24) & 0xff));
    addr++;
    BF_PUT_BYTE(addr, (uint16)((data >> 16) & 0xff));
    addr++;
    BF_PUT_BYTE(addr, (uint16)((data >> 8) & 0xff));
    addr++;
    BF_PUT_BYTE(addr, (uint16)((data ) & 0xff));
    addr++;
    return;
  }

  if (remain_len == 0) {
    //TODO: report error, not enough memory
    return;
  }

  if (remain_len == 2) {
    //BF_PUT_WORD(addr, (UINT16)(data >> 16) & 0xffff);
    BF_PUT_BYTE(addr, (uint16)((data >> 24) & 0xff));
    addr++;
    BF_PUT_BYTE(addr, (uint16)((data >> 16) & 0xff));
    addr++;
    offset = offset+2;
    status = SBM_get_addr_in_packet(pbd, offset, &addr, &remain_len);
    //BF_PUT_WORD(addr, (UINT16)(data & 0xffff));
    BF_PUT_BYTE(addr, (uint16)((data >> 8) & 0xff));
    addr++;
    BF_PUT_BYTE(addr, (uint16)((data ) & 0xff));
    addr++;
    return;
  }

  //TODO: report error, remain_len shouldn't be 1 or 3
  proj_assert(0);  
  return;
}


/*===========================================================================
  Function     : SBM_get_long_bigend
  Description  :
   quick and dirty method to get a long value from a SBD_s chain.
   need to optimize it to take advantage of BD knowledge so we are not
   walking the chain with each iteration by calling SBM_get_addr_in_packet().

  Parameters:
   input             
      pbd      : the BD chain to copy from
      offset   : byte offset into the BD chain 

   output      : 

   return value: the value
   NOTE: this API will get the bytes in a big edian way, significant 
   byte is get from low octet address, so it is different from 
   the BF_GET_LONG
      
===========================================================================*/
UINT32 SBM_get_long_bigend(SBD_s *pbd, UINT16 offset)
{
  UINT32 addr;
  UINT16 remain_len;
  UINT16 old_remain_len;
  SBM_STATUS_t status;
  UINT32 data;
  int i;

  status = SBM_get_addr_in_packet(pbd, offset, &addr, &remain_len);
  if (status != SBM_SUCCESS)
  {
    //TODO: report error 
  }

  if (remain_len >= 4) {
    //BF_PUT_LONG(addr, data);
    data = (uint32)(BF_GET_BYTE(addr)) << 24;
    addr++;
    data |= (uint32)(BF_GET_BYTE(addr)) << 16;
    addr++;
    data |= (uint32)(BF_GET_BYTE(addr)) << 8;
    addr++;
    data |= (uint32)(BF_GET_BYTE(addr));
    addr++;
    return data;
  }

  //if (remain_len == 0) {
  //  //TODO: report error, not enough memory
  //  return;
  //}

  data = 0x0;
  for (i=0; i< remain_len; i++) {
    data = data << 8;
    data |= (uint32)(BF_GET_BYTE(addr));
    addr++;
  }
  offset += remain_len;
  old_remain_len = remain_len;
  status = SBM_get_addr_in_packet(pbd, offset, &addr, &remain_len);
  if (status != SBM_SUCCESS)
  {
    //TODO: report error 
  }
  for (i=0; i< (4-old_remain_len); i++) {
    data = data << 8;
    data |= (uint32)(BF_GET_BYTE(addr));
    addr++;
  }

  //if (remain_len == 2) {
  //  data = BF_GET_BYTE(addr) << 24;
  //  addr++;
  //  data |= BF_GET_BYTE(addr) << 16;
  //  addr++;
  //  offset = offset+2;
  //  status = SBM_get_addr_in_packet(pbd, offset, &addr, &remain_len);
  //  //BF_PUT_WORD(addr, (UINT16)(data & 0xffff));
  //  data |= BF_GET_BYTE(addr) << 8;
  //  addr++;
  //  data |= BF_GET_BYTE(addr);
  //  addr++;
  //  return;
  //}

  return data;
}

//create another bd chain contains the same content
//TODO: right now, we only handle small packets
SBD_s * SBM_copy_bd_chain(MP_SPOOL_HND_s *pHnd, SBD_s *pInput)
{
  UINT32 addr;
  UINT16 bd_len, cpSize, *pbuf;
  UINT16 len_remain, offset;
  SBD_s *pOutput;
  SBM_STATUS_t status;

  len_remain = SBM_get_size_in_BD_chain(pInput);
  pOutput = MP_SBM_pool_malloc(pHnd, len_remain, 0);
  if (NULL == pOutput) {
    return NULL;
  }

  offset = 0;
  while (len_remain > 0) {
    status = SBM_get_addr_in_packet(pOutput, offset, &addr, &bd_len);
    if (status != SBM_SUCCESS) {
      break;
    }

    if (len_remain < bd_len) {
      bd_len = len_remain;
    }
 
    //since it is new allocated buffer chain, 
    //byte starting address of every block should always be even
    
    //proj_assert( (addr & 0x1) == 0);
    if ((addr & 0x1) != 0)
    {
      return NULL;
    }

#if defined (CM3)
    pbuf = (UINT16 *)(addr );
#else
    pbuf = (UINT16 *)(addr >> 1);
#endif

    cpSize = SBM_copy_buffer(pInput, offset, pbuf, bd_len);
    if (cpSize != bd_len) {
      break;
    }

    len_remain -= cpSize;
    offset += cpSize;
  }

  if (len_remain > 0) {
    MP_SBM_pool_free(pHnd, pOutput);
    return NULL;
  }

  return pOutput;

 
#if 0  
  UINT16 blkSize, cpSize, *pbuf;
  UINT32 totalSize = SBM_get_size_in_BD_chain(pInput);
  SBD_s *pOutput;

  blkSize = MP_SBM_pool_get_block_size(mp_buf_pool);
  if (totalSize > blkSize) {
    mac_cm_handle.err_block_mem_size++;
    return NULL;
  }
  
  pOutput = MP_SBM_pool_malloc(mp_buf_pool, totalSize, 0);
  if (NULL == pOutput) {
    mac_cm_handle.err_mem++;
    return NULL;
  }

  pbuf = (UINT16 *)(SBM_get_buf_addr(pOutput) >> 1);
  cpSize = SBM_copy_buffer(pInput, 0, pbuf, totalSize);
  if (cpSize != totalSize) {
    mac_cm_handle.err_mem_copy++;
    MP_SBM_pool_free(mp_buf_pool, pOutput);
    return NULL;
  }

  return pOutput;
#endif  
}

/******************************************************************************
* FUNCTION NAME: MP_SBM_dup_pkt
*
* DESCRIPTION:  This function duplicates one packet
*
* Return Value:  none
*
* Input Parameters:   
*   none
*   
*
* Output Parameters:
*   None
*
* 
******************************************************************************/  
SBD_s *MP_SBM_dup_pkt(MP_SPOOL_HND_s *pHnd,SBD_s *pkt_p)
{
  UINT16 i,num_buffer,block_size;
  SBD_s * pbd,*ptemp,*phead,*pprev;
  UINT16 *psrc,*pdst;
  
  // traverse the BD chain to find out how many BDs are needed
  num_buffer =0;
  ptemp = pkt_p;
  
  if (ptemp==NULL)
  {
    return NULL;
  }
  
  while (ptemp)
  {
    num_buffer++;

    ptemp = ptemp->pnext;
  }

  // check to see if we have enough buffer and BD in the pool

  if (!( (pHnd->num_free_buf>num_buffer) && (pHnd->num_free_bd >num_buffer)))
  {
    // no enough buffer
    return NULL;
  }
  
  // start to duplcuaite each buffer
  ptemp = pkt_p;
  pprev = NULL;
  block_size = pHnd->block_size;
  
  for (i=0;i<num_buffer;i++)
  {
    // allocate one buffer
    pbd = MP_SBM_pool_malloc(pHnd,block_size,0);

#if defined (CM3)
    // copy every from the pkt to new alloated BD
    pdst = (UINT16 *)(SBM_get_buf_addr(pbd) );
    psrc = (UINT16 *)(SBM_get_buf_addr(ptemp) );

    memcpy(pdst,psrc,block_size);

#else
    // copy every from the pkt to new alloated BD
    pdst = (UINT16 *)(SBM_get_buf_addr(pbd)>>1);
    psrc = (UINT16 *)(SBM_get_buf_addr(ptemp)>>1);

    memcpy(pdst,psrc,block_size>>1);
#endif

    // update the data length and data offset
    pbd->dat_offset = ptemp->dat_offset;
    pbd->len        = ptemp->len;

    if (pprev!=NULL)
    {
      pprev->pnext = pbd;
    }
    else
    { // this is the first BD
      phead = pbd;
    }
    /* update the pprev */
    pprev = pbd;     

    ptemp = ptemp->pnext;
  }

  return phead;
}

