/**************************************************************************************************
  Filename:       boot_api.c
  
  Description:    boot and reset related API
  
  Copyright 2005-2010 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.

  HISTORY:

  12/29/2010 R.Liang           initial version  


  LIST OF FUNCTION POTOTYPES:

          
          
******************************************************************************/

/* ------------------------------------------------------------------------------------------------
 *                                          Includes
 * ------------------------------------------------------------------------------------------------
 */

#include <boot_api.h>
#include <crc32.h>

#ifdef F2806X
#include <F2806x_Device.h>
#elif defined(F28M35X)
#include <F28M35x_Device.h>
#else
#include <DSP2833x_Device.h>
#endif
/* ------------------------------------------------------------------------------------------------
 *                                           Local Variables
 * ------------------------------------------------------------------------------------------------
 */

/* ------------------------------------------------------------------------------------------------
 *                                           global Variables
 * ------------------------------------------------------------------------------------------------
 */
UINT32 idle_cnt;

void boot_shutDownClock(void)
{
  PieCtrlRegs.PIEIER1.bit.INTx4 = 0; 
  PieCtrlRegs.PIEIER1.bit.INTx5 = 0; 

  SysCtrlRegs.PCLKCR0.all =0;
  SysCtrlRegs.PCLKCR1.all =0;
#if defined(F2806X) || defined(F28M35X)
  SysCtrlRegs.PCLKCR2.all =0;
#endif
  SysCtrlRegs.PCLKCR3.all =0;

}

/******************************************************************************
 * FUNCTION NAME: soft_reset
 *
 * DESCRIPTION:   soft reset the application. It jumps to the start address of application
 *                Note: F28335 and Octave start address are different.
 *
 * Return Value:
 *
 * Input Parameters:
 *
 * Output Parameters:
 ******************************************************************************/
void soft_reset(void)
{
  //R.liang: the application starts at 0x320000
  //directly jmup the system start address
  DINT;   //disable iterrupt

  // put SCI-A and SCI-B in the reset state
  SciaRegs.SCICTL1.all = 0x0000;
#ifndef F28M35X  
  ScibRegs.SCICTL1.all = 0x0000;
#endif
  // shut down all clock
  boot_shutDownClock();
  
#ifdef F2806X
  asm(" LB 0x3dC000");
#elif defined (F28M35X)
  asm(" LB 0x108000");  
#else  
  asm(" LB 0x310000");
#endif
}

/******************************************************************************
 * FUNCTION NAME: Idle_Task
 *
 * DESCRIPTION:   when DSP/BIOS is idle, this function is called and go to
 *                idle mode
 *
 * Return Value:
 *
 * Input Parameters:
 *
 * Output Parameters:
 ******************************************************************************/
void Idle_Task(void)
{
  idle_cnt++;
  // put in the low power mode
  asm("  idle");
}

/******************************************************************************
 * FUNCTION NAME: probe_Flash_upgrade
 *
 * DESCRIPTION:   This function is used to probe if the firmware upgrade
 *                function exists in the Flash.
 *
 * Return Value:
 *
 * Input Parameters:
 *
 * Output Parameters:
 ******************************************************************************/
UINT16 probe_Flash_upgrade(void)
{
   UINT16 *pdata;
   UINT16 temp[2];

   // check if there are some thing in Firm Upgrade
   // goto 0x320000 to read data back if data rae 
   // 0x561F 0x7622

   pdata = (UINT16 *) FIRMWARE_UPGRADE_APP_START_ADDRESS;
   
   //read data
   temp[0] = *pdata++;
   temp[1] = *pdata;

   if ( (temp[0]==0x561F) && (temp[1]==0x7622))
   {  // firmware upgrade utility exists
      return 1;
   }

   return 0;
}

/******************************************************************************
 * FUNCTION NAME: jump_flash_upgrade_start
 *
 * DESCRIPTION:   Jump to firmware upgrade function
 *
 * Return Value:
 *
 * Input Parameters:
 *
 * Output Parameters:
 ******************************************************************************/
void jump_flash_upgrade_start(int port)
{
  //R.liang: the application starts at 0x320000
  //directly jmup the system start address
  DINT;   //disable interrupt

  // put SCI-A and SCI-B in the reset state
  SciaRegs.SCICTL1.all = 0x0000;
#ifndef F28M35X  
  ScibRegs.SCICTL1.all = 0x0000;
#endif
  // shut down all clock
  boot_shutDownClock();

#ifdef F2806X
  asm(" LB 0x3D8000");
#elif defined (F28M35X)
  asm(" LB 0x100000");   // DFU is at sector M and N
#else
  // sms - write the port to the section of RAM that will
  //  be the port.  I did this by mapping a ram section
  //  specifically for this variable and thus the address
  //  is known.

  // 0000c000   _g_port
  (*(UINT16 *) 0x0000c000) = port;

  asm(" LB 0x300000");

#endif

}

void Jump_Start_BIT(void)
{
#ifdef F2806X

#else
  //R.liang: the application starts at 0x328000 (sector C)
  //directly jmup the system start address
  DINT;   //disable iterrupt

  // put SCI-A and SCI-B in the reset state
  SciaRegs.SCICTL1.all = 0x0000;
#ifndef F28M35X  
  ScibRegs.SCICTL1.all = 0x0000;
#endif
  // shut down all clock
  boot_shutDownClock();
	
  asm(" LB 0x330000");
#endif  
}

#ifdef FRM_UPGRADE
#pragma CODE_SECTION(Jump_Start_DFU,"ramfuncs");
#endif
void Jump_Start_DFU(void)
{
  DINT;   //disable iterrupt
  // put SCI-A and SCI-B in the reset state
  SciaRegs.SCICTL1.all = 0x0000;
#ifndef F28M35X  
  ScibRegs.SCICTL1.all = 0x0000;
#endif  

  // shut down all clock
  boot_shutDownClock();
	
#ifdef F2806X
  //R.liang: the application starts at 0x320000 (Sector D)
  //directly jmup the system start address
  
  asm(" LB 0x3D8000");

#else

#ifdef F28M35X
  asm(" LB 0x100000");  // sector M and N for DFU
#else
  //R.liang: the application starts at 0x320000 (Sector D)
  //directly jmup the system start address

  asm(" LB 0x300000");
#endif
  
#endif

}

#ifdef FRM_UPGRADE
#pragma CODE_SECTION(JUMP_Start_PLC_App,"ramfuncs");
#endif

void JUMP_Start_PLC_App(void)
{
  DINT;   //disable iterrupt
  // put SCI-A and SCI-B in the reset state
  SciaRegs.SCICTL1.all = 0x0000;
#ifndef F28M35X    
  ScibRegs.SCICTL1.all = 0x0000;
#endif

  // shut down all clock
  boot_shutDownClock();

#ifdef F2806X
  //R.liang: the application starts at 0x320000
  //directly jmup the system start address

  asm(" LB 0x3DC000");
#elif defined (F28M35X)
  asm(" LB 0x108000");  
#else
  //R.liang: the application starts at 0x320000
  //directly jmup the system start address

  asm(" LB 0x310000");
#endif  
}

// this function will check the flash firmware CRC32
// return 0: CRC32 is bad
//        1: CRC32 is good
UINT16 Flash_Check_Firmware_CRC32(void)
{
  UINT16 *startAddr;
  UINT32 len; //in byte
  UINT32 crc32,storedCRC;
  UINT32 *pFlashAddr;
  
  // assume the application is 220K bytes
  // assume Flash store length and CRC32
  // address : 0x31FFD0: length (32 bit)
  //           0x31FFD2: CRC32

  pFlashAddr = (UINT32 *)FLASH_STORED_CRC;
  
  len = *pFlashAddr++;
  storedCRC = *pFlashAddr;

  if ((len== 0xFFFFFFFF) || (storedCRC== 0xFFFFFFFF))
  { 
    return 0;
  }
    
  startAddr = (UINT16*)FLASH_PLC_IMAGE_START;
  
  crc32 = INIT_CRC32;
  // use the length in bytes
  len = len*2;
  
  crc32 = getCRC32 (crc32,startAddr,(parity_t)0,len);

  if (crc32 == storedCRC)
  { // it is good, we jump o the application;
    return 1;
  }

  // CRC32 is bad
  return 0;

}

void Flash_Get_Firmware_Info(UINT32 *flash_start,UINT32 *len)
{
  UINT32 *pFlashAddr;

  pFlashAddr = (UINT32 *)FLASH_STORED_CRC;
  
  *flash_start  = FLASH_PLC_IMAGE_START;
  *len          = *pFlashAddr;

}

// CQ 833
UINT32 Flash_getStoredFirmwareCRC32(void)
{  
  UINT32 storedCRC;
  UINT32 *pFlashAddr;
  
  // assume the application is 220K bytes
  // assume Flash store length and CRC32
  // address : 0x31FFD0: length (32 bit)
  //           0x31FFD2: CRC32

  pFlashAddr = (UINT32 *)(FLASH_STORED_CRC+2);
    
  storedCRC = *pFlashAddr;

  return storedCRC;

}

