/******************************************************************************
* FILE PURPOSE: Module for CRC-8 calculation
*******************************************************************************
*
* FILE NAME: crc8.c
*
* DESCRIPTION:
*  This file contains the functions to do the CRC32
*
*  Copyright (c) 2009 Texas Instruments Inc.
*  All Rights Reserved This program is the confidential and proprietary
*  product of Texas Instruments Inc.  Any Unauthorized use, reproduction or
*  transfer of this program is strictly prohibited.
*
* HISTORY:
*
*  03/02/2009  : Shu Du initial implementation for PRIME 
*  05/20/2009  : Shu Du changed the crc tables to be in flash 
*  03/04/2010  : Robert liang, ported to G3 for CRC16
*                added table generation and help functions
*
* LIST OF FUNCTION POTOTYPES:
*
* Notes: 
*
*  o CRC implementation is based on tables. If the tables
*    are calculated on the fly, the macro GEN_CRC_TABLES shall be defined
*    and genCRC32Tables() shall be called during 
*    initialization. If the macro is not defined, pre-calculated tables will
*    will be used. Using the latter option can put the tables to flash.
******************************************************************************/
#include <typedefs.h>
#include <mp_sbfm.h>

#include "crc8.h"

#ifdef VCU
// we don't need the table
#else
#pragma DATA_SECTION(crc8_table,"CRC8_TABLE");
#endif

#ifdef GEN_CRC_TABLES
uint16 crc8_table[256];
#else
#ifdef VCU
// we don't need the table
#else
/*===========================================================================
  STATIC TABLE DEFINATIONS
===========================================================================*/
static const uint16 crc8_table[256] = 
{
  0x0000, 0x0007, 0x000E, 0x0009, 0x001C, 0x001B, 0x0012, 0x0015,
  0x0038, 0x003F, 0x0036, 0x0031, 0x0024, 0x0023, 0x002A, 0x002D,
  0x0070, 0x0077, 0x007E, 0x0079, 0x006C, 0x006B, 0x0062, 0x0065,
  0x0048, 0x004F, 0x0046, 0x0041, 0x0054, 0x0053, 0x005A, 0x005D,
  0x00E0, 0x00E7, 0x00EE, 0x00E9, 0x00FC, 0x00FB, 0x00F2, 0x00F5,
  0x00D8, 0x00DF, 0x00D6, 0x00D1, 0x00C4, 0x00C3, 0x00CA, 0x00CD,
  0x0090, 0x0097, 0x009E, 0x0099, 0x008C, 0x008B, 0x0082, 0x0085,
  0x00A8, 0x00AF, 0x00A6, 0x00A1, 0x00B4, 0x00B3, 0x00BA, 0x00BD,
  0x00C7, 0x00C0, 0x00C9, 0x00CE, 0x00DB, 0x00DC, 0x00D5, 0x00D2,
  0x00FF, 0x00F8, 0x00F1, 0x00F6, 0x00E3, 0x00E4, 0x00ED, 0x00EA,
  0x00B7, 0x00B0, 0x00B9, 0x00BE, 0x00AB, 0x00AC, 0x00A5, 0x00A2,
  0x008F, 0x0088, 0x0081, 0x0086, 0x0093, 0x0094, 0x009D, 0x009A,
  0x0027, 0x0020, 0x0029, 0x002E, 0x003B, 0x003C, 0x0035, 0x0032,
  0x001F, 0x0018, 0x0011, 0x0016, 0x0003, 0x0004, 0x000D, 0x000A,
  0x0057, 0x0050, 0x0059, 0x005E, 0x004B, 0x004C, 0x0045, 0x0042,
  0x006F, 0x0068, 0x0061, 0x0066, 0x0073, 0x0074, 0x007D, 0x007A,
  0x0089, 0x008E, 0x0087, 0x0080, 0x0095, 0x0092, 0x009B, 0x009C,
  0x00B1, 0x00B6, 0x00BF, 0x00B8, 0x00AD, 0x00AA, 0x00A3, 0x00A4,
  0x00F9, 0x00FE, 0x00F7, 0x00F0, 0x00E5, 0x00E2, 0x00EB, 0x00EC,
  0x00C1, 0x00C6, 0x00CF, 0x00C8, 0x00DD, 0x00DA, 0x00D3, 0x00D4,
  0x0069, 0x006E, 0x0067, 0x0060, 0x0075, 0x0072, 0x007B, 0x007C,
  0x0051, 0x0056, 0x005F, 0x0058, 0x004D, 0x004A, 0x0043, 0x0044,
  0x0019, 0x001E, 0x0017, 0x0010, 0x0005, 0x0002, 0x000B, 0x000C,
  0x0021, 0x0026, 0x002F, 0x0028, 0x003D, 0x003A, 0x0033, 0x0034,
  0x004E, 0x0049, 0x0040, 0x0047, 0x0052, 0x0055, 0x005C, 0x005B,
  0x0076, 0x0071, 0x0078, 0x007F, 0x006A, 0x006D, 0x0064, 0x0063,
  0x003E, 0x0039, 0x0030, 0x0037, 0x0022, 0x0025, 0x002C, 0x002B,
  0x0006, 0x0001, 0x0008, 0x000F, 0x001A, 0x001D, 0x0014, 0x0013,
  0x00AE, 0x00A9, 0x00A0, 0x00A7, 0x00B2, 0x00B5, 0x00BC, 0x00BB,
  0x0096, 0x0091, 0x0098, 0x009F, 0x008A, 0x008D, 0x0084, 0x0083,
  0x00DE, 0x00D9, 0x00D0, 0x00D7, 0x00C2, 0x00C5, 0x00CC, 0x00CB,
  0x00E6, 0x00E1, 0x00E8, 0x00EF, 0x00FA, 0x00FD, 0x00F4, 0x00F3,
};
#endif

#endif

#ifdef GEN_CRC_TABLES
/*===========================================================================
  Function   : genCRC8Table
  Description :
    This function is to generate the CRC8 table for every possible byte

  Parameters:
  intput : none
  output : none
  return : SUCCESS or FAILURE
===========================================================================*/
void genCRC8Table()
{ 
  register uint16 i;
  register uint16 j;
  register uint16 crc8_accum;


  for ( i = 0; i < 256;  i++ )
  { 
    crc8_accum = i;
    for ( j = 0; j < 8;  j++ )
    { 
      if ( crc8_accum & 0x80L )
        crc8_accum = ( (crc8_accum << 1) & 0xff ) ^ POLYNOMIAL8;
      else
        crc8_accum = ( crc8_accum << 1 ) & 0xff; 
    }
    crc8_table[i] = crc8_accum; 
  }
  return PRIME_STAT_SUCCESS; 
}
#endif

/*===========================================================================
  Function   : getCRC8
  Description :
    This function is to calculate the CRC8 from a linear bufffer

  Parameters:
  intput :
      input_crc8_accum: the init value of crc, in case the message has been
          chopped into several parts, you can use the crc8 of the previsou
          segment as the init value for the current segment crc8 calculation
          until the final crc is derived.
      msg         : pointer of the word stream of the message
      offset      : ODD or EVEN to tell the starting byte in the word stream
      rxLen       : number of BYTES of the input msg
  output : none
  return : crc value
===========================================================================*/
uint16 getCRC8 (uint16 input_crc8_accum, uint16 * msg, parity_t parity, 
    uint16 rxLen)
{
#ifdef VCU
  return getCRC8_vcu_asm(input_crc8_accum, msg, parity, rxLen);
#else
  register uint16 i;
  register uint16 j;
  uint16 crc8_accum;
  SINT16 *pdata;

  crc8_accum = input_crc8_accum;
  pdata = (SINT16 *)msg;

  for (j = 0; j < rxLen; j++, parity++)
  {
    i = crc8_accum ^ (__byte(pdata, parity));
    crc8_accum = crc8_table[i]; 
  }

  return crc8_accum;
#endif  
}

/*===========================================================================
  Function   : getCRC8FromBD
  Description :
    This function is to calculate the CRC8 from a BD chain

  Parameters:
  intput :
      input_crc8_accum: the init value of crc, in case the message has been
          chopped into several parts, you can use the crc8 of the previsou
          segment as the init value for the current segment crc8 calculation
          until the final crc is derived.
      pbd         : pointer of the first BD of the BD chain
      rxLen       : number of BYTES of the input msg
  output : none
  return : crc value
===========================================================================*/
uint16 getCRC8FromBD (uint16 input_crc8_accum, SBD_s * pbd, uint16 rxLen)
{
  uint16 crc8_accum = input_crc8_accum;
  SBM_STATUS_t status;
  UINT32 addr;
  UINT16 bd_len, remain_len;
  UINT32 phy_addr;
  parity_t offset;

  remain_len = rxLen;
  while (remain_len > 0)
  { 
    status = SBM_get_addr_in_packet(pbd, 0, &addr, &bd_len);
    if (status != SBM_SUCCESS)
    {
      //todo report errors
      break;
    }
    if (remain_len < bd_len) 
      bd_len = remain_len;
 
    phy_addr = (addr>>1);
    offset = (parity_t)(addr & 0x01);
    crc8_accum = getCRC8(crc8_accum, (uint16 *)phy_addr, offset, bd_len);

    remain_len -= bd_len;
    pbd = pbd->pnext;
  }

  return crc8_accum; 
}
