/** Set functions for YAFFS
 *
 * @file yaffs_setfuncs.c
 * @author Igor Kondrashenkov
 * @date 2012-08-02
 *
 * Copyright (c) 2012 Kelvatek Ltd. All rights reserved.
 */

#include "yaffs_setfuncs.h"
#include "yaffs_trace.h"
#include "hal_nand.h"

/** if some initialisation is needed */
static int  CheckInit(void)
{
    static int initialised = 0;
    
    if (initialised) {
        return YAFFS_OK;
    }
    initialised = 1;

    if (Hal_Nand_IoInit() == FLASH_OP_FAIL)
        return YAFFS_FAIL;

    return YAFFS_OK;
}


int InitialiseNAND(yaffs_Device *dev)
{
    return CheckInit();
}


int DeinitialiseNAND(yaffs_Device *dev)
{
    return 1;
}


int WriteChunkWithTagsToNAND(yaffs_Device *dev, int chunkInNAND, const __u8 *data, const yaffs_ExtendedTags *tags)
{
    T(YAFFS_TRACE_MTD, (TSTR("write chunk %d data %x tags %x" TENDSTR), chunkInNAND, (unsigned)data, (unsigned)tags));

    CheckInit();

    if (!dev->param.isYaffs2 || tags == 0)
        return YAFFS_FAIL;

    if (!dev->param.inbandTags) {

        // so far it is not supported
        return YAFFS_FAIL;

    } else {
        // our tag is inside the data, so get address for a chunk
        yaffs_inbandtag_t oob;
        
        // set structure to erased state
        memset(&oob, 0xFF, sizeof(yaffs_inbandtag_t));
        // pack the tags
        yaffs_PackTags2(&oob.pt2, tags, !dev->param.noTagsECC);

        // write the data 
        if (data) {
            int i;
            const uint8_t *pCh;
            int x256 = dev->param.totalBytesPerChunk / 256;
            
            // have to set the outband area to 0xFF
            memset((void*) &data[dev->nDataBytesPerChunk], 0xFF, sizeof(yaffs_inbandtag_t));
            
            // calculate ECC for data section
            pCh = data;
            for (i = 0; i < x256; i++) {
                uint32_t ecc;
                // calculate the data ECC
                yaffs_ECCCalculate(pCh, (uint8_t*) &ecc);
                oob.ecc_data[i] = ecc;
                pCh += 256;
            }

            // add tags to the data
            memcpy((void*) &data[dev->nDataBytesPerChunk], &oob, sizeof(yaffs_inbandtag_t));
            // write data and tags to the flash
            if (Hal_Nand_IoWrite((uint32_t) dev->driverContext, chunkInNAND, 0, data, dev->param.totalBytesPerChunk) != FLASH_OP_OK)
                return YAFFS_FAIL;
        } else {
            /* just write tags only */
            if (Hal_Nand_IoWrite((uint32_t) dev->driverContext, chunkInNAND, dev->nDataBytesPerChunk, &oob, sizeof(yaffs_inbandtag_t)) != FLASH_OP_OK)
                return YAFFS_FAIL;
        }
    }

    return YAFFS_OK;
}


int ReadChunkWithTagsFromNAND(yaffs_Device *dev, int chunkInNAND, __u8 *data, yaffs_ExtendedTags *tags)
{
    int retval = YAFFS_OK;

    CheckInit();

    if (!dev->param.isYaffs2)
        return YAFFS_FAIL;
    
    if (!dev->param.inbandTags){ 
    
        // so far it is not supported
        retval = YAFFS_FAIL;

    } else {
        // our tag is inside the data
        int ecc;
        int ecc_result, i;

        yaffs_inbandtag_t oob;
        int x256 = dev->param.totalBytesPerChunk / 256;
      
        // read tags
        if (Hal_Nand_IoRead((uint32_t) dev->driverContext, chunkInNAND, dev->nDataBytesPerChunk, &oob, sizeof(yaffs_inbandtag_t)) != FLASH_OP_OK)
            return YAFFS_FAIL;
        // check TAG information
        if (tags) {
            // unpack tags and check ecc if stated
            yaffs_UnpackTags2(tags, &(oob.pt2), !dev->param.noTagsECC);
            if (oob.blockState != -1)
                tags->blockBad = 1;
        }

        // read data
        ecc_result = YAFFS_ECC_RESULT_NO_ERROR;
        if (data) {
            uint8_t *pCh;
            // set outband data to 0xFF
            memset(&data[dev->nDataBytesPerChunk], 0xFF, sizeof(yaffs_inbandtag_t));
            // read only data section
            if (Hal_Nand_IoRead((uint32_t) dev->driverContext, chunkInNAND, 0, data, dev->nDataBytesPerChunk) != FLASH_OP_OK)
                return YAFFS_FAIL;
            pCh = data;
            for (i = 0; i < x256; i++) {
                int result;
                // calculate the data ECC
                yaffs_ECCCalculate(pCh, (uint8_t*) &ecc);
                result = yaffs_ECCCorrect(pCh, (uint8_t*) &oob.ecc_data[i], (uint8_t*) &ecc);
                // stupid part because author have strange enum values
                if (result < 0)
                    ecc_result = YAFFS_ECC_RESULT_UNFIXED;
                else if (result == 1 && ecc_result > 0) {
                    ecc_result = YAFFS_ECC_RESULT_FIXED;
                }
                pCh += 256;
            }
            // copy inbandTag data inside the buffer
            memcpy(&data[dev->nDataBytesPerChunk], &oob, sizeof(yaffs_inbandtag_t));
        } 

        // update tags->eccResult if needed
        if (tags && (ecc_result > tags->eccResult))
            tags->eccResult = ecc_result;
    }

    return retval;

}


int MarkNANDBlockBad(struct yaffs_DeviceStruct *dev, int blockNo)
{
    uint32_t pageno = blockNo * dev->param.nChunksPerBlock;
    uint32_t offset = dev->nDataBytesPerChunk;
    yaffs_inbandtag_t oob;
    
    oob.blockState = 0;

    if (Hal_Nand_IoWrite((uint32_t) dev->driverContext, pageno, offset, &oob.blockState, sizeof(oob.blockState)) == FLASH_OP_FAIL) {
        T(YAFFS_TRACE_ALWAYS, (TSTR("Bad block marking op is failed %d\n"), blockNo));
        return YAFFS_FAIL;
    }
    
    return YAFFS_OK;
}

/** EraseBlockInNAND implementation */
int EraseBlockInNAND(yaffs_Device *dev, int blockNo)
{
    CheckInit();

    if (blockNo < 0 || blockNo > dev->param.endBlock) {
        T(YAFFS_TRACE_ALWAYS,(TSTR("Attempt to erase non-existant block %d\n"), blockNo));
        return YAFFS_FAIL;
    }

    // get block addr
    if (Hal_Nand_IoEraseBlock((uint32_t) dev->driverContext, blockNo) == FLASH_OP_FAIL) {
        T(YAFFS_TRACE_ALWAYS,(TSTR("Block erase op is failed %d\n"), blockNo));
        return YAFFS_FAIL;
    }

    return YAFFS_OK;
}


/** QueryNANDBlock implementation */
int QueryNANDBlock(struct yaffs_DeviceStruct *dev, int blockNo, yaffs_BlockState *state, __u32 *sequenceNumber)
{
    int retval = YAFFS_OK;
    yaffs_ExtendedTags tags;
    int chunkNo = blockNo * dev->param.nChunksPerBlock;

    if (!sequenceNumber || !state)
        return YAFFS_FAIL;

    // get extended tags
    if (ReadChunkWithTagsFromNAND(dev, chunkNo, NULL, &tags) != YAFFS_OK)
        return YAFFS_FAIL;

    // should be set to zero if DEAD or EMPTY
    *sequenceNumber = 0;
    if (tags.blockBad) {
        *state = YAFFS_BLOCK_STATE_DEAD;
        retval = YAFFS_FAIL;
    } else if (!tags.chunkUsed) {
        *state = YAFFS_BLOCK_STATE_EMPTY;
    } else if (tags.chunkUsed) {
        *state = YAFFS_BLOCK_STATE_NEEDS_SCANNING;
        *sequenceNumber = tags.sequenceNumber;
    }

    return retval;
}
