/******************************************************************************\
* Copyright (C) 2006 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... Board Support Library for SPM176431
* VERSION.......... (Defined in README.TXT)
*------------------------------------------------------------------------------
* CONTENT.......... Header file for API of peripheral AC97
* FILENAME......... bsl_flash.h
*------------------------------------------------------------------------------
* PERIPHERAL NAME.. AC97
* PERIPHERAL TYPE.. Single-device peripheral
* ACCESS MODE...... Direct (no handle)
* REGISTER ACCESS.. 16-bit wide
*
*   NOTE:   All functions and variables in this file are only intended to be
*           utilized by a single thread in any .out file or project.  These
*           functions are not "thread safe."  To explain, in a multithreaded
*           application you might fork off a single thread with the sole purpose
*           of pulling audio out of the audio buffers and placing audio into
*           the audio buffer.
*
\******************************************************************************/
#ifndef _BSL_AC97_H_
#define _BSL_AC97_H_

#include <bsl.h>


#if (BSL_AC97_SUPPORT)
/******************************************************************************\
* scope and inline control macros
\******************************************************************************/

#ifdef  __cplusplus
    #define BSLAPI extern "C" far
#else
    #define BSLAPI extern far
#endif

#undef  USEDEFS
#undef  IDECL
#undef  IDEF

#ifdef _BSL_AC97_MOD_
    #define IDECL BSLAPI
    #define USEDEFS
    #define IDEF
#else
    #ifdef  _INLINE
        #define IDECL static inline
        #define USEDEFS
        #define IDEF  static inline
    #else
        #define IDECL BSLAPI
    #endif
#endif


/******************************************************************************\
* global macro declarations
\******************************************************************************/

#define LM4550_MASTER_VOLUME_REG                            0x02
#define LM4550_HEADPHONE_VOLUME_REG                         0x04
#define LM4550_MONO_VOLUME_REG                              0x06
#define LM4550_MIC_VOLUME_REG                               0x0E
#define LM4550_CD_VOLUME_REG                                0x12
#define LM4550_PCM_OUT_VOLUME_REG                           0x18
#define LM4550_RECORD_SELECT_REG                            0x1A
#define LM4550_RECORD_GAIN_REG                              0x1C
#define LM4550_GENERAL_PURPOSE_REG                          0x20
#define LM4550_PWRDOWNCTRL_STAT_REG                         0x26
#define LM4550_EXT_AUDIO_CONTROL_REG                        0x2A
#define LM4550_PCM_DAC_RATE_REG                             0x2C
#define LM4550_PCM_ADC_RATE_REG                             0x32

#define RECIEVE_PING_EDMA_INT_NUM                   15
#define RECIEVE_PONG_EDMA_INT_NUM                   23
#define TRANSMIT_PING_EDMA_INT_NUM                  14
#define TRANSMIT_PONG_EDMA_INT_NUM                  22

#define DO_NOT_CHANGE_VALUE                     0xFFFFFFFF

#ifndef FALSE
#define FALSE                                       0
#endif
#ifndef FALSE
#define TRUE                                        1
#endif

#define AC97_ERROR_INVALID_PARAMETER                1
#define AC97_ERROR_IN_READING_PARAMETER_REGISTER    2
#define AC97_ERROR_IN_WRITING_PARAMETER_REGISTER    3

// do not change these unless the linker file is changed
// if this is changed be sure to change the linker file also
// below 1024*2*24 32-bit-words -- means it actually holds 1024*2*24 samples
// below 1024*2 32-bit-words -- means it actually holds 1024*2 samples
#define SIZE_OF_CIRC_BUFF                       1024*2*24
#define SIZE_OF_PP_BUFF                         1024*2

#define AC97_STARTING_ADDRESS                   0x6C0800A2


/******************************************************************************\
* global typedef declarations
\******************************************************************************/

/******************************************************************************\
* global variable declarations
\******************************************************************************/

/******************************************************************************\
* global function declarations
\******************************************************************************/

//=============================================================================
// AC97_open
//  This function resets the audio chip.  It clears all 4 buffers used in the
//  ping pong buffering.  It clears the users two buffers.  It also opens the
//  McBsp and configures it for communications with the adio chip(via the FPGA)
//
//
//=============================================================================
BSLAPI void AC97_open();
//=============================================================================
// AC97_start_edma
//  This function sets all the necessary parameters in EDMA to use ping pong
//  buffering for transmission to and from the audio chip.  Then it actually
//  Enables the EDMA which begins the ping pong buffering.  This function is
//  designed to work in pair with the stop function.  Therefore you can ...
//  Start...Stop....Start....Stop....Start....Stop...etc.
//
//=============================================================================
BSLAPI void AC97_start_edma(void);
//=============================================================================
// AC97_start_edma_recieve
//  This function does exactly the same as the AC97_start_edma, but this
//  only works with the recieve buffers.  It allows the recieve and transmit
//  EDMA to be controlled seperately
//
//=============================================================================
BSLAPI void AC97_start_edma_recieve(void);
//=============================================================================
// AC97_start_edma_transmit
//  This function does exactly the same as the AC97_start_edma, but this
//  only works with the transmit buffers.  It allows the recieve and transmit
//  EDMA to be controlled seperately
//
//=============================================================================
BSLAPI void AC97_start_edma_transmit(void);
//=============================================================================
// AC97_stop_edma
//  This function stops the transmit and recieve EDMA.  This function just sets
//  two flags that are read by the ISR.  The ISR will then stop the EDMA at the
//  next possible stoping point.  This time to stop could be as long as it takes
//  to fill upto two more ping-pong buffers.
//
//  NOTICE:  This function will return before the actual EDMA has stopped.  To
//  determine if either EDMA is running test if EDMA_Rcv_stopped_flag/
//  EDMA_Tx_stopped_flag are True.  These are false while the EDMA is still
//  running.
//
//=============================================================================
BSLAPI void AC97_stop_edma(void);
//=============================================================================
// AC97_stop_edma_recieve
//  This function does exactly the same as the AC97_stop_edma, but this
//  only works with the recieve buffers.  It allows the recieve and transmit
//  EDMA to be controlled seperately
//
//=============================================================================
BSLAPI void AC97_stop_edma_recieve(void);
//=============================================================================
// AC97_stop_edma_transmit
//  This function does exactly the same as the AC97_stop_edma, but this
//  only works with the transmit buffers.  It allows the recieve and transmit
//  EDMA to be controlled seperately
//
//=============================================================================
BSLAPI void AC97_stop_edma_transmit(void);
//=============================================================================
// AC97_close_edma
//  This function release all services utilized to operate the AC97
//
//=============================================================================
BSLAPI void AC97_close_edma(void);
//=============================================================================
// AC97_set_output_volumes
//  This function set registers inside the AC97 by calling the read/write
//  functions above.  This function sets the mute, left volume, right volume
//  for the master volume and headphone volume, and the mute and mono volume
//  for the mono channel.  See National Semiconductors, LM4550 Rev 2.1
//  data sheet on AC97
//
//  Inputs:
//      MasterMute:     0)unmutes the volume    1)mutes the volume
//      MasterLeft:
//      MasterRight:  00000(0dB) -- 11111(46.5dB) in increments of 1.5 dB
//      HeadphoneMute:  0)unmutes the volume    1)mutes the volume
//      HeadphoneLeft:
//      HeadphoneRight:  00000(0dB) -- 11111(46.5dB) in increments of 1.5 dB
//      MonoMute:       0)unmutes the volume    1)mutes the volume
//      MonoVolume:      00000(0dB) -- 11111(46.5dB) in increments of 1.5 dB
//
//  Outputs:
//  AC97_ERROR_INVALID_PARAMETER:  inputs invalid
//  AC97_ERROR_IN_READING_PARAMETER_REGISTER: unable to read register in ac97
//  AC97_ERROR_IN_WRITING_PARAMETER_REGISTER: last value written to register
//      in ac97 read back incorrectly
//
//=============================================================================
BSLAPI Uint32 AC97_set_output_volumes(Uint32 MasterMute, Uint32 MasterLeft,
    Uint32 MasterRight, Uint32 HeadphoneMute, Uint32 HeadphoneLeft,
    Uint32 HeadphoneRight, Uint32 MonoMute, Uint32 MonoVolume);
//=============================================================================
// AC97_set_input_volumes
//  This function set registers inside the AC97 by calling the read/write
//  functions above.  This function sets the mute, left volume, right volume
//  for the CD(input) volume and Digital to Analog converter output volume,
//  and the mute and mono volume for the microphone channel.
//  See National Semiconductors, LM4550 Rev 2.1 data sheet on AC97
//
//  Inputs:
//      MicMute:        0)unmutes the volume    1)mutes the volume
//      Mic20dBBoost:   0)no extra Mic volume   1)20dB Mic Boost
//      MicVolume:  00000(+12dB)-01000(0dB)-11111(34.5dB) increments of 1.5 dB
//      CDMute:         0)unmutes the volume    1)mutes the volume
//      CDLeft:
//      CDRight:  00000(+12dB)-01000(0dB)-11111(34.5dB) increments of 1.5 dB
//      DACOutMute:         0)unmutes the volume    1)mutes the volume
//      DACOutRight:
//      DACOutLeft: 00000(+12dB)-01000(0dB)-11111(34.5dB) increments of 1.5 dB
//
//  Outputs:
//  AC97_ERROR_INVALID_PARAMETER:  inputs invalid
//  AC97_ERROR_IN_READING_PARAMETER_REGISTER: unable to read register in ac97
//  AC97_ERROR_IN_WRITING_PARAMETER_REGISTER: last value written to register
//      in ac97 read back incorrectly
//
//=============================================================================
BSLAPI Uint32 AC97_set_input_volumes(Uint32 MicMute, Uint32 Mic20dBBoost,
    Uint32 MicVolume, Uint32 CDMute, Uint32 CDLeft, Uint32 CDRight,
    Uint32 DACOutMute, Uint32 DACOutLeft, Uint32 DACOutRight);
//=============================================================================
// AC97_record_setup
//  This function set registers inside the AC97 by calling the read/write
//  functions above.  This function sets the recording parameters.  It sets
//  the mux select for which channel to record from on the left and right side
//  and it sets the volume to record with.
//  See National Semiconductors, LM4550 Rev 2.1 data sheet on AC97
//
//  Inputs:
//      RecordSelectLeft:
//      RecordSelectRight:  0) Microphone
//                          1) CD (line in)
//                          5) Stereo Mix
//                          6) Mono Mix
//      RecordGainMute: 0)unmutes the volume    1)mutes the volume
//      RecordGainRight:
//      RecordGainLeft: 0000(0dB) -- 1111(22.5dB)
//
//  Outputs:
//  AC97_ERROR_INVALID_PARAMETER:  inputs invalid
//  AC97_ERROR_IN_READING_PARAMETER_REGISTER: unable to read register in ac97
//  AC97_ERROR_IN_WRITING_PARAMETER_REGISTER: last value written to register
//      in ac97 read back incorrectly
//
//=============================================================================
BSLAPI Uint32 AC97_record_setup(Uint32 RecordSelectLeft, Uint32 RecordSelectRight,
    Uint32 RecordGainMute, Uint32 RecordGainLeft, Uint32 RecordGainRight);
//=============================================================================
// AC97_set_miscellaneous
//  This function set registers inside the AC97 by calling the read/write
//  functions above.  This function enables/disables loopback mode.  It sets
//  internal muxes in the ac97(Mix & Pop) and it sets whether the output of
//  the DSP board is tied to the master line output or the headphone output
//  See National Semiconductors, LM4550 Rev 2.1 data sheet on AC97
//
//  Inputs:
//      Pop:        SEE DATASHEET
//      Mix:        SEE DATASHEET
//      Loopback:   SEE DATASHEET
//      OutputDevice: 0)Line Output 1)Headphone Output
//
//  Outputs:
//  AC97_ERROR_INVALID_PARAMETER:  inputs invalid
//  AC97_ERROR_IN_READING_PARAMETER_REGISTER: unable to read register in ac97
//  AC97_ERROR_IN_WRITING_PARAMETER_REGISTER: last value written to register
//      in ac97 read back incorrectly
//
//=============================================================================
BSLAPI Uint32 AC97_set_miscellaneous(Uint32 Pop, Uint32 Mix, Uint32 LoopBack,
    Uint32 OutputDevice);
//=============================================================================
// AC97_set_variable_rate_control
//  This function set registers inside the AC97 by calling the read/write
//  functions above.  This function sets up the variable rate control of the
//  AC97.  By default it runs at 48KHz but any value can be inputed from 4Khz
//  to 48Khz.
//  See National Semiconductors, LM4550 Rev 2.1 data sheet on AC97
//
//  Inputs:
//      VariableRateOn:  0)default of 48Khz 1)RateKhz
//      Rate:  any value from 4000 to 48000
//
//  Outputs:
//  AC97_ERROR_INVALID_PARAMETER:  inputs invalid
//  AC97_ERROR_IN_READING_PARAMETER_REGISTER: unable to read register in ac97
//  AC97_ERROR_IN_WRITING_PARAMETER_REGISTER: last value written to register
//      in ac97 read back incorrectly
//
//=============================================================================
BSLAPI Uint32 AC97_set_variable_rate_control(Uint32 VariableRateOn, Uint32 Rate);
//=============================================================================
// AC97_data_in_recieve_buffer
//  This function returns the amount of data currently in the user's recieve
//  buffer.(32 bit words -- each piece of left and right data take up a word)
//
//=============================================================================
BSLAPI Uint32 AC97_data_in_recieve_buffer(void);
//=============================================================================
// AC97_room_in_transmit_buffer
//  This function returns the amount of room currently in the user's transmit
//  buffer.(32 bit words -- each piece of left and right data take up a word)
//
//=============================================================================
BSLAPI Uint32 AC97_room_in_transmit_buffer(void);
//=============================================================================
// AC97_flush_recieve_buffer
//  Removes all data from the recieve buffer.
//
//=============================================================================
BSLAPI void AC97_flush_recieve_buffer(void);
//=============================================================================
// AC97_flush_transmit_buffer
//  Removes all data from the transmit buffer.
//
//=============================================================================
BSLAPI void AC97_flush_transmit_buffer(void);
//=============================================================================
// AC97_flush_transmit_ping_pong_buffers
//  writes zero's over the transmit ping & pong buffers
//
//=============================================================================
BSLAPI void AC97_flush_transmit_ping_pong_buffers(void);
//=============================================================================
// AC97_flush_recieve_ping_pong_buffers
//  writes zero's over the recieve ping & pong buffers
//
//=============================================================================
BSLAPI void AC97_flush_recieve_ping_pong_buffers(void);
//=============================================================================
// AC97_read_register needed for inline compile of status registers.
//
//=============================================================================
BSLAPI Uint32 AC97_read_register(Uint32 AddressToRead);

/******************************************************************************\
* inline function declarations
\******************************************************************************/

IDECL BOOL Is_AC97_EDMA_Rcv_stopped(void);
IDECL BOOL Is_AC97_EDMA_Tx_stopped(void);
IDECL BOOL Is_AC97_Rcv_Buff_Overflow(void);
IDECL BOOL Is_AC97_Tx_Buff_Underflow(void);
IDECL BOOL Is_AC97_Colliding_Interrupts(void);
IDECL BOOL Is_AC97_Powered_Up(void);

/******************************************************************************\
* inline function definitions
\******************************************************************************/
#ifdef USEDEFS

/*----------------------------------------------------------------------------*/
IDEF BOOL Is_AC97_EDMA_Rcv_stopped(void)
{
    extern volatile far BOOL EDMA_Rcv_stopped_flag;
    return EDMA_Rcv_stopped_flag;
}
/*----------------------------------------------------------------------------*/
IDEF BOOL Is_AC97_EDMA_Tx_stopped(void)
{
    extern volatile far BOOL EDMA_Tx_stopped_flag;
    return EDMA_Tx_stopped_flag;
}
/*----------------------------------------------------------------------------*/
IDEF BOOL Is_AC97_Rcv_Buff_Overflow(void)
{
    extern volatile far BOOL Recieve_Buff_Overflow;
    return Recieve_Buff_Overflow;
}
/*----------------------------------------------------------------------------*/
IDEF BOOL Is_AC97_Tx_Buff_Underflow(void)
{
    extern volatile far BOOL Transmit_Buff_Underflow;
    return Transmit_Buff_Underflow;
}
/*----------------------------------------------------------------------------*/
IDEF BOOL Is_AC97_Colliding_Interrupts(void)
{
    extern volatile far BOOL Colliding_Interrupts_flag;
    return Colliding_Interrupts_flag;
}
/*----------------------------------------------------------------------------*/
IDEF BOOL Is_AC97_Powered_Up(void)
{
    //
    //  The low 4 bits should all be 1 indicating that REF,ANL,DAC, and ADC
    //  are all online.
    //  Bit 0 = 1 - ADC section ready to transmit data
    //  Bit 1 = 1 - DAC section ready to accept data
    //  Bit 2 = 1 - Analog Mixers REady
    //  Bit 3 = 1 -Vref is up to nominal level
    //
    return (0x0000000f == (0x0000000f & AC97_read_register(LM4550_PWRDOWNCTRL_STAT_REG)));
}
/*----------------------------------------------------------------------------*/

#endif /* USEDEFS */


#endif /* BSL_AC97_SUPPORT */
#endif /* _BSL_AC97_H_ */
/******************************************************************************\
* End of bsl_ac97.h
\******************************************************************************/
