/******************************************************************************\
* Copyright (C) 2006 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... Board Support Library for SPM176431
* VERSION.......... (Defined in README.TXT)
*------------------------------------------------------------------------------
* CONTENT.......... Header file for API of peripheral Digital IO
* FILENAME......... bsl_adio.h
*------------------------------------------------------------------------------
* PERIPHERAL NAME.. ADIO
* PERIPHERAL TYPE.. Single-device peripheral
* ACCESS MODE...... Direct (no handle)
* REGISTER ACCESS.. 16-bit wide
*
*   NOTE:   All functions and variables in this file are only intended to be
*           utilized by a single thread in any .out file or project.  These
*           functions are not "thread safe."  To explain, in a multithreaded
*           application you might fork off a single thread with the sole purpose
*           of controlling all digitial i\o output and input
*
\******************************************************************************/
#ifndef _BSL_ADIO_H_
#define _BSL_ADIO_H_

#include <bsl.h>

#if(BSL_ADIO_SUPPORT)
/******************************************************************************\
* scope and inline control macros
\******************************************************************************/

#ifdef  __cplusplus
    #define BSLAPI extern "C" far
#else
    #define BSLAPI extern far
#endif

#undef  USEDEFS
#undef  IDECL
#undef  IDEF

#ifdef _BSL_ADIO_MOD_
    #define IDECL BSLAPI
    #define USEDEFS
    #define IDEF
#else
    #ifdef  _INLINE
        #define IDECL static inline
        #define USEDEFS
        #define IDEF  static inline
    #else
        #define IDECL BSLAPI
    #endif
#endif

/******************************************************************************\
* global macro declarations
\******************************************************************************/

#ifndef FALSE
#define FALSE                                       0
#endif

#ifndef FALSE
#define TRUE                                        1
#endif

#define DO_NOT_CHANGE_VALUE                     0xFFFFFFFF

#define ADIO_ERROR_INVALID_PARAMETER            1
#define ADIO_NO_ERRORS                          0

#define FPGA_STARTING_ADDRESS                   0x6C080000

/* addresses of registers in the fpga */
#define FPGA_INT_STAT_OFFSET                    0x00000044
#define FPGA_INT_6_ENABLE                       0x00000050
#define FPGA_ADIO_INOUT_OFFSET                  0x00000090
#define FPGA_ADIO_DIR_OFFSET                    0x00000092
#define FPGA_ADIO_ENABLE_OFFSET                 0x00000094
#define FPGA_ADIO_INT_MODE_OFFSET               0x00000098
#define FPGA_ADIO_MASK_OFFSET                   0x0000009A
#define FPGA_ADIO_COMP_OFFSET                   0x0000009C
#define FPGA_ADIO_CAPT_OFFSET                   0x0000009E

/******************************************************************************\
* global typedef declarations
\******************************************************************************/

/******************************************************************************\
* global variable declarations
\******************************************************************************/


/******************************************************************************\
* global function declarations
\******************************************************************************/

//=============================================================================
// ADIO_enable
//  This function sets the interrupt mode, the debounce mode in hardware and
//  then enables the ADIO output( instead of McBSP0)
//
//  Input:
//      InterruptMode:  0)  Disabled
//                      1)  Reserved
//                      2)  Match
//                      3)  Event
//      Debounce:       0) 20ms debounce    1) 40ns debounce
//
//  Returns:
//      ADIO_ERROR_INVALID_PARAMETER
//      ADIO_NO_ERRORS
//
//=============================================================================
BSLAPI Uint32 ADIO_enable(Uint32 InterruptMode, Uint32 Debounce);
//=============================================================================
// ADIO_disable
//  This function disables the ADIO pins and enables McBSP 0 instead.
//
//  Returns:
//      ADIO_NO_ERRORS
//
//=============================================================================
BSLAPI Uint32 ADIO_disable(void);
//=============================================================================
// ADIO_Read_bit
//  This function reads the current state of a bit (high/low) and returns it
//
//  Inputs:
//      Bit: 0-7 are valid
//
//  Returns:    0) bit is low   1) bit is high
//
//=============================================================================
BSLAPI BOOL ADIO_Read_bit(Uint32 Bit);
//=============================================================================
// ADIO_Write_bit
//  This function writes to the current state of a bit (high/low)
//
//  Inputs:
//      Bit: 0-7 are valid
//      value: 0,1  (low, high)
//
//=============================================================================
BSLAPI void ADIO_Write_bit(Uint32 Bit, BOOL value);
//=============================================================================
// ADIO_Read_Capture_bit
//  This function reads the captured state of a bit (high/low) and returns it.
//  This captured value is obtained/stored after the last successfully
//  debounced interrupt.
//
//  Inputs:
//      Bit: 0-7 are valid
//
//  Returns:    0) bit is low   1) bit is high
//
//=============================================================================
BSLAPI BOOL ADIO_Read_Capture_bit(Uint32 Bit);
//=============================================================================
// ADIO_Write_Compare_bit
//  This function writes the a bit to the compare register that will be used
//  in the match mode interrupt.  If the debounced value of the port is equal
//  at all bits excluding the ones that are masked off then an interrupt will
//  be generated
//
//  Input:
//      Bit:  0-7 are valid
//      value:  0-1  the bit will be correctly compared when low(0) or high(1)
//
//=============================================================================
BSLAPI void ADIO_Write_Compare_bit(Uint32 Bit, BOOL value);
//=============================================================================
// ADIO_Set_Direction_Output_bit
//  This function turns the given bit to an output.
//
//  Input:
//      Bit:  0-7 are valid
//
//=============================================================================
BSLAPI void ADIO_Set_Direction_Output_bit(Uint32 Bit);
//=============================================================================
// ADIO_Set_Direction_Input_bit
//  This function turns the given bit to an input.
//
//  Input:
//      Bit:  0-7 are valid
//
//=============================================================================
BSLAPI void ADIO_Set_Direction_Input_bit(Uint32 Bit);
//=============================================================================
// ADIO_Mask_bit
//  This function masks the given bit from being analyzed to generate possible
//  interrupts.
//
//  Input:
//      Bit:  0-7 are valid
//
//=============================================================================
BSLAPI void ADIO_Mask_bit(Uint32 Bit);
//=============================================================================
// ADIO_Unmask_bit
//  This function unmasks the given bit so that it will analyzed to generate
//  possible interrupts.
//
//  Input:
//      Bit:  0-7 are valid
//
//=============================================================================
BSLAPI void ADIO_Unmask_bit(Uint32 Bit);


/******************************************************************************\
* inline function declarations
\******************************************************************************/

IDECL BOOL Is_State_of_Button(void);
IDECL BOOL Is_Interrupt_Ocurred(void);
IDECL void Reset_Interrupt_Ocurred(void);

/******************************************************************************\
* inline function definitions
\******************************************************************************/
#ifdef USEDEFS

/*----------------------------------------------------------------------------*/
IDEF BOOL Is_State_of_Button(void)
{
    extern far volatile BOOL State_of_Button;
    return State_of_Button;
}
/*----------------------------------------------------------------------------*/
IDEF BOOL Is_Interrupt_Ocurred(void)
{
    extern far volatile BOOL Interrupt_Ocurred;
    return Interrupt_Ocurred;

}
/*----------------------------------------------------------------------------*/
IDEF void Reset_Interrupt_Ocurred(void)
{
    extern far volatile BOOL Interrupt_Ocurred;
    extern far Uint32 Current_InterruptMode;
    extern far Uint32 Current_Debounce;

    Interrupt_Ocurred = FALSE;
    /* if using a match rather than event then reenable the interrupt */
    if(Current_InterruptMode == 2)
        ADIO_enable(Current_InterruptMode, Current_Debounce);

}
/*----------------------------------------------------------------------------*/
#endif /* USEDEFS */


#endif /* BSL_ADIO_SUPPORT */
#endif /* _BSL_ADIO_H_ */
/******************************************************************************\
* End of bsl_adio.h
\******************************************************************************/
