/******************************************************************************\
* Copyright (C) 2004 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... Board Support Library for SPM6420
* VERSION.......... (Defined in README.TXT)
*------------------------------------------------------------------------------
* CONTENT.......... Source file for API of DSP-Host Communication
* FILENAME......... bsl_comm.c
\******************************************************************************/
#define _BSL_COMM_MOD_

#include <bsl_comm.h>
#include <bsl_ucomm.h>
#include <bsl_uint.h>
#include <bsl_umisc.h>
#include <csl_irq.h>
#include <sem.h>


/******************************************************************************\
*                         L O C A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* static macro declarations
\******************************************************************************/


/******************************************************************************\
* static typedef declarations
\******************************************************************************/


/******************************************************************************\
* static function declarations
\******************************************************************************/

Uint32 COMM_SYSfnx_CheckLive(Uint32 a, Uint32 b);


/******************************************************************************\
* static variable definitions
\******************************************************************************/

// semaphore to call the tskFunctionCalls() function
SEM_Obj _COMM_semForFuncCall;

// it contains the last Function Call ID that has been received
Uint32 _COMM_funcCallID = 0x00000000u;


/******************************************************************************\
* static function definitions
\******************************************************************************/

//==============================================================================
// Initializes the COMM module of the BSL.
//
// Called from the BSL_init().
//==============================================================================
void COMM_init()
{
	// Clear PCI-to-Local Doorbell register
	UCOMM_RSET( P2LDBCLR, 0xFFFFFFFF );

	// enable the PCI-to-Local Doorbell IT source for ExtInt7
	UINT_FSETS( EIT7SRC, P2LDBIT, ENABLE );

	// Clearing of Local-to-PCI Doorbell register and
	// enabling of Local-to-PCI Doorbell Interrupt
	// must be done by the host application
	// before downloading and starting the DSP program.

	// Clear previous Interrupt Active status of PCI-to-Local Doorbell
	UINT_CLEAR_STATUS_S( P2LDBIA );

	COMM_hSemForFuncCall = &_COMM_semForFuncCall;
	SEM_new( COMM_hSemForFuncCall, 0 );
}

//==============================================================================
// Interrupt Service Routine for the Host-DSP communication
//
// The host application always sends a PCI-to-Local Doorbell Interrupt to the
// DSP program, if a DSP Function Call has required.
// That is, the host application writes the parameters of the DSP function
// to be executed into the communication buffer, after that, the Host-to-DSP
// Message Mailbox is written with the Identifier of the DSP fucntion to be
// called and some control bits, and then, the PCI-to-Local Doorbell is written
// by the same content with the Host-to-DSP Message Mailbox. The DSP program has
// enable PCI-to-Local Doorbell Interrupt at its initialization process,
// therfore, a PCI-to-Local Doorbell Interrupt can be occured in the DSP
// processor. The PCI-to-Local Doorbell Interrupt has been directed to the
// External Interrupt Pin 7, which is the interrupt 7 of the DSP processor by
// default. After the receiving the interrupt, the PCI-to-Local Doorbell
// register must be cleared (writing by 0xFFFFFFFF);
//==============================================================================
void COMM_ISR_CB()	// DO NOT USE interrupt keyword!!
{
	// clearing the PCI-to-Local Doorbell register to stop interrupt request
	// the content of the register don't care (The message is in the
	// communication data buffer's Host-to-DSP Mailbox.)
	UCOMM_RSET( P2LDBCLR, 0xFFFFFFFF );

	// clearing the sticky bit of the PCI-to-Local Doorbell Interrupt from the
	// Interrupt Status register
	UINT_CLEAR_STATUS_S( P2LDBIA );

	// posting semaphore to sign that a DSP fucntion call request has arrived
	// from the host
	SEM_ipost( COMM_hSemForFuncCall );
}

/*----------------------------------------------------------------------------*/
Uint32 COMM_SYSfnx_CheckLive(Uint32 a, Uint32 b)
{
	return (a+b);
}


/******************************************************************************\
*                        G L O B A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* global variable definitions
\******************************************************************************/

SEM_Handle COMM_hSemForFuncCall;


/******************************************************************************\
* global function definitions
\******************************************************************************/

/*----------------------------------------------------------------------------*/
void COMM_handshake()
{
	// writing Handshake Message to the DSP-to-Host Message Mailbox
	COMM_WRITE_MBOX( TO_HOST_MSG,
		_COMM_MM_SYSTEM_FCID | FCID_SYSTEM_HANDSHAKE |
		_COMM_MM_FUNC_IS_COMPLETED
	);

	// sending IT to the Host
	// Writing Local-to-PCI Doorbell register by the Handshake message to
	// generate a PCI IT, if it has been set by the host application
	UCOMM_RSET( L2PDBSET, COMM_READ_MBOX(TO_HOST_MSG) );
}

/*----------------------------------------------------------------------------*/
void COMM_waitForFunctionCall()
{
	// waiting for function call by semaphore
	// the semaphore is set by COMM_ISR()
	SEM_pend( COMM_hSemForFuncCall, SYS_FOREVER );

	// DSP gets a Function Call Request
	_COMM_funcCallID = COMM_getFCID();

	COMM_WRITE_MBOX( TO_HOST_MSG, 0x00000000u );
	COMM_WRITE_MBOX( TO_HOST_ERR, 0x00000000u );
}

/*----------------------------------------------------------------------------*/
void COMM_funcCallCompleted()
{
	Uint32	msgToHost;
	Uint32	bITtoHost;

	msgToHost = COMM_READ_MBOX( TO_HOST_MSG );

	if( msgToHost & _COMM_MM_UNKNOWN_FCID )
	{
		// The last DSP Function Call Request was not handled, because
		// the given FCID is unknown.
		msgToHost &= ~_COMM_MM_FCID_MASK;
		msgToHost |= _COMM_funcCallID & _COMM_MM_FCID_MASK;
	}

	if( COMM_READ_MBOX(TO_DSP_MSG) & _COMM_MM_IT_REQ ) bITtoHost = TRUE;
	else bITtoHost = FALSE;

	// clear Host-to-DSP Message Mailbox
	COMM_WRITE_MBOX( TO_DSP_MSG, 0 );

	// DSP tells to Host it is ready to receive a new Function Call Request
	COMM_WRITE_MBOX( TO_HOST_MSG,
		msgToHost						// message (FCID)
		| _COMM_MM_FUNC_IS_COMPLETED	// this is a response, not a request
		| _COMM_MM_SYSTEM_FCID			// DSP always sends System Messages
	);

	if( bITtoHost )
	{
		// writing Local-to-PCI Doorbell with the current message
		// to generate an interrupt to the host.
		// IT will be generated, only if the the host has been enabled the
		// Local-to-PCI Doorbell Interrupt generation and the main PCI
		// Interrupt generation previously.
		UCOMM_RSET( L2PDBSET, COMM_READ_MBOX(TO_HOST_MSG) );
	}
}

/*----------------------------------------------------------------------------*/
void COMM_systemFuncCall()
{
	switch( _COMM_funcCallID )
	{
	case FCID_SYSTEM_CHECK_LIVE:
		// Check live:
		// It adds values in the Host-to-DSP Data 0 and Data 1 Mailboxes,
		// and writes the result into the DSP-to-Host Data 0 Mailbox.
		FUNC_RET( Uint32,
			FUNC_CALL_2( COMM_SYSfnx_CheckLive, Uint32, Uint32)
		);
		break;

	default:
		// When you get an unknown Function Call Identifier, you should
		// call COMM_unknownFCID() function to indicate to the Host
		// application your DSP program does not support the required
		// DSP function.
		COMM_unknownFCID();
		break;
	}
}

/*----------------------------------------------------------------------------*/

/******************************************************************************\
* End of bsl_comm.c
\******************************************************************************/

