/******************************************************************************\
* Copyright (C) 2004 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... Board Support Library for SPM6420
* VERSION.......... (Defined in README.TXT)
*------------------------------------------------------------------------------
* CONTENT.......... C source file for API of peripheral FLASH
* FILENAME......... bsl_flash.c
* GENERATED BY..... Peripheral Generator v1.1
* GENERATED AT..... 02/10/2004 08:50:53
*------------------------------------------------------------------------------
* PERIPHERAL NAME.. FLASH
* PERIPHERAL TYPE.. Single-device peripheral
* ACCESS MODE...... Direct (no handle)
* REGISTER ACCESS.. 16-bit wide
\******************************************************************************/
#define _BSL_FLASH_MOD_

#include <bsl_flash.h>
#include <csl_irq.h>
#include <csl_dat.h>
#include <bsl_uint.h>


#if (BSL_FLASH_SUPPORT)
/******************************************************************************\
*                         L O C A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* static macro declarations
\******************************************************************************/

// valid values for Flash_lock
#define FLASH_UNLOCKED			0
#define FLASH_LOCKED			1

// flash write macro using flash relative offset
#define FLASH_WRITE16(flashByteOffset,val16) \
	*(volatile Uint16 *)(((Uint32)(FLASH_STARTING_ADDRESS)+(Uint32)(flashByteOffset))) \
	= ((Uint16)(val16))

// flash read macro using flash relative offset
#define FLASH_READ16(flashByteOffset) \
	(*(volatile Uint16 *)((Uint32)(FLASH_STARTING_ADDRESS)+(Uint32)(flashByteOffset)))

// DSP read macro using DSP absolute address
#define DSP_READ16(dspByteAddress) \
	(*(volatile Uint16 *)((Uint32)(dspByteAddress)))


/******************************************************************************\
* static typedef declarations
\******************************************************************************/


/******************************************************************************\
* static function declarations
\******************************************************************************/

// this declaration is in the application's main <app_name>cfg.h header file
#ifdef __cplusplus
extern "C" far SWI_Obj SWIFlashController;
#else
extern far SWI_Obj SWIFlashController;
#endif	// #ifdef __cplusplus

void	FLASH_swiFC_blockRead16();
Uint32	FLASH_swiFC_checkSectorBlank();
Uint32	FLASH_swiFC_blockComp16();

void	FLASH_swiFC_signalForEndOfRequest();
void	FLASH_swiFC_getNextRequest();
void	FLASH_swiFC_setCompleted(Uint32 reterr);
void	FLASH_swiFC_setInProcess();

void	FLASH_cmdUnlockBypass();
void	FLASH_cmdUnlockBypassProg16(Uint32 flashOffset, Uint16 data16);
void	FLASH_cmdUnlockBypassReset();
void	FLASH_cmdChipErase();
void	FLASH_cmdSectorErase(Uint32 sectorIndex);


/******************************************************************************\
* static variable definitions
\******************************************************************************/

// for flash device dimension
Uint32	FLASH_sizeInBytes;		// flash size in bytes
Uint32	FLASH_lastAddress;		// flash device's last byte address in the DSP memory space

// pointer to the RARO (Resource Access Request Object), which is currently in process
volatile FLASH_AccessReqObj *FLASH_pCurrentAccessReqObj = NULL;

// Multi-priority static FIFO queues to store the RAROs
QUE_Obj FLASH_accessReqQueL;	// low priority level for normal requests
QUE_Obj FLASH_accessReqQueH;	// high priority level for urgent requests

// variables used, when a flash operation requires more interrupt passes
// It is typically at the short CPU burst and short/long wait time for result of
// operations such as programming, sector and chip erasing. (Write a flash
// command, then wait short/long time for response of flash device.)
Uint32	FLASH_OP_param[3];


/******************************************************************************\
* static function definitions
\******************************************************************************/

//==============================================================================
// Initializes the flash controller module.
//
// Called from the BLS_init().
//
// The state of flash device's Ready/Busy# pin can be accessed through an
// EPLD register, and an interrupt can be generated for the Busy#-to-Ready
// transition on the ExtInt4 to ExtInt7.
//
// The Ready/Busy# pin is redirected to the ExtInt7. The ExtInt7 provides an
// interrupt controlled flash access method.
//==============================================================================
void FLASH_init()
{
	// initialize queues
	QUE_new( &FLASH_accessReqQueL );
	QUE_new( &FLASH_accessReqQueH );

	// enable the Flash Ready/Busy# IT for ExtInt7
	UINT_FSETS( EIT7SRC, FRYBYIT, ENABLE );

	// init FLASH module variables
	FLASH_sizeInBytes = BSL_boardDescriptor.flashSize << 20;
	FLASH_lastAddress = FLASH_STARTING_ADDRESS + FLASH_sizeInBytes - 2;

	// reset flash device (does not generate Busy# signal --> no IT)
	FLASH_WRITE16( 0x000, 0xF0 );

	// clear the sticky bit
	UINT_CLEAR_STATUS_S( FRYBYIA );

	// open DAT (if it has not been opened); it will be used for ...blockRead16()
	DAT_open( DAT_CHAANY, DAT_PRI_LOW, 0 );
}

//==============================================================================
// Copies a block of data from the flash memory to the DSP memory
// (it can cross sector boundaries)
//
// This function is called by the FLASH_swiControllerEngine() that is an SWI.
//==============================================================================
void FLASH_swiFC_blockRead16()
{
	//Uint16	*pBuf16InDSP;
	//Uint16	*pBuf16InFlash;
	Uint32	cnt;
	Uint32	xfrId;

	Uint32	flashOffset	= FLASH_pCurrentAccessReqObj->param[FLASH_IOP_flashOffset];
	Uint32	dspAddress	= FLASH_pCurrentAccessReqObj->param[FLASH_IOP_dspAddress];
	Uint32	numOf16bWs	= FLASH_pCurrentAccessReqObj->param[FLASH_IOP_numOf16bWs];

	// transfer block from flash to DSP
	cnt = numOf16bWs << 1;
	while( cnt >= 0xFFF8 )
	{
		xfrId = DAT_copy(
			(void *)(FLASH_STARTING_ADDRESS + flashOffset),
			(void *)(dspAddress),
			0xFFF8	// transfer byte count
		);
		cnt -= 0xFFF8;
		dspAddress += 0xFFF8;
		flashOffset += 0xFFF8;
		DAT_wait( xfrId );
	}
	if( cnt != 0 )
	{
		xfrId = DAT_copy(
			(void *)(FLASH_STARTING_ADDRESS + flashOffset),
			(void *)(dspAddress),
			(Uint16)(cnt)	// transfer byte count
		);
		DAT_wait( xfrId );
	}

	// data transfer with no DAT
	//pBuf16InDSP = (Uint16 *)(dspAddress);
	//pBuf16InFlash = (Uint16 *)(FLASH_STARTING_ADDRESS + flashOffset);
	//for( cnt = numOf16bWs; cnt != 0; cnt-- )
	//{
	//	*pBuf16InDSP++ = *pBuf16InFlash++;
	//}
}

//==============================================================================
// Checks the given sectors of the flash for blank
//
// This function is called by the FLASH_swiControllerEngine() that is an SWI.
//
// returns:
//		0		if there is a non-blank sector
//		1		if all the selected sectors are blank
//==============================================================================
Uint32 FLASH_swiFC_checkSectorBlank()
{
	Uint32	*pB;
	register Uint32	d0,d1,d2,d3;
	Uint32	cnt16;

	Uint32	startSI	= FLASH_pCurrentAccessReqObj->param[FLASH_IOP_startSI];
	Uint32	endSI	= FLASH_pCurrentAccessReqObj->param[FLASH_IOP_endSI];

	// start address
	cnt16 = FLASH_STARTING_ADDRESS + FLASH_getSectorStartOffset(startSI);
	pB = (Uint32 *)(cnt16);	// pointer to the beginning of the sector block

	// number of bytes to be checked
	cnt16 = (FLASH_STARTING_ADDRESS + FLASH_getSectorStartOffset(endSI) +
		FLASH_getSectorLength(endSI) - cnt16) >> 4;
	// the algorithm checks 16 bytes in a round
	// do not worry, each sector's length is a multiple of 16 bytes

	d0 = *pB++;
	d1 = *pB++;
	d2 = *pB++;
	d3 = *pB++;
	for( ; cnt16 != 0; cnt16-- )
	{
		if( d0 != 0xFFFFFFFF ) break;
		d0 = *pB++;
		if( d1 != 0xFFFFFFFF ) break;
		d1 = *pB++;
		if( d2 != 0xFFFFFFFF ) break;
		d2 = *pB++;
		if( d3 != 0xFFFFFFFF ) break;
		d3 = *pB++;
	}

	// cnt16 != 0 --> the checking failed --> there is a non-blank sector
	// cnt16 == 0 --> checking was successful --> all the selected sectors are blank
	return (cnt16 == 0);
}

//==============================================================================
// Reads the given block of the flash memory to compare its content with the
// given memory block located in the DSP's memory.
//
// This function is called by the FLASH_swiControllerEngine() that is an SWI.
//
// returns:
//		0		comparison failed
//		1		comparison was successful
//==============================================================================
Uint32 FLASH_swiFC_blockComp16()
{
	Uint16	*pBuf16InDSP;
	Uint16	*pBuf16InFlash;
	Uint32	cnt16;

	Uint32	flashOffset	= FLASH_pCurrentAccessReqObj->param[FLASH_IOP_flashOffset];
	Uint32	dspAddress	= FLASH_pCurrentAccessReqObj->param[FLASH_IOP_dspAddress];
	Uint32	numOf16bWs	= FLASH_pCurrentAccessReqObj->param[FLASH_IOP_numOf16bWs];

	// start addresses
	pBuf16InDSP		= (Uint16 *)(dspAddress);
	pBuf16InFlash	= (Uint16 *)(FLASH_STARTING_ADDRESS + flashOffset);

	// doing comparing
	for( cnt16 = numOf16bWs; cnt16 != 0; cnt16 -- )
	{
		if( *pBuf16InDSP++ != *pBuf16InFlash++ ) break;
	}

	// cnt16 != 0 --> comparison failed
	// cnt16 == 0 --> comparison was successful
	return (cnt16 == 0);
}

//==============================================================================
// At the end of each flash operation, this FLASH_swiFC_signalForEndOfTransfer()
// function is called to post the proper SWI or SEM defined in the Access
// Request Object.
//
// This function is called by the FLASH_swiControllerEngine() that is an SWI.
//==============================================================================
void FLASH_swiFC_signalForEndOfRequest()
{
	if( ((FLASH_pCurrentAccessReqObj->control) & _FLASH_TOS_MASK) == FLASH_TOS_SWI )
	{
		SWI_post( (SWI_Handle)(FLASH_pCurrentAccessReqObj->pSemOrSwiObj) );
	}
	else // if( ((FLASH_pCurrentAccessReqObj->operation) & _FLASH_TOS_MASK) == FLASH_TOS_SEM )
	{
		// in an ISR (SWI/HWI), you should call SEM_ipost() instead of SEM_post()
		SEM_ipost( (SEM_Handle)(FLASH_pCurrentAccessReqObj->pSemOrSwiObj) );
	}

	// to sign to the flash controller function,
	// there is no flash operation currently in process
	FLASH_pCurrentAccessReqObj = NULL;
}

//==============================================================================
// Sends a signal to the user (the current request has been served out), then
// checks the queues and if necessary, calls the FLASH_swiControllerEngine()
// function by posting its SWI object.
//==============================================================================
void FLASH_swiFC_getNextRequest()
{
	Uint32	gie;

	// signal to the user
	FLASH_swiFC_signalForEndOfRequest();
	// sets the FLASH_pCurrentAccessReqObj to NULL

	// Check whether or not there are more requests in the queues. It is true, if
	// the queues are not empty, at the next call of this FLASH_swiControllerEngine()
	// function, the queues will be checked again. However, the flash device is used
	// rarely typically. There are requests in queues not too often. It is much fast
	// to check twice the queues than post the SWI function twice and wait for the
	// DSP/BIOS scheduler to make the proper context switch for "nothing", because
	// the queues are emtpy.
	gie = IRQ_globalDisable();
	if( !QUE_empty(&FLASH_accessReqQueH) || !QUE_empty(&FLASH_accessReqQueL) )
	{
		// post the SWIFlashController to call this FLASH_swiControllerEngine()
		// function to perform the next request
		SWI_post( &SWIFlashController );
	}
	else
	{
		// no more requests (at the moment)
	}
	IRQ_globalRestore(gie);
}

//==============================================================================
// Sets that the current flash request has been completed and sets the return
// error code.
//
// parameters:
//		reterr			return error code
//						use the FLASH_RETERR_x constants
//==============================================================================
void FLASH_swiFC_setCompleted(Uint32 reterr)
{
	// set up control bit fields
	FLASH_pCurrentAccessReqObj->control =
		// keep the previously set fields:
		(FLASH_pCurrentAccessReqObj->control &
			(_FLASH_OP_MASK | _FLASH_TOQ_MASK | _FLASH_TOS_MASK))
		// initialize the others:
		| ((reterr)	& _FLASH_RETERR_MASK)	// return error code = no error
		| FLASH_SOR_COMPLETED;				// state of request = completed
}

//==============================================================================
// Sets that the current flash request is in process.
//==============================================================================
void FLASH_swiFC_setInProcess()
{
	// set up control bit fields
	FLASH_pCurrentAccessReqObj->control =
		// keep the previously set fields:
		(FLASH_pCurrentAccessReqObj->control &
			(_FLASH_OP_MASK | _FLASH_TOQ_MASK | _FLASH_TOS_MASK | _FLASH_RETERR_MASK))
		// initialize the others:
		| FLASH_SOR_IN_PROCESS;	// state of request = in process
}

//==============================================================================
// "Flash Controller" Engine part 2
//
// ("Flash Controller" Engine part 1 is the FLASH_ISR_CB() HWI.)
//
// This FLASH_swiControllerEngine() function is called by the Flash's Interrupt
// Service Routine, if necessary. The Flash's ISR is always called, when a
// low-to-high change is on the Busy#/Ready pin of the flash device. The Flash's
// ISR does not call always this FLASH_swiControllerEngine() function. If there
// is a very short task to be done (e.g. sending a programming command to the
// flash device), the task is run in the Flash's ISR (in a HW Interrupt Routine)
// avoided unnecessary task switches (the administration code running of the
// task switching).
//
// This is an SWI function that cannot be interrupted by tasks nor by Task
// Scheduler of the DSP/BIOS. (The Task Scheduler is the lowest priority SWI in
// the application.)
//==============================================================================
void FLASH_swiControllerEngine()
{
	Uint32	gie;
	Uint32	*pOrigParams, *pWorkParams;
	register Uint32 r0, r1, r2;

	// Is there a flash operation (access request) in process?
	if( FLASH_pCurrentAccessReqObj == NULL )
	{
		// No flash request in process, get the next request.
		gie = IRQ_globalDisable();
		if( !QUE_empty( &FLASH_accessReqQueH ) )
		{
			FLASH_pCurrentAccessReqObj = QUE_get( &FLASH_accessReqQueH );
		}
		else if( !QUE_empty( &FLASH_accessReqQueL ) )
		{
			FLASH_pCurrentAccessReqObj = QUE_get( &FLASH_accessReqQueL );
		}
		else
		{
			// no more requests (at the moment)
			//FLASH_pCurrentAccessReqObj = NULL;	// it is NULL currently
			IRQ_globalRestore(gie);
			return;
		}
		IRQ_globalRestore(gie);

		//----------------------------------------------------------------------
		// This section is reached, if there is a new flash access request.
		//----------------------------------------------------------------------

		// start the requested (defined) operation:
		switch( FLASH_pCurrentAccessReqObj->control & _FLASH_OP_MASK )
		{
		case FLASH_OP_READ_FROM_FLASH:	// performs the entire data transfer
			// set: the current request is in process
			//FLASH_swiFC_setInProcess();	// unnecessary

			// perform the entire data transfer
			FLASH_swiFC_blockRead16();

			// set: the request completed and there was no error
			FLASH_swiFC_setCompleted( FLASH_RETERR_SUCCESSFUL );

			// send a signal to the user and perform the next request, if there
			FLASH_swiFC_getNextRequest();
			break;

		case FLASH_OP_WRITE_TO_FLASH:	// performs the write of the first selected 16-bit word only!
			// set: the current request is in process
			FLASH_swiFC_setInProcess();

			// copy parameters to the work variables
			// all the three parameters are used (for flashOffset, dspAddress, numOf16bWs)
			pOrigParams = (Uint32 *)FLASH_pCurrentAccessReqObj->param;
			r0 = *pOrigParams++;	// flashOffset
			r1 = *pOrigParams++;	// dspAddress
			r2 = *pOrigParams;		// numOf16bWs
			pWorkParams = FLASH_OP_param;
			*pWorkParams++ = r0;	// flashOffset
			*pWorkParams++ = r1;	// dspAddress
			*pWorkParams = r2;		// numOf16bWs

			// enter Unlock Bypass Programming mode
			FLASH_cmdUnlockBypass();

			// start the programming in unlock bypass mode
			// by programming the first 16-bit word
			FLASH_cmdUnlockBypassProg16(
				FLASH_OP_param[FLASH_IOP_flashOffset],
				DSP_READ16( FLASH_OP_param[FLASH_IOP_dspAddress] )
			);
			// after the physical programming, a Busy#-to-Ready low-to-high rising edge
			// will be generated on the pin Ready/Busy# of the flash device, which can
			// generate an interrupt.

			// There is no FLASH_swiFC_getNextRequest()!
			// We have to wait for the response of the flash device. If the programming
			// of the set single 16-bit word is done, an IT will be generated, and the
			// FLASH_ISR_CB(), the Flash Controller Engine part 1, will be called.
			break;

		case FLASH_OP_ERASE_SECTORS:	// starts the erasing of the first selected sector
			// set: the current request is in process
			FLASH_swiFC_setInProcess();

			// copy parameters to the work variables
			// only two parameters are used (for startSI = currSI, endSI)
			pOrigParams = (Uint32 *)FLASH_pCurrentAccessReqObj->param;
			r0 = *pOrigParams++;	// startSI
			r1 = *pOrigParams;		// endSI
			pWorkParams = FLASH_OP_param;
			*pWorkParams++ = r0;	// currSI = startSI
			*pWorkParams = r1;		// endSI

			// start the first selected sector erasing
			FLASH_cmdSectorErase( FLASH_OP_param[FLASH_IOP_currSI] );
			// after the physical erasing, a Busy#-to-Ready low-to-high rising edge
			// will be generated on the pin Ready/Busy# of the flash device, which can
			// generate an interrupt.

			// There is no FLASH_swiFC_getNextRequest()!
			// We have to wait for the response of the flash device. If the programming
			// of the set single 16-bit word is done, an IT will be generated, and the
			// FLASH_ISR_CB(), the Flash Controller Engine part 1, will be called.
			break;

		case FLASH_OP_ERASE_CHIP:	// starts the chip erasing
			// set: the current request is in process
			FLASH_swiFC_setInProcess();

			// start the chip erasing
			FLASH_cmdChipErase();
			// after the physical erasing, a Busy#-to-Ready low-to-high rising edge
			// will be generated on the pin Ready/Busy# of the flash device, which can
			// generate an interrupt.

			// There is no FLASH_swiFC_getNextRequest()!
			// We have to wait for the response of the flash device. If the erasing of
			// the entire chip is done, an IT will be generated, and the FLASH_ISR_CB(),
			// the Flash Controller Engine part 1, will be called.
			break;

		case FLASH_OP_CHECK_BLANK:	// checks all the given sectors for blank
			// set: the current request is in process
			//FLASH_swiFC_setInProcess();	// unnecessary

			// copy parameters to the work variables
			// only two parameters are used (for startSI = currSI, endSI)
			pOrigParams = (Uint32 *)FLASH_pCurrentAccessReqObj->param;
			r0 = *pOrigParams++;	// startSI
			r1 = *pOrigParams;		// endSI
			pWorkParams = FLASH_OP_param;
			*pWorkParams++ = r0;	// currSI = startSI
			*pWorkParams = r1;		// endSI

			// checks all the given sectors for blank
			if( FLASH_swiFC_checkSectorBlank() )
			{
				// set: the request completed and all the given sectors are blank
				FLASH_swiFC_setCompleted( FLASH_RETERR_SUCCESSFUL );
			}
			else
			{
				// set: the request completed, but there was a dirty sector
				FLASH_swiFC_setCompleted( FLASH_RETERR_DIRTY_SECTOR );
			}

			// send a signal to the user and perform the next request, if there
			FLASH_swiFC_getNextRequest();
			break;

		case FLASH_OP_COMPARE_BLOCK:	// compare the entire block
			// set: the current request is in process
			//FLASH_swiFC_setInProcess();	// unnecessary

			// copy parameters to the work variables
			// all the three parameters are used (for flashOffset, dspAddress, numOf16bWs)
			pOrigParams = (Uint32 *)FLASH_pCurrentAccessReqObj->param;
			r0 = *pOrigParams++;	// flashOffset
			r1 = *pOrigParams++;	// dspAddress
			r2 = *pOrigParams;		// numOf16bWs
			pWorkParams = FLASH_OP_param;
			*pWorkParams++ = r0;	// flashOffset
			*pWorkParams++ = r1;	// dspAddress
			*pWorkParams = r2;		// numOf16bWs

			// compare the entire block
			if( FLASH_swiFC_blockComp16() )
			{
				// set: the request completed and the comparison was successful
				FLASH_swiFC_setCompleted( FLASH_RETERR_SUCCESSFUL );
			}
			else
			{
				// set: the request completed, but the comparison failed
				FLASH_swiFC_setCompleted( FLASH_RETERR_COMPARISON_FAILED );
			}

			// send a signal to the user and perform the next request, if there
			FLASH_swiFC_getNextRequest();
			break;
		}

		// A new flash operation has been started (or performed completely).
		// If it was performed completely, there was an FLASH_swiFC_getNextRequest() call
		// to recall this FLASH_swiControllerEngine() function again, if necessary,
		// to receive a new request.
		return;
	}
	else
	{
		// this part of the code should never be reached normally
	}
}

//==============================================================================
// FLASH module's Callback Interrupt Service Routine
// "Flash Controller" Engine part 1
//
// ("Flash Controller" Engine part 2 is the FLASH_swiControllerEngine() SWI.)
//
// This function is called, when there is a rising edge on the flash device's
// Ready/Busy# pin (that is, there is a Busy#-to-Ready change).
//==============================================================================
void FLASH_ISR_CB()		// DO NOT USE interrupt keyword!!
{
	// clearing the sticky bit of the Flash's Ready/Busy# Interrupt from the
	// Interrupt Status register to enable further interrupts
	UINT_CLEAR_STATUS_S( FRYBYIA );

	// Is there an operation in process?
	if( FLASH_pCurrentAccessReqObj != NULL )
	{
		// continue/finish the current operation
		switch( FLASH_pCurrentAccessReqObj->control & _FLASH_OP_MASK )
		{
		case FLASH_OP_WRITE_TO_FLASH:	// the physical programming of a 16-bit word has been completed
			// checking the last programmed word
			if( FLASH_READ16( FLASH_OP_param[FLASH_IOP_flashOffset] ) !=
				DSP_READ16( FLASH_OP_param[FLASH_IOP_dspAddress] ) )
			{
				// the programming failed

				// the end of the block programming
				// leave the Unlock Bypass mode, back to the normal read mode
				FLASH_cmdUnlockBypassReset();

				// set: the request completed, but there was a programming error
				FLASH_swiFC_setCompleted( FLASH_RETERR_PROGRAMMING_ERROR );

				// send a signal to the user and perform the next request, if there
				//FLASH_swiFC_getNextRequest();
				//return;
				// it is at the end of this FLASH_ISR_CB() function
			}

			// ok, the last 16-bit word programming was successful
			// are there more words to be programmed?
			FLASH_OP_param[FLASH_IOP_numOf16bWs]--;
			if( FLASH_OP_param[FLASH_IOP_numOf16bWs] != 0 )
			{
				// jump to the next address
				FLASH_OP_param[FLASH_IOP_flashOffset]	+= 2;
				FLASH_OP_param[FLASH_IOP_dspAddress]	+= 2;

				// start a new 16-bit word programming in unlock bypass mode
				FLASH_cmdUnlockBypassProg16(
					FLASH_OP_param[FLASH_IOP_flashOffset],
					DSP_READ16( FLASH_OP_param[FLASH_IOP_dspAddress] )
				);

				// after the physical programming, a Busy#-to-Ready low-to-high rising edge
				// will be generated on the pin Ready/Busy# of the flash device, which can
				// generate an interrupt again
				return;
			}
			else
			{
				// the end of the block programming
				// leave the Unlock Bypass mode, back to the normal read mode
				FLASH_cmdUnlockBypassReset();

				// set: the request completed and the programming was successful
				FLASH_swiFC_setCompleted( FLASH_RETERR_SUCCESSFUL );

				// send a signal to the user and perform the next request, if there
				//FLASH_swiFC_getNextRequest();
				//return;
				// it is at the end of this FLASH_ISR_CB() function
			}
			break;

		case FLASH_OP_ERASE_SECTORS:	// the physical erasing of a sector has been finished
			// there is no check whether or not the erasing was successful
			// are there more sectors to be programmed?
			if( FLASH_OP_param[FLASH_IOP_currSI] != FLASH_OP_param[FLASH_IOP_endSI] )
			{
				// jump to the next sector
				FLASH_OP_param[FLASH_IOP_currSI]++;

				// start a new sector erasing
				FLASH_cmdSectorErase( FLASH_OP_param[FLASH_IOP_currSI] );

				// after the physical erasing, a Busy#-to-Ready low-to-high rising edge
				// will be generated on the pin Ready/Busy# of the flash device, which can
				// generate an interrupt again
				return;
			}
			else
			{
				// all the necessary sectors were erased

				// set: the request completed and the erasing was successful
				FLASH_swiFC_setCompleted( FLASH_RETERR_SUCCESSFUL );

				// send a signal to the user and perform the next request, if there
				//FLASH_swiFC_getNextRequest();
				//return;
				// it is at the end of this FLASH_ISR_CB() function
			}
			break;

		case FLASH_OP_ERASE_CHIP:	// chip erase has been finished
			// set: the request completed and the erasing was successful
			FLASH_swiFC_setCompleted( FLASH_RETERR_SUCCESSFUL );
			// There is no check, whether the erasing was successful or not.

			// send a signal to the user and perform the next request, if there
			//FLASH_swiFC_getNextRequest();
			//return;
			// it is at the end of this FLASH_ISR_CB() function
			break;
		}

		// send a signal to the user (because the last flash operation has been
		// finished), and perform the next request, if there
		FLASH_swiFC_getNextRequest();
	}
	else
	{
		// this part of the code should never be reached normally
	}
}

//==============================================================================
// "Unlock bypass" flash command
// Enters unlock bypass mode
//==============================================================================
void FLASH_cmdUnlockBypass()
{
	FLASH_WRITE16( 0xAAA, 0xAA );
	FLASH_WRITE16( 0x555, 0x55 );
	FLASH_WRITE16( 0xAAA, 0x20 );
}

//==============================================================================
// "A single 16-bit word programming in unlock bypass mode" flash command
// Starts the programming of a single 16-bit word in unlock bypass mode
//
// parameters:
//		flashOffset			Byte offset within the flash address space
//		data16				16-bit word to be written in the flash at the
//							address flashOffset
//==============================================================================
void FLASH_cmdUnlockBypassProg16(Uint32 flashOffset, Uint16 data16)
{
	FLASH_WRITE16( flashOffset, 0xA0 );
	FLASH_WRITE16( flashOffset, data16 );
}

//==============================================================================
// "Unlock bypass reset" flash command
// Leaves unlock bypass mode
//==============================================================================
void FLASH_cmdUnlockBypassReset()
{
	FLASH_WRITE16( 0x000, 0x90 );
	FLASH_WRITE16( 0x000, 0x00 );
}

//==============================================================================
// "Chip erase" flash command
// Starts the erasing of the entire flash memory
//==============================================================================
void FLASH_cmdChipErase()
{
	FLASH_WRITE16( 0xAAA, 0xAA );
	FLASH_WRITE16( 0x555, 0x55 );
	FLASH_WRITE16( 0xAAA, 0x80 );
	FLASH_WRITE16( 0xAAA, 0xAA );
	FLASH_WRITE16( 0x555, 0x55 );
	FLASH_WRITE16( 0xAAA, 0x10 );
}

//==============================================================================
// "A single sector erasing" flash command
// Starts the erasing of a single sector's all the 16-bit words
//
// parameters:
//		sectorIndex			Index of the sector to be erased. Counted from 0.
//							The 0-index sector is at the beginning of the flash
//							memory.
//==============================================================================
void FLASH_cmdSectorErase(Uint32 sectorIndex)
{
	FLASH_WRITE16( 0xAAA, 0xAA );
	FLASH_WRITE16( 0x555, 0x55 );
	FLASH_WRITE16( 0xAAA, 0x80 );
	FLASH_WRITE16( 0xAAA, 0xAA );
	FLASH_WRITE16( 0x555, 0x55 );
	FLASH_WRITE16( FLASH_getSectorStartOffset(sectorIndex), 0x30 );
}


/******************************************************************************\
*                        G L O B A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* global variable definitions
\******************************************************************************/


/******************************************************************************\
* global function definitions
\******************************************************************************/

//==============================================================================
// Checking parameters (address ranges) for data transfer
//
// parameters:
//		flashOffset			Starting byte offset within the flash address space
//		dspAddress			Starting byte address within the DSP address space
//							(must be an 16-bit aligned address)
//		numOf16bWs			Number of 16-bit words to be transferred
//
// return:
//		FLASH_CHKPRM_SIZE	too many 16-bit words should be transferred
//		FLASH_CHKPRM_FLASH	flash offset is too big
//		FLASH_CHKPRM_DSP	DSP address (range) falls into flash memory address region
//		FLASH_CHKPRM_OK		all right
//==============================================================================
Uint32 FLASH_checkTrParams(
	Uint32 flashOffset, Uint32 dspAddress, Uint32 numOf16bWs)
{
	Uint32	numOfBytes;

	if( (numOf16bWs == 0) || (numOf16bWs  > (FLASH_sizeInBytes >> 1)) )
	{
		// either the size of buffer to be transferred is larger than the flash
		// memory's whole size, or it is zero length
		return FLASH_CHKPRM_SIZE;
	}

	if( (flashOffset & 1) || (flashOffset >= FLASH_sizeInBytes) )
	{
		// either not 16-bit word aligned address (offset),
		// or too big flash offset (over the flash size)
		return FLASH_CHKPRM_FLASH;
	}

	// create transfer block size in bytes
	// calculation requirement: max flash device size less than 2 GBytes (not MB!)
	numOfBytes = numOf16bWs << 1;

	if( (flashOffset + numOfBytes) > FLASH_sizeInBytes )
	{
		// the end of the flash buffer to be transferred is over the end of the flash
		return FLASH_CHKPRM_SIZE;
	}

	if( dspAddress < FLASH_STARTING_ADDRESS )
	{
		if( (dspAddress + numOfBytes) > FLASH_STARTING_ADDRESS )
		{
			// the end of the DSP buffer to be transferred falls into the flash memory
			return FLASH_CHKPRM_SIZE;
		}
	}
	else if( (dspAddress & 1) || (dspAddress <= FLASH_lastAddress) )
	{
		// either not 16-bit word aligned address,
		// or the DSP starting address range falls into the flash memory range
		return FLASH_CHKPRM_DSP;
	}

	return FLASH_CHKPRM_OK;
}

//==============================================================================
// Checking parameters (sector indexes) for operations that works on sectors
//
// parameters:
//		startSI		Index of the first sector in the selected sector block
//					Counter from 0.
//		endSI		Index of the last sector in the selected sector block
//					Counter from 0.
//
// return:
//		FLASH_CHKPRM_START_SI	Too big sector index.
//		FLASH_CHKPRM_END_SI		endSI less then startSI
//		FLASH_CHKPRM_OK			all right
//==============================================================================
Uint32 FLASH_checkScParams(Uint32 startSI, Uint32 endSI)
{
	if( startSI > FLASH_getLastSectorIndex() ) return FLASH_CHKPRM_START_SI;
	if( endSI < startSI ) return FLASH_CHKPRM_END_SI;
	return FLASH_CHKPRM_OK;
}

//==============================================================================
// Put a previously allocated (by the user) and filled up (by a call of one of
// the FLASH_makeReq_x() functions) FLASH_AccessReqObj into the defined queue to
// request a flash operation in fact.
//
// See the Common considerations description for more details.
//
// parameters:
//		pReqObj				Pointer to the previously allocated and filled up
//							FLASH_AccessReqObj structure.
//		bResetSEM			If the pReqObj defines a semaphore signaling type
//							and the parameter bResetSEM is true, the semaphore
//							will be reset. Can be TRUE, called from TSK!
//
// LOOK OUT! If you call this FLASH_accessReq() function from a context of SWI,
// the parameter bResetSEM MUST BE FALSE; and before the call of this function,
// you should call the SEM_reset() function from a context of TSK for that
// semaphore, which is pointed by the pReqObj.
//==============================================================================
void FLASH_accessReq(FLASH_AccessReqObj *pReqObj, BOOL bResetSEM)
{
	Uint32	gie;

	// set up control bit fields
	pReqObj->control =
		// keep the previously set fields:
		(pReqObj->control &
			(_FLASH_OP_MASK | _FLASH_TOQ_MASK | _FLASH_TOS_MASK))
		// initialize the others:
		| FLASH_RETERR_BUSY		// return error code = no error, just busy
		| FLASH_SOR_IN_QUEUE;	// state of request = in queue

	// If the signaling type is semaphore and the bResetSEM is TRUE,
	// the semaphore will be reset.
	// LOOK OUT! SEM_reset() can be called from TSK only!
	if( bResetSEM && ((pReqObj->control & _FLASH_TOS_MASK) == FLASH_TOS_SEM) )
	{
		SEM_reset( (SEM_Handle)(pReqObj->pSemOrSwiObj), 0 );
	}

	// put the request into the proper queue
	if( ((pReqObj->control) & _FLASH_TOQ_MASK) == FLASH_TOQ_NORMAL )
	{
		QUE_put( &FLASH_accessReqQueL, pReqObj );
	}
	else // if( ((pReqObj->control) & _FLASH_TOQ_MASK) == FLASH_TOQ_EMERGENCY )
	{
		QUE_put( &FLASH_accessReqQueH, pReqObj );
	}

	gie = IRQ_globalDisable();

	// The Flash Controller function looks at the queues automatically
	// after serving out of each request.
	// If there is no flash operation performance in process currently,
	// we have to kick the Flash Controller Engine function to look at the queues,
	// there is (at least) one new request.
	if( FLASH_pCurrentAccessReqObj == NULL ) SWI_post( &SWIFlashController );

	IRQ_globalRestore(gie);
}

//==============================================================================
// Returns the flash byte offset of the beginning of the given sector
//
// parameters:
//		sectorIndex			Selects the desired sector. Counted from 0.
//							The 0-index sector is at the beginning of the flash
//							memory.
// return:
//		The starting flash byte offset of the given sector.
//==============================================================================
Uint32 FLASH_getSectorStartOffset(Uint32 sectorIndex)
{
	if( FLASH_sizeInBytes == 0x00400000 )
	{
		// there is an Am29LV320DB flash device installed on the board

		if( sectorIndex > 7 ) return ((sectorIndex - 7) << 16);
		return (sectorIndex << 13);
	}
	else // if( FLASH_sizeInBytes == 0x00200000 )
	{
		// there is an Am29LV160DB flash device installed on the board

		if( sectorIndex > 3 ) return ((sectorIndex - 3) << 16);
		else if( sectorIndex == 0 ) return 0;
		else if( sectorIndex != 3 ) return 0x004000 + ((sectorIndex - 1) << 13);
		else return 0x008000;
	}
}

//==============================================================================
// Returns the length of the selected sector in bytes
//
// parameters:
//		sectorIndex			Selects the desired sector. Counted from 0.
//							The 0-index sector is at the beginning of the flash
//							memory.
// return:
//		The length of the selected sector in bytes.
//==============================================================================
Uint32 FLASH_getSectorLength(Uint32 sectorIndex)
{
	if( FLASH_sizeInBytes == 0x00400000 )
	{
		// there is an Am29LV320DB flash device installed on the board

		if( sectorIndex > 7 ) return 0x10000;
		return 0x02000;
	}
	else // if( FLASH_sizeInBytes == 0x00200000 )
	{
		// there is an Am29LV160DB flash device installed on the board

		if( sectorIndex > 3 ) return 0x10000;
		else if( sectorIndex == 0 ) return 0x04000;
		else if( sectorIndex != 3 ) return 0x02000;
		else return 0x08000;
	}
}

//==============================================================================
// Returns the last sector index of the flash memory
//
// return:
//		The last sector index of the installed flash device
//==============================================================================
Uint32 FLASH_getLastSectorIndex()
{
	if( FLASH_sizeInBytes == 0x00400000 )
	{
		// there is an Am29LV320DB flash device installed on the board
		return 70;
	}
	else // if( FLASH_sizeInBytes == 0x00200000 )
	{
		// there is an Am29LV160DB flash device installed on the board
		return 34;
	}

}

//==============================================================================
// Returns the size of the flash memory in bytes
//
// return:
//		flash memory size in bytes
//==============================================================================
Uint32 FLASH_getFlashSizeInBytes()
{
	return FLASH_sizeInBytes;
}


#endif /* BSL_FLASH_SUPPORT */
/******************************************************************************\
* End of bsl_flash.c
\******************************************************************************/
