/******************************************************************************\
* Copyright (C) 2004 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... DSP/BIOS Board Support Library for SPM6420 and SPM6430
* VERSION.......... (Defined in README.TXT)
*------------------------------------------------------------------------------
* CONTENT.......... Header file for API of DSP-Host Communication
* FILENAME......... bsl_comm.h
\******************************************************************************/
#ifndef _BSL_COMM_H_
#define _BSL_COMM_H_

#include <bsl.h>
#include <bsl_commhal.h>
#include <sem.h>


/******************************************************************************\
* scope and inline control macros
\******************************************************************************/
#ifdef __cplusplus
	#define BSLAPI extern "C" far
#else
	#define BSLAPI extern far
#endif

#undef	USEDEFS
#undef	IDECL
#undef	IDEF

#ifdef	_BSL_COMM_MOD_
	#define IDECL BSLAPI
	#define USEDEFS
	#define IDEF
#else
	#ifdef	_INLINE
		#define	IDECL static inline
		#define	USEDEFS
		#define	IDEF  static inline
	#else
		#define	IDECL BSLAPI
	#endif
#endif


/******************************************************************************\
* global macro declarations
\******************************************************************************/

// Communication Function Call Loop BEGIN (prolog)
#define COMM_FUNC_CALL_LOOP_BEGIN \
COMM_handshake(); \
for(;;) \
{ \
	COMM_waitForFunctionCall(); \
	if( COMM_isUserFCID() ) \
	{ \
		switch( COMM_getFCID() ) \
		{

// Communication Function Call Loop END (epilog)
#define COMM_FUNC_CALL_LOOP_END \
		default: \
			COMM_unknownFCID(); \
			break; \
		} \
	} \
	else \
	{ \
		COMM_systemFuncCall(); \
	} \
	COMM_funcCallCompleted(); \
}


/******************************************************************************\
* global typedef declarations
\******************************************************************************/


/******************************************************************************\
* global variable declarations
\******************************************************************************/

BSLAPI SEM_Handle COMM_hSemForFuncCall;


/******************************************************************************\
* global function declarations
\******************************************************************************/

//==============================================================================
//	Initializes the COMM module of the BSL.
//
//	This function can be called after initialization of the BSL.
//	This function have to be called before calls of any COMM_ functions.
//==============================================================================
BSLAPI void COMM_init();

//==============================================================================
//	Sends a 'handshake' to the host application.
//
//	This function have to be called before any communication between the DSP
//	program and the host application.
//==============================================================================
BSLAPI void COMM_handshake();

//==============================================================================
//	Waits for a DSP Function Call Request from the host application.
//
//	The function is waiting for the DSP Function Call Request with an assebly
//	instruction IDLE. (The host application sends an IT to the DSP program, if
//	there is a downloaded DSP Function Call Request in the Communication
//	Buffer.)
//==============================================================================
BSLAPI void COMM_waitForFunctionCall();

//==============================================================================
//	Sends a "DSP Function Call has completed" message to the host application.
//
//	You should call this function after a DSP Function Call.
//==============================================================================
BSLAPI void COMM_funcCallCompleted();

//==============================================================================
//	Performs a built-in "System" DSP Function Call.
//
//	You should call this function, if the received DSP Function Call Identifier
//	belongs to a "System" DSP function according to the COMM_isUserFCID() or
//	COMM_isSystemFCID().
//==============================================================================
BSLAPI void COMM_systemFuncCall();


/******************************************************************************\
* inline function declarations
\******************************************************************************/

//==============================================================================
//	Returns the DSP Function Call Identifier part of the last received
//	Host-to-DSP message from Host-to-DSP Message Mailbox.
//
//	You can call different DSP functions according to the got value.
//
//	returns:
//		The DSP Function Call Identifier part of the last received Host-to-DSP
//		message. The lower 24 bits are valid.
//==============================================================================
IDECL Uint32 COMM_getFCID();

//==============================================================================
//	Sets the Function Call Identifier part of the DSP-to-Host Message Mailbox.
//
//	parameters:
//		fcid			the new Function Call Identifier for the DSP-to-Host
//						Message Mailbox
//						The lower 24 bits are valid.
//==============================================================================
IDECL void COMM_setFCID(Uint32 fcid);

//==============================================================================
//	Writes the DSP-to-Host Error Message Mailbox with the given error message
//	and sets the Error Indicator Bit in the DSP-to-Host Message Mailbox.
//
//	parameters:
//		error			Error Message word
//						All the 32 bits are valid.
//==============================================================================
IDECL void COMM_setError(Uint32 error);

//==============================================================================
//	Clears the DSP-to-Host Message Mailbox and set the Unknown Function Call
//	Indentifier flag in the DSP-to-Host Message Mailbox.
//==============================================================================
IDECL void COMM_unknownFCID();

//==============================================================================
//	Returns whether the last receive Host-to-DSP message contains a User DSP
//	Function Call Identifier or a "System" DSP Function Call Identifier.
//
//	returns:
//		TRUE	the last receive Host-to-DSP message contains a User DSP
//				Function Call Identifier.
//		FALSE	the last receive Host-to-DSP message contains a "System" DSP
//				Function Call Identifier.
//==============================================================================
IDECL Uint32 COMM_isUserFCID();

//==============================================================================
//	Returns whether the last receive Host-to-DSP message contains a User DSP
//	Function Call Identifier or a "System" DSP Function Call Identifier.
//
//	returns:
//		TRUE	the last receive Host-to-DSP message contains a "System" DSP
//				Function Call Identifier.
//		FALSE	the last receive Host-to-DSP message contains a User DSP
//				Function Call Identifier.
//==============================================================================
IDECL Uint32 COMM_isSystemFCID();


/******************************************************************************\
* inline function definitions
\******************************************************************************/
#ifdef USEDEFS

/*----------------------------------------------------------------------------*/
/* reads COMM_TO_DSP_MSG_MBOX_ADDR */
IDEF Uint32 COMM_getFCID()
{
	return ( COMM_READ_MBOX(TO_DSP_MSG) & _COMM_MM_FCID_MASK );
}

/*----------------------------------------------------------------------------*/
/* writes COMM_TO_HOST_MSG_MBOX_ADDR */
IDEF void COMM_setFCID(Uint32 fcid)
{
	COMM_WRITE_MBOX( TO_HOST_MSG,
		( COMM_READ_MBOX( TO_HOST_MSG ) & ~_COMM_MM_FCID_MASK ) |
		(fcid & _COMM_MM_FCID_MASK)
	);
}

/*----------------------------------------------------------------------------*/
/* writes COMM_TO_HOST_ERR_MBOX_ADDR */
IDEF void COMM_setError(Uint32 errorMessage)
{
	/* write DSP-to-Host Error Message Mailbox */
	COMM_WRITE_MBOX( TO_HOST_ERR, errorMessage );

	/* set Error Indicator Bit in the DSP-to-Host Messaeg Mailbox */
	COMM_WRITE_MBOX( TO_HOST_MSG,
		COMM_READ_MBOX( TO_HOST_MSG ) | _COMM_MM_FUNCTION_FAILED );
}

/*----------------------------------------------------------------------------*/
IDEF void COMM_unknownFCID()
{
	/* clear the DSP-to-Host Message Mailbox and set the Unknown FCID flag. */
	COMM_WRITE_MBOX( TO_HOST_MSG, _COMM_MM_UNKNOWN_FCID );
}

/*----------------------------------------------------------------------------*/
IDEF Uint32 COMM_isUserFCID()
{
	return ( COMM_READ_MBOX(TO_DSP_MSG) & _COMM_MM_USER_FCID);
}

/*----------------------------------------------------------------------------*/
IDEF Uint32 COMM_isSystemFCID()
{
	return ( COMM_READ_MBOX(TO_DSP_MSG) & _COMM_MM_SYSTEM_FCID);
}

/*----------------------------------------------------------------------------*/
#endif /* USEDEFS */

#endif /* _BSL_COMM_H_ */
/******************************************************************************\
* End of bsl_comm.h
\******************************************************************************/

