/******************************************************************************\
* Copyright (C) 2004 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... Board Support Library for SPM6420
* VERSION.......... (Defined in README.TXT)
*------------------------------------------------------------------------------
* CONTENT.......... Header file for API of peripheral PCI Bus
* FILENAME......... bsl_pcibus.h
\******************************************************************************/
#ifndef _BSL_PCIBUS_H_
#define _BSL_PCIBUS_H_

#include <bsl.h>
#include <csl_pci.h>

#include <stddef.h>
#include <sem.h>
#include <swi.h>
#include <que.h>


#if (BSL_PCIBUS_SUPPORT)
/******************************************************************************\
* scope and inline control macros
\******************************************************************************/
#ifdef  __cplusplus
    #define BSLAPI extern "C" far
#else
    #define BSLAPI extern far
#endif

#undef  USEDEFS
#undef  IDECL
#undef  IDEF

#ifdef  _BSL_PCIBUS_MOD_
    #define IDECL BSLAPI
    #define USEDEFS
    #define IDEF
#else
    #ifdef  _INLINE
        #define IDECL static inline
        #define USEDEFS
        #define IDEF  static inline
    #else
        #define IDECL BSLAPI
    #endif
#endif


/******************************************************************************\
* global macro declarations
\******************************************************************************/

// transfer directions
// (direct 'Start' field values for PCI Master Control Register)
#define PCIBUS_TRANSFER_FROM_DSP            PCI_PCIMC_START_WRITE
#define PCIBUS_TRANSFER_TO_PCI              PCI_PCIMC_START_WRITE

#define PCIBUS_TRANSFER_TO_DSP              PCI_PCIMC_START_READPREF
#define PCIBUS_TRANSFER_FROM_PCI            PCI_PCIMC_START_READPREF

// In the PCI Master Control Register, there is a reserved bit field
// from bit 3 to bit 15. This region of bits is used to store control
// status bits for PCI Module.

#define PCIBUS_AUTOINCREMENT_ON             (0x00000000u)
#define PCIBUS_AUTOINCREMENT_OFF            (0x00000008u)
#define PCIBUS_AUTOINCREMENT_ENABLE         (0x00000000u)
#define PCIBUS_AUTOINCREMENT_DISABLE        (0x00000008u)

// Mask for "Type Of Signaling" values
#define PCIBUS_TOS_MASK                     (0x000000F0u)

// "Type of Signaling" values
#define PCIBUS_TOS_NONE                     (0x00000000u)
#define PCIBUS_TOS_SWI                      (0x00000010u)
#define PCIBUS_TOS_SEM_FOR_TSK              (0x00000020u)
#define PCIBUS_TOS_SEM_FOR_NO_TSK           (0x00000040u)
#define PCIBUS_TOS_SEM_FOR_PRD              (0x00000040u)
#define PCIBUS_TOS_SEM_FOR_CLK              (0x00000040u)
#define PCIBUS_TOS_SEM_FOR_SWI              (0x00000040u)
#define PCIBUS_TOS_SEM_FOR_HWI              (0x00000040u)
#define PCIBUS_TOS_ANY_SEM                  (0x00000060u)

// Mask for "Type Of Error" values
#define PCIBUS_TOE_MASK                     (0x0000F000u)
#define PCIBUS_TOE_MASK_INV                 (0xFFFF0FFFu)

// "Type Of Error" values
#define PCIBUS_TOE_SUCCESSFUL               (0x00000000u)
#define PCIBUS_TOE_NO_ERROR                 (0x00000000u)
#define PCIBUS_TOE_REC_TARGET_ABORT         (0x00002000u)
#define PCIBUS_TOE_REC_MASTER_ABORT         (0x00004000u)
#if !(C64_SUPPORT)
#define PCIBUS_TOE_DMAHALTED                (0x00008000u)
#endif

//==============================================================================
// Macro for easier setting of the parameters of the PCIBUS_makeReq_Mem()
// function.
//
// Calls either the PCIBUS_makeReq_Mem_SEM() or the PCIBUS_makeReq_Mem_SWI() or
// the PCIBUS_makeReq_Mem_NONE() function according to the given parameters.
//
// According to the selected type of the signaling, the macro can call
// different real functions to check type casting. (SEM_Handle or SWI_Handle)
//
// parameters:
//      pReqObj         pointer to a PCIBUS_AccessReqObj type access request
//                      object
//                      The object has to be unmodified until the request
//                      has been executed.
//      pIntBuf         32-bit unsigned integer type pointer to the internal
//                      buffer that is in the DSP's (or DSP board's) memory
//                      This buffer can be either the source or the
//                      destination of the data transfer according to the
//      pciAddr         32-bit PCI Memory address
//                      This address is the starting address of either the
//                      source or the destination of the data transfer
//                      according to the direction of the transfer.
//      lenIn32bWs      16-bit unsigned integer value for number of the
//                      32-bit words to be transferred (copied)
//      trDir_end       select the direction of the data transfer
//                      Characters from the PCIBUS_TRANSFER_x constants after
//                      the PCIBUS_TRANSFER_:
//                          FROM_DSP, TO_PCI, TO_DSP, FROM_PCI
//      tos_end         select the type of signaling
//                      Determines the signaling type (semaphore or software
//                      interrupt) and the location of the waiting in case of
//                      semaphore signaling.
//                      Characters from the PCIBUS_TOS_ constants after the
//                      PCIBUS_TOS_:
//                          NONE, SWI, SEM_FOR_TSK, SEM_FOR_NO_TSK,
//                          SEM_FOR_PRD, SEM_FOR_CLK, SEM_FOR_SWI, SEM_FOR_HWI
//      pSemSwiObj      SEM_Handle (pointer to a SEM_Obj) or SWI_Handle (pointer
//                      to a SWI_Obj) type variable. It can be NULL.
//==============================================================================
#define PCIBUS_MAKE_REQ_MEM(                                                        \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir_end,tos_end,pSemSwiObj)    \
    _PCIBUS_MAKE_REQ_MEM_##tos_end(                                                 \
        pReqObj,pIntBuf,pciAddr,lenIn32bWs,PCIBUS_TRANSFER_##trDir_end,pSemSwiObj)

// _PCIBUS_MAKE_REQ_MEM_x() macro varities for PCIBUS_MAKE_REQ_MEM():
#define _PCIBUS_MAKE_REQ_MEM_NONE( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSemSwiObj) \
    PCIBUS_makeReq_Mem(pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,PCIBUS_TOS_NONE,pSemSwiObj)

#define _PCIBUS_MAKE_REQ_MEM_SWI( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSwiObj) \
    PCIBUS_makeReq_Mem_SWI(pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSwiObj)

#define _PCIBUS_MAKE_REQ_MEM_SEM( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,tos,pSemObj) \
    PCIBUS_makeReq_Mem_SEM(pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,tos,pSemObj)

#define _PCIBUS_MAKE_REQ_MEM_SEM_FOR_TSK( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_MEM_SEM( \
        pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,PCIBUS_TOS_SEM_FOR_TSK,pSemObj)

#define _PCIBUS_MAKE_REQ_MEM_SEM_FOR_PRD( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_MEM_SEM( \
        pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,PCIBUS_TOS_SEM_FOR_PRD,pSemObj)

#define _PCIBUS_MAKE_REQ_MEM_SEM_FOR_CLK( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_MEM_SEM( \
        pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,PCIBUS_TOS_SEM_FOR_CLK,pSemObj)

#define _PCIBUS_MAKE_REQ_MEM_SEM_FOR_SWI( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_MEM_SEM( \
        pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,PCIBUS_TOS_SEM_FOR_SWI,pSemObj)

#define _PCIBUS_MAKE_REQ_MEM_SEM_FOR_HWI( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_MEM_SEM( \
        pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,PCIBUS_TOS_SEM_FOR_HWI,pSemObj)

//==============================================================================
// Macro for easier setting of the parameters of the PCIBUS_makeReq_IO()
// function.
//
// Calls either the PCIBUS_makeReq_IO_SEM() or the PCIBUS_makeReq_IO_SWI() or
// the PCIBUS_makeReq_IO_NONE() function according to the given parameters.
//
// According to the selected type of the signaling, the macro can call
// different real functions to check type casting. (SEM_Handle or SWI_Handle)
//
// parameters:
//      pReqObj         pointer to a PCIBUS_AccessReqObj type access request
//                      object
//                      The object has to be unmodified until the request
//                      has been executed.
//      pIntBuf         32-bit unsigned integer type pointer to the internal
//                      buffer that is in the DSP's (or DSP board's) memory
//                      This buffer can be either the source or the
//                      destination of the data transfer according to the
//      pciAddr         16-bit PCI I/O address
//                      This address is the starting address of either the
//                      source or the destination of the data transfer
//                      according to the direction of the transfer.
//      lenIn32bWs      16-bit unsigned integer value for number of the
//                      32-bit words to be transferred (copied)
//      trDir_end       select the direction of the data transfer
//                      Characters from the PCIBUS_TRANSFER_x constants after
//                      the PCIBUS_TRANSFER_:
//                          FROM_DSP, TO_PCI, TO_DSP, FROM_PCI
//      tos_end         select the type of signaling
//                      Determines the signaling type (semaphore or software
//                      interrupt) and the location of the waiting in case of
//                      semaphore signaling.
//                      Characters from the PCIBUS_TOS_ constants after the
//                      PCIBUS_TOS_:
//                          NONE, SWI, SEM_FOR_TSK, SEM_FOR_NO_TSK,
//                          SEM_FOR_PRD, SEM_FOR_CLK, SEM_FOR_SWI, SEM_FOR_HWI
//      pSemSwiObj      SEM_Handle (pointer to a SEM_Obj) or SWI_Handle (pointer
//                      to a SWI_Obj) type variable. It can be NULL.
//==============================================================================
#define PCIBUS_MAKE_REQ_IO(                                                     \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir_end,tos_end,pSemSwiObj)    \
    _PCIBUS_MAKE_REQ_IO_##tos_end(                                              \
        pReqObj,pIntBuf,pciAddr,lenIn32bWs,PCIBUS_TRANSFER_##trDir_end,pSemSwiObj)

// _PCIBUS_MAKE_REQ_IO_x() macro varities for PCIBUS_MAKE_REQ_IO():
#define _PCIBUS_MAKE_REQ_IO_NONE( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSemSwiObj) \
    PCIBUS_makeReq_IO(pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,PCIBUS_TOS_NONE,pSemSwiObj)

#define _PCIBUS_MAKE_REQ_IO_SWI( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSwiObj) \
    PCIBUS_makeReq_IO_SWI(pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSwiObj)

#define _PCIBUS_MAKE_REQ_IO_SEM( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,tos,pSemObj) \
    PCIBUS_makeReq_IO_SEM(pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,tos,pSemObj)

#define _PCIBUS_MAKE_REQ_IO_SEM_FOR_TSK( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_IO_SEM( \
        pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,PCIBUS_TOS_SEM_FOR_TSK,pSemObj)

#define _PCIBUS_MAKE_REQ_IO_SEM_FOR_PRD( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_IO_SEM( \
        pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,PCIBUS_TOS_SEM_FOR_PRD,pSemObj)

#define _PCIBUS_MAKE_REQ_IO_SEM_FOR_CLK( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_IO_SEM( \
        pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,PCIBUS_TOS_SEM_FOR_CLK,pSemObj)

#define _PCIBUS_MAKE_REQ_IO_SEM_FOR_SWI( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_IO_SEM( \
        pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,PCIBUS_TOS_SEM_FOR_SWI,pSemObj)

#define _PCIBUS_MAKE_REQ_IO_SEM_FOR_HWI( \
            pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_IO_SEM( \
        pReqObj,pIntBuf,pciAddr,lenIn32bWs,trDir,PCIBUS_TOS_SEM_FOR_HWI,pSemObj)

//==============================================================================
// Macro for easier setting of the parameters of the PCIBUS_makeReq_Cfg()
// function.
//
// Calls either the PCIBUS_makeReq_Cfg_SEM() or the PCIBUS_makeReq_Cfg_SWI() or
// the PCIBUS_makeReq_Cfg_NONE() function according to the given parameters.
//
// According to the selected type of the signaling, the macro can call
// different real functions to check type casting. (SEM_Handle or SWI_Handle)
//
// parameters:
//      pReqObj         pointer to a PCIBUS_AccessReqObj type access request
//                      object
//                      The object has to be unmodified until the request
//                      has been executed.
//      reg             ordinal number of the register to be read
//      func            PCI function number of the PCI deivce to be reached
//      dev             PCI device number of the PCI deivce to be reached
//      bus             PCI bus number of the PCI deivce to be reached
//      pVal            pointer to an Uint32-type variable that is the
//                      source or the destination of the transfer according
//                      to the direction of the transfer
//      trDir_end       select the direction of the data transfer
//                      Characters from the PCIBUS_TRANSFER_x constants after
//                      the PCIBUS_TRANSFER_:
//                          FROM_DSP, TO_PCI, TO_DSP, FROM_PCI
//      tos_end         select the type of signaling
//                      Determines the signaling type (semaphore or software
//                      interrupt) and the location of the waiting in case of
//                      semaphore signaling.
//                      Characters from the PCIBUS_TOS_ constants after the
//                      PCIBUS_TOS_:
//                          NONE, SWI, SEM_FOR_TSK, SEM_FOR_NO_TSK,
//                          SEM_FOR_PRD, SEM_FOR_CLK, SEM_FOR_SWI, SEM_FOR_HWI
//      pSemSwiObj      SEM_Handle (pointer to a SEM_Obj) or SWI_Handle (pointer
//                      to a SWI_Obj) type variable. It can be NULL.
//==============================================================================
#define PCIBUS_MAKE_REQ_CFG(                                                        \
            pReqObj,reg,func,dev,bus,pVal,trDir_end,tos_end,pSemSwiObj) \
    _PCIBUS_MAKE_REQ_CFG_##tos_end(                                                 \
        pReqObj,reg,func,dev,bus,pVal,PCIBUS_TRANSFER_##trDir_end,pSemSwiObj)

// _PCIBUS_MAKE_REQ_CFG_x() macro varities for PCIBUS_MAKE_REQ_CFG():
#define _PCIBUS_MAKE_REQ_CFG_NONE( \
            pReqObj,reg,func,dev,bus,pVal,trDir,pSemSwiObj) \
    PCIBUS_makeReq_Cfg(pReqObj,reg,func,dev,bus,pVal,trDir,PCIBUS_TOS_NONE,pSemSwiObj)

#define _PCIBUS_MAKE_REQ_CFG_SWI( \
            pReqObj,reg,func,dev,bus,pVal,trDir,pSwiObj) \
    PCIBUS_makeReq_Cfg_SWI(pReqObj,reg,func,dev,bus,pVal,trDir,pSwiObj)

#define _PCIBUS_MAKE_REQ_CFG_SEM( \
            pReqObj,reg,func,dev,bus,pVal,trDir,tos,pSemObj) \
    PCIBUS_makeReq_Cfg_SEM(pReqObj,reg,func,dev,bus,pVal,trDir,tos,pSemObj)

#define _PCIBUS_MAKE_REQ_CFG_SEM_FOR_TSK( \
            pReqObj,reg,func,dev,bus,pVal,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_CFG_SEM( \
        pReqObj,reg,func,dev,bus,pVal,trDir,PCIBUS_TOS_SEM_FOR_TSK,pSemObj)

#define _PCIBUS_MAKE_REQ_CFG_SEM_FOR_PRD( \
            pReqObj,reg,func,dev,bus,pVal,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_CFG_SEM( \
        pReqObj,reg,func,dev,bus,pVal,trDir,PCIBUS_TOS_SEM_FOR_PRD,pSemObj)

#define _PCIBUS_MAKE_REQ_CFG_SEM_FOR_CLK( \
            pReqObj,reg,func,dev,bus,pVal,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_CFG_SEM( \
        pReqObj,reg,func,dev,bus,pVal,trDir,PCIBUS_TOS_SEM_FOR_CLK,pSemObj)

#define _PCIBUS_MAKE_REQ_CFG_SEM_FOR_SWI( \
            pReqObj,reg,func,dev,bus,pVal,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_CFG_SEM( \
        pReqObj,reg,func,dev,bus,pVal,trDir,PCIBUS_TOS_SEM_FOR_SWI,pSemObj)

#define _PCIBUS_MAKE_REQ_CFG_SEM_FOR_HWI( \
            pReqObj,reg,func,dev,bus,pVal,trDir,pSemObj) \
    _PCIBUS_MAKE_REQ_CFG_SEM( \
        pReqObj,reg,func,dev,bus,pVal,trDir,PCIBUS_TOS_SEM_FOR_HWI,pSemObj)

//==============================================================================
// Macro to call the PCIBUS_accessReq() with normal priority as there is the
// EXPBUS_ACCESS_REQ() macro in the Expansion Bus Module of the BSL.
//
// In order to fill up a PCIBUS_AccessReqObj type access request object, call
// the PCIBUS_MAKE_REQ() macro.
//
// parameters:
//      pReqObj         pointer for a PCIBUS_AccessReqObj type access request
//                      object that has been filled up previously
//==============================================================================
#define PCIBUS_ACCESS_REQ(pReqObj)      PCIBUS_accessReq(pReqObj)

//==============================================================================
// Macro to create the proper code to wait for a semaphore of an Access Request.
//
// parameters:
//      towt            select the type of the thread where this macro is
//                      Character string: TSK, PRD, CLK, SWI, or HWI
//      hsem            handle of the semaphore what the code is waiting for
//      timeout         timeout count for SEM_pend() function
//                      If the type of the thread -- where this macro is -- is a
//                      task (TSK), you can give a timeuot value for a
//                      SEM_pend() function. (You can use SYS_FOREVER constant.)
//                      If the type of the thread -- where this macro is -- is
//                      not a task (TSK), the code is waiting for the semaphore
//                      forever independently from the current value of timeout.
//==============================================================================
#define PCIBUS_ACCESS_WAIT(towt,hsem,timeout) \
    _PCIBUS_ACCESS_WAIT_IN_##towt(hsem,timeout)

// _PCIBUS_ACCESS_WAIT_IN__x() macro varities for PCIBUS_ACCESS_WAIT():
#define _PCIBUS_ACCESS_WAIT_IN_TSK(hsem,timeout) \
    SEM_pend( hsem, (Uns)(timeout) )

#define _PCIBUS_ACCESS_WAIT_IN_NO_TSK(hsem,timeout) \
    while( !SEM_pend( hsem, 0 ) )

#define _PCIBUS_ACCESS_WAIT_IN_PRD(hsem,timeout) \
    _PCIBUS_ACCESS_WAIT_IN_NO_TSK(hsem,timeout)

#define _PCIBUS_ACCESS_WAIT_IN_CLK(hsem,timeout) \
    _PCIBUS_ACCESS_WAIT_IN_NO_TSK(hsem,timeout)

#define _PCIBUS_ACCESS_WAIT_IN_SWI(hsem,timeout) \
    _PCIBUS_ACCESS_WAIT_IN_NO_TSK(hsem,timeout)

#define _PCIBUS_ACCESS_WAIT_IN_HWI(hsem,timeout) \
    _PCIBUS_ACCESS_WAIT_IN_NO_TSK(hsem,timeout)


/******************************************************************************\
* global typedef declarations
\******************************************************************************/

// Access Request Object
// Descriptor to define a data transfer between an internal DSP memory buffer
// and the Expansion Bus of the DSP processor.
typedef struct
{
    QUE_Elem    link;           // for queue
    Uint32      regDSPMA;       // DSP Master Address reg.
    Uint32      regPCIMA;       // PCI Master Address reg.
    Uint32      regPCIMC;       // PCI Master Control reg. + PCI Module Control
    Void *      pSemOrSwiObj;   // SEM or SWI Handler according to Type Of Signaling
} PCIBUS_AccessReqObj;


/******************************************************************************\
* global variable declarations
\******************************************************************************/


/******************************************************************************\
* global function declarations
\******************************************************************************/

//==============================================================================
// Adds some sources to the enabled sources of the DSPINT to call the User's ISR
// Callback function.
//
// parameters:
//      srcEnMask   Bit mask to select additional sources of DSPINT to enable
//                  them.
//                  You can use the PCIBUS_PCIIS_USER_x macros OR-ed.
//==============================================================================
BSLAPI void PCIBUS_enableUserDSPINTsources( Uint32 srcEnMask );

//==============================================================================
// Adds some sources to the disabled sources of the DSPINT for User.
//
// parameters:
//      srcDisMask  Bit mask to select additional sources of DSPINT to disable
//                  them.
//                  You can use the PCIBUS_PCIIS_USER_x macros OR-ed.
//==============================================================================
BSLAPI void PCIBUS_disableUserDSPINTsources( Uint32 srcDisMask );

//==============================================================================
// Installs the User's DSPINT ISR Callback function
//
// It does not modify the enabled sources of DSPINT. By default, all the sources
// of DSPINT for User is disabled.
//
// parameters:
//      userISR     '*void(void)'-type function to be called if the DSPINT has
//                  been generated because of a source what the user had
//                  enabled.
//
// returns:
//      TRUE        Everything is OK. The given function will be called if
//                  necessary.
//      FALSE       Actually, there is another function installed. It has not
//                  been overwritten.
//==============================================================================
BSLAPI BOOL PCIBUS_installUserDSPINTisr( VFxn userISR );

//==============================================================================
// Uninstalls the User's DSPINT ISR Callback function
//
// Disables all the sources of DSPINT for the User before uninstalling the
// User's ISR Callback function.
//==============================================================================
BSLAPI void PCIBUS_uninstallUserDSPINTisr();

//==============================================================================
//  Fills up an PCIBUS_AccessReqObj type access request object to define a data
//  transfer between an internal DSP memory buffer and the PCI Bus of the DSP
//  processor.
//
//  In order to request the defined access, call the PCIBUS_accessReq() function.
//
//  parameters:
//      pReqObj             pointer to a PCIBUS_AccessReqObj type access request
//                          object
//                          The object has to be unmodified until the request
//                          has been executed.
//      pInternalBuffer     32-bit unsigned integer type pointer to the internal
//                          buffer that is in the DSP's (or DSP board's) memory
//                          This buffer can be either the source or the
//                          destination of the data transfer according to the
//                          direction of the transfer.
//      pciAddress          32-bit PCI Memory address
//                          This address is the starting address of either the
//                          source or the destination of the data transfer
//                          according to the direction of the transfer.
//      lengthIn32bWs       16-bit unsigned integer value for number of the
//                          bytes to be transferred (copied)
//      trDir_autoIncr      selects the direction of the data transfer and
//                          enables/disables the address autoincrement in the
//                          DSP. Use constants from PCIBUS_TRANSFER_x and from
//                          PCIBUS_AUTOINCREMENT_x OR-ed.
//      typeOfSignaling     select the type of signaling
//                          Determines the signaling type (semaphore or software
//                          interrupt) and the location of the waiting in case
//                          of semaphore signaling.
//                          constants from PCIBUS_TOS_
//      pSemOrSwiObj        void type pointer to the casted Semaphore or SWI
//                          object
//                          This pointer will be casted back according to the
//                          typeOfSignaling parameter. The pointer can be NULL.
//==============================================================================
BSLAPI void PCIBUS_makeReq_Mem( PCIBUS_AccessReqObj * pReqObj,
    Uint32 * pInternalBuffer, Uint32 pciAddress,
    Uint16 lengthIn32bWs, Uint32 trDir_autoIncr,
    Uint32 typeOfSignaling, Void * pSemOrSwiObj);

//==============================================================================
//  Fills up an PCIBUS_AccessReqObj type access request object to define a data
//  transfer between an internal DSP memory buffer and the PCI Bus of the DSP
//  processor.
//
//  In order to request the defined access, call the PCIBUS_accessReq() function.
//
//  parameters:
//      pReqObj             pointer to a PCIBUS_AccessReqObj type access request
//                          object
//                          The object has to be unmodified until the request
//                          has been executed.
//      val                 pointer to an Uint32-type variable that is the
//                          source or the destination of the transfer according
//                          to the direction of the transfer
//      pciAddress          16-bit PCI I/O address
//                          This address is the starting address of either the
//                          source or the destination of the data transfer
//                          according to the direction of the transfer.
//      trDir               selects the direction of the data transfer
//                          Use constants from PCIBUS_TRANSFER_x.
//      typeOfSignaling     select the type of signaling
//                          Determines the signaling type (semaphore or software
//                          interrupt) and the location of the waiting in case
//                          of semaphore signaling.
//                          constants from PCIBUS_TOS_
//      pSemOrSwiObj        void type pointer to the casted Semaphore or SWI
//                          object
//                          This pointer will be casted back according to the
//                          typeOfSignaling parameter. The pointer can be NULL.
//==============================================================================
BSLAPI void PCIBUS_makeReq_IO( PCIBUS_AccessReqObj * pReqObj,
    Uint32 *val, Uint32 pciAddress, Uint32 trDir,
    Uint32 typeOfSignaling, Void * pSemOrSwiObj);

//==============================================================================
// Fills up a PCIBUS_AccessReqObj type access request object to define a single
// 32-bit word read/write by a PCI Configuration I/O Cycle.
//
// In order to request the defined access, call the PCIBUS_accessReq() function.
//
// parameters:
//      pReqObj             pointer to a PCIBUS_AccessReqObj type access request
//                          object
//                          The object has to be unmodified until the request
//                          has been executed.
//      reg                 ordinal number of the register to be reached
//      func                PCI function number of the PCI deivce to be reached
//      dev                 PCI device number of the PCI deivce to be reached
//      val                 pointer to an Uint32-type variable that is the
//                          source or the destination of the transfer according
//                          to the direction of the transfer
//      trDir               select the direction of the data transfer
//                          constants from PCIBUS_TRANSFER_x
//      typeOfSignaling     select the type of signaling
//                          Determines the signaling type (semaphore or software
//                          interrupt) and the location of the waiting in case
//                          of semaphore signaling.
//                          constants from PCIBUS_TOS_
//      pSemOrSwiObj        void type pointer to the casted Semaphore or SWI
//                          object
//                          This pointer will be casted back according to the
//                          typeOfSignaling parameter. The pointer can be NULL.
//==============================================================================
BSLAPI void PCIBUS_makeReq_Cfg( PCIBUS_AccessReqObj * pReqObj,
    Uint32 reg, Uint32 func, Uint32 dev, Uint32 *val, Uint32 trDir,
    Uint32 typeOfSignaling, Void * pSemOrSwiObj);

//==============================================================================
//  Places the previously filled up PCIBUS_AccessReqObj type access request
//  object at the end of the given queue of the access requests.
//
//  In order to fill up a PCIBUS_AccessReqObj type access request object, call
//  the PCIBUS_makeReq() function.
//
//  parameters:
//      pReqObj         pointer for a PCIBUS_AccessReqObj type access request
//                      object that has been filled up previously
//==============================================================================
BSLAPI void PCIBUS_accessReq( PCIBUS_AccessReqObj * pReqObj);

//==============================================================================
//  Places the previously filled up PCIBUS_AccessReqObj type access request
//  objects at the end of the given queue of the access requests after each
//  other directly.
//
//  In order to fill up a PCIBUS_AccessReqObj type access request object, call
//  the PCIBUS_makeReq() function.
//
//  parameters:
//      pReqObjArray    pointer for the array of PCIBUS_AccessReqObj type access
//                      request objects that have been filled up previously
//      numberOfReqs    number of elements in the array above
//==============================================================================
BSLAPI void PCIBUS_accessMoreReq(
    PCIBUS_AccessReqObj * pReqObjArray, Uint32 numberOfReqs);

//==============================================================================
//  Executes the previously filled up PCIBUS_AccessReqObj request.
//  Waits for the request to complete before returning
//
//  In order to fill up a PCIBUS_AccessReqObj type access request object, call
//  the PCIBUS_makeReq() function.
//
//  parameters:
//      pReqObj         pointer for a PCIBUS_AccessReqObj type access request
//                      object that has been filled up previously
//==============================================================================
BSLAPI BOOL PCIBUS_accessReqWithWait( PCIBUS_AccessReqObj * pReqObj );


/******************************************************************************\
* inline function declarations
\******************************************************************************/

//==============================================================================
// Service functions of the PCIBUS_makeReq_Mem() function.
//==============================================================================
IDECL void PCIBUS_makeReq_Mem_SEM( PCIBUS_AccessReqObj * pReqObj,
    Uint32 * pInternalBuffer, Uint32 pciAddress,
    Uint16 lengthIn32bWs, Uint32 trDir_autoIncr,
    Uint32 typeOfSignaling, SEM_Obj *pSemObj);

IDECL void PCIBUS_makeReq_Mem_SWI( PCIBUS_AccessReqObj * pReqObj,
    Uint32 * pInternalBuffer, Uint32 pciAddress,
    Uint16 lengthIn32bWs, Uint32 trDir_autoIncr,
    SWI_Obj *pSwiObj);

//==============================================================================
// Service functions of the PCIBUS_makeReq_IO() function.
//==============================================================================
IDECL void PCIBUS_makeReq_IO_SEM( PCIBUS_AccessReqObj * pReqObj,
    Uint32 *val, Uint32 pciAddress, Uint32 trDir,
    Uint32 typeOfSignaling, SEM_Obj *pSemObj);

IDECL void PCIBUS_makeReq_IO_SWI( PCIBUS_AccessReqObj * pReqObj,
    Uint32 *val, Uint32 pciAddress, Uint32 trDir,
    SWI_Obj *pSwiObj);

//==============================================================================
// Service functions of the PCIBUS_makeReq_Cfg() function.
//==============================================================================
IDECL void PCIBUS_makeReq_Cfg_SEM( PCIBUS_AccessReqObj * pReqObj,
    Uint32 reg, Uint32 func, Uint32 dev, Uint32 *val, Uint32 trDir,
    Uint32 typeOfSignaling, SEM_Obj *pSemObj);

IDECL void PCIBUS_makeReq_Cfg_SWI( PCIBUS_AccessReqObj * pReqObj,
    Uint32 reg, Uint32 func, Uint32 dev, Uint32 *val, Uint32 trDir,
    SWI_Obj *pSwiObj);

//==============================================================================
//  Defines an Expansion Bus read according to the given parameters, requests
//  the access, and then waits the end of the defined data transfer.
//
//  parameters:
//      pInternalBuffer     32-bit unsigned integer type pointer to the internal
//                          buffer that is in the DSP's (or DSP board's) memory
//                          This buffer is the destination of the data transfer.
//      pciAddress          32-bit Expansion Bus address
//                          This address is the starting address of the source
//                          of the data transfer.
//      lengthIn32bWs       16-bit unsigned integer value for number of the
//                          32-bit words to be transferred (copied)
//
//  return:
//      error codes:
//      PCIBUS_TOE_NO_ERROR         = no error happened
//      PCIBUS_TOE_REC_TARGET_ABORT = PCI Receiver Target Abort Error
//      PCIBUS_TOE_REC_MASTER_ABORT = PCI Receiver Master Abort Error
//==============================================================================
IDECL Uint32 PCIBUS_read(Uint32 * pInternalBuffer, Uint32 pciAddress,
    Uint16 lengthIn32bWs);

//==============================================================================
//  Defines an Expansion Bus write according to the given parameters, requests
//  the access, and then waits the end of the defined data transfer.
//
//  parameters:
//      pInternalBuffer     32-bit unsigned integer type pointer to the internal
//                          buffer that is in the DSP's (or DSP board's) memory
//                          This buffer is the source of the data transfer.
//      pciAddress      32-bit Expansion Bus address
//                          This address is the starting address of the destination
//                          of the data transfer.
//      lengthIn32bWs       16-bit unsigned integer value for number of the
//                          32-bit words to be transferred (copied)
//
//  return:
//      error codes:
//      PCIBUS_TOE_NO_ERROR         = no error happened
//      PCIBUS_TOE_REC_TARGET_ABORT = PCI Receiver Target Abort Error
//      PCIBUS_TOE_REC_MASTER_ABORT = PCI Receiver Master Abort Error
//==============================================================================
IDECL Uint32 PCIBUS_write(Uint32 * pInternalBuffer, Uint32 pciAddress,
    Uint16 lengthIn32bWs);

//==============================================================================
//  Defines an Expansion Bus data transfer to read a single 32-bit word
//  according to the given parameter, requests the access, and then waits the
//   end of the defined data transfer.
//
//  parameters:
//      pciAddress      32-bit Expansion Bus address
//                          This address will be read.
//      value           pointer to the 32-bit word to be written by the read
//                          value
//
//  return:
//      error codes:
//      PCIBUS_TOE_NO_ERROR         = no error happened
//      PCIBUS_TOE_REC_TARGET_ABORT = PCI Receiver Target Abort Error
//      PCIBUS_TOE_REC_MASTER_ABORT = PCI Receiver Master Abort Error
//==============================================================================
IDECL Uint32 PCIBUS_readOneWord(Uint32 pciAddress, Uint32 *value);

//==============================================================================
//  Defines an Expansion Bus data transfer to write a single 32-bit word
//  according to the given parameters, requests the access, and then waits the
//  end of the defined data transfer.
//
//  Callable from TSK only.
//
//  parameters:
//      pciAddress      32-bit Expansion Bus address
//                          This address will be written.
//      value           new 32-bit value of the memory at the given address
//
//  return:
//      error codes:
//      PCIBUS_TOE_NO_ERROR         = no error happened
//      PCIBUS_TOE_REC_TARGET_ABORT = PCI Receiver Target Abort Error
//      PCIBUS_TOE_REC_MASTER_ABORT = PCI Receiver Master Abort Error
//==============================================================================
IDECL Uint32 PCIBUS_writeOneWord(Uint32 pciAddress, Uint32 value);

//==============================================================================
//  Returns the error code of an Expansion Bus access.
//
//  parameters:
//      pReqObj             pointer to the PCIBUS_AccessReqObj type access
//                          request object that has already been executed
//
//  return:
//      error codes:
//      PCIBUS_TOE_NO_ERROR         = no error happened
//      PCIBUS_TOE_REC_TARGET_ABORT = PCI Receiver Target Abort Error
//      PCIBUS_TOE_REC_MASTER_ABORT = PCI Receiver Master Abort Error
//==============================================================================
IDECL Uint32 PCIBUS_getError(PCIBUS_AccessReqObj * pReqObj);

//==============================================================================
//  Returns whether or not the data transfer has completed
//
//  parameters:
//      pReqObj             pointer to the PCIBUS_AccessReqObj type access
//                          request object that is in process
//
//  return:
//      TRUE                The data transfer has completed.
//      FALSE               The data transfer is in process.
//==============================================================================
IDECL BOOL PCIBUS_isCompleted(PCIBUS_AccessReqObj * pReqObj);


/******************************************************************************\
* inline function definitions
\******************************************************************************/
#ifdef USEDEFS

//==============================================================================
// Function to check type casting.
// If typeOfSignaling is PCIBUS_TOS_NONE, you should call this function with
// parameters typeOfSignaling = PCIBUS_TOS_NONE and pSemObj = NULL.
//==============================================================================
IDEF void PCIBUS_makeReq_Mem_SEM( PCIBUS_AccessReqObj * pReqObj,
    Uint32 * pInternalBuffer, Uint32 pciAddress,
    Uint16 lengthIn32bWs, Uint32 trDir_autoIncr,
    Uint32 typeOfSignaling, SEM_Obj *pSemObj)
{
    // the semaphore must be initialized by a call of the SEM_new() function
    SEM_reset(pSemObj,0);   // reset semaphore
    PCIBUS_makeReq_Mem( pReqObj, pInternalBuffer, pciAddress, lengthIn32bWs,
        trDir_autoIncr, typeOfSignaling, (Void *)(pSemObj) );
}

//==============================================================================
// Function to check type casting.
// If typeOfSignaling is PCIBUS_TOS_NONE, you should call PCIBUS_makeReq_Mem_SEM
// function with parameters typeOfSignaling = PCIBUS_TOS_NONE and pSemObj = NULL.
//==============================================================================
IDEF void PCIBUS_makeReq_Mem_SWI( PCIBUS_AccessReqObj * pReqObj,
    Uint32 * pInternalBuffer, Uint32 pciAddress,
    Uint16 lengthIn32bWs, Uint32 trDir_autoIncr, SWI_Obj *pSwiObj)
{
    PCIBUS_makeReq_Mem( pReqObj, pInternalBuffer, pciAddress, lengthIn32bWs,
        trDir_autoIncr, PCIBUS_TOS_SWI, (Void *)(pSwiObj) );
}

//==============================================================================
// Function to check type casting.
// If typeOfSignaling is PCIBUS_TOS_NONE, you should call this function with
// parameters typeOfSignaling = PCIBUS_TOS_NONE and pSemObj = NULL.
//==============================================================================
IDEF void PCIBUS_makeReq_IO_SEM( PCIBUS_AccessReqObj * pReqObj,
    Uint32 *val, Uint32 pciAddress, Uint32 trDir,
    Uint32 typeOfSignaling, SEM_Obj *pSemObj)
{
    // the semaphore must be initialized by a call of the SEM_new() function
    SEM_reset(pSemObj,0);   // reset semaphore
    PCIBUS_makeReq_IO( pReqObj, val, pciAddress, trDir,
        typeOfSignaling, (Void *)(pSemObj) );
}

//==============================================================================
// Function to check type casting.
// If typeOfSignaling is PCIBUS_TOS_NONE, you should call PCIBUS_makeReq_IO_SEM
// function with parameters typeOfSignaling = PCIBUS_TOS_NONE and pSemObj = NULL.
//==============================================================================
IDEF void PCIBUS_makeReq_IO_SWI( PCIBUS_AccessReqObj * pReqObj,
    Uint32 *val, Uint32 pciAddress, Uint32 trDir, SWI_Obj *pSwiObj)
{
    PCIBUS_makeReq_IO( pReqObj, val, pciAddress, trDir,
        PCIBUS_TOS_SWI, (Void *)(pSwiObj) );
}

//==============================================================================
// Function to check type casting.
// If typeOfSignaling is PCIBUS_TOS_NONE, you should call this functCfgn with
// parameters typeOfSignaling = PCIBUS_TOS_NONE and pSemObj = NULL.
//==============================================================================
IDEF void PCIBUS_makeReq_Cfg_SEM( PCIBUS_AccessReqObj * pReqObj,
    Uint32 reg, Uint32 func, Uint32 dev, Uint32 *val, Uint32 trDir,
    Uint32 typeOfSignaling, SEM_Obj *pSemObj)
{
    // the semaphore must be initialized by a call of the SEM_new() function
    SEM_reset(pSemObj,0);   // reset semaphore
    PCIBUS_makeReq_Cfg( pReqObj, reg, func, dev, val, trDir,
        typeOfSignaling, (Void *)(pSemObj) );
}

//==============================================================================
// Function to check type casting.
// If typeOfSignaling is PCIBUS_TOS_NONE, you should call PCIBUS_makeReq_Cfg_SEM
// function with parameters typeOfSignaling = PCIBUS_TOS_NONE and pSemObj = NULL.
//==============================================================================
IDEF void PCIBUS_makeReq_Cfg_SWI( PCIBUS_AccessReqObj * pReqObj,
    Uint32 reg, Uint32 func, Uint32 dev, Uint32 *val, Uint32 trDir,
    SWI_Obj *pSwiObj)
{
    PCIBUS_makeReq_Cfg( pReqObj, reg, func, dev, val, trDir,
        PCIBUS_TOS_SWI, (Void *)(pSwiObj) );
}

/*----------------------------------------------------------------------------*/
/* You can call this function from a TSK only.
 * Do *NOT* call this function from the main(), or SWI (PRD, CLK), or HWI.
 */
IDEF Uint32 PCIBUS_read(Uint32 * pInternalBuffer, Uint32 pciAddress,
    Uint16 lengthIn32bWs)
{
    PCIBUS_AccessReqObj reqObj;
    SEM_Obj sem;

    SEM_new( &sem, 0 );
    PCIBUS_makeReq_Mem_SEM( &reqObj, pInternalBuffer, pciAddress, lengthIn32bWs,
        PCIBUS_TRANSFER_TO_DSP | PCIBUS_AUTOINCREMENT_ON,
        PCIBUS_TOS_SEM_FOR_TSK, &sem );
    PCIBUS_accessReq( &reqObj );
    SEM_pend( &sem, SYS_FOREVER );

    return PCIBUS_getError(&reqObj);
}

/*----------------------------------------------------------------------------*/
/* You can call this function from a TSK only.
 * Do *NOT* call this function from the main(), or SWI (PRD, CLK), or HWI.
 */
IDEF Uint32 PCIBUS_write(Uint32 * pInternalBuffer, Uint32 pciAddress,
    Uint16 lengthIn32bWs)
{
    PCIBUS_AccessReqObj reqObj;
    SEM_Obj sem;

    SEM_new( &sem, 0 );
    PCIBUS_makeReq_Mem_SEM(&reqObj, pInternalBuffer, pciAddress, lengthIn32bWs,
        PCIBUS_TRANSFER_FROM_DSP | PCIBUS_AUTOINCREMENT_ON,
        PCIBUS_TOS_SEM_FOR_TSK, &sem );
    PCIBUS_accessReq( &reqObj );
    SEM_pend( &sem, SYS_FOREVER );

    return PCIBUS_getError(&reqObj);
}

/*----------------------------------------------------------------------------*/
/* You can call this function from a TSK only.
 * Do *NOT* call this function from the main(), or SWI (PRD, CLK), or HWI.
 */
IDEF Uint32 PCIBUS_readOneWord(Uint32 pciAddress, Uint32 *buf)
{
    return PCIBUS_read( buf, pciAddress, 1 );
}

/*----------------------------------------------------------------------------*/
/* You can call this function from a TSK only.
 * Do *NOT* call this function from the main(), or SWI (PRD, CLK), or HWI.
 */
IDEF Uint32 PCIBUS_writeOneWord(Uint32 pciAddress, Uint32 value)
{
    Uint32 buf = value;

    return PCIBUS_write( &buf, pciAddress, 1 );
}

/*---------------------------------------------------------------------------*/
IDEF Uint32 PCIBUS_getError(PCIBUS_AccessReqObj * pReqObj)
{
    return ((pReqObj->regPCIMC) & PCIBUS_TOE_MASK);
}

/*----------------------------------------------------------------------------*/
#endif /* USEDEFS */


#endif /* BSL_PCIBUS_SUPPORT */
#endif /* _BSL_PCIBUS_H_ */
/******************************************************************************\
* End of bsl_pcibus.h
\******************************************************************************/

