
// *
// * Copyright (c) 2013, Texas Instruments Incorporated
// * All rights reserved.
// *
// * Redistribution and use in source and binary forms, with or without
// * modification, are permitted provided that the following conditions
// * are met:
// *
// * *  Redistributions of source code must retain the above copyright
// *    notice, this list of conditions and the following disclaimer.
// *
// * *  Redistributions in binary form must reproduce the above copyright
// *    notice, this list of conditions and the following disclaimer in the
// *    documentation and/or other materials provided with the distribution.
// *
// * *  Neither the name of Texas Instruments Incorporated nor the names of
// *    its contributors may be used to endorse or promote products derived
// *    from this software without specific prior written permission.
// *
// * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
// * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
// * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
// * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
// * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
// * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// * *


// *
// * This script automates the loading of an OpenMP project.
// * It performs the following functions:
// * - connects to all 8 cores of the C6678 EVM
// * - resets each core
// * - loads symbols from the same .out file into all other CPUs (the slave CPUs) and restarts the slave CPUs 
// * - loads the .out file into CPU 0 (the master CPU)
// * - displays instructions on the console on how to run the project and collect events in System Analyzer
// *
// * The launch for the device should be configured as follows:
// * ON ALL CORES:
// * - disable software breakpoints
// * - reset on connect
// *
// * On CPU 0 only:
// * - on program load, run to main
// * On the slave CPUs:
// * - disable running on program load
// *
// * More information on the APIs used in this script and
// * on DSS in general is provided in the ccsv5\ccs_base\scripting 
// * subfolder within the folder that CCS is installed in
// *
// * NOTE: you will need to customize the following variables in order to
// * make this script work for your project and your target connection:
// * - strOutFileName
// * - strEmulator
// * - strIsaCPUIDprefix
// * - numCPUs 
// *
 
//Import the DSS packages into our namespace to save on typing
importPackage(Packages.com.ti.debug.engine.scripting)
importPackage(Packages.com.ti.ccstudio.scripting.environment)
importPackage(Packages.java.lang)

//To customize this script to support a different target, project or emulator,
//please change the following lines within the script to appropriate values
//var strOutFileName = "C:/Users/YourUserName/workspace_v5_3/OpenMP with UIA Example C6678_8cores/Release/OpenMP with UIA Example C6678_8cores.out";
var strOutFileName = "C:/CCSv5_Dev/workspace_run-E2E/OpenMP with UIA Example C6678_8cores/Debug/OpenMP with UIA Example C6678_8cores.out";

var strEmulator = "Blackhawk XDS560v2-USB Mezzanine Emulator_0";  // See the DebugView in CCS after launching your target for the exact string you should use
var strIsaCPUIDprefix = "C66xx_";  // See the DebugView in CCS after launching your target for the exact string you should use
var numCPUs = 8;


//some global variables
var debugServer = null;
var debugSessionArray = new Array();
var slaveSessionArray = new Array();
var isDebugSession = false;

var script = ScriptingEnvironment.instance();

//Run this script.
testEnv = {};
run();

/**
 * Send message to the console and log (if logging is enabled)
 * @param {String} The string to output to the console/log.
 */
function printTrace(string)
{
	script.traceWrite(string);
}

/**
 * Get error code from the given exception.
 * @param {exception} The exception from which to get the error code.
 */
function getErrorCode(exception)
{
	var ex2 = exception.javaException;
	if (ex2 instanceof Packages.com.ti.ccstudio.scripting.environment.ScriptingException) {
		return ex2.getErrorID();
	}
	return 0;
}



/*
 * Main function.
 */
function run()
{
	var inst;
	var ok = true;
	var errCode = 0;
	var retVal = 0;
	var date = 0;
	var defaultTimeout = -1;
	var PI = 3.14157
	var wasteTime = PI;
	var delayCtr = 0;

	isDebugServer = false;
	isDebugSession = false;


	// Create base scripting environment.
	dssScriptEnv = Packages.com.ti.ccstudio.scripting.environment.ScriptingEnvironment.instance();

	// Set overall script timeout value.
	dssScriptEnv.setScriptTimeout(defaultTimeout);

	printTrace("\n***** OpenMP program Loader *****\n");

	date = new Date();
	printTrace("START: " + date.toTimeString() + "\n");

	// Open Debug Server session.
	if (!isDebugServer)
	{
		debugServer = dssScriptEnv.getServer("DebugServer.1");
		isDebugServer = true;
	}

	// get debugSession objects for each CPU in the target
	var cpuCtr;
	for (cpuCtr = 0; cpuCtr < numCPUs; cpuCtr++)
	{

		try{
			debugSessionArray[cpuCtr] = debugServer.openSession(strEmulator+"/"+strIsaCPUIDprefix+Number(cpuCtr).toString(10));
			if (cpuCtr != 0){
				slaveSessionArray[cpuCtr-1] = debugSessionArray[cpuCtr];
			}
			if (debugSessionArray[cpuCtr] != null){
				isDebugSession = true;
				showOnConsole("Found CPU "+Number(cpuCtr).toString(10)+"...");
				printTrace("TARGET: " + debugSessionArray[cpuCtr].getBoardName());
				//connect if necessary
				if (!debugSessionArray[cpuCtr].target.isConnected()) {
					printTrace("Connecting to target...");

					// Connect to target. If target is simulator or already connected, a warning will be reported.
					try
					{
						debugSessionArray[cpuCtr].target.connect();
					}
					catch (ex)
					{
						errCode = getErrorCode(ex);
						dssScriptEnv.traceWrite("Error code #" + errCode + ", could not connect to target!\nAborting!");
						ok = false;
						break;
					}				

				}
				
				// Reset target.
				printTrace("Resetting target...");
				try
				{
					debugSessionArray[cpuCtr].target.reset();
				}
				catch (ex)
				{
					errCode = getErrorCode(ex);
					dssScriptEnv.traceWrite("Error code #" + errCode + ", could reset target!\nAborting!");
					ok = false;
					break;
				}

			}
		}
		catch(err1) {
			debugSessionArray[cpuCtr] = null;
		}
	}

    // Do a system reset on CPU 0 if possible
	// Query the number of supported resets
	var numReset = debugSessionArray[0].target.getNumResetTypes();


	// Access one of the resetType objects
	var reset1 = debugSessionArray[0].target.getResetType(1);

	// determine if the reset is allowed, and issue a reset
	if ( reset1.isAllowed() ) { 
		showOnConsole("Issuing system reset...");
		reset1.issueReset(); 
	} else {
		showOnConsole("Not able to do a system reset.");
	}

	// Load the program into core 0 and the symbols for the program on all other cores.  Don't run any CPU yet!

	var filePath = new java.io.File(strOutFileName);
	showOnConsole("Loading "+strOutFileName);
	var outFileName = filePath.getName();
	if (ok) {

		for (cpuCtr = numCPUs-1; cpuCtr >= 0; cpuCtr--)
		{
			// Load program and pass arguments to main (if applicable).
			try
			{
				if (cpuCtr != 0){
					showOnConsole("Loading symbols on CPU "+Number(cpuCtr).toString(10));
					debugSessionArray[cpuCtr].symbol.load(strOutFileName);
					debugSessionArray[cpuCtr].target.restart();
				} else {
					showOnConsole("OpenMP_Load.js:\nAbout to load program on Core0. The launch should be configured to auto-run CPU 0 to main.\n"+
								"- Once CPU 0 is running, please group the slave CPUs and run the group.\n");
					// CPU0 is loaded last - it is configured to auto-run to main.
					showOnConsole("Loading program on CPU "+Number(cpuCtr).toString(10));
					debugSessionArray[cpuCtr].memory.loadProgram(strOutFileName);
				} 

			}
			catch (ex)
			{
				errCode = getErrorCode(ex);
				showOnConsole("Error code #" + errCode + ", " + strOutFileName + " load failed!\nAborting!");
				ok = false;
				break;
			}
		}
	}

	if (ok) {
		showOnConsole("OpenMP_Load.js:\nOnce CPU0 halts at main, halt the group of slave CPUs, open Tools/System Analyzer->Live, and run core 0 and then the group.\n"+
				        "- Terminate session before reloading.");
	}

}



/**
 * Print command-line argument error along with correct usage and exit.
 */
function exitWithArgError(message)
{
	print("ERROR: " + message);
	print();

	printHelp();

	delete testEnv;
	java.lang.System.exit(1);
}

/**
 * Print help and exit.
 */
function exitWithHelp()
{
	printHelp();

	delete testEnv;
	java.lang.System.exit(0);
}

/**
 * Print help to console.
 */
function printHelp()
{
	print(
			"Usage: loadJSFile \"<full path to OpenMP_Load.js/OpenMP_Load.js\"\n" +
			"Loads an OpenMP project into all cores on a C6678 EVM.\n" +
	"\n");
}

/**
 * showOnConsole (for debugging only)
 * @param outputStr string to be displayed in the CCS console view.
 */
function showOnConsole(outputStr){
	// enable the following line for debugging purposes:
	if(isDebugSession){
		debugSessionArray[0].expression.evaluate("GEL_TextOut\(\""+outputStr+"\\n\"\)");
	}
	script.traceWrite(outputStr);
}