/*
 *  Copyright 2007 b7 Spectrum Digital Incorporated.
 *  All rights reserved. Property of Spectrum Digital Incorporated.
 */

/*
 *  SPI ROM interface
 *
 */

#include "usbstk5505.h"
#include "spirom.h"
#include "stdio.h"

static Uint16 spirombuf[spirom_PAGESIZE + 8];
static Uint16 statusbuf[8];


/* ------------------------------------------------------------------------ *
 *  spirom_init( )                                                            *
 * ------------------------------------------------------------------------ */
void spirom_init( )
{
	/* Enable SPI */
	SYS_EXBUSSEL = (SYS_EXBUSSEL & 0x0fff) | 0x1000;
	SYS_PRCNTR = 0x04;
	SYS_PRCNTRLR = 0x0080;

    /* SPI Clock Control */
    SPI_SPICC2 = 0x0000; // SPI clock disabled
    SPI_SPICC1 = 0x00b0; // 12MHz input clock divided by 120


    SPI_SPIDC1 = 0x0000; // SPI0 low idle clock, active low CS, shift out on rising clock, no clock delay
    SPI_SPIDC2 = 0x0000; 
    
    SPI_SPICC2 = 0x8000; // SPI clock enabled
}



/* ------------------------------------------------------------------------ *
 *  spirom_cycle(buf, len)                                                    *
 *                                                                          *
 *  Execute a SPI spirom data transfer cycle.  Each byte in buf is shifted    *
 *  out and replaced with data coming back from the spirom.                   *
 * ------------------------------------------------------------------------ */

void spirom_cycle(Uint16 *buf, Uint16 len)
{
    Uint16 i;

    SPI_SPICR1 = 0x0000 | len - 1;
    i = SPI_SPIDR2;  // Dummy read

   	/* Spirom access cycle */
	for (i = 0; i < len; i++)
	{
		SPI_SPIDR2 = buf[i] << 8;
		SPI_SPIDR1 = 0x0000;
        SPI_SPICR2 = 0x0039;  //  8-bit words, read
        while((SPI_SPISR1 & 0x0002) == 0) {};
        buf[i] = SPI_SPIDR2 & 0xff;
	}
	
	/* Wait for transfer to complete */
//	while((SPI_SPISR1 & 0x0008) == 0) {};
	
	USBSTK5505_waitusec(10000);
}

/* ------------------------------------------------------------------------ *
 *  spirom_status( )                                                          *
 * ------------------------------------------------------------------------ */
Uint16 spirom_status( )
{
    /* Issue read status command */
    statusbuf[0] = spirom_CMD_RDSR;
    statusbuf[1] = 0;

	spirom_cycle(statusbuf, 5);

    return statusbuf[3];
}


/* ------------------------------------------------------------------------ *
 *  spirom_read( src, dst, len )                                              *
 * ------------------------------------------------------------------------ */
void spirom_read( Uint16 src, Uint32 dst, Uint32 length )
{
    Int32 i;
    Uint16 *psrc, *pdst;

	// Setup command
    spirombuf[0] = spirom_CMD_READ;
    spirombuf[1] = ( src >> 8 );
    spirombuf[2] = ( src >> 0 );

	// Execute spirom read cycle
	spirom_cycle(spirombuf, length + 5);

    // Copy returned data
    pdst = ( Uint16 * )dst;
    psrc = spirombuf + 5;
    for ( i = 0 ; i < length ; i++ )
        *pdst++ = *psrc++;

}

/* ------------------------------------------------------------------------ *
 *  spirom_write( src, dst, len )                                             *
 * ------------------------------------------------------------------------ */
void spirom_write( Uint32 src, Uint16 dst, Uint32 length )
{
    Int32 i;
    Int32 bytes_left;
    Int32 bytes_to_program;
    Uint16 *psrc;

    /* Establish source */
    psrc = ( Uint16 * )src;
    bytes_left = length;

    while ( bytes_left > 0 )
    {
        bytes_to_program = bytes_left;

        /* Most to program is spirom_CMD_BLOCKSIZE */
        if ( bytes_to_program > spirom_PAGESIZE )
             bytes_to_program = spirom_PAGESIZE;

        /* Make sure you don't run off the end of a block */
        if ( ( dst & spirom_PAGEMASK ) != ( ( dst + bytes_to_program ) & spirom_PAGEMASK ) )
            bytes_to_program -= ( dst + bytes_to_program ) - ( ( dst + bytes_to_program ) & spirom_PAGEMASK );

        /* Issue WPEN */
        spirombuf[0] = spirom_CMD_WREN;
        spirom_cycle(spirombuf, 1);

        /* Create command block for program operation */
        spirombuf[0] = spirom_CMD_WRITE;
        spirombuf[1] = ( Uint16 )( dst >> 8 );
        spirombuf[2] = ( Uint16 )( dst );

        for ( i = 0 ; i < bytes_to_program ; i++ )
            spirombuf[3+i] = *psrc++;

        /* Execute write command */
        spirom_cycle(spirombuf, bytes_to_program + 3 );

        /* Wait while busy */
        while( ( spirom_status( ) & 0x01 ) );

        /* Get ready for next iteration */
        bytes_left -= bytes_to_program;
        dst += bytes_to_program;
    }
}


