/******************************************************************************\
* Copyright (C) 2006 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... Board Support Library for SPM176431
* VERSION.......... (Defined in README.TXT)
*------------------------------------------------------------------------------
* CONTENT.......... C source file for API of peripheral AC97
* FILENAME......... bsl_ac97.c
*------------------------------------------------------------------------------
* PERIPHERAL NAME.. AC97
* PERIPHERAL TYPE.. Single-device peripheral
* ACCESS MODE...... Direct (no handle)
* REGISTER ACCESS.. 16-bit wide
\******************************************************************************/
#define _BSL_AC97_MOD_

#include <bsl_ac97.h>
#include <csl_edma.h>
#include <csl_irq.h>
#include <csl_mcbsp.h>
#include <csl_timer.h>

#if (BSL_AC97_SUPPORT)
/******************************************************************************\
*                         L O C A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* static macro declarations
\******************************************************************************/

/* audio write macro using relative offset*/
#define AC97_WRITE16(flashByteOffset,val16) \
    *(volatile Uint16 *)(((Uint32)(AC97_STARTING_ADDRESS)+(Uint32)(flashByteOffset))) \
    = ((Uint16)(val16))

/* audio read macro using relative offset*/
#define AC97_READ16(flashByteOffset) \
    (*(volatile Uint16 *)((Uint32)(AC97_STARTING_ADDRESS)+(Uint32)(flashByteOffset)))


/******************************************************************************\
* static typedef declarations
\******************************************************************************/

/******************************************************************************\
* static function declarations
\******************************************************************************/

void AC97_set_interrupts_edma(void);
void Ac97_Isr(void);

/******************************************************************************\
* static variable definitions
\******************************************************************************/


/* the buffers defined below are mapped into specific memory locations
   specified in the linker file */
#ifdef __cplusplus

extern "C" {
#endif

    #pragma DATA_SECTION(RxPingBuffer,".ReceivePingBuffer");
    volatile Uint32 RxPingBuffer[SIZE_OF_PP_BUFF];

    #pragma DATA_SECTION(TxPingBuffer,".TransmitPingBuffer");
    volatile Uint32 TxPingBuffer[SIZE_OF_PP_BUFF];

    #pragma DATA_SECTION(RxPongBuffer,".ReceivePongBuffer");
    volatile Uint32 RxPongBuffer[SIZE_OF_PP_BUFF];

    #pragma DATA_SECTION(TxPongBuffer,".TransmitPongBuffer");
    volatile Uint32 TxPongBuffer[SIZE_OF_PP_BUFF];

    #pragma DATA_SECTION(CircularBufferTransmit, ".TransmitCircBuff");
    volatile Uint32 CircularBufferTransmit[SIZE_OF_CIRC_BUFF];

    #pragma DATA_SECTION(CircularBufferRecieve, ".RecieveCircBuff");
    volatile Uint32 CircularBufferRecieve[SIZE_OF_CIRC_BUFF];

#ifdef __cplusplus
}
#endif  // extern "C" {

/* these flags are used to test if edma is still running */
volatile far BOOL EDMA_Rcv_stopped_flag = FALSE;
volatile far BOOL EDMA_Tx_stopped_flag = FALSE;

/* these flags are used to test for errors */
volatile far BOOL Recieve_Buff_Overflow = FALSE;
volatile far BOOL Transmit_Buff_Underflow = FALSE;
volatile far BOOL Colliding_Interrupts_flag = FALSE;

/* counts how many buffers are serviced durning edma interrupt */
volatile far int Num_of_Interrupts_serviced;

/* these flags are used to signal edma to shut down */
volatile int stop_EDMA_recieve_flag = FALSE;
volatile int stop_EDMA_transmit_flag = FALSE;

/* these flags are used to shut down edma */
volatile int Linked_Rcv_Ping_to_NULL = FALSE;
volatile int Linked_Tx_Ping_to_NULL = FALSE;
volatile int Linked_Rcv_Pong_to_NULL = FALSE;
volatile int Linked_Tx_Pong_to_NULL = FALSE;

/* these pointer mark the begining and end of data
   in the Transmit/Recieve circular ques */
volatile far Uint32 * TransmitOut;
volatile far Uint32 * TransmitIn;
volatile far Uint32 * RecieveOut;
volatile far Uint32 * RecieveIn;

/* these handles are defined and used to set up the ping-pong buffering
   in the EDMA */
MCBSP_Handle hMcbsp1; /* this is the mcbsp tied to the audio chip */
EDMA_Handle hEdmaRcv;
EDMA_Handle hEdmaTx;
EDMA_Handle hEdmaRcvPing;
EDMA_Handle hEdmaTxPing;
EDMA_Handle hEdmaRcvPong;
EDMA_Handle hEdmaTxPong;
EDMA_Handle hEdmadummy;
EDMA_Handle hEdmadummy1;


/******************************************************************************\
* static function definitions
\******************************************************************************/

//==============================================================================
// Ac97_Isr
//  This function should be called during an EDMA interrupt.  This function
//  determines the EDMA buffer that has finished being read/written and
//  via the McBsp port and either moves incoming data from an incoming buffer
//  that the McBsp has finished writing into the users input buffer or moves
//  outgoing data from the users output buffer to an outgoing buffer that the
//  McBsp has finished reading.
//
//  Output flags:
//  EDMA_Rcv_stopped_flag, EDMA_Tx_stopped_flag: signifies the according buffers
//                                                  have finished.
//  Recieve_Buff_Overflow, Transmit_Buff_Underflow, Colliding_Interrupts_flag:
//
//  Num_of_Interrupts_serviced: how buffers were during current function call
//
//  Input flags:
//  stop_EDMA_recieve_flag, stop_EDMA_transmit_flag:
//
//  Utilized Global Ptrs
//  TransmitOut, RecieveIn
//
//==============================================================================
void Ac97_Isr(void)     /* for the EDMA */
{

  volatile Uint32 * ParseBuffer; /* pointer for ping/pong buffer to parse */
  int RoomInBuffer, DataInBuffer;/* remaining room in circular buffers */
  /* pointer to know when to wrap around circ buffers*/
  volatile Uint32 * EndofBuffer;
  int i,j;/* counter in for loop */
  /* number of interrupts serviced in while loop */
  int Num_of_Interrupts_serviced = 0;

  /* set this interrupt up so it can be interrupted by any interrupt currently
     enabled except for itself */
  volatile Uint32 CopyIRP, CopyIER, CopyCSR, CopyIE, CopyIER2;

  CopyIRP = CHIP_CRGET(IRP);/* back up interrupt return pointer */
  CopyIER = CHIP_CRGET(IER);/* back up current interrupts */
  CopyCSR = CHIP_CRGET(CSR);/* back up current status register */

  IRQ_disable(IRQ_EVT_EDMAINT);/* turn off edma interrupts */

  IRQ_globalEnable(); /* turn on interrupts */

while ( ( EDMA_intTest(TRANSMIT_PING_EDMA_INT_NUM) ||
          EDMA_intTest(RECIEVE_PING_EDMA_INT_NUM) ||
          EDMA_intTest(TRANSMIT_PONG_EDMA_INT_NUM) ||
          EDMA_intTest(RECIEVE_PONG_EDMA_INT_NUM) )
          && Num_of_Interrupts_serviced < 2 )
{
        /* ---- Transmit Ping Buffer is exhausted ---- */
    if(EDMA_intTest(TRANSMIT_PING_EDMA_INT_NUM))
    {
        /* increment the number of interrupts serviced */
        Num_of_Interrupts_serviced++;

        /* if stop flag has been set from outside interrupt */
        if(stop_EDMA_transmit_flag && !(Linked_Tx_Pong_to_NULL) && !(Linked_Tx_Ping_to_NULL))
        {/* change link in transmit ping buffer to a null buffer */

            EDMA_link(hEdmaTxPing, hEdmadummy1);
            Linked_Tx_Ping_to_NULL = TRUE;
        }
        else
        {
            if(Linked_Tx_Ping_to_NULL)
            {/* link was changed, and this is the last buffer to process */
                EDMA_Tx_stopped_flag = TRUE;

                /* disable EDMA transmitter channel */
                EDMA_disableChannel(hEdmaTx);

                /* put transmit in reset on our serial port */
                MCBSP_FSETSH(hMcbsp1,SPCR,XRST,YES);

                /* if mcbsp is not being used supress fsync signal */
                if(EDMA_Rcv_stopped_flag == TRUE)
                    AC97_WRITE16(8, 0);
            }

        }
        /* clear CIPR bit so future interrupts can be recognized */
        EDMA_intClear(TRANSMIT_PING_EDMA_INT_NUM);

        /* set appropriate buffer to process and appropriate end of buffer */
        ParseBuffer = TxPingBuffer;
        EndofBuffer = &CircularBufferTransmit[SIZE_OF_CIRC_BUFF-1];

        /* calculate how much room is in the buffer */
        if( TransmitOut == TransmitIn)
            DataInBuffer = 0;
        if( TransmitOut < TransmitIn)
            DataInBuffer = (TransmitIn - TransmitOut);
        if( TransmitIn < TransmitOut)
            DataInBuffer = SIZE_OF_CIRC_BUFF - (TransmitOut - TransmitIn) -1;

        if(DataInBuffer < SIZE_OF_PP_BUFF)
        {/* check if there is enough data in circular buffer */
            Transmit_Buff_Underflow = TRUE;

            for(i = 0; i < DataInBuffer; i++)
            {
                /* write to the transmit buffer */
                *ParseBuffer = *TransmitOut;

                /* increment the pointers */
                ParseBuffer++;
                TransmitOut++;

                /* wrap around if necessary */
                if(TransmitOut > EndofBuffer)
                    TransmitOut -= SIZE_OF_CIRC_BUFF;

            }/* for loop that parsebuffer */

            for( j = 0; j < SIZE_OF_PP_BUFF - i; j++){
                *ParseBuffer = 0x80000000;
                ParseBuffer++;
            }


            /* fill the rest of the buffer with zeros */
            //memset(ParseBuffer, 0x0, (SIZE_OF_PP_BUFF - i)*4);

            break;
        }
        else
        {/* put the data into the ping-pong buffer */

            Transmit_Buff_Underflow = FALSE;
            for(i = 0; i < SIZE_OF_PP_BUFF; i++)
            {
                /* write to the transmit buffer */
                *ParseBuffer = *TransmitOut;

                /* increment the pointers */
                ParseBuffer++;
                TransmitOut++;

                /* wrap around if necessary */
                if(TransmitOut > EndofBuffer)
                    TransmitOut -= SIZE_OF_CIRC_BUFF;

            }/* for loop that parsebuffer */


        }/* if there is room in the buffer */

    }/* interrupt TRANSMIT_PING_EDMA_INT_NUM */

    /* ---- Recieve Ping Buffer is full ----- */
    if(EDMA_intTest(RECIEVE_PING_EDMA_INT_NUM))
    {
        /* increment the number of interrupts serviced */
        Num_of_Interrupts_serviced++;

        /* if stop flag has been set from outside interrupt */
        if(stop_EDMA_recieve_flag && !(Linked_Rcv_Pong_to_NULL) && !(Linked_Rcv_Ping_to_NULL))
        {/* change link in recieve ping buffer to a null buffer */

            EDMA_link(hEdmaRcvPing, hEdmadummy);
            Linked_Rcv_Ping_to_NULL = TRUE;
        }
        else
        {
            if(Linked_Rcv_Ping_to_NULL)
            {/* link was changed, and this is the last buffer to process */
                EDMA_Rcv_stopped_flag = TRUE;

                /* disable EDMA reciever channel */
                EDMA_disableChannel(hEdmaRcv);

                /* put reciever in reset on our serial port */
                MCBSP_FSETSH(hMcbsp1,SPCR,RRST,YES);

                /* if mcbsp is not being used supress fsync signal */
                if(EDMA_Tx_stopped_flag == TRUE)
                    AC97_WRITE16(8, 0);
            }
        }

        /* clear CIPR bit so future interrupts can be recognized */
        EDMA_intClear(RECIEVE_PING_EDMA_INT_NUM);

        /* set appropriate buffer to process and appropriate end of buffer */
        ParseBuffer = RxPingBuffer;
        EndofBuffer = &CircularBufferRecieve[SIZE_OF_CIRC_BUFF-1];

        /* calculate how much room is in the buffer */
        if( RecieveOut == RecieveIn)
            RoomInBuffer = SIZE_OF_CIRC_BUFF - 1;
        if( RecieveOut < RecieveIn)
            RoomInBuffer = SIZE_OF_CIRC_BUFF - (RecieveIn - RecieveOut) - 1;
        if( RecieveIn < RecieveOut)
            RoomInBuffer = (RecieveIn - RecieveOut) - 1;

        if(RoomInBuffer < SIZE_OF_PP_BUFF)
        {/* check if there is enough room in circular buffer */
            Recieve_Buff_Overflow = TRUE;
            break;
        }
        else
        {/* put the data into the circular buffer */

            Recieve_Buff_Overflow = FALSE;
            for(i = 0; i < SIZE_OF_PP_BUFF; i++)
            {
                /* read from the recieve buffer */
                *RecieveIn = *ParseBuffer;

                /* increment the pointers */
                RecieveIn++;
                ParseBuffer++;

                /* wrap around if necessary */
                if(RecieveIn > EndofBuffer)
                    RecieveIn -= SIZE_OF_CIRC_BUFF;
            }

        }/* if there is room in the buffer */

    }/* interrupt RECIEVE_PING_EDMA_INT_NUM */

    /* ---- Transmit Pong Buffer is exhausted ---- */
    if (EDMA_intTest(TRANSMIT_PONG_EDMA_INT_NUM))
    {
        /* increment the number of interrupts serviced */
        Num_of_Interrupts_serviced++;

        /* if stop flag has been set from outside interrupt */
        if(stop_EDMA_transmit_flag && !(Linked_Tx_Ping_to_NULL) && !(Linked_Tx_Pong_to_NULL))
        {/* change link in transmit pong buffer to a null buffer */

            EDMA_link(hEdmaTxPong, hEdmadummy1);
            Linked_Tx_Pong_to_NULL = TRUE;
        }
        else
        {
            if(Linked_Tx_Pong_to_NULL)
            {/* link was changed, and this is the last buffer to process */
                EDMA_Tx_stopped_flag = TRUE;

                /* disable EDMA transmitter channel */
                EDMA_disableChannel(hEdmaTx);

                /* put transmit in reset on our serial port */
                MCBSP_FSETSH(hMcbsp1,SPCR,XRST,YES);

                /* if mcbsp is not being used supress fsync signal */
                if(EDMA_Rcv_stopped_flag == TRUE)
                    AC97_WRITE16(8, 0);
            }

        }
        /* clear CIPR bit so future interrupts can be recognized */
        EDMA_intClear(TRANSMIT_PONG_EDMA_INT_NUM);

        /* set appropriate buffer to process and appropriate end of buffer */
        ParseBuffer = TxPongBuffer;
        EndofBuffer = &CircularBufferTransmit[SIZE_OF_CIRC_BUFF-1];

        /* calculate how much room is in the buffer */
        if( TransmitOut == TransmitIn)
            DataInBuffer = 0;
        if( TransmitOut < TransmitIn)
            DataInBuffer = (TransmitIn - TransmitOut);
        if( TransmitIn < TransmitOut)
            DataInBuffer = SIZE_OF_CIRC_BUFF - (TransmitOut - TransmitIn) -1;

        if(DataInBuffer < SIZE_OF_PP_BUFF)
        {/* check if there is enough data in circular buffer */
            Transmit_Buff_Underflow = TRUE;

            for(i = 0; i < DataInBuffer; i++)
            {
                /* write to the transmit buffer */
                *ParseBuffer = *TransmitOut;

                /* increment the pointers */
                ParseBuffer++;
                TransmitOut++;

                /* wrap around if necessary */
                if(TransmitOut > EndofBuffer)
                    TransmitOut -= SIZE_OF_CIRC_BUFF;

            }/* for loop that parsebuffer */


            for( j = 0; j < SIZE_OF_PP_BUFF - i; j++){
                *ParseBuffer = 0x80000000;
                ParseBuffer++;
            }

            /* fill the rest of the buffer with zeros */
            //memset(ParseBuffer, 0x0, (SIZE_OF_PP_BUFF - i)*4);

            break;
        }
        else
        {/* put the data into the ping-pong buffer */

            Transmit_Buff_Underflow = FALSE;
            for(i = 0; i < SIZE_OF_PP_BUFF; i++)
            {
                /* write to the transmit buffer */
                *ParseBuffer = *TransmitOut;

                /* increment the pointers */
                ParseBuffer++;
                TransmitOut++;

                /* wrap around if necessary */
                if(TransmitOut > EndofBuffer)
                    TransmitOut -= SIZE_OF_CIRC_BUFF;

            }/* for loop that parsebuffer */


        }/* if there is room in the buffer */

    }/* interrupt TRANSMIT_PONG_EDMA_INT_NUM */

    /* ---- Recieve Pong Buffer is full ---- */
    if (EDMA_intTest(RECIEVE_PONG_EDMA_INT_NUM))
    {
        /* increment the number of interrupts serviced */
        Num_of_Interrupts_serviced++;

        /* if stop flag has been set from outside interrupt */
        if(stop_EDMA_recieve_flag && !(Linked_Rcv_Ping_to_NULL) && !(Linked_Rcv_Pong_to_NULL))
        {/* change link in recieve pong buffer to a null buffer */

            EDMA_link(hEdmaRcvPong, hEdmadummy);
            Linked_Rcv_Pong_to_NULL = TRUE;
        }
        else
        {
            if(Linked_Rcv_Pong_to_NULL)
            {/* link was changed, and this is the last buffer to process */
                EDMA_Rcv_stopped_flag = TRUE;

                /* disable EDMA reciever channel */
                EDMA_disableChannel(hEdmaRcv);

                /* put recever in reset on our serial port */
                MCBSP_FSETSH(hMcbsp1,SPCR,RRST,YES);

                /* if mcbsp is not being used supress fsync signal */
                if(EDMA_Tx_stopped_flag == TRUE)
                    AC97_WRITE16(8, 0);
            }

        }

        /* clear CIPR bit so future interrupts can be recognized */
        EDMA_intClear(RECIEVE_PONG_EDMA_INT_NUM);

        /* set appropriate buffer to process and appropriate end of buffer */
        ParseBuffer = RxPongBuffer;
        EndofBuffer = &CircularBufferRecieve[SIZE_OF_CIRC_BUFF-1];

        /* calculate how much room is in the buffer */
        if( RecieveOut == RecieveIn)
            RoomInBuffer = SIZE_OF_CIRC_BUFF - 1;
        if( RecieveOut < RecieveIn)
            RoomInBuffer = SIZE_OF_CIRC_BUFF - (RecieveIn - RecieveOut) - 1;
        if( RecieveIn < RecieveOut)
            RoomInBuffer = (RecieveIn - RecieveOut) - 1;

        if(RoomInBuffer < SIZE_OF_PP_BUFF)
        {/* check if there is enough room in circular buffer */
            Recieve_Buff_Overflow = TRUE;
            break;
        }
        else
        {/* put the data into the circular buffer */

            Recieve_Buff_Overflow = FALSE;
            for(i = 0; i < SIZE_OF_PP_BUFF; i++)
            {
                /* read from the recieve buffer */
                *RecieveIn = *ParseBuffer;

                /* increment the pointers */
                RecieveIn++;
                ParseBuffer++;

                /* wrap around if necessary */
                if(RecieveIn > EndofBuffer)
                    RecieveIn -= SIZE_OF_CIRC_BUFF;
            }

        }/* if there is room in the buffer */

    }/* interrupt RECIEVE_PONG_EDMA_INT_NUM */



} /* while no more interrupts */

    /* check reason for jumping out of the loop was too many interrupts */
    if( Num_of_Interrupts_serviced > 2)
        /* set flag if too many interrupts were serviced */
        Colliding_Interrupts_flag = TRUE;

  /* set this interrupt up so it can be interrupted by any interrupt currently
     enabled except for itself */

  CHIP_CRSET(CSR, CopyCSR);/* turn interrupts back off */
  CHIP_CRSET(IER, CopyIER);/* turns edma interrupt back on */
  CHIP_CRSET(IRP, CopyIRP);/* restore the interrupt return address pointer */

  /* below we will force a new rising edge on the edma interrupt channel
    incase a new interrupt has occured that has not been serviced */


  /* turn off all edma intterupts to force EDMA intterupt low */
  EDMA_intDisable(TRANSMIT_PING_EDMA_INT_NUM);
  EDMA_intDisable(RECIEVE_PING_EDMA_INT_NUM);
  EDMA_intDisable(TRANSMIT_PONG_EDMA_INT_NUM);
  EDMA_intDisable(RECIEVE_PONG_EDMA_INT_NUM);

  IRQ_reset(IRQ_EVT_EDMAINT);
  IRQ_disable(IRQ_EVT_EDMAINT);
  IRQ_clear(IRQ_EVT_EDMAINT);
  IRQ_enable(IRQ_EVT_EDMAINT);

  /* now turn interrupts back on which will force a new rising edge */
  EDMA_intEnable(TRANSMIT_PING_EDMA_INT_NUM);
  EDMA_intEnable(RECIEVE_PING_EDMA_INT_NUM);
  EDMA_intEnable(TRANSMIT_PONG_EDMA_INT_NUM);
  EDMA_intEnable(RECIEVE_PONG_EDMA_INT_NUM);


return;
} /*EDMA interrupt routine */
//=============================================================================
// AC97_set_interrupts_edma
//  This function is called by the AC97_open() function to turn reset and
//  enable the appropriate interrupts for the EDMA and ping pong buffering
//
//=============================================================================
void AC97_set_interrupts_edma(void)
{
 IRQ_nmiEnable();
 IRQ_globalEnable();
 IRQ_reset(IRQ_EVT_EDMAINT);
 IRQ_disable(IRQ_EVT_EDMAINT);
 EDMA_intDisable(TRANSMIT_PING_EDMA_INT_NUM);/*McBSP transmit event PING done*/
 EDMA_intDisable(RECIEVE_PING_EDMA_INT_NUM); /*McBSP receive event PING done*/
 EDMA_intDisable(TRANSMIT_PONG_EDMA_INT_NUM);/*McBSP transmit event PONG done*/
 EDMA_intDisable(RECIEVE_PONG_EDMA_INT_NUM); /*McBSP receive event PONG done*/
 IRQ_clear(IRQ_EVT_EDMAINT);
 EDMA_intClear(TRANSMIT_PING_EDMA_INT_NUM);
 EDMA_intClear(RECIEVE_PING_EDMA_INT_NUM);
 EDMA_intClear(TRANSMIT_PONG_EDMA_INT_NUM);
 EDMA_intClear(RECIEVE_PONG_EDMA_INT_NUM);
 IRQ_enable(IRQ_EVT_EDMAINT);
 EDMA_intEnable(TRANSMIT_PING_EDMA_INT_NUM);
 EDMA_intEnable(RECIEVE_PING_EDMA_INT_NUM);
 EDMA_intEnable(TRANSMIT_PONG_EDMA_INT_NUM);
 EDMA_intEnable(RECIEVE_PONG_EDMA_INT_NUM);

 return;
}/*AC97_set_interrupts_edma(void)*/
//=============================================================================
// AC97_write_register
//  This function writes a register inside the Audio chip.
//  See National Semiconductors, LM4550 Rev 2.1
//  This is accomplished through hardware in the FPGA
//
//=============================================================================
void AC97_write_register(Uint32 AddressToWrite, Uint32 DataToWrite)
{

    Uint16 Read_Command;

    /*poll it until it is ready for another command*/
    Read_Command = AC97_READ16(2);
    while(!(Read_Command & 0x0100))
        Read_Command = AC97_READ16(2);

    /* put data into FPGA to write to Codec */
    AC97_WRITE16(4,(DataToWrite & 0xFFFF) );

    /* give it command to write */
    AC97_WRITE16(2,( 0x0100 | (AddressToWrite & 0x7F) ) );

    /*poll it until it is ready for another command*/
    Read_Command = AC97_READ16(2);
    while(!(Read_Command & 0x0100))
        Read_Command = AC97_READ16(2);

    return;

}/*Uint32 AC97_write_register*/
//=============================================================================
// AC97_read_register
//  This function reads a register inside the Audio chip.
//  See National Semiconductors, LM4550 Rev 2.1
//  This is accomplished through hardware in the FPGA
//
//=============================================================================
Uint32 AC97_read_register(Uint32 AddressToRead)
{

    Uint16 Read_Command;

    /*poll it until it is ready for another command*/
    Read_Command = AC97_READ16(2);
    while(!(Read_Command & 0x0100))
        Read_Command = AC97_READ16(2);

    /* give it the FPGA the command to read Codec */
    AC97_WRITE16(2,( 0x0300 | (AddressToRead & 0x7F) ) );

    /*poll it until it is ready for another command*/
    Read_Command = AC97_READ16(2);
    while(!(Read_Command & 0x0100))
        Read_Command = AC97_READ16(2);

    /* return the value read from the FPGA */
    return AC97_READ16(6);

}/*Uint32 AC97_read_register*/

/******************************************************************************\
*                        G L O B A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* global variable definitions
\******************************************************************************/

TIMER_Handle hTimer0;  /*handle for Timer0*/

EDMA_Config CfgEdmaRcvPing ={
     EDMA_OPT_RMK(
        EDMA_OPT_PRI_HIGH,       /* High priority EDMA */
        EDMA_OPT_ESIZE_32BIT,    /* Element size 32 bits */
        EDMA_OPT_2DS_DEFAULT,
        EDMA_OPT_SUM_DEFAULT,
        EDMA_OPT_2DD_DEFAULT,
        EDMA_OPT_DUM_INC,        /* Destination increment by element size */
        EDMA_OPT_TCINT_YES,      /* Enable Transfer Complete Interrupt */
        EDMA_OPT_TCC_OF(RECIEVE_PING_EDMA_INT_NUM),     /* TCCINT = 0xD, REVT0*/
        EDMA_OPT_TCCM_OF(((RECIEVE_PING_EDMA_INT_NUM & 0x30) >> 4 )),
        EDMA_OPT_ATCINT_DEFAULT,
        EDMA_OPT_ATCC_DEFAULT,
        EDMA_OPT_PDTS_DEFAULT,
        EDMA_OPT_PDTD_DEFAULT,
        EDMA_OPT_LINK_YES,       /* Enable linking */
        EDMA_OPT_FS_NO
        ),

        0,//EDMA_SRC_RMK(MCBSP_ADDRH(hMcbsp1, DRR)), /* src to DRR0 */
        EDMA_CNT_RMK(SIZE_OF_PP_BUFF/2 - 1, 2),     /* no. of elements */
        0,//EDMA_DST_RMK((Uint32)RxPingBuffer), /* dst addr to edmaInbuff */
        EDMA_IDX_RMK(0,0),
        EDMA_RLD_RMK(2,0)

};/* end of CfgEdmaRcvPing */

EDMA_Config CfgEdmaRcvPong ={
     EDMA_OPT_RMK(
        EDMA_OPT_PRI_HIGH,       /* High priority EDMA */
        EDMA_OPT_ESIZE_32BIT,    /* Element size 32 bits */
        EDMA_OPT_2DS_DEFAULT,
        EDMA_OPT_SUM_DEFAULT,
        EDMA_OPT_2DD_DEFAULT,
        EDMA_OPT_DUM_INC,        /* Destination increment by element size */
        EDMA_OPT_TCINT_YES,      /* Enable Transfer Complete Interrupt */
        EDMA_OPT_TCC_OF(RECIEVE_PONG_EDMA_INT_NUM),
        EDMA_OPT_TCCM_OF(((RECIEVE_PONG_EDMA_INT_NUM & 0x30) >> 4 )),
        EDMA_OPT_ATCINT_DEFAULT,
        EDMA_OPT_ATCC_DEFAULT,
        EDMA_OPT_PDTS_DEFAULT,
        EDMA_OPT_PDTD_DEFAULT,
        EDMA_OPT_LINK_YES,       /* Enable linking */
        EDMA_OPT_FS_NO
        ),

        0,//EDMA_SRC_RMK(MCBSP_ADDRH(hMcbsp1, DRR)), /* src to DRR0 */
        EDMA_CNT_RMK(SIZE_OF_PP_BUFF/2 - 1, 2), /* no. of elements */
        0,//EDMA_DST_RMK((Uint32)RxPongBuffer), /* dst addr to edmaInbuff */
        EDMA_IDX_RMK(0,0),
        EDMA_RLD_RMK(2,0)

};/* end of CfgEdmaRcvPong */

EDMA_Config CfgEdmaTxPing ={
        EDMA_OPT_RMK(            /* For 64x devices only */
        EDMA_OPT_PRI_HIGH,       /* High priority EDMA */
        EDMA_OPT_ESIZE_32BIT,    /* Element size 32 bits */
        EDMA_OPT_2DS_DEFAULT,
        EDMA_OPT_SUM_INC,        /* Source increment by element size */
        EDMA_OPT_2DD_DEFAULT,
        EDMA_OPT_DUM_DEFAULT,
        EDMA_OPT_TCINT_YES,      /* Enable Transfer Complete Interrupt */
        EDMA_OPT_TCC_OF((TRANSMIT_PING_EDMA_INT_NUM & 0x0F)),
        EDMA_OPT_TCCM_OF(((TRANSMIT_PING_EDMA_INT_NUM & 0x30) >> 4 )),
        EDMA_OPT_ATCINT_DEFAULT,
        EDMA_OPT_ATCC_DEFAULT,
        EDMA_OPT_PDTS_DEFAULT,
        EDMA_OPT_PDTD_DEFAULT,
        EDMA_OPT_LINK_YES,       /* Enable linking */
        EDMA_OPT_FS_NO

        ),
        0,//EDMA_SRC_RMK((Uint32)TxPingBuffer), /*src to edmaOutbuff */
        EDMA_CNT_RMK(SIZE_OF_PP_BUFF/2 - 1, 2), /* set no. of elements */
        0,//EDMA_DST_RMK(MCBSP_ADDRH(hMcbsp1, DXR)), /* dst addr to DXR0 */
        EDMA_IDX_RMK(0,0),
        EDMA_RLD_RMK(2,0)

};/* end of CfgEdmaTxPing */

EDMA_Config CfgEdmaTxPong ={
        EDMA_OPT_RMK(            /* For 64x devices only             */
        EDMA_OPT_PRI_HIGH,       /* High priority EDMA               */
        EDMA_OPT_ESIZE_32BIT,    /* Element size 32 bits             */
        EDMA_OPT_2DS_DEFAULT,
        EDMA_OPT_SUM_INC,        /* Source increment by element size */
        EDMA_OPT_2DD_DEFAULT,
        EDMA_OPT_DUM_DEFAULT,
        EDMA_OPT_TCINT_YES,      /* Enable Transfer Complete Interrupt      */
        EDMA_OPT_TCC_OF((TRANSMIT_PONG_EDMA_INT_NUM & 0x0F)),
        EDMA_OPT_TCCM_OF(((TRANSMIT_PONG_EDMA_INT_NUM & 0x30) >> 4 )),
        EDMA_OPT_ATCINT_DEFAULT,
        EDMA_OPT_ATCC_DEFAULT,
        EDMA_OPT_PDTS_DEFAULT,
        EDMA_OPT_PDTD_DEFAULT,
        EDMA_OPT_LINK_YES,       /* Enable linking */
        EDMA_OPT_FS_NO

        ),
        0,//EDMA_SRC_RMK((Uint32)TxPongBuffer), /*src to edmaOutbuff */
        EDMA_CNT_RMK(SIZE_OF_PP_BUFF/2 - 1, 2), /* set no. of elements */
        0,//EDMA_DST_RMK(MCBSP_ADDRH(hMcbsp1, DXR)), /* dst addr to DXR0 */
        EDMA_IDX_RMK(0,0),
        EDMA_RLD_RMK(2,0)

};/* end of CfgEdmaTxPong */


MCBSP_Config mcbspCfg1 = {
    MCBSP_SPCR_RMK(
           MCBSP_SPCR_FREE_YES,
           MCBSP_SPCR_SOFT_DEFAULT,
           MCBSP_SPCR_FRST_DEFAULT,
           MCBSP_SPCR_GRST_DEFAULT,
           MCBSP_SPCR_XINTM_FRM,
           MCBSP_SPCR_XSYNCERR_DEFAULT,
           MCBSP_SPCR_XRST_DEFAULT,
           MCBSP_SPCR_DLB_DEFAULT,
           MCBSP_SPCR_RJUST_DEFAULT,
           MCBSP_SPCR_CLKSTP_DEFAULT,
           MCBSP_SPCR_DXENA_DEFAULT,
           MCBSP_SPCR_RINTM_DEFAULT,
           MCBSP_SPCR_RSYNCERR_DEFAULT,
           MCBSP_SPCR_RRST_DEFAULT
           ),

    MCBSP_RCR_RMK(
           MCBSP_RCR_RPHASE_DEFAULT,
           MCBSP_RCR_RFRLEN2_DEFAULT,
           MCBSP_RCR_RWDLEN2_DEFAULT,
           MCBSP_RCR_RCOMPAND_DEFAULT,
           MCBSP_RCR_RFIG_DEFAULT,
           MCBSP_RCR_RDATDLY_1BIT,    /* 1-bit receive data delay  */
           MCBSP_RCR_RFRLEN1_OF(0x1), /* frame length of 2 element */
           MCBSP_RCR_RWDLEN1_32BIT,   /* receive elements = 20 bits */
           MCBSP_RCR_RWDREVRS_DEFAULT
           ),

    MCBSP_XCR_RMK(
           MCBSP_XCR_XPHASE_DEFAULT,
           MCBSP_XCR_XFRLEN2_DEFAULT,
           MCBSP_XCR_XWDLEN2_DEFAULT,
           MCBSP_XCR_XCOMPAND_DEFAULT,
           MCBSP_XCR_XFIG_DEFAULT,
           MCBSP_XCR_XDATDLY_1BIT,    /* 1-bit transmit data delay */
           MCBSP_XCR_XFRLEN1_OF(0x1), /* frame length of 2 element */
           MCBSP_XCR_XWDLEN1_32BIT,   /* receive elements = 20 bits */
           MCBSP_XCR_XWDREVRS_DEFAULT
           ),
      MCBSP_SRGR_RMK(
           MCBSP_SRGR_GSYNC_DEFAULT,
           MCBSP_SRGR_CLKSP_DEFAULT,
           MCBSP_SRGR_CLKSM_DEFAULT,
           MCBSP_SRGR_FSGM_DEFAULT,
           MCBSP_SRGR_FPER_DEFAULT,
           MCBSP_SRGR_FWID_DEFAULT,
           MCBSP_SRGR_CLKGDV_DEFAULT
           ),

     MCBSP_MCR_RMK(                  /* only for 64x */
           MCBSP_MCR_XMCME_DEFAULT,/* All fields in MCR set to default values*/
           MCBSP_MCR_XPBBLK_DEFAULT,
           MCBSP_MCR_XPABLK_DEFAULT,
           MCBSP_MCR_XMCM_DEFAULT,
           MCBSP_MCR_RPBBLK_DEFAULT,
           MCBSP_MCR_RMCME_DEFAULT,
           MCBSP_MCR_RPABLK_DEFAULT,
           MCBSP_MCR_RMCM_DEFAULT
           ),
           MCBSP_RCERE0_RMK(0),      /* Additional registers only for 64x */
           MCBSP_RCERE1_RMK(0),
           MCBSP_RCERE2_RMK(0),
           MCBSP_RCERE3_RMK(0),

           MCBSP_XCERE0_RMK(0),       /* Additional registers only for 64x */
           MCBSP_XCERE1_RMK(0),
           MCBSP_XCERE2_RMK(0),
           MCBSP_XCERE3_RMK(0),

     MCBSP_PCR_RMK(
           MCBSP_PCR_XIOEN_DEFAULT,
           MCBSP_PCR_RIOEN_DEFAULT,
           MCBSP_PCR_FSXM_DEFAULT,  /* Frame sync generated externally */
           MCBSP_PCR_FSRM_DEFAULT,  /* Frame sync generated externally */
           MCBSP_PCR_CLKXM_DEFAULT, /* clks are inputs */
           MCBSP_PCR_CLKRM_DEFAULT,
           MCBSP_PCR_CLKSSTAT_DEFAULT,
           MCBSP_PCR_DXSTAT_DEFAULT,
           MCBSP_PCR_FSXP_DEFAULT, /* FSX is active high */
           MCBSP_PCR_FSRP_DEFAULT, /* FSR is active high */
           MCBSP_PCR_CLKXP_DEFAULT,
           MCBSP_PCR_CLKRP_DEFAULT
           )
};


/******************************************************************************\
* global function definitions
\******************************************************************************/

void AC97_open()
{
    Uint32 delay_count;

    /* clear out ping pong buffers*/
    memset(RxPingBuffer, 0x0, SIZE_OF_PP_BUFF*4);
    memset(TxPingBuffer, 0x0, SIZE_OF_PP_BUFF*4);
    memset(RxPongBuffer, 0x0, SIZE_OF_PP_BUFF*4);
    memset(TxPongBuffer, 0x0, SIZE_OF_PP_BUFF*4);

    /* these are begin enough for half second of audio*/
    memset(CircularBufferTransmit, 0x0, SIZE_OF_CIRC_BUFF*4);
    memset(CircularBufferRecieve, 0x0, SIZE_OF_CIRC_BUFF*4);

    AC97_flush_recieve_buffer();
    AC97_flush_transmit_buffer();

    TransmitIn = TransmitOut = CircularBufferTransmit;
    RecieveIn = RecieveOut = CircularBufferRecieve;

    /* reset the audio chip */
    hTimer0 = TIMER_open(TIMER_DEV0, TIMER_OPEN_RESET);
    TIMER_setDatOut(hTimer0, 0);    /* Write a 0 to TOUT0 */

    /* open the port here */
    hMcbsp1 = MCBSP_open(MCBSP_DEV1, MCBSP_OPEN_RESET);

    /* make sure TOUT0 was low for > 1 usec */
    for(delay_count = 0; delay_count < 5000; delay_count++);

    EDMA_clearPram(0x00000000);     /* clear PaRAM RAM of the EDMA */
    AC97_set_interrupts_edma();


    /* pull codec out of reset */
    TIMER_setDatOut(hTimer0, 1);

    /* McBsp configured to talk to the codec */
    MCBSP_config(hMcbsp1, &mcbspCfg1);


    hEdmaRcv = EDMA_open(EDMA_CHA_REVT1, EDMA_OPEN_RESET);
    hEdmaTx = EDMA_open(EDMA_CHA_XEVT1, EDMA_OPEN_RESET);
    hEdmaRcvPing = EDMA_allocTable(-1);
    hEdmaTxPing = EDMA_allocTable(-1);
    hEdmaRcvPong = EDMA_allocTable(-1);
    hEdmaTxPong = EDMA_allocTable(-1);

    hEdmadummy = EDMA_allocTable(-1); /* Dynamically allocates PaRAM RAM table */
    EDMA_configArgs(hEdmadummy, /* Dummy or Terminating Table in PaRAM    */
         0x00000001,          /* Terminate EDMA transfers by linking to */
         0x00000000,          /* this NULL table                        */
         0x00000000,
         0x00000000,
         0x00000000,
         0x00000000
         );
    hEdmadummy1 = EDMA_allocTable(-1); /* Dynamically allocates PaRAM RAM table */
    EDMA_configArgs(hEdmadummy1, /* Dummy or Terminating Table in PaRAM    */
         0x00000001,          /* Terminate EDMA transfers by linking to */
         0x00000000,          /* this NULL table                        */
         0x00000000,
         0x00000000,
         0x00000000,
         0x00000000
         );

    /* reset all values in the audio chip */
    AC97_write_register(0, 0);





}/*AC97_open()*/

void AC97_start_edma(void)
{

  CfgEdmaRcvPing.src = MCBSP_ADDRH(hMcbsp1, DRR);
  CfgEdmaRcvPing.dst = (Uint32)RxPingBuffer;
  EDMA_config(hEdmaRcv, &CfgEdmaRcvPing);

  CfgEdmaTxPing.src = (Uint32)TxPingBuffer;
  CfgEdmaTxPing.dst = MCBSP_ADDRH(hMcbsp1, DXR);
  EDMA_config(hEdmaTx, &CfgEdmaTxPing);

  CfgEdmaRcvPing.src = MCBSP_ADDRH(hMcbsp1, DRR);
  CfgEdmaRcvPing.dst = (Uint32)RxPingBuffer;
  EDMA_config(hEdmaRcvPing, &CfgEdmaRcvPing);

  CfgEdmaTxPing.src = (Uint32)TxPingBuffer;
  CfgEdmaTxPing.dst = MCBSP_ADDRH(hMcbsp1, DXR);
  EDMA_config(hEdmaTxPing, &CfgEdmaTxPing);

  CfgEdmaRcvPong.src = MCBSP_ADDRH(hMcbsp1, DRR);
  CfgEdmaRcvPong.dst = (Uint32)RxPongBuffer;
  EDMA_config(hEdmaRcvPong, &CfgEdmaRcvPong);

  CfgEdmaTxPong.src = (Uint32)TxPongBuffer;
  CfgEdmaTxPong.dst = MCBSP_ADDRH(hMcbsp1, DXR);
  EDMA_config(hEdmaTxPong, &CfgEdmaTxPong);

  if(Linked_Rcv_Ping_to_NULL)
  {/*above it is setup to start on Ping
    if ping was the last buffer read
    here we will change the setup to start on Pong*/
    EDMA_config(hEdmaRcv, &CfgEdmaRcvPong);
  }

  if(Linked_Tx_Ping_to_NULL)
  {/*above it is setup to start on Ping
    if ping was the last buffer transmitted
    here we will change the setup to start on Pong*/
    EDMA_config(hEdmaTx, &CfgEdmaTxPong);
  }



  EDMA_link(hEdmaRcv, hEdmaRcvPong);     /* Link terminating event */
  EDMA_link(hEdmaTx, hEdmaTxPong);
  EDMA_link(hEdmaRcvPing, hEdmaRcvPong); /* Link terminating event */
  EDMA_link(hEdmaTxPing, hEdmaTxPong);
  EDMA_link(hEdmaRcvPong, hEdmaRcvPing); /* Link terminating event */
  EDMA_link(hEdmaTxPong, hEdmaTxPing);

  if(Linked_Rcv_Ping_to_NULL)
  {/*above it is setup to start on Ping
    if ping was the last buffer read
    here we will change the setup to start on Pong*/
    EDMA_link(hEdmaRcv, hEdmaRcvPing);
  }

  if(Linked_Tx_Ping_to_NULL)
  {/*above it is setup to start on Ping
    if ping was the last buffer transmitted
    here we will change the setup to start on Pong*/
      EDMA_link(hEdmaTx, hEdmaTxPing);
  }

  /*set all flags back to false*/
  Linked_Rcv_Ping_to_NULL = FALSE;
  Linked_Tx_Ping_to_NULL = FALSE;
  Linked_Rcv_Pong_to_NULL = FALSE;
  Linked_Tx_Pong_to_NULL = FALSE;

  /*set all flags back to false*/
  stop_EDMA_recieve_flag = FALSE;
  stop_EDMA_transmit_flag = FALSE;
  EDMA_Rcv_stopped_flag = FALSE;
  EDMA_Tx_stopped_flag = FALSE;
  Colliding_Interrupts_flag = FALSE;

  /* pull recever bit out of reset on our serial port */
  MCBSP_enableRcv(hMcbsp1);

  EDMA_enableChannel(hEdmaRcv);    /* Enable EDMA channels */
  EDMA_enableChannel(hEdmaTx);

  /* pull transmit bit out of reset on our serial port */
  MCBSP_enableXmt(hMcbsp1);

  AC97_WRITE16(8, 1); //unsupress the fsync signal

}/*AC97_start_edma(void)*/

void AC97_start_edma_recieve(void)
{

  CfgEdmaRcvPing.src = MCBSP_ADDRH(hMcbsp1, DRR);
  CfgEdmaRcvPing.dst = (Uint32)RxPingBuffer;
  EDMA_config(hEdmaRcv, &CfgEdmaRcvPing);

  CfgEdmaRcvPing.src = MCBSP_ADDRH(hMcbsp1, DRR);
  CfgEdmaRcvPing.dst = (Uint32)RxPingBuffer;
  EDMA_config(hEdmaRcvPing, &CfgEdmaRcvPing);

  CfgEdmaRcvPong.src = MCBSP_ADDRH(hMcbsp1, DRR);
  CfgEdmaRcvPong.dst = (Uint32)RxPongBuffer;
  EDMA_config(hEdmaRcvPong, &CfgEdmaRcvPong);

  if(Linked_Rcv_Ping_to_NULL)
  {/*above it is setup to start on Ping
    if ping was the last buffer read
    here we will change the setup to start on Pong*/
    EDMA_config(hEdmaRcv, &CfgEdmaRcvPong);
  }

  EDMA_link(hEdmaRcv, hEdmaRcvPong);     /* Link terminating event */
  EDMA_link(hEdmaRcvPing, hEdmaRcvPong); /* Link terminating event */
  EDMA_link(hEdmaRcvPong, hEdmaRcvPing); /* Link terminating event */

  if(Linked_Rcv_Ping_to_NULL)
  {/*above it is setup to start on Ping
    if ping was the last buffer read
    here we will change the setup to start on Pong*/
    EDMA_link(hEdmaRcv, hEdmaRcvPing);
  }

  /*set all flags back to false*/
  Linked_Rcv_Ping_to_NULL = FALSE;
  Linked_Rcv_Pong_to_NULL = FALSE;

  /*set all flags back to false*/
  stop_EDMA_recieve_flag = FALSE;
  EDMA_Rcv_stopped_flag = FALSE;
  Colliding_Interrupts_flag = FALSE;

  /* pull recever bit out of reset on our serial port */
  MCBSP_enableRcv(hMcbsp1);

  /* Enable EDMA channels */
  EDMA_enableChannel(hEdmaRcv);

}/*AC97_start_edma_recieve(void)*/

void AC97_start_edma_transmit(void)
{

  //int delay_count = 0;

  CfgEdmaTxPing.src = (Uint32)TxPingBuffer;
  CfgEdmaTxPing.dst = MCBSP_ADDRH(hMcbsp1, DXR);
  EDMA_config(hEdmaTx, &CfgEdmaTxPing);

  CfgEdmaTxPing.src = (Uint32)TxPingBuffer;
  CfgEdmaTxPing.dst = MCBSP_ADDRH(hMcbsp1, DXR);
  EDMA_config(hEdmaTxPing, &CfgEdmaTxPing);

  CfgEdmaTxPong.src = (Uint32)TxPongBuffer;
  CfgEdmaTxPong.dst = MCBSP_ADDRH(hMcbsp1, DXR);
  EDMA_config(hEdmaTxPong, &CfgEdmaTxPong);

  if(Linked_Tx_Ping_to_NULL)
  {/*above it is setup to start on Ping
    if ping was the last buffer transmitted
    here we will change the setup to start on Pong*/
    EDMA_config(hEdmaTx, &CfgEdmaTxPong);
  }



  EDMA_link(hEdmaTx, hEdmaTxPong);      /* Link terminating event */
  EDMA_link(hEdmaTxPing, hEdmaTxPong);  /* Link terminating event */
  EDMA_link(hEdmaTxPong, hEdmaTxPing);  /* Link terminating event */

  if(Linked_Tx_Ping_to_NULL)
  {/*above it is setup to start on Ping
    if ping was the last buffer transmitted
    here we will change the setup to start on Pong*/
      EDMA_link(hEdmaTx, hEdmaTxPing);
  }

  /*set all flags back to false*/
  Linked_Tx_Ping_to_NULL = FALSE;
  Linked_Tx_Pong_to_NULL = FALSE;

  /*set all flags back to false*/
  stop_EDMA_transmit_flag = FALSE;
  EDMA_Tx_stopped_flag = FALSE;
  Colliding_Interrupts_flag = FALSE;

  /* Enable EDMA channels */
  EDMA_enableChannel(hEdmaTx);

  /* pull transmit bit out of reset on our serial port */
  MCBSP_enableXmt(hMcbsp1);

  AC97_WRITE16(8, 1); //unsupress the fsync signal


}/*AC97_start_edma_transmit(void)*/

void AC97_stop_edma(void)
{

    /* set a flag to request a stop */
    stop_EDMA_recieve_flag = 1;
    stop_EDMA_transmit_flag = 1;

}/*AC97_stop_edma(void)*/

void AC97_stop_edma_recieve(void)
{

    /* set a flag to request a stop */
    stop_EDMA_recieve_flag = 1;

}/*AC97_stop_edma_recieve(void)*/

void AC97_stop_edma_transmit(void)
{

    /* set a flag to request a stop */
    stop_EDMA_transmit_flag = 1;

}/*AC97_stop_edma_transmit(void)*/

void AC97_close_edma(void)
{
    MCBSP_close(hMcbsp1);

    EDMA_close(hEdmaRcv);
    EDMA_close(hEdmaTx);
    EDMA_close(hEdmaRcvPing);
    EDMA_close(hEdmaTxPing);
    EDMA_close(hEdmaRcvPong);
    EDMA_close(hEdmaTxPong);
    EDMA_close(hEdmadummy);

    TIMER_close(hTimer0);

}/*AC97_close_edma(void)*/

Uint32 AC97_set_output_volumes(Uint32 MasterMute, Uint32 MasterLeft, Uint32 MasterRight,
                Uint32 HeadphoneMute, Uint32 HeadphoneLeft, Uint32 HeadphoneRight,
                        Uint32 MonoMute, Uint32 MonoVolume)
{
    Uint32 CurrentValue, ModifiedValue;

    /* validate parameters */
    if(MasterMute & 0xFFFFFFFE  && MasterMute != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(MasterLeft & 0xFFFFFFE0  && MasterLeft != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(MasterRight & 0xFFFFFFE0  && MasterRight != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(HeadphoneMute & 0xFFFFFFFE  && HeadphoneMute != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(HeadphoneLeft & 0xFFFFFFE0  && HeadphoneLeft != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(HeadphoneRight & 0xFFFFFFE0  && HeadphoneRight != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(MonoMute & 0xFFFFFFFE  && MonoMute != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(MonoVolume & 0xFFFFFFE0  && MonoVolume != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;



        /* read current value back */
        CurrentValue = AC97_read_register(LM4550_MASTER_VOLUME_REG);
        if(CurrentValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        /* change current value */
        if(MasterMute != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFF7FFF;
            CurrentValue |= (MasterMute << 15);
        }

        if(MasterLeft != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFE0FF;
            CurrentValue |= (MasterLeft << 8);
        }

        if(MasterRight != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFFFE0;
            CurrentValue |= (MasterRight << 0);
        }

        /* write new value out */
        AC97_write_register(LM4550_MASTER_VOLUME_REG, CurrentValue);

        /* make sure value was written */
        ModifiedValue = AC97_read_register(LM4550_MASTER_VOLUME_REG);
        if(ModifiedValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        if(ModifiedValue != CurrentValue)
            return AC97_ERROR_IN_WRITING_PARAMETER_REGISTER;





        /* read current value back */
        CurrentValue = AC97_read_register(LM4550_HEADPHONE_VOLUME_REG);
        if(CurrentValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        /* change current value */
        if(HeadphoneMute != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFF7FFF;
            CurrentValue |= (HeadphoneMute << 15);
        }

        if(HeadphoneLeft != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFE0FF;
            CurrentValue |= (HeadphoneLeft << 8);
        }

        if(HeadphoneRight != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFFFE0;
            CurrentValue |= (HeadphoneRight << 0);
        }

        /* write new value out */
        AC97_write_register(LM4550_HEADPHONE_VOLUME_REG, CurrentValue);

        /* make sure value was written */
        ModifiedValue = AC97_read_register(LM4550_HEADPHONE_VOLUME_REG);
        if(ModifiedValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        if(ModifiedValue != CurrentValue)
            return AC97_ERROR_IN_WRITING_PARAMETER_REGISTER;



        /* read current value back */
        CurrentValue = AC97_read_register(LM4550_MONO_VOLUME_REG);
        if(CurrentValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        /* change current value */
        if(MonoMute != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFF7FFF;
            CurrentValue |= (MonoMute << 15);
        }

        if(MonoVolume != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFFFE0;
            CurrentValue |= (MonoVolume << 0);
        }

        /* write new value out */
        AC97_write_register(LM4550_MONO_VOLUME_REG, CurrentValue);

        /* make sure value was written */
        ModifiedValue = AC97_read_register(LM4550_MONO_VOLUME_REG);
        if(ModifiedValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        if(ModifiedValue != CurrentValue)
            return AC97_ERROR_IN_WRITING_PARAMETER_REGISTER;


    return 0;


}/* AC97_set_output_volumes */

Uint32 AC97_set_input_volumes(Uint32 MicMute, Uint32 Mic20dBBoost, Uint32 MicVolume,
                Uint32 CDMute, Uint32 CDLeft, Uint32 CDRight,
                Uint32 DACOutMute, Uint32 DACOutLeft, Uint32 DACOutRight)
{
    Uint32 CurrentValue, ModifiedValue;

    /* validate parameters */
    if(CDMute & 0xFFFFFFFE  && CDMute != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(CDLeft & 0xFFFFFFE0  && CDLeft != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(CDRight & 0xFFFFFFE0  && CDRight != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(DACOutMute & 0xFFFFFFFE  && DACOutMute != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(DACOutLeft & 0xFFFFFFE0  && DACOutLeft != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(DACOutRight & 0xFFFFFFE0  && DACOutRight != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(MicMute & 0xFFFFFFFE  && MicMute != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(Mic20dBBoost & 0xFFFFFFFE  && Mic20dBBoost != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(MicVolume & 0xFFFFFFE0  && MicVolume != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;



        /* read current value back */
        CurrentValue = AC97_read_register(LM4550_CD_VOLUME_REG);
        if(CurrentValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        /* change current value */
        if(CDMute != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFF7FFF;
            CurrentValue |= (CDMute << 15);
        }

        if(CDLeft != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFE0FF;
            CurrentValue |= (CDLeft << 8);
        }

        if(CDRight != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFFFE0;
            CurrentValue |= (CDRight << 0);
        }

        /* write new value out */
        AC97_write_register(LM4550_CD_VOLUME_REG, CurrentValue);

        /* make sure value was written */
        ModifiedValue = AC97_read_register(LM4550_CD_VOLUME_REG);
        if(ModifiedValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        if(ModifiedValue != CurrentValue)
            return AC97_ERROR_IN_WRITING_PARAMETER_REGISTER;





        /* read current value back */
        CurrentValue = AC97_read_register(LM4550_PCM_OUT_VOLUME_REG);
        if(CurrentValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        /* change current value */
        if(DACOutMute != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFF7FFF;
            CurrentValue |= (DACOutMute << 15);
        }

        if(DACOutLeft != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFE0FF;
            CurrentValue |= (DACOutLeft << 8);
        }

        if(DACOutRight != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFFFE0;
            CurrentValue |= (DACOutRight << 0);
        }

        /* write new value out */
        AC97_write_register(LM4550_PCM_OUT_VOLUME_REG, CurrentValue);

        /* make sure value was written */
        ModifiedValue = AC97_read_register(LM4550_PCM_OUT_VOLUME_REG);
        if(ModifiedValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        if(ModifiedValue != CurrentValue)
            return AC97_ERROR_IN_WRITING_PARAMETER_REGISTER;



        /* read current value back */
        CurrentValue = AC97_read_register(LM4550_MIC_VOLUME_REG);
        if(CurrentValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        /* change current value */
        if(MicMute != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFF7FFF;
            CurrentValue |= (MicMute << 15);
        }

        if(Mic20dBBoost != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFFFBF;
            CurrentValue |= (Mic20dBBoost << 6);
        }

        if(MicVolume != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFFFE0;
            CurrentValue |= (MicVolume << 0);
        }

        /* write new value out */
        AC97_write_register(LM4550_MIC_VOLUME_REG, CurrentValue);

        /* make sure value was written */
        ModifiedValue = AC97_read_register(LM4550_MIC_VOLUME_REG);
        if(ModifiedValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        if(ModifiedValue != CurrentValue)
            return AC97_ERROR_IN_WRITING_PARAMETER_REGISTER;


    return 0;

}/* AC97_set_input_volumes */

Uint32 AC97_record_setup(Uint32 RecordSelectLeft, Uint32 RecordSelectRight,
                Uint32 RecordGainMute, Uint32 RecordGainLeft, Uint32 RecordGainRight)
{
    Uint32 CurrentValue, ModifiedValue;

    /* validate parameters */
    if(RecordGainMute & 0xFFFFFFFE  && RecordGainMute != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(RecordSelectLeft & 0xFFFFFFF8  && RecordSelectLeft != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(RecordSelectRight & 0xFFFFFFF8  && RecordSelectRight != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(RecordGainLeft & 0xFFFFFFF0  && RecordGainLeft != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(RecordGainRight & 0xFFFFFFF0  && RecordGainRight != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;




        /* read current value back */
        CurrentValue = AC97_read_register(LM4550_RECORD_GAIN_REG);
        if(CurrentValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        /* change current value */
        if(RecordGainMute != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFF7FFF;
            CurrentValue |= (RecordGainMute << 15);
        }

        if(RecordGainLeft != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFF0FF;
            CurrentValue |= (RecordGainLeft << 8);
        }

        if(RecordGainRight != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFFFF0;
            CurrentValue |= (RecordGainRight << 0);
        }

        /* write new value out */
        AC97_write_register(LM4550_RECORD_GAIN_REG, CurrentValue);

        /* make sure value was written */
        ModifiedValue = AC97_read_register(LM4550_RECORD_GAIN_REG);
        if(ModifiedValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        if(ModifiedValue != CurrentValue)
            return AC97_ERROR_IN_WRITING_PARAMETER_REGISTER;





        /* read current value back */
        CurrentValue = AC97_read_register(LM4550_RECORD_SELECT_REG);
        if(CurrentValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        /* change current values */
        if(RecordSelectLeft != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFF8FF;
            CurrentValue |= (RecordSelectLeft << 8);
        }

        if(RecordSelectRight != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFFFF8;
            CurrentValue |= (RecordSelectRight << 0);
        }

        /* write new value out */
        AC97_write_register(LM4550_RECORD_SELECT_REG, CurrentValue);

        /* make sure value was written */
        ModifiedValue = AC97_read_register(LM4550_RECORD_SELECT_REG);
        if(ModifiedValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        if(ModifiedValue != CurrentValue)
            return AC97_ERROR_IN_WRITING_PARAMETER_REGISTER;

    return 0;


}/* AC97_record_setup */

Uint32 AC97_set_miscellaneous(Uint32 Pop, Uint32 Mix, Uint32 LoopBack, Uint32 OutputDevice)
{
    Uint32 CurrentValue, ModifiedValue;

    /* validate parameters */
    if(Pop & 0xFFFFFFFE  && Pop != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(Mix & 0xFFFFFFFE  && Mix != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(LoopBack & 0xFFFFFFFE  && LoopBack != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(OutputDevice & 0xFFFFFFFE  && OutputDevice != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

        /* read current value back */
        CurrentValue = AC97_read_register(LM4550_GENERAL_PURPOSE_REG);
        if(CurrentValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        /* change current value */
        if(Pop != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFF7FFF;
            CurrentValue |= (Pop << 15);
        }

        if(Mix != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFFDFF;
            CurrentValue |= (Mix << 9);
        }

        if(LoopBack != DO_NOT_CHANGE_VALUE)
        {
            CurrentValue &= 0xFFFFFF7F;
            CurrentValue |= (LoopBack << 7);
        }

        /* go ahead and select between line out and head phone */
        if(OutputDevice != DO_NOT_CHANGE_VALUE)
        {
            AC97_WRITE16(0 ,OutputDevice);
        }

        /* write new value out */
        AC97_write_register(LM4550_GENERAL_PURPOSE_REG, CurrentValue);

        /* make sure value was written */
        ModifiedValue = AC97_read_register(LM4550_GENERAL_PURPOSE_REG);
        if(ModifiedValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        if(ModifiedValue != CurrentValue)
            return AC97_ERROR_IN_WRITING_PARAMETER_REGISTER;

    return 0;

}/* AC97_set_miscellaneous */

Uint32 AC97_set_variable_rate_control(Uint32 VariableRateOn, Uint32 Rate)
{
    Uint32 CurrentValue, ModifiedValue;

    /* validate parameters */
    if(VariableRateOn & 0xFFFFFFFE  && VariableRateOn != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;

    if(((Rate < 4000) || (Rate > 48000)) && Rate != DO_NOT_CHANGE_VALUE)
        return AC97_ERROR_INVALID_PARAMETER;


    if(VariableRateOn != DO_NOT_CHANGE_VALUE)
    {
        /* read current value back */
        CurrentValue = AC97_read_register(LM4550_EXT_AUDIO_CONTROL_REG);
        if(CurrentValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        /* change current value */
        CurrentValue &= 0xFFFFFFFE;
        CurrentValue |= VariableRateOn;

        /* write new value out */
        AC97_write_register(LM4550_EXT_AUDIO_CONTROL_REG, CurrentValue);

        /* make sure value was written */
        ModifiedValue = AC97_read_register(LM4550_EXT_AUDIO_CONTROL_REG);
        if(ModifiedValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        if(ModifiedValue != CurrentValue)
            return AC97_ERROR_IN_WRITING_PARAMETER_REGISTER;
    }/* if Variable rate is going to be changed */

    if(Rate != DO_NOT_CHANGE_VALUE && VariableRateOn & 1)
    {
        /* change current value */
        CurrentValue = Rate;

        /* write new value out */
        AC97_write_register(LM4550_PCM_DAC_RATE_REG, CurrentValue);

        /* make sure value was written */
        ModifiedValue = AC97_read_register(LM4550_PCM_DAC_RATE_REG);
        if(ModifiedValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        if(ModifiedValue != CurrentValue)
            return AC97_ERROR_IN_WRITING_PARAMETER_REGISTER;

        CurrentValue = Rate;
        /* write new value out */
        AC97_write_register(LM4550_PCM_ADC_RATE_REG, CurrentValue);

        /* make sure value was written */
        ModifiedValue = AC97_read_register(LM4550_PCM_ADC_RATE_REG);
        if(ModifiedValue == 0xFFFFFFFF)
            return AC97_ERROR_IN_READING_PARAMETER_REGISTER;

        if(ModifiedValue != CurrentValue)
            return AC97_ERROR_IN_WRITING_PARAMETER_REGISTER;
    }/* if ADC rate is going to be changed */

    return 0;

}/* AC97_set_variable_rate_control */

Uint32 AC97_data_in_recieve_buffer(void)
{
    Uint32 DataInBuffer;

    /* calculate how much room is in the buffer */
    if( RecieveOut == RecieveIn)
        DataInBuffer = 0;
    if( RecieveOut < RecieveIn)
        DataInBuffer = (RecieveIn - RecieveOut);
    if( RecieveIn < RecieveOut)
        DataInBuffer = SIZE_OF_CIRC_BUFF - (RecieveOut - RecieveIn);


    return DataInBuffer;

}/* AC97_data_in_recieve_buffer */

Uint32 AC97_room_in_transmit_buffer(void)
{
    Uint32 RoomInBuffer;


    /* calculate how much room is in the buffer */
    if( TransmitOut == TransmitIn)
        RoomInBuffer = SIZE_OF_CIRC_BUFF - 1;
    if( TransmitOut < TransmitIn)
        RoomInBuffer = SIZE_OF_CIRC_BUFF - (TransmitIn - TransmitOut) - 1;
    if( TransmitIn < TransmitOut)
        RoomInBuffer = (TransmitOut - TransmitIn) - 1;


    return RoomInBuffer;

}/* AC97_room_in_transmit_buffer */

void AC97_flush_recieve_buffer(void)
{

    RecieveOut = RecieveIn;
    Recieve_Buff_Overflow = FALSE;

}/* AC97_flush_recieve_buffer(void) */

void AC97_flush_transmit_buffer(void)
{
    TransmitOut = TransmitIn;
    Transmit_Buff_Underflow = FALSE;

}/* AC97_flush_transmit_buffer(void) */

void AC97_flush_transmit_ping_pong_buffers(void)
{
   /* write zeros over the ping pong buffers */
   volatile Uint32 * BufferPtr;
   int i;

   BufferPtr = TxPingBuffer;
   for( i = 0; i < SIZE_OF_PP_BUFF; i++){
        *BufferPtr = 0x80000000;
        BufferPtr++;
   }
   BufferPtr = TxPongBuffer;
   for( i = 0; i < SIZE_OF_PP_BUFF; i++){
        *BufferPtr = 0x80000000;
        BufferPtr++;
   }

}/* AC97_flush_tranmit_ping_pong_buffers */

void AC97_flush_recieve_ping_pong_buffers(void)
{
   /* write zeros over the ping pong buffers */
   volatile Uint32 * BufferPtr;
   int i;

   BufferPtr = RxPingBuffer;
   for( i = 0; i < SIZE_OF_PP_BUFF; i++){
        *BufferPtr = 0x80000000;
        BufferPtr++;
   }
   BufferPtr = RxPongBuffer;
   for( i = 0; i < SIZE_OF_PP_BUFF; i++){
        *BufferPtr = 0x80000000;
        BufferPtr++;
   }

}/* AC97_flush_recieve_ping_pong_buffers */
#endif /* BSL_AC97_SUPPORT */
/******************************************************************************\
* End of bsl_ac97.c
\******************************************************************************/

