/******************************************************************************\
* Copyright (C) 2006 by RTD Embedded Technologies, Inc.   All rights reserved.
* Confidential and Proprietary, Not for Public Release
*------------------------------------------------------------------------------
* PROJECT.......... Board Support Library for SPM176431
* VERSION.......... (Defined in README.TXT)
*------------------------------------------------------------------------------
* CONTENT.......... C source file for API of peripheral Digital IO
* FILENAME......... bsl_adio.c
*------------------------------------------------------------------------------
* PERIPHERAL NAME.. ADIO
* PERIPHERAL TYPE.. Single-device peripheral
* ACCESS MODE...... Direct (no handle)
* REGISTER ACCESS.. 16-bit wide
\******************************************************************************/
#define _BSL_ADIO_MOD_

#include <bsl_adio.h>
#include <bsl_uint.h>
#include <csl_irq.h>

#if(BSL_ADIO_SUPPORT)
/******************************************************************************\
*                         L O C A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* static macro declarations
\******************************************************************************/

/* states of a switch/button */
#define LOW                                     0
#define HIGH                                    1

/* digital I/O write macro using relative offset*/
#define ADIO_WRITE16(fpgaByteOffset,val16) \
    *(volatile Uint16 *)(((Uint32)(FPGA_STARTING_ADDRESS)+(Uint32)(fpgaByteOffset))) \
    = ((Uint16)(val16))

/* digital I/O read macro using relative offset*/
#define ADIO_READ16(fpgaByteOffset) \
    (*(volatile Uint16 *)((Uint32)(FPGA_STARTING_ADDRESS)+(Uint32)(fpgaByteOffset)))

/******************************************************************************\
* static typedef declarations
\******************************************************************************/

/* holds current state of switch/button */
volatile far BOOL State_of_Button = LOW;
volatile far BOOL Interrupt_Ocurred = FALSE;

/* these are set during enable/disable function calls*/
far Uint32 Current_InterruptMode = 0;
far Uint32 Current_Debounce = 0;



/******************************************************************************\
* static function declarations
\******************************************************************************/

void Adio_set_interrupts(void);
void Adio_Isr(void);

/******************************************************************************\
* static function definitions
\******************************************************************************/

void Adio_set_interrupts(void)
{
 IRQ_nmiEnable();
 IRQ_globalEnable();
 IRQ_reset(IRQ_EVT_EXTINT6);
 IRQ_disable(IRQ_EVT_EXTINT6);
 IRQ_clear(IRQ_EVT_EXTINT6);
 IRQ_enable(IRQ_EVT_EXTINT6);

 /* clear the interupt */
 UINT_CLEAR_STATUS_S(PBTIN1IA);

 /* set the enable register for interrupt 6 to map this inerrupt */
 UINT_FSETS(EIT6SRC, PBTIN1IT, ENABLE);

 return;
}/*Adio_set_interrupts_edma(void)*/

void Adio_Isr(void)
{
    Uint16 capture_bit;

    /* get current status of button */
    State_of_Button = ADIO_Read_bit(0);
    Interrupt_Ocurred = TRUE;

    /* determine the last captured value on the line */
    capture_bit = ADIO_Read_Capture_bit(0);

    /* set compare value here */
    ADIO_Write_Compare_bit(0, ~capture_bit);

    //if using a match rather than event then disable interrupts
    if((ADIO_READ16(FPGA_ADIO_INT_MODE_OFFSET) & 0x0003 ) == 2)
        ADIO_disable();

    /* reset the interupt in the FPGA */
    UINT_CLEAR_STATUS_S(PBTIN1IA);

}

/******************************************************************************\
*                        G L O B A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* global variable definitions
\******************************************************************************/


/******************************************************************************\
* global function definitions
\******************************************************************************/

Uint32 ADIO_enable(Uint32 InterruptMode, Uint32 Debounce)
{

    Uint16 CurrentValue;

    /* validate parameters */
    if(InterruptMode & 0xFFFFFFFC  && InterruptMode != DO_NOT_CHANGE_VALUE)
        return ADIO_ERROR_INVALID_PARAMETER;

    if(Debounce & 0xFFFFFFFE  && Debounce != DO_NOT_CHANGE_VALUE)
        return ADIO_ERROR_INVALID_PARAMETER;

    /*backing up copy of current interrupt mode and debounce */
    if(InterruptMode != DO_NOT_CHANGE_VALUE)
        Current_InterruptMode = InterruptMode;
    if(Debounce != DO_NOT_CHANGE_VALUE)
        Current_Debounce = Debounce;


    if(InterruptMode != DO_NOT_CHANGE_VALUE && Debounce != DO_NOT_CHANGE_VALUE)
    {

        CurrentValue = ADIO_READ16(FPGA_ADIO_INT_MODE_OFFSET);
        /* disable the interrupt no matter what*/
        ADIO_WRITE16(FPGA_ADIO_INT_MODE_OFFSET, 0);
        /* set interupt type to event with 20 ms debounce */
        ADIO_WRITE16(FPGA_ADIO_INT_MODE_OFFSET, (Debounce << 2) | InterruptMode);
    }
    else
    {
        if(InterruptMode == DO_NOT_CHANGE_VALUE)
        {
            /* read the current value of register */
            CurrentValue = ADIO_READ16(FPGA_ADIO_INT_MODE_OFFSET);

            /* disable the interrupt no matter what*/
            ADIO_WRITE16(FPGA_ADIO_INT_MODE_OFFSET, 0);

            /* change debounce mode */
            CurrentValue &= 0xFFFB;
            CurrentValue |= (Debounce << 2);

            /* set interupt type to event with 20 ms debounce */
            ADIO_WRITE16(FPGA_ADIO_INT_MODE_OFFSET, CurrentValue);
        }/* if */
        else
        {
            /* read the current value of register */
            CurrentValue = ADIO_READ16(FPGA_ADIO_INT_MODE_OFFSET);

            /* disable the interrupt no matter what*/
            ADIO_WRITE16(FPGA_ADIO_INT_MODE_OFFSET, 0);

            /* change debounce mode */
            CurrentValue &= 0xFFFC;
            CurrentValue |= (InterruptMode);

            /* set interupt type to event with 20 ms debounce */
            ADIO_WRITE16(FPGA_ADIO_INT_MODE_OFFSET, CurrentValue);

        }/* else */
    }/* nested if else end */

    /* set the disable the McBsp and use I/O */
    ADIO_WRITE16(FPGA_ADIO_ENABLE_OFFSET, 1);


    return ADIO_NO_ERRORS;

}/* ADIO_enable */

Uint32 ADIO_disable(void)
{
    Uint16 InterruptRegister;

    /* save current settings before disabling */
    InterruptRegister = ADIO_READ16(FPGA_ADIO_INT_MODE_OFFSET);
    Current_InterruptMode = (InterruptRegister & 0x0003);
    Current_Debounce = (InterruptRegister >> 2) & 0x0001;

    /* set the disable the McBsp and use I/O */
    ADIO_WRITE16(FPGA_ADIO_INT_MODE_OFFSET, 0x0000);
    ADIO_WRITE16(FPGA_ADIO_ENABLE_OFFSET, 0x0000);

    return ADIO_NO_ERRORS;

}/* ADIO_disable */

BOOL ADIO_Read_bit(Uint32 Bit)
{
    Uint16 ReadRegister;

    ReadRegister = ADIO_READ16(FPGA_ADIO_INOUT_OFFSET);

    if(ReadRegister & (1 << (Bit + 8) ) )
        return 1;
    else
        return 0;

}/* ADIO_Read_bit */

void ADIO_Write_bit(Uint32 Bit, BOOL value)
{
    Uint16 CurrentValue;

    CurrentValue = ADIO_READ16(FPGA_ADIO_INOUT_OFFSET);

    /* update value */
    CurrentValue &= ~(1 << Bit );
    CurrentValue |= (value << Bit);

    ADIO_WRITE16(FPGA_ADIO_INOUT_OFFSET, CurrentValue);

}/* ADIO_Write_bit */

BOOL ADIO_Read_Capture_bit(Uint32 Bit)
{
    Uint16 ReadRegister;

    ReadRegister = ADIO_READ16(FPGA_ADIO_CAPT_OFFSET);

    if(ReadRegister & (1 << Bit) )
        return 1;
    else
        return 0;

}/* ADIO_Read_Capture_bit */

void ADIO_Write_Compare_bit(Uint32 Bit, BOOL value)
{

    Uint16 CurrentValue;

    CurrentValue = ADIO_READ16(FPGA_ADIO_COMP_OFFSET);

    /* update value */
    CurrentValue &= ~(1 << Bit );
    CurrentValue |= (value << Bit);

    ADIO_WRITE16(FPGA_ADIO_COMP_OFFSET, CurrentValue);

}/* ADIO_Write_Compare_bit */

void ADIO_Set_Direction_Output_bit(Uint32 Bit)
{

    Uint16 CurrentValue;

    CurrentValue = ADIO_READ16(FPGA_ADIO_DIR_OFFSET);

    /* update value */
    CurrentValue |= (1 << Bit);

    ADIO_WRITE16(FPGA_ADIO_DIR_OFFSET, CurrentValue);


}/* ADIO_Set_Dierction_Output_bit */

void ADIO_Set_Direction_Input_bit(Uint32 Bit)
{

    Uint16 CurrentValue;

    CurrentValue = ADIO_READ16(FPGA_ADIO_DIR_OFFSET);

    /* update value */
    CurrentValue &= ~(1 << Bit );

    ADIO_WRITE16(FPGA_ADIO_DIR_OFFSET, CurrentValue);


}/* ADIO_Set_Direction_Input_bit */

void ADIO_Mask_bit(Uint32 Bit)
{

    Uint16 CurrentValue;

    CurrentValue = ADIO_READ16(FPGA_ADIO_MASK_OFFSET);

    /* update value */
    CurrentValue |= (1 << Bit);

    ADIO_WRITE16(FPGA_ADIO_MASK_OFFSET, CurrentValue);


}/* ADIO_Mask_bit */

void ADIO_Unmask_bit(Uint32 Bit)
{

    Uint16 CurrentValue;

    CurrentValue = ADIO_READ16(FPGA_ADIO_MASK_OFFSET);

    /* update value */
    CurrentValue &= ~(1 << Bit );

    ADIO_WRITE16(FPGA_ADIO_MASK_OFFSET, CurrentValue);


}/* ADIO_Unmask_bit */

#endif /* BSL_ADIO_SUPPORT */
/******************************************************************************\
* End of bsl_adio.c
\******************************************************************************/

