/******************************************************************************\
* File Name        : bsl_aim6020.c
* Target           : AIM6020 Analog Front-end (with SPM6020 dspModule) Board
* Compiler         : Code Composer Studio 2.0, 2.2
* Version          : 1.0
*
* This file provides functions to handle the I2C commication to the digital
* potentiometers of the analog input/output channels of the AIM6020 board.
*
* Copyright (c) 2003, RTD Embedded Technologies, Inc.
\******************************************************************************/
/*
 *  Copyright 2002 by Texas Instruments Incorporated.
 *  All rights reserved. Property of Texas Instruments Incorporated.
 *  Restricted rights to use, duplicate or disclose this code are
 *  granted through contract.
 *  
 */
#define _BSL_AIM6020_MOD_
#include <bsl_aim6020.h>


#if (BSL_AIM6020_SUPPORT)
/******************************************************************************\
*                         L O C A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* static macro declarations
\******************************************************************************/
#define DEVICE_TABLE_ENTRY(devNum) { \
	/* allocated */ FALSE, \
	/* baseAddr  */ (Uint32)(_AIM6020##devNum##_BASE_ADDRESS), \
	/* recorded registers */ { \
		AIM6020_CONFIG_DEFAULT, \
		AIM6020_FLTCLK1_DEFAULT, \
		AIM6020_FLTCLK2_DEFAULT, \
		AIM6020_FLTCFG_DEFAULT, \
		AIM6020_INTCFG_DEFAULT, \
		AIM6020_DISPLAY_DEFAULT \
	}, \
	/* typeOfBoard */ 0x6020, \
	/* pDisplayTable */ _AIM6020_display_normal \
}

/******************************************************************************\
* static typedef declarations
\******************************************************************************/


/******************************************************************************\
* static function declarations
\******************************************************************************/
void AIM6020_I2C_reset(AIM6020_Handle hAIMboard);
void AIM6020_I2C_start(AIM6020_Handle hAIMboard);
void AIM6020_I2C_stop(AIM6020_Handle hAIMboard);
void AIM6020_I2C_sendBit(AIM6020_Handle hAIMboard, Uint32 bit);
Uint32 AIM6020_I2C_getBit(AIM6020_Handle hAIMboard);
Uint32 AIM6020_I2C_isAcknowledged(AIM6020_Handle hAIMboard);
Uint32 AIM6020_I2C_sendByte(AIM6020_Handle hAIMboard, Uint32 dataByte);
Uint32 AIM6020_I2C_getByte(AIM6020_Handle hAIMboard);
Uint32 AIM6020_I2C_sendSlaveAddress(AIM6020_Handle hAIMboard, Uint32 potID);
Uint32 AIM6020_I2C_checkCmdFinished(
	AIM6020_Handle hAIMboard, Uint32 potID, Uint32 timout_ms );


/******************************************************************************\
* static variable definitions
\******************************************************************************/

/* tables for display register's values (17th = dot) */

/* table for AIM6020 */
#pragma DATA_ALIGN(_AIM6020_display_normal,4)
Uint8 _AIM6020_display_normal[17] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
	0x10
};

/* table for AIM6021 */
#pragma DATA_ALIGN(_AIM6020_display_direct,4)
Uint8 _AIM6020_display_direct[17] = {
	0x3F, 0x06, 0x5B, 0x4F, 0x66, 0x6D, 0x7D, 0x07,
	0x7F, 0x6F, 0x77, 0x7C, 0x39, 0x5E, 0x79, 0x71,
	0x80
};


/******************************************************************************\
* static function definitions
\******************************************************************************/

/*----------------------------------------------------------------------------*/
void AIM6020_I2C_reset(AIM6020_Handle hAIMboard)
{
	/* default state of the I2C wires */
	AIM6020_delay_us(5);
	AIM6020_disableI2CDataOut(hAIMboard);
	AIM6020_delay_us(5);
	AIM6020_setI2CClockLow(hAIMboard);
	AIM6020_delay_us(5);
	AIM6020_setI2CDataHigh(hAIMboard);
	AIM6020_delay_us(5);
}

/*----------------------------------------------------------------------------*/
void AIM6020_I2C_start(AIM6020_Handle hAIMboard)
{
	/* I assume that the I2C Clock is low. */
	AIM6020_delay_us(5);
	
	/* drive I2C Data wire */
	AIM6020_enableI2CDataOut(hAIMboard);
	AIM6020_delay_us(5);
	
	/* START condition: Data goes down while the Clock is high */
	AIM6020_setI2CDataHigh(hAIMboard);
	AIM6020_delay_us(5);
	AIM6020_setI2CClockHigh(hAIMboard);
	AIM6020_delay_us(5);
	AIM6020_setI2CDataLow(hAIMboard);
	AIM6020_delay_us(5);
	AIM6020_setI2CClockLow(hAIMboard);
	AIM6020_delay_us(5);
}

/*----------------------------------------------------------------------------*/
void AIM6020_I2C_stop(AIM6020_Handle hAIMboard)
{
	/* I assume that the I2C Clock is low. */
	AIM6020_delay_us(5);

	/* STOP condition: Data goes up while the Clock is high */
	AIM6020_setI2CDataLow(hAIMboard);
	AIM6020_delay_us(5);
	AIM6020_setI2CClockHigh(hAIMboard);
	AIM6020_delay_us(5);
	AIM6020_setI2CDataHigh(hAIMboard);
	AIM6020_delay_us(5);
	AIM6020_setI2CClockLow(hAIMboard);
	AIM6020_delay_us(5);

	/* release I2C Data wire */
	AIM6020_disableI2CDataOut(hAIMboard);
	AIM6020_delay_us(5);
}

/*----------------------------------------------------------------------------*/
/* bit = the bit to be sent (0 or 1)
 */
void AIM6020_I2C_sendBit(AIM6020_Handle hAIMboard, Uint32 bit)
{
	/* I assume that the I2C Clock is low and I2C Data Output is enabled. */
	AIM6020_delay_us(5);
	if( bit ) AIM6020_setI2CDataHigh(hAIMboard);
	else AIM6020_setI2CDataLow(hAIMboard);
	AIM6020_delay_us(1);
	AIM6020_setI2CClockHigh(hAIMboard);
	AIM6020_delay_us(10);
	AIM6020_setI2CClockLow(hAIMboard);
	AIM6020_delay_us(5);
}

/*----------------------------------------------------------------------------*/
/* <-- received bit: 0 or 1 
 */
Uint32 AIM6020_I2C_getBit(AIM6020_Handle hAIMboard)
{
	/* I assume that the I2C Clock is low and I2C Data Output is disabled. */
	Uint32 bit;

	AIM6020_delay_us(5);
	AIM6020_setI2CClockHigh(hAIMboard);
	AIM6020_delay_us(5);
	bit = AIM6020_readI2CData(hAIMboard);
	AIM6020_delay_us(5);
	AIM6020_setI2CClockLow(hAIMboard);
	AIM6020_delay_us(5);

	return (bit ? 1 : 0);
}

/*----------------------------------------------------------------------------*/
/* <-- 1 = accepted, 0 = rejected
 */
Uint32 AIM6020_I2C_isAcknowledged(AIM6020_Handle hAIMboard)
{
	/* I assume that the I2C Clock is low and I2C Data Output is enabled. */
	Uint32 ackBit;

	AIM6020_delay_us(5);
	AIM6020_disableI2CDataOut(hAIMboard);		/* release I2C Data wire */
	ackBit = AIM6020_I2C_getBit(hAIMboard);		/* get bit */
	AIM6020_enableI2CDataOut(hAIMboard);		/* drive I2C Data wire */
	AIM6020_delay_us(5);
	
	if( ackBit == 0 ) return 1;					/* success (accepted) */

	AIM6020_I2C_stop(hAIMboard);				/* failed (rejected) */
	AIM6020_delay_us(5);
	return 0;
}

/*----------------------------------------------------------------------------*/
/* dataByte = 8-bit data word
 * <-- 1 = successful, 0 = failed
 */
Uint32 AIM6020_I2C_sendByte(AIM6020_Handle hAIMboard, Uint32 dataByte)
{
	/* I assume that the I2C Clock is low and I2C Data Output is enabled. */
	Uint32 i;

	/* send 8 bits of the 8-bit data word... */
	for( i = 8; i != 0; i-- )
	{
		AIM6020_I2C_sendBit(hAIMboard, dataByte & 0x80);
		dataByte <<= 1;
	}

	/* ...after that, an acknowledge bit is received */
	return AIM6020_I2C_isAcknowledged(hAIMboard);
}

/*----------------------------------------------------------------------------*/
/* <-- 1 = successful, 0 = failed
 */
Uint32 AIM6020_I2C_getByte(AIM6020_Handle hAIMboard)
{
	/* I assume that the I2C Clock is low and I2C Data Output is enabled. */
	Uint32 i;
	Uint32 dataByte = 0;

	/* disable I2C Data Output */
	AIM6020_disableI2CDataOut(hAIMboard);
	
	/* receive 8 bits of an 8-bit data word... */
	for( i = 8; i != 0; i-- )
	{
		dataByte <<= 1;
		dataByte |= AIM6020_I2C_getBit(hAIMboard);
	}
	
	/* ...enable I2C Data Output... */
	AIM6020_enableI2CDataOut(hAIMboard);

	/* ...send a Master Acknowledge */	
	AIM6020_I2C_sendBit(hAIMboard, 0);

	/* return received 8-bit data word */
	return dataByte;
}

/*----------------------------------------------------------------------------*/
/* potID = AIM6020_DIGPOT_POTID_xxx
 * <-- 1 = successful, 0 = failed
 */
Uint32 AIM6020_I2C_sendSlaveAddress(AIM6020_Handle hAIMboard, Uint32 potID)
{
	if( potID & 0x10 )
	{
		/* Potentiometer of an Analog Output (DAC) */
		return AIM6020_I2C_sendByte( hAIMboard, AIM6020_DIGPOT_DEVID_DAC );
	}
	else
	{
		/* Potentiometer of an Analog Input (ADC) */
		return AIM6020_I2C_sendByte( hAIMboard, AIM6020_DIGPOT_DEVID_ADC );
	}
}

/*----------------------------------------------------------------------------*/
/* potID = AIM6020_DIGPOT_POTID_xxx
 * timeout_ms = time limit for waiting (in msec)
 * <-- 1 = successful, 0 = failed
 */
Uint32 AIM6020_I2C_checkCmdFinished(
	AIM6020_Handle hAIMboard, Uint32 potID, Uint32 timeout_ms )
{
	Uint32	t;
	Uint32	isAck;
	
	isAck = 0;
	for( t = 2*timeout_ms; t != 0; t-- )
	{
		AIM6020_delay_us(157);

		/* start: 30 us */
		AIM6020_I2C_start(hAIMboard);
		
		/* send slave address: 8*31 us + get ACK: 35 us = 283 us*/
		if( AIM6020_I2C_sendSlaveAddress(hAIMboard,potID) )
		{
			isAck = 1;
			break;
		}
		
		/* stop: 30 us */
		AIM6020_I2C_stop(hAIMboard);
	}
	
	return isAck;
}

/******************************************************************************\
*                        G L O B A L   S E C T I O N
\******************************************************************************/


/******************************************************************************\
* global variable definitions
\******************************************************************************/
//AIM6020_Obj _AIM6020_boardTable[4] = {
	//BOARD_TABLE_ENTRY(0),
	//BOARD_TABLE_ENTRY(1),
	//BOARD_TABLE_ENTRY(2),
	//BOARD_TABLE_ENTRY(3)
//};

AIM6020_Obj _AIM6020_deviceTable[_AIM6020_DEVICE_COUNT] = {
	DEVICE_TABLE_ENTRY(0),
	DEVICE_TABLE_ENTRY(1),
	DEVICE_TABLE_ENTRY(2),
	DEVICE_TABLE_ENTRY(3)
};



/******************************************************************************\
* global function definitions
\******************************************************************************/

/*----------------------------------------------------------------------------*/
/* potID = AIM6020_DIGPOT_POTID_xxx
 * position = [0,63]
 * <-- 1 = successful, 0 = failed
 */
Uint32 AIM6020_setDigPotPosition(AIM6020_Handle hAIMboard, Uint32 potID,
	Uint32 position)
{
	/* I assume that the I2C Clock is low. */
	
	/* send START condition */
	AIM6020_I2C_start(hAIMboard);

	/* send Slave Address */
	if( !AIM6020_I2C_sendSlaveAddress( hAIMboard, potID ) )
	{
		return 0;	/* it means error */
	}

	/* send Instruction Word (1 0 1 0 P1 P0 - -) */
	if( !AIM6020_I2C_sendByte( hAIMboard, 0xA0 | ((potID & 0x03) << 2) ) )
	{
		return 0;	/* it means error */
	}

	/* send data word */
	if( !AIM6020_I2C_sendByte( hAIMboard, position ) )
	{
		return 0;	/* it means error */
	}
	
	/* send STOP condition */
	AIM6020_I2C_stop(hAIMboard);

	/* the new position of the given digital potentiometer has been set properly */
	return 1;
}

/*----------------------------------------------------------------------------*/
/* potID = AIM6020_DIGPOT_POTID_xxx
 * <-- position or error: [0,63] = position, 0xFFFFFFFF = failed
 */
Uint32 AIM6020_getDigPotPosition(AIM6020_Handle hAIMboard, Uint32 potID)
{
	/* I assume that the I2C Clock is low. */
	Uint32 position;

	/* send START condition */
	AIM6020_I2C_start(hAIMboard);

	/* send Slave Address */
	if( !AIM6020_I2C_sendSlaveAddress( hAIMboard, potID ) )
	{
		return 0xFFFFFFFFu;	/* it means error */
	}

	/* send Instruction Word (1 0 0 1 P1 P0 - -) */
	if( !AIM6020_I2C_sendByte( hAIMboard, 0x90 | ((potID & 0x03) << 2) ) )
	{
		return 0xFFFFFFFFu;	/* it means error */
	}

	/* get data word */
	position = AIM6020_I2C_getByte(hAIMboard) & 0x3F;

	/* send STOP condition */
	AIM6020_I2C_stop(hAIMboard);

	/* return position of the queried digital potentiometer */
	return position;
}

/*----------------------------------------------------------------------------*/
/* potID = AIM6020_DIGPOT_POTID_xxx
 * memory = AIM6020_DIGPOT_MEM_xxx
 * <-- 1 = successful, 0 = failed
 */
Uint32 AIM6020_saveDigPotPosition(AIM6020_Handle hAIMboard, Uint32 potID,
	Uint32 memory)
{
	Uint32	repeat;
	Uint32	ret;
	
	for( repeat = 2; repeat != 0; repeat-- )
	{
		/* I assume that the I2C Clock is low. */
		
		/* send START condition */
		AIM6020_I2C_start(hAIMboard);
	
		/* send Slave Address */
		/* Note:
		 * potID & 0x10 == 0x00 ---> Analog Input (ADC)
		 * potID & 0x10 == 0x10 ---> Analog Output (DAC)
		 */
		if( !AIM6020_I2C_sendSlaveAddress( hAIMboard, potID & 0x10 ) )
		{
			return 0;	/* it means error */
		}
	
		/* send Instruction Byte (1 1 1 0 P1 P0 R1 R0) */
		if( !AIM6020_I2C_sendByte(
				hAIMboard,				/* AIM board handle								*/
				0xE0 |					/* the upper 4 bits of the instrucion byte		*/
				((potID << 2) & 0x0C) |	/* [P1 P0] bits = potentiometer					*/
				(memory & 0x03) )		/* [R1 R0] bits = non-volatile memory register	*/
			)
		{
			return 0;	/* it means error */
		}
		
		/* send STOP condition */
		AIM6020_I2C_stop(hAIMboard);
	
		/* the internal write cycle has been started
		 * check the end of the write
		 */
		ret = AIM6020_I2C_checkCmdFinished(hAIMboard, potID & 0x10, 10);
	}
	return ret;
}

/*----------------------------------------------------------------------------*/
/* potID = AIM6020_DIGPOT_POTID_xxx
 * memory = AIM6020_DIGPOT_MEM_xxx
 * <-- 1 = successful, 0 = failed
 */
Uint32 AIM6020_loadDigPotPosition(AIM6020_Handle hAIMboard, Uint32 potID,
	Uint32 memory)
{
	Uint32	repeat;
	
	for( repeat = 2; repeat != 0; repeat-- )
	{
		/* I assume that the I2C Clock is low. */
		
		/* send START condition */
		AIM6020_I2C_start(hAIMboard);
	
		/* send Slave Address */
		/* Note:
		 * potID & 0x10 == 0x00 ---> Analog Input (ADC)
		 * potID & 0x10 == 0x10 ---> Analog Output (DAC)
		 */
		if( !AIM6020_I2C_sendSlaveAddress( hAIMboard, potID & 0x10 ) )
		{
			return 0;	/* it means error */
		}
	
		/* send Instruction Byte (1 1 0 1 P1 P0 R1 R0) */
		if( !AIM6020_I2C_sendByte(
				hAIMboard,				/* AIM board handle								*/
				0xD0 |					/* the upper 4 bits of the instrucion byte		*/
				((potID << 2) & 0x0C) |	/* [P1 P0] bits = potentiometer					*/
				(memory & 0x03) )		/* [R1 R0] bits = non-volatile memory register	*/
			)
		{
			return 0;	/* it means error */
		}
		
		/* send STOP condition */
		AIM6020_I2C_stop(hAIMboard);
	
		/* the selected potentiometer's position has been loaded from the given
		 * non-volatile memory cell
		 */
		if( repeat != 1 )
		{
			AIM6020_waitForDigPotLoad();
		}
	}
	return 1;
}

/*----------------------------------------------------------------------------*/
/* devID = AIM6020_DIGPOT_DEVID_xxx
 * memory = AIM6020_DIGPOT_MEM_xxx
 * <-- 1 = successful, 0 = failed
 */
Uint32 AIM6020_saveAllDigPotPositions(AIM6020_Handle hAIMboard, Uint32 devID,
	Uint32 memory)
{
	/* There is a SW (HW?) bug? Do not use the "Save All" instrument of the
	 * digital potmeter IC.
	 */
	
	if( devID == AIM6020_DIGPOT_DEVID_ADC )
	{
		if( !AIM6020_saveDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AI1O, memory) ) return 0;
		if( !AIM6020_saveDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AI2O, memory) ) return 0;
		if( !AIM6020_saveDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AI1G, memory) ) return 0;
		if( !AIM6020_saveDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AI2G, memory) ) return 0;
	}
	else // if( devID == AIM6020_DIGPOT_DEVID_DAC )
	{
		if( !AIM6020_saveDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AO1O, memory) ) return 0;
		if( !AIM6020_saveDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AO2O, memory) ) return 0;
		if( !AIM6020_saveDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AO1G, memory) ) return 0;
		if( !AIM6020_saveDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AO2G, memory) ) return 0;
	}
	return 1;	// successful

	/*
	// I assume that the I2C Clock is low.
	
	// send START condition
	AIM6020_I2C_start(hAIMboard);

	// send Slave Address
	if( devID == AIM6020_DIGPOT_DEVID_ADC )
	{
		// Potentiometer of an Analog Input (ADC)
		if( !AIM6020_I2C_sendSlaveAddress( hAIMboard, 0x00 ) )
		{
			return 0;	// it means error
		}
	}
	else
	{
		// Potentiometer of an Analog Output (DAC)
		if( !AIM6020_I2C_sendSlaveAddress( hAIMboard, 0x10 ) )
		{
			return 0;	// it means error
		}
	}

	// send Instruction Byte (1 0 0 0 - - R1 R0)
	if( !AIM6020_I2C_sendByte( hAIMboard, 0x80 | (memory & 0x03) ) )
	{
		return 0;	// it means error
	}

	// send STOP condition
	AIM6020_I2C_stop(hAIMboard);

	// the internal write cycle has been started
	// check the end of the write
	if( devID == AIM6020_DIGPOT_DEVID_ADC )
	{
		return AIM6020_I2C_checkCmdFinished(hAIMboard, 0x00, 10);
	}
	return AIM6020_I2C_checkCmdFinished(hAIMboard, 0x10, 10);
	*/
}

/*----------------------------------------------------------------------------*/
/* devID = AIM6020_DIGPOT_DEVID_xxx
 * memory = AIM6020_DIGPOT_MEM_xxx
 * <-- 1 = successful, 0 = failed
 */
Uint32 AIM6020_loadAllDigPotPositions(AIM6020_Handle hAIMboard, Uint32 devID,
	Uint32 memory)
{
	/* There is a SW (HW?) bug? Do not use the "Load All" instrument of the
	 * digital potmeter IC.
	 */
	
	if( devID == AIM6020_DIGPOT_DEVID_ADC )
	{
		if( !AIM6020_loadDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AI1O, memory) ) return 0;
		if( !AIM6020_loadDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AI2O, memory) ) return 0;
		if( !AIM6020_loadDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AI1G, memory) ) return 0;
		if( !AIM6020_loadDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AI2G, memory) ) return 0;
	}
	else // if( devID == AIM6020_DIGPOT_DEVID_DAC )
	{
		if( !AIM6020_loadDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AO1O, memory) ) return 0;
		if( !AIM6020_loadDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AO2O, memory) ) return 0;
		if( !AIM6020_loadDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AO1G, memory) ) return 0;
		if( !AIM6020_loadDigPotPosition(
			hAIMboard, AIM6020_DIGPOT_POTID_AO2G, memory) ) return 0;
	}
	return 1;	// successful
	
	/*
	// I assume that the I2C Clock is low.
	
	// send START condition
	AIM6020_I2C_start(hAIMboard);

	// send Slave Address
	if( devID == AIM6020_DIGPOT_DEVID_ADC )
	{
		// Potentiometer of an Analog Input (ADC)
		if( !AIM6020_I2C_sendSlaveAddress( hAIMboard, 0x00 ) )
		{
			return 0;	// it means error
		}
	}
	else
	{
		// Potentiometer of an Analog Output (DAC)
		if( !AIM6020_I2C_sendSlaveAddress( hAIMboard, 0x10 ) )
		{
			return 0;	// it means error
		}
	}

	// send Instruction Byte (0 0 0 1 - - R1 R0)
	if( !AIM6020_I2C_sendByte( hAIMboard, 0x10 | (memory & 0x03) ) )
	{
		return 0;	// it means error
	}
	
	// send STOP condition
	AIM6020_I2C_stop(hAIMboard);
	
	// positions of all the digital potentiometers of the selected converter
	// (A/D or D/C) have been set according to the given memory cell
	return 1;
	*/
}

/*----------------------------------------------------------------------------*/
void AIM6020_reset(AIM6020_Handle hAIMboard)
{
	Uint32 gie;

	gie = IRQ_globalDisable();

	AIM6020_RSETH( hAIMboard, CONFIG, AIM6020_CONFIG_DEFAULT );
	hAIMboard->recordedRegs.regCONFIG = AIM6020_CONFIG_DEFAULT;

	AIM6020_RSETH( hAIMboard, FLTCLK1, AIM6020_FLTCLK1_DEFAULT );
	hAIMboard->recordedRegs.regFLTCLK1 = AIM6020_FLTCLK1_DEFAULT;

	AIM6020_RSETH( hAIMboard, FLTCLK2, AIM6020_FLTCLK2_DEFAULT );
	hAIMboard->recordedRegs.regFLTCLK2 = AIM6020_FLTCLK2_DEFAULT;

	AIM6020_RSETH( hAIMboard, FLTCFG, AIM6020_FLTCFG_DEFAULT );
	hAIMboard->recordedRegs.regFLTCFG = AIM6020_FLTCFG_DEFAULT;

	AIM6020_RSETH( hAIMboard, INTCFG, AIM6020_INTCFG_DEFAULT );
	hAIMboard->recordedRegs.regINTCFG = AIM6020_INTCFG_DEFAULT;

	AIM6020_RSETH( hAIMboard, DISPLAY, AIM6020_DISPLAY_DEFAULT );
	hAIMboard->recordedRegs.regDISPLAY = AIM6020_DISPLAY_DEFAULT;

	/* after the hardware reset, the D/A converters show negative limit */
	/* writing 0 to D/A converters to set them to middle */
	AIM6020_writeDA1( hAIMboard, 0 );
	AIM6020_writeDA2( hAIMboard, 0 );
	AIM6020_updateDA12( hAIMboard );
	
	/* setting display to digit 0 and no dot */
	AIM6020_setDisplay( hAIMboard, 0x00 );
	
	IRQ_globalRestore(gie);
}

/*----------------------------------------------------------------------------*/
void AIM6020_resetAll()
{
	AIM6020_reset( &(_AIM6020_deviceTable[0]) );
	AIM6020_reset( &(_AIM6020_deviceTable[1]) );
	AIM6020_reset( &(_AIM6020_deviceTable[2]) );
	AIM6020_reset( &(_AIM6020_deviceTable[3]) );
}

/*----------------------------------------------------------------------------*/
AIM6020_Handle AIM6020_open(Int32 devNum, Uint32 flags)
{
	AIM6020_Handle hAim6020 = INV;
	Uint32	gie;
	Uint32	devid;
	Int32	brdID;

	if( devNum == (Uint32)(AIM6020_BOARD_FIRST_AVAILABLE) )
	{
		// open the first available (free) AIM board
		brdID = 0;
		while(
			(brdID < 4) &&
			((hAim6020 = AIM6020_open(brdID,flags)) == INV)
		) brdID++;
	}
	else if( devNum < 4 )
	{
		gie = IRQ_globalDisable();
		
		if( !_AIM6020_deviceTable[devNum].allocated )
		{
			// the selected ID is free
			
			hAim6020 = (AIM6020_Handle) &(_AIM6020_deviceTable[devNum]);
			
			// dummy write on the platform bus to discharge physically the bus
			AIM6020_RSETH( hAim6020, DUMMY, 0x0000 );
			
			// reading the Device ID
			//devid = AIM6020_RGETH( hAim6020, DEVID ) & 0x0000FFFFu;
			devid = AIM6020_RGETH( hAim6020, DEVID );

			if( (devid == 0x6020u) || (devid == 0x6021u) )
			{
				// there is a board at the selected ID
				hAim6020->allocated = TRUE;

				switch(devid)
				{
				case 0x6020:	// it is an AIM6020 board
					hAim6020->typeOfBoard = 0x6020;
					hAim6020->pDisplayTable = _AIM6020_display_normal;
					break;
				
				case 0x6021:	// it is an AIM6021 board
					hAim6020->typeOfBoard = 0x6021;
					hAim6020->pDisplayTable = _AIM6020_display_direct;
					break;
				}
				
				if( flags & AIM6020_OPEN_RESET ) AIM6020_reset( hAim6020 );
				
//				if( flags & AIM6020_OPEN_ID_ON_DISPLAY ) AIM6020_setDisplay( hAim6020, devNum );
			}
			else
			{
				// there is no board at the selected ID; clear back the handle
				hAim6020 = INV;
			}
		}

		IRQ_globalRestore(gie);
	}
	else
	{
		// bad boardID --> hAim6020 = INV
	}

	return hAim6020;
}

/*----------------------------------------------------------------------------*/
/* close and reset */
void AIM6020_close(AIM6020_Handle hAIMboard, Uint32 flags)
{
	Uint32 gie;

	gie = IRQ_globalDisable();

	hAIMboard->allocated = FALSE;
	if( flags & AIM6020_CLOSE_RESET ) AIM6020_reset( hAIMboard );

	IRQ_globalRestore(gie);
}


/*----------------------------------------------------------------------------*/


#endif /* BSL_AIM6020_SUPPORT */
/******************************************************************************\
* End of bsl_aim6020.c
\******************************************************************************/


